/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef _DECIBEL_TYPES_H_
#define _DECIBEL_TYPES_H_

#include <QtCore/QList>
#include <QtCore/QMap>

#include <QtDBus/QDBusArgument>
#include <QtDBus/QDBusMetaType>

#include <QtTelepathy/Common/BaseTypes>

namespace QtTapioca
{
class Connection;
class Channel;
}

namespace Decibel {

/**
 * @brief A structure to pass all the information required to connect
 * to a channel over D-Bus.
 * @author Tobias Hunger <info@basyskom.de>
 */
struct Q_DECL_EXPORT ChannelInfo
{
    /** @brief Constructor. */
    ChannelInfo();
    /** @brief Constructor. */
    ChannelInfo(const QtTapioca::Connection *,
                const QtTapioca::Channel *);
    /** @brief Copy constructor. */
    ChannelInfo(const ChannelInfo &);

    /** @brief The service name of the connection to handle. */
    QString connection_service;
    /** @brief The object path of the connection to handle. */
    QDBusObjectPath connection_path;
    /** @brief The service name of the channel to handle. */
    QString channel_service;
    /** @brief The object path of the channel to handle. */
    QDBusObjectPath channel_path;
    /** @brief The type of the channel. */
    int channel_type;
    /** @brief The type of the handle of the channel's target. */
    uint target_handle_type;
    /** @brief The handle of the channel's target. */
    uint target_handle;

    /**
     * @brief Retruns true if this ChannelInfo is a Null-object.
     *
     * Retruns true if this ChannelInfo is a Null object and false otherwise.
     * This method makes it easy to test whether it is possible to get a
     * connection/channel from this channel info.
     */
    bool isNull() const;

    /**
     * @brief Create a connection object from the DBus information stored in
     *        this channel info.
     * @param parent The parent of the newly created connection.
     *
     * This method examines the DBus data held by this channel info and tries
     * to construct a Connection object from it. This method returns a null
     * pointer if the construction was not possible.
     */
    QtTapioca::Connection * connection(QObject * parent = 0) const;

    /**
     * @brief Create a channel object from the DBus information stored in
     *        this channel info.
     *
     * This method examines the DBus data held by this channel info and tries
     * to construct a Channel object from it. This method returns a null
     * pointer if the construction was not possible.
     */
    QtTapioca::Channel * channel() const;

    /** @brief Assignment operator */
    ChannelInfo & operator = (const ChannelInfo &);
};

/**
 * @brief A structure to pass information on Components around.
 * @author Tobias Hunger <info@basyskom.de>
 */
struct Q_DECL_EXPORT ComponentInfo
{
    /** @brief Defaultconstructor. */
    ComponentInfo();
    /** @brief Constructor. */
    ComponentInfo(const QStringList &, const QList<int> &, const QList<int> &,
                  const QString &, const QDBusObjectPath &);

    /** @brief Copy Constructor. */
    ComponentInfo(const ComponentInfo &);

    /**
     * @brief Checks whether this ComponentInfo refers to the same D-Bus
     * object as the other ComponentInfo structure.
     * @param info The other ComponentInfo.
     * @return true if both refer to the same object on D-Bus und false
     * otherwise.
     */
    bool sameObject(const ComponentInfo & info) const;

    /** @brief List of allowed protocols. */
    QStringList possible_protocol_list;
    /** @brief List of allowed channel types. */
    QList<int> possible_type_list;
    /** @brief List of allowed target types. */
    QList<int> possible_target_type_list;
    /** @brief The D-Bus service name of this component. */
    QString service_name;
    /** @brief The D-Bus object path of this component. */
    QDBusObjectPath object_path;
};

/**
 * @brief A structure to pass Components around.
 * @author Tobias Hunger <info@basyskom.de>
 */
struct Q_DECL_EXPORT Component
{
    /** @brief Defaultconstructor. */
    Component();
    /** @brief Constructor. */
    Component(const uint);
    /** @brief Constructor. */
    Component(const QStringList &, const QList<int> &, const QList<int> &,
              const uint);

    /** @brief Copy Constructor. */
    Component(const Component &);

    /** @brief List of configured protocols. */
    QStringList protocol_list;
    /** @brief List of configured types. */
    QList<int> type_list;
    /** @brief List of configured target types. */
    QList<int> target_type_list;
    /** @brief The handle to the ComponentInfo struct. */
    uint component_handle;
};

/**
 * @brief A helper function to register our types with QtDBus.
 *
 * Call this function before accessing or offering services of Decibel.
 *
 * @author Tobias Hunger <info@basyskom.de>
 */
void Q_DECL_EXPORT registerTypes();

} // namespace

Q_DECLARE_METATYPE(Decibel::ChannelInfo)
Q_DECLARE_METATYPE(Decibel::ComponentInfo)
Q_DECLARE_METATYPE(Decibel::Component)
Q_DECLARE_METATYPE(QList<Decibel::Component>)

/** @brief Required convertor function for Qt D-Bus. */
const QDBusArgument & operator << (QDBusArgument &,
                                   const Decibel::ChannelInfo &);
/** @brief Required convertor function for Qt D-Bus. */
const QDBusArgument & operator >> (const QDBusArgument &,
                                   Decibel::ChannelInfo &);

/** @brief Required convertor function for Qt D-Bus. */
const QDBusArgument & operator << (QDBusArgument &,
                                   const Decibel::ComponentInfo &);
/** @brief Required convertor function for Qt D-Bus. */
const QDBusArgument & operator >> (const QDBusArgument &,
                                   Decibel::ComponentInfo &);

/** @brief Required convertor function for Qt D-Bus. */
const QDBusArgument & operator << (QDBusArgument &,
                                   const Decibel::Component &);
/** @brief Required convertor function for Qt D-Bus. */
const QDBusArgument & operator >> (const QDBusArgument &,
                                   Decibel::Component &);

#endif
