/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "accountconnector.h"

#include <QtCore/QStringList>
#include <QtCore/QSettings>
#include <QtCore/QHash>

/// @cond FALSE

namespace
{
    static const QString accountmanager_group("AccountManager");
    static const QString account_array("Accounts");
}

AccountConnector::AccountConnector() :
    m_currentHandle(1)
{
    // read account data:
    QSettings settings;
    settings.beginGroup(accountmanager_group);

    int num_accounts = settings.beginReadArray(account_array);
    for (int i = 0; i < num_accounts; ++i)
    {
        settings.setArrayIndex(i);
        QVariantMap account_data;

        QStringList keys = settings.allKeys();
        QString current_key;
        foreach (current_key, keys)
        {
            QVariant value = settings.value(current_key);
            if (current_key == Decibel::name_presence)
            { value = QVariant(value.toInt()); }
            account_data.insert(current_key, value);
        }
        storeAccount(account_data);
    }
    settings.endArray();
    settings.endGroup();

    Q_ASSERT(m_accounts.count() == num_accounts);
}

AccountConnector::~AccountConnector()
{
    // Store account data:
    QSettings settings;
    settings.beginGroup(accountmanager_group);

    settings.beginWriteArray(account_array, m_accounts.size());
    uint current_id;
    int i = 0;
    foreach (current_id, m_accounts.keys())
    {
        settings.setArrayIndex(i);

        QString current_key;
        foreach (current_key, m_accounts[current_id].keys())
        { settings.setValue(current_key, m_accounts[current_id][current_key]); }
        ++i;
    }
    settings.endArray();
    settings.endGroup();
}

bool AccountConnector::hasAccount(const uint id) const
{
    if (id != 0)
    { return m_accounts.contains(id); }
    else
    { return false; }
}

uint AccountConnector::storeAccount(const QVariantMap & nv_pairs)
{
    int num_accounts = m_accounts.count();
    m_accounts.insert(m_currentHandle, nv_pairs);
    Q_ASSERT((num_accounts + 1) == m_accounts.count());
    Q_ASSERT(!m_accounts.contains(0));

    return m_currentHandle++;
}

void AccountConnector::updateAccount(const uint id, const QVariantMap & data)
{
    Q_ASSERT(hasAccount(id));
    int num_accounts = m_accounts.count();
    m_accounts[id] = data;
    Q_ASSERT(num_accounts == m_accounts.count());
    Q_ASSERT(!m_accounts.contains(0));
}

QVariantMap AccountConnector::getAccount(const uint id) const
{
    QVariantMap result;
    if (hasAccount(id)) { result = m_accounts[id]; }
    return result;
}

bool AccountConnector::deleteAccount(const uint id)
{
    int num_accounts = m_accounts.count();
    bool result(false);
    if (hasAccount(id)) { m_accounts.remove(id); result = true; }
    Q_ASSERT((result && num_accounts == m_accounts.count() + 1) ||
             (!result && num_accounts == m_accounts.count()));
    return result;
}

QList<uint> AccountConnector::accountIds() const
{ return m_accounts.keys(); }

QList<uint> AccountConnector::findAccounts(const QVariantMap & nv_pairs) const
{
    QList<uint> results;

    uint current_key;
    foreach (current_key, m_accounts.keys())
    {
        bool does_match(true);

        QString nv_key;
        foreach (nv_key, nv_pairs.keys())
        {
            if (!m_accounts[current_key].contains(nv_key) ||
                nv_pairs[nv_key].typeName() != m_accounts[current_key][nv_key].typeName() ||
                nv_pairs[nv_key] != m_accounts[current_key][nv_key])
            {
                does_match = false;
                break;
            }
        }
        if (does_match) { results.append(current_key); }
    }
    Q_ASSERT(results.count() <= m_accounts.count());
    return results;
}

bool AccountConnector::setValue(const uint id,
                                const QString & key, const QVariant & value)
{
    bool result(false);
    if (hasAccount(id))
    {
        if (value.isNull())
        { m_accounts[id].remove(key); }
        else
        { m_accounts[id][key] = value; }
        result = true;
    }
    return result;
}

/// @endcond
