#!/usr/bin/python
#
# deluge.py
# Copyright (C) Zach Tibbitts 2006 <zach@collegegeek.org>
# Copyright (C) Alon Zakai    2006 <kripkensteiner@gmail.com>
# 
# Deluge is free software.
# 
# You may redistribute this file and/or modify it under the terms of the
# GNU General Public License, as published by the Free Software
# Foundation; either version 2 of the License, or (at your option)
# any later version.
# 
# This file is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this file.  If not, write to:
# 	The Free Software Foundation, Inc.,
# 	51 Franklin Street, Fifth Floor
# 	Boston, MA  02110-1301, USA.

#
# delugecommon.py
# A set of common variables and functions to
# be used throughout the Deluge program



import sys, os, gtk

import xdg, xdg.BaseDirectory

import decimal, webbrowser


#
# Set to true to log debug output to console
#
DEBUG = True

#
# Global variables for use throughout the program
#
APP_NAME = "Deluge"
APP_VERSION = "0.4"
APP_PATH = os.path.abspath(os.path.dirname(sys.argv[0]))
PIXMAP_PATH = APP_PATH + "/pixmaps"
GLADE_PATH = APP_PATH + "/glade"
HOME_DIR = os.path.expanduser("~")
CONF_DIR_NAME = ".deluge"
CONFIG_DIR = xdg.BaseDirectory.save_config_path("deluge")
CONFIG_XML = CONFIG_DIR + "/preferences.xml"
SEARCH_XML = CONFIG_DIR + "/searchengines.xml"
DHTfile = CONFIG_DIR  + "/dht.state"

V1 = 0
V2 = 4
V3 = 0
V4 = 0

UPDATE_INTERVAL = 1000 #milliseconds
MIN_TIME_TO_ETA = 4.0 #seconds
ETA_GAMMA       = 0.33
MAX_ETA_WEEKS   = 4

MAX_DOWNLOAD_HISTORY = int(5*60*(1000./UPDATE_INTERVAL))

STATUSBAR_DIVIDER = "   "

#
# Global functions
#
def debugmsg(arg):
	if DEBUG:
		print arg

def get_icon(arg):
	return PIXMAP_PATH + "/" + arg

# To be changed when glade files are given their own directory
def get_glade(arg):
	return GLADE_PATH + "/" + arg

#
# 
#
#
def file_browse(dialog_action, file_name=""):
	dialog_buttons = (gtk.STOCK_CANCEL
				, gtk.RESPONSE_CANCEL
				, gtk.STOCK_OPEN
				, gtk.RESPONSE_OK)
	file_dialog = gtk.FileChooserDialog(title=_("Select Torrent")
				, action=dialog_action
				, buttons=dialog_buttons)

	filter = gtk.FileFilter()
	filter.set_name(".torrent Files")
	filter.add_pattern("*." + "torrent")
	file_dialog.add_filter(filter)

	filter = gtk.FileFilter()
	filter.set_name(_("All files"))
	filter.add_pattern("*")
	file_dialog.add_filter(filter)

	result = ""
	if file_dialog.run() == gtk.RESPONSE_OK:
		result = file_dialog.get_filename()
	file_dialog.destroy()
	
	return result

def directory_browse(dialog_title, file_name=""):

	dialog_buttons = (gtk.STOCK_CANCEL
				, gtk.RESPONSE_CANCEL
				, gtk.STOCK_OK
				, gtk.RESPONSE_OK)
				
	dir_dialog = gtk.FileChooserDialog(title=dialog_title
				, action=gtk.FILE_CHOOSER_ACTION_SELECT_FOLDER
				, buttons=dialog_buttons)
	

	if (file_name != ""):
		dir_dialog.set_current_name(file_name)

	result = ""
	
	if dir_dialog.run() == gtk.RESPONSE_OK:
		result = dir_dialog.get_filename()
	dir_dialog.destroy()
	
	return result
	
def open_url_in_browser(url):
	try:
		webbrowser.open(url, 1, 1)
	except webbrowser.Error:
		ShowPopupWarning(None, "Cannot automatically find a web browser to run. Please copy and paste the following command into your web browser:\n\n" + url)


def getDataRate(value):
	return getDataAmount(value) + "/s"

def getDataAmount(value):
	val = 0
	units = ""

	if(value < 1048576):
		val = value / 1024.
		units = "KB"
	elif(value < 1073741824):
		val = value / 1048576.
		units = "MB"
	else:
		val = value / 1073741824.
		units = "GB"

	value = str(decimal.Decimal(str(val)).quantize(decimal.Decimal(10)** -2))

	return value + " " + units

def decimal_to_percentage(value):
	value = value * 100
	str_val = str(value)
	d = decimal.Decimal(str_val)
	pstr = str(d.quantize(decimal.Decimal(10)** -2))

	return pstr + "%"

def getETA(total_size, done_so_far, download_speeds):
	if (len(download_speeds) * UPDATE_INTERVAL) / 1000 < MIN_TIME_TO_ETA:
		return _("Unknown")

	weighted_speed = 0.
	weights = 0.
	curr_weight = 1.

	for i in range(len(download_speeds)):
		weights        = weights + curr_weight
		weighted_speed = weighted_speed + (curr_weight * download_speeds[i])
		curr_weight    = curr_weight * (ETA_GAMMA * UPDATE_INTERVAL / 1000.)

	weighted_speed = weighted_speed / weights

	if weighted_speed == 0:
		return _("infinity")

	time_left = (total_size - done_so_far)/weighted_speed

	if time_left < 60:
		if time_left <= 2:
			return _("1 second")
		return str(int(time_left)) + " " + _("seconds")

	if time_left < 60*60:
		if time_left <= 2*60:
			return _("1 minute")
		return str(int(time_left/60)) + " " + _("minutes")

	if time_left < 60*60*24:
		if time_left <= 2*60*60:
			return _("1 hour")
		return str(int(time_left/(60*60))) + " " + _("hours")

	if time_left < 60*60*24*7:
		if time_left <= 2*60*60*24:
			return _("1 day")
		return str(int(time_left/(60*60*24))) + " " + _("days")

	if time_left < 60*60*24*7*MAX_ETA_WEEKS:
		if time_left <= 2*60*60*24*7:
			return _("1 week")
		return str(int(time_left/(60*60*24*7))) + " " + _("weeks")

	return _("infinity")

# Internally used by the torrentclass.update* functions
def biographer(model, path, iter, dictionary):
	assert(model.get_value(iter, 0) not in dictionary.keys())
	dictionary[model.get_value(iter, 0)] = model.get_string_from_iter(iter)

def AddTextColumn(view, title, columnId, backgroundColumnID = -1, hideme=False, sortColumn=-1):
	#By default, sort by the same column, unless told otherwise
	if sortColumn == -1:
		sortColumn = columnId

	renderer = gtk.CellRendererText()
	if backgroundColumnID is not -1:
		renderer.set_property('foreground', 'gray')
		column = gtk.TreeViewColumn(title, renderer, text=columnId, foreground_set = backgroundColumnID)
	else:
		column = gtk.TreeViewColumn(title, renderer, text=columnId)
	column.set_resizable(True)		
	column.set_sort_column_id(sortColumn)
	column.set_expand(False)
	view.append_column(column)
	if(hideme):
		debugmsg("Hiding" + title)
		column.set_visible(False)

def AddToggleColumn(view, title, columnId, toggledSignal):
	renderer = gtk.CellRendererToggle()
	renderer.set_property('activatable', True)
	column = gtk.TreeViewColumn(title, renderer, active=columnId)
	column.set_resizable(True)		
	column.set_sort_column_id(columnId)
	column.set_expand(False)
	view.append_column(column)
	renderer.connect("toggled", toggledSignal, columnId)
		
def AddProgressColumn(view, title, columnId):
	renderer = gtk.CellRendererProgress()
	##renderer.set_property('value', 50)
	column = gtk.TreeViewColumn(title, renderer, value=columnId, text=(columnId+1))
	column.set_resizable(True)		
	column.set_sort_column_id(columnId)
	column.set_expand(False)
	view.append_column(column)

def ShowPopupWarning(window, message):
	warner = gtk.MessageDialog(
						 parent = window,
						 flags  = gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
						 buttons= gtk.BUTTONS_OK,
						 message_format=message,
						 type   = gtk.MESSAGE_WARNING)
	warner.run()
	warner.destroy()

def ShowPopupInfo(window, message):
	teller = gtk.MessageDialog(
						 parent = window,
						 flags  = gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
						 buttons= gtk.BUTTONS_OK,
						 message_format=message,
						 type   = gtk.MESSAGE_INFO)
	teller.run()
	teller.destroy()

def ShowProgressDialog(window, message, workThread, params):
	gladefile = get_glade("progress.glade")
	wTree     = gtk.glade.XML(gladefile, "ProgressDialog")
	dlg       = wTree.get_widget("ProgressDialog")

	messageWidget  = wTree.get_widget("message")
	progressWidget = wTree.get_widget("progress")

	messageWidget.set_text(message)

	dlg.show_all()

	theThread = workThread(params, dlg, progressWidget)

	theThread.start()

	dlg.run()

	dlg.destroy()

	theThread.cancel() # if dlg stopped because of a 'cancel' press, then cancel the operation
	# Note that we call cancel() INSTEAD of join(), this is a forced quit, not a wait

def RoundedFloatString(x, places):
	y = str(round(x,places))
	index = y.find(".")
	if index == -1:
		return y
	else:
		return y[:index + places + 1]
		
		
class DelugeColumn(gtk.TreeViewColumn):
	def __init__(self, title=None, renderer=None):
		gtk.TreeViewColumn.__init__(self, title, renderer)
	
	def show(self):
		self.set_visible(True)
	
	def hide(self):
		self.set_visible(False)	
		

class TextColumn(DelugeColumn):
	def __init__(self, title=None):
		DelugeColumn.__init__(self, title, gtk.CellRendererText())
		
	def set_value(self, string):
		pass
		
class ToggleColumn(DelugeColumn):
	def __init__(self, title=None):
		DelugeColumn.__init__(self, title, gtk.CellRendererToggle())

		
	def set_value(self, string):
		pass
		
class ProgressColumn(DelugeColumn):
	def __init__(self, title=None):
		DelugeColumn.__init__(self, title, gtk.CellRendererProgress())
		
	def set_value(self, string):
		val = int(string)
		
		




