#!/usr/bin/python
#
# deluge.py
# Copyright (C) Zach Tibbitts 2006 <zach@collegegeek.org>
# Copyright (C) Alon Zakai    2006 <kripkensteiner@gmail.com>
# 
# Deluge is free software.
# 
# You may redistribute this file and/or modify it under the terms of the
# GNU General Public License, as published by the Free Software
# Foundation; either version 2 of the License, or (at your option)
# any later version.
# 
# This file is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this file.  If not, write to:
# 	The Free Software Foundation, Inc.,
# 	51 Franklin Street, Fifth Floor
# 	Boston, MA  02110-1301, USA.


# dpref.py
# The Preferences Class
#
# Contains three classes:
#	1: DelugePreferences
#	    an object that stores the user's preferences
#	    so they can be accessed at runtime.
#	2: DXMLParser
#	    a class to parse XML files for storing settings
#	3: DPreferencesDialog
#	    a dialog to get preferences from the user, and
#	    write those settings back to a preferences obj


import sys, os, shutil

from xml.sax import saxutils
from xml.sax import make_parser
from xml.sax.handler import feature_namespaces
from xml.sax import ContentHandler

try:
 	import pygtk
  	pygtk.require("2.0")
except:
  	pass
try:
	import gtk
  	import gtk.glade
except:
	sys.exit(1)

from delugecommon import *
import torrent

## Stock keys in the preferences files, and their default values
##
## global_save = False		## If all downloads will be saved to the same location
## global_loc = ""		## The location to save to if above is true
## tcp_port_min = 0		## The lower port number of the TCP range
## tcp_port_max = 65535		## The upper port number of the TCP range
## max_ul_rate = 0		## The maximum upload rate, 0 is unlimited
## max_num_ul = 0		## The maximum number of uploads per torrent, 0 is unlimited
## max_dl_rate = 0		## The maximum download rate, 0 is unlimited
## max_num_dl = 0		## The maximum number of downloads per torrent, 0 is unlimited
## stop_seeding = False		## If the client should stop seeding once a completed
## 					torrent reaches a certain share ratio
##seed_cap_ratio = 0		## The share ratio to stop at if the above is true

class DelugePreferences:
	def __init__(self, filepath=None):
		debugmsg("DelugePreferences object created")
		self.settings = {'global_save': False,
				'global_loc': HOME_DIR,
				'tcp_port_min': 6881,
				'tcp_port_max': 6999,
				'max_ul_rate': 0,
				'max_num_ul': 0,
				'max_dl_rate': 0,
				'max_num_dl': 0,
				'stop_seeding': False,
				'dht' : 1,
				'seed_cap_ratio': 0,
				'max_connections': 80,
				'compact_allocation': 1 }
		if (filepath):
			self.load_from_file(filepath)

		

	def load_from_file(self, filepath):
		debugmsg("Loading from file "+filepath)
		# First, check to see if the file exists
		if not os.path.basename(filepath) in os.listdir(CONFIG_DIR):
			self.save_to_file(filepath)
			debugmsg("config file not found, creating")
		try:
			# Create a parser
			parser = make_parser()
		
			# Tell the parser we are not interested in XML namespaces
			parser.setFeature(feature_namespaces, 0)
		
			# Create the handler
			dh = DXMLParser()
		
			# Tell the parser to use our handler
			parser.setContentHandler(dh)
		
			# Parse the input
			parser.parse(filepath)

			for key in dh.options.keys():
				self.settings[key] = dh.options[key]
		except:
			debugmsg("An error occured while loading config, creating new file")
			self.save_to_file(filepath)
	
	def save_to_file(self, filepath):
		debugmsg("writing pref to file")
		f = open(filepath, 'w')
		f.write('<settings>\n')
		
		for k in self.settings.keys():
			debugmsg("Writing key "+str(k)+"with a value of:"+str(self.settings[k]))
			f.write('\t<option key="')
			f.write(k)
			f.write('">')
			f.write(str(self.settings[k]))
			f.write('</option>\n')
					
		f.write('</settings>\n')
		
	def show_dlg(self):
		return DPreferencesDialog(self).run()

	def get(self, key):
		try:
			return str(self.settings[key])
		except KeyError:
			return None

	#A function to set a value in the dict structure	
	def set(self, key, value):
		self.settings[key] = value

	def delete(self, key):
		try:
			del self.settings[key]
		except KeyError:
			pass

	## A set of accessor functions for the standard preferences.
	## These are different from the 'get' method because they
	## will return values in the appropriate variable type,
	## while the get_value method always returns a string
	def get_global_save(self):
		debugmsg("Global Save Boolean")
		debugmsg(self.settings['global_save'])
		debugmsg(bool(self.settings['global_save'] == "True"))
		return bool(self.settings['global_save'] == "True")
	def get_global_loc(self):
		return str(self.settings['global_loc'])	
	def get_tcp_port_min(self):
		return int(self.settings['tcp_port_min'])
	def get_tcp_port_max(self):
		return int(self.settings['tcp_port_max'])
	def get_max_ul_rate(self):
		return int(self.settings['max_ul_rate'])
	def get_max_num_ul(self):
		return int(self.settings['max_num_ul'])
	def get_max_dl_rate(self):
		return int(self.settings['max_dl_rate'])
	def get_max_num_dl(self):
		return int(self.settings['max_num_dl'])
	def get_stop_seeding(self):
		return bool(self.settings['stop_seeding'] == "True")
	def get_seed_cap_ratio(self):
		return float(self.settings['seed_cap_ratio'])
		
	
		
	
class DXMLParser(ContentHandler):
	def __init__(self):
		self.inOptionContent = 0
		self.options = {}

	def startElement(self, name, attrs):
		if name == 'option':
			self.inOptionContent = 1
			self.optType = self.normalize_whitespace(attrs.get('key', None))
			self.optVal = ""

	def characters(self, ch):
		if self.inOptionContent:
			self.optVal = self.optVal + ch

	def endElement(self, name):
		if name == 'option':
			self.inOptionContent = 0
			## Removing this line to fix error with spaces in 
			## .torrent files.  Needs to be tested.
			##self.optVal = self.normalize_whitespace(self.optVal)
			debugmsg(self.optType + ": "+ self.optVal)
			self.options[self.optType] = self.optVal
	
	def normalize_whitespace(self,text):
		return ' '.join(text.split())

class DPreferencesDialog:
	def __init__(self, dpref=None):
		self.settings = dpref
		self.gladefile = get_glade("prefdialog.glade")
		self.wTree = gtk.glade.XML(self.gladefile, "PrefDialog")
		self.dlg = self.wTree.get_widget("PrefDialog")
		self.dlg.set_icon_from_file(get_icon("deluge-32.png"))
		## Set increment of the share ratio spinner, b/c glade does not
		## allow us to do this
		self.wTree.get_widget("ShareRatioSpinner").set_increments(0.1,1)

		self.wTree.signal_autoconnect({ "test_port_clicked" : self.testPort })

		if (self.settings):
			global_bool = self.settings.get_global_save()
			debugmsg("global_save_bool: " + str(global_bool))
			if(global_bool):
				debugmsg("Setting save_radio2 as enabled")
				self.wTree.get_widget("save_radio2").set_active(1)
			else:
				debugmsg("Setting save_radio2 as disabled")
				self.wTree.get_widget("save_radio1").set_active(1)
			self.wTree.get_widget("SaveLocationButton").set_current_folder(self.settings.get_global_loc())
			self.wTree.get_widget("tcpmin_spinner").set_value(float(self.settings.get_tcp_port_min()))
			self.wTree.get_widget("tcpmax_spinner").set_value(float(self.settings.get_tcp_port_max()))
			self.wTree.get_widget("maxul_spinner").set_value(float(self.settings.get_max_ul_rate()))
			self.wTree.get_widget("maxulc_spinner").set_value(float(self.settings.get_max_num_ul()))
			self.wTree.get_widget("maxdl_spinner").set_value(float(self.settings.get_max_dl_rate()))
			self.wTree.get_widget("maxdlc_spinner").set_value(float(self.settings.get_max_num_dl()))
			if(self.settings.get_stop_seeding()):
				self.wTree.get_widget("aseed_check").set_active(1)
			else:
				self.wTree.get_widget("aseed_check").set_active(0)

			print "ALLOCATION: ", self.settings.get("compact_allocation")

			if(int(self.settings.get("compact_allocation")) == 1):
				self.wTree.get_widget("compact_allocation").set_active(1)
			else:
				self.wTree.get_widget("compact_allocation").set_active(0)

			self.wTree.get_widget("ShareRatioSpinner").set_value(float(self.settings.get_seed_cap_ratio()))

			self.wTree.get_widget("curr_port").set_text(str(torrent.listeningPort()))

			self.wTree.get_widget("dht_button").set_active(self.settings.get("dht") == "1")

			self.wTree.get_widget("maxcon_spinner").set_value(float(self.settings.get("max_connections")))

	def testPort(self, args):
		open_url_in_browser("https://www.grc.com/x/portprobe=" + self.wTree.get_widget("curr_port").get_text())

	def run(self):	  
		self.dlg.show_all()
		debugmsg("Running dlg")
		response = self.dlg.run()
		if (response == 1):
			debugmsg("getactive " + str(self.wTree.get_widget("save_radio2").get_active()))
			self.settings.set('global_save', str(self.wTree.get_widget("save_radio2").get_active()))
			self.settings.set('global_loc', self.wTree.get_widget("SaveLocationButton").get_current_folder())
			self.settings.set('tcp_port_min', self.wTree.get_widget("tcpmin_spinner").get_value_as_int())
			self.settings.set('tcp_port_max', self.wTree.get_widget("tcpmax_spinner").get_value_as_int())
			self.settings.set('max_ul_rate', self.wTree.get_widget("maxul_spinner").get_value_as_int())
			self.settings.set('max_num_ul', self.wTree.get_widget("maxulc_spinner").get_value_as_int())
			self.settings.set('max_dl_rate', self.wTree.get_widget("maxdl_spinner").get_value_as_int())
			self.settings.set('max_num_dl', self.wTree.get_widget("maxdlc_spinner").get_value_as_int())
			self.settings.set('stop_seeding', str(self.wTree.get_widget("aseed_check").get_active()))

			self.settings.set('compact_allocation', str(int(self.wTree.get_widget("compact_allocation").get_active())))
			print "ALLOCATION: ", self.settings.get("compact_allocation")

			self.settings.set('seed_cap_ratio', self.wTree.get_widget("ShareRatioSpinner").get_value())

			self.settings.set("dht", int(self.wTree.get_widget("dht_button").get_active()))

			self.settings.set('max_connections', self.wTree.get_widget("maxcon_spinner").get_value_as_int())
		else:
			debugmsg("Cancel was clicked")		
		self.dlg.destroy()
		return response
