#!/usr/bin/python
#
# deluge.py
# Copyright (C) Zach Tibbitts 2006 <zach@collegegeek.org>
# Copyright (C) Alon Zakai    2006 <kripkensteiner@gmail.com>
# 
# Deluge is free software.
# 
# You may redistribute this file and/or modify it under the terms of the
# GNU General Public License, as published by the Free Software
# Foundation; either version 2 of the License, or (at your option)
# any later version.
# 
# This file is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this file.  If not, write to:
# 	The Free Software Foundation, Inc.,
# 	51 Franklin Street, Fifth Floor
# 	Boston, MA  02110-1301, USA.

import sys
import os
import decimal
import gtk
import torrent
from delugecommon import *

class TorrentError(Exception):
	def __init__(self, code):
		self.code = code

		if code == torrent.constants()['ERROR_INVALID_ENCODING']:
			self.message = _("The torrent file does not contain valid information.")
		if code == torrent.constants()['ERROR_INVALID_TORRENT']:
			self.message = _("The torrent file does not contain all necessary valid information.")
		elif code == torrent.constants()['ERROR_FILESYSTEM']:
			self.message = _("A filesystem error occurred when trying to open the torrent file.")
		elif code == torrent.constants()['ERROR_DUPLICATE_TORRENT']:
			self.message = _("The selected torrent is already present in the system.")

	def __str__(self):
		return "Torrent Error (Exception): " + repr(self.code) + " : " + self.message

class Torrent:
	def __init__(self, the_torrent, save_dir, compact_allocation):
		self.fileName	= os.path.split(the_torrent)[1]
		self.saveDir	= save_dir
#		print "allocation: ", compact_allocation
		self.compact_allocation = compact_allocation
		self.uniqueID 	= torrent.addTorrent(the_torrent, save_dir, compact_allocation)
		if self.uniqueID < 0:
			raise TorrentError(self.uniqueID)

#		ShowPopupInfo(None, "Added ID: " + str(self.uniqueID))

		self.pause()   # All torrents begin paused! This is important, so that you can
							# set up filtering and so forth before they actually start
		self.status   	= torrent.getState(self.uniqueID)

		self.downloadHistory = []

		self.uploadedMemory = 0

		# Create the torrent's personal data stores (peers, etc.)
		self.peerList = gtk.ListStore(str, str, str, str, str, float, float, float)

		self.fileList = gtk.ListStore(str, str, str, int, bool)
		self.initFileList() # We load the filenames, prepare the ground

		self.userPause = False

		self.trackerStatus  = ""
		self.trackerMessage = ""

		self.savedPeerInfo = None

#		debugmsg("New torrent added, unique ID: "+ self.uniqueID)

	def update(self):
		self.status = torrent.getState(self.uniqueID)
#		debugmsg(self.status)

		# Wipe some data that is now outdated:
		self.savedPeerInfo = None

		# NOTE: the download history should ONLY be recorded for ACTIVE torrents
		# So when we implement active vs. paused torrents, be aware
		self.downloadHistory.insert(0, self.status["downloadRate"])
		if len(self.downloadHistory) > MAX_DOWNLOAD_HISTORY:
			self.downloadHistory.pop()

	def updateSummary(self, parent):

		if not self.status['tracker']:
			self.status['tracker'] = _("N-A")

		parent.text_summary_title.set_text(self.getName())
		parent.text_summary_total_size.set_text(self.getSize())
		parent.text_summary_pieces.set_text(str(self.status['numPieces']) + " x " + getDataAmount(self.status['pieceLength']))
		parent.text_summary_total_downloaded.set_text(getDataAmount(self.status['totalDone']))
		parent.text_summary_total_uploaded.set_text(getDataAmount(self.status['totalUpload'] + self.uploadedMemory))
		parent.text_summary_percentage_done.set_text(self.getProgress())
		parent.text_summary_share_ratio.set_text(self.getShareRatio())
		parent.text_summary_downloaded_this_session.set_text(getDataAmount(self.status['totalDownload']))
		parent.text_summary_uplodaded_this_session.set_text(getDataAmount(self.status['totalUpload']))
		parent.text_summary_tracker.set_text(self.status['tracker'])
		parent.text_summary_tracker_response.set_text(self.trackerMessage)
		parent.text_summary_tacker_status.set_text(self.trackerStatus)
		parent.text_summary_next_announce.set_text(self.status['nextAnnounce'])
		if self.compact_allocation:
			compactText = _("Yes")
		else:
			compactText = _("No")
		parent.text_summary_compact_allocation.set_text(compactText)

	def initFileList(self):
		newFileInfo = torrent.getFileInfo(self.uniqueID)

		self.filePaths = {}

		for i in range(len(newFileInfo)):
			self.fileList.insert(i,(	newFileInfo[i]['path'], # Add None to start for tree
													getDataAmount(newFileInfo[i]['size']),
													str(int(newFileInfo[i]['offset'])),
													int(newFileInfo[i]['progress']),
													not newFileInfo[i]['filteredOut']
													))
			self.filePaths[newFileInfo[i]['path']] = i
			i = i + 1

	def updateFileListInternally(self): # Only update the list from libtorrent data

		newFileInfo = torrent.getFileInfo(self.uniqueID)

		currFiles = {}

		self.fileList.foreach(biographer, currFiles)

		assert(self.fileList.iter_n_children(None) == len(currFiles.keys()))

		for currFile in newFileInfo:
			self.fileList.set(self.fileList.get_iter_from_string(currFiles[currFile['path']]),
									1,	getDataAmount(currFile['size']),
									2,	str(int(currFile['offset'])),
									3, int(currFile['progress']),
									4, not currFile['filteredOut']
									)

	def updateFileList(self, view):
		view.set_model(self.fileList)
		self.updateFileListInternally()

	def loadPeerInfo(self):
		if self.savedPeerInfo is None:
			self.savedPeerInfo = torrent.getPeerInfo(self.uniqueID)

	def updatePeerList(self, view):
		view.set_model(self.peerList)

		newPeerInfo = self.getPeerInfo()

		newIPs = {}

		for index in range(len(newPeerInfo)):
			if not newPeerInfo[index]['client'] == "":
				assert(newPeerInfo[index]['ip'] not in newIPs.keys())
				newIPs[newPeerInfo[index]['ip']] = index

		# Remove IPs no longer appearing
		class remover_data:
			def __init__(self, newIPs):
				self.newIPs  = newIPs
				self.removed = False

		def remover(model, path, iter, data):
			if model.get_value(iter, 0) not in data.newIPs:
				model.remove(iter)
				data.removed = True
				return True
			else:
				return False

		while True:
			data = remover_data(newIPs.keys())
			self.peerList.foreach(remover, data)
			if not data.removed:
				break

		# Update IPs that were and still are here
		currIPs = {}

		self.peerList.foreach(biographer, currIPs)

		assert(self.peerList.iter_n_children(None) == len(currIPs.keys()))

		for peer in newPeerInfo:
			if peer['ip'] in currIPs.keys():
				self.peerList.set(self.peerList.get_iter_from_string(currIPs[peer['ip']]),
										1, 	unicode(peer['client'], 'Latin-1'),
										2,		RoundedFloatString(peer['peerHas'],1),
										3,		getDataRate(peer['downloadSpeed']),
										4,		getDataRate(peer['uploadSpeed']),
										5,		peer['downloadSpeed'],
										6,		peer['uploadSpeed'],
										7,		peer['peerHas'])

		# We have only to insert the new IPs, and we are done. Note this this is AFTER
		# the updates, since inserts may invalidate the iters.
		for peer in newPeerInfo:
			if peer['ip'] not in currIPs.keys() and peer['client'] is not "":
				self.peerList.append([	peer['ip'],
												unicode(peer['client'], 'Latin-1'),
												RoundedFloatString(peer['peerHas'],1),
												getDataRate(peer['downloadSpeed']),
												getDataRate(peer['uploadSpeed']),
												peer['downloadSpeed'],
												peer['uploadSpeed'],
												peer['peerHas']
											])

	def getPeerInfo(self):
		self.loadPeerInfo()
		return self.savedPeerInfo

	def getVisualFilterOuts(self):
		numFiles = self.fileList.iter_n_children(None)
		temp = range(numFiles)

		for i in range(numFiles):
			the_iter = self.fileList.get_iter(i)
			path = self.fileList.get_value(the_iter, 0)
			temp[self.filePaths[path]] = not self.fileList.get_value(the_iter, 4)

		return temp

	def getFilterOuts(self):
		# The filelist may have changed without gui change (e.g. by preferences,
		# and never looking at the gui, ever. So update the filelist from libtorrent data
		self.updateFileListInternally()

		return self.getVisualFilterOuts()

	def applyFilterOuts(self, temp):
		torrent.setFilterOut(self.uniqueID, temp)

	def applyVisualFilterOuts(self):
		self.applyFilterOuts(self.getVisualFilterOuts())

	def setUploadedMemory(self, upload):
		if upload is not None:
			self.uploadedMemory = int(upload)

	def getUploadedMemory(self):
		return self.uploadedMemory

	def updateTracker(self):
		debugmsg(str(self.getName()) + _(" is updating its tracker"))
		torrent.reannounce(self.uniqueID)

	def start(self):
		torrent.resume(self.uniqueID)

	def pause(self):
		torrent.pause(self.uniqueID)

	def forcePause(self):
		self.userPause = True
		self.pause()

	def forceStart(self):
		self.userPause = False
		self.start()

	def getList(self):
		myList = [self.uniqueID, self.getFileName(),
				self.getName(),self.getSize(),self.getProgressFloat(),
				self.getStatus(),self.getNumSeeders(),
				self.getNumPeers(),self.getDLSpeed(),self.getULSpeed(),
				self.getETA(),self.getShareRatio(),
				self.status["downloadRate"]<1024,
				self.status["uploadRate"]<1024]
		return myList

	def getUniqueID(self):
		return self.uniqueID

	def getFileName(self):
		return self.fileName

	def getSaveDir(self):
		return self.saveDir

	def getName(self):
		return torrent.getName(self.uniqueID)

	def getSize(self):
		totalSize = self.status["totalSize"]

		return getDataAmount(totalSize)
		
	def getProgressFloat(self):
		p = float(self.status["progress"]) * 100
		return p
	
	def getProgressBar(self):
		debugmsg("Making Progress Bar")
		p = float(self.status["progress"])
		p100 = float(p * 100)
		debugmsg(str(p) + str(p100))
		renderer = gtk.CellRendererProgress()
		renderer.set_property('value', 50)
		return renderer

	
	def getProgress(self):
		p = self.status["progress"]
		return decimal_to_percentage(p)
		
	def get_is_seeding(self):
		if self.status["isSeed"]:
			return True
		else:
			return False

	def getStatus(self):
		if self.userPause:
			return _("Paused")
		elif self.isPaused():
			return _("Queued")
		elif self.status["isSeed"]:
			return _("Seeding")
		elif self.status['state'] is torrent.constants()["STATE_QUEUED"]:
			return _("Waiting to check")
		elif self.status['state'] is torrent.constants()["STATE_CHECKING"]:
			return _("Checking")
		elif self.status['state'] is torrent.constants()["STATE_CONNECTING"]:
			return _("Downloading (T?)")
		elif self.status['state'] is torrent.constants()["STATE_DOWNLOADING_META"]:
			return _("Downloading Meta")
		elif self.status['state'] is torrent.constants()["STATE_DOWNLOADING"]:
			return _("Downloading")
		elif self.status['state'] is torrent.constants()["STATE_FINISHED"]:
			return _("Finished")
		elif self.status['state'] is torrent.constants()["STATE_ALLOCATING"]:
			return _("Allocating")

	def getNumSeeders(self):
		totalSeeds = self.status["numComplete"]
		if totalSeeds == -1:
			totalSeeds = self.status["totalSeeds"]
		return str(self.status["numSeeds"]) + " (" + str(totalSeeds) + ")"

	def getNumPeers(self):
		totalPeers = self.status["numIncomplete"]
		if totalPeers == -1:
			totalPeers = self.status["totalPeers"]
		return str(self.status["numPeers"] - self.status["numSeeds"]) + " (" + str(totalPeers) + ")"

	def inProgress(self):
		return (self.status['state'] is torrent.constants()["STATE_DOWNLOADING"] or \
			(self.status['state'] is torrent.constants()["STATE_CONNECTING"] and \
			 self.status["progress"] < 100.0 and not self.isPaused())) and \
			not self.isPaused()

	def getDLSpeed(self):
		if self.inProgress():
			return getDataRate(self.status["downloadRate"])
		else:
			return "-"

	def getULSpeed(self):
		if self.inProgress() or self.status["isSeed"]:
			return getDataRate(self.status["uploadRate"])
		else:
			return "-"

	def getETA(self):
		if self.inProgress():
			return getETA(self.status["totalSize"],
							self.status["totalSize"]*self.status["progress"],
							self.downloadHistory)
		else:
			return _("N/A")

	def isPaused(self):
		return self.status["isPaused"]

	def isForcePaused(self):
		return self.userPause

	def useCompactAllocation(self):
		return self.compact_allocation

	def getShareRatio(self):
		try:	
			re = decimal_to_percentage(float(self.status['totalUpload'] + self.uploadedMemory) / float(self.status["totalDone"]))
		except:
			re = _("INF")

		return re

	def getShareDec(self):
		try:
			ratio = float(self.status["totalUpload"]) / (float(self.status["totalDownload"]))
		except:
			ratio = -1
		debugmsg(ratio)
		return ratio

	def getSessionUpload(self):
		return self.status['totalUpload']

	def setTrackerMessage(self, trackerStatus, message):
		self.trackerStatus  = trackerStatus
		self.trackerMessage = message

	def getFileNames(self):
		ret = []
		the_iter = self.fileList.get_iter_first()

		while the_iter is not None:
			ret.append(self.fileList.get_value(the_iter, 0))
			the_iter = self.fileList.iter_next(the_iter)

		return ret
