import unittest
import database
from os import remove
from os.path import expanduser
import random
import config
import prefs
import time
import storedatabase
from threading import Thread

from test.framework import DemocracyTestCase

class EmptyViewTestCase(DemocracyTestCase):
    def setUp(self):
        DemocracyTestCase.setUp(self)
        self.everything = database.defaultDatabase
    def testCur(self):
        self.everything.resetCursor()
        self.assertEqual(self.everything.cur(),None)
    def testNext(self):
        self.everything.resetCursor()
        self.assertEqual(self.everything.getNext(),None)
    def testGetItem(self):
        self.assertEqual(self.everything[0],None)
    def testGetPrev(self):
        self.everything.resetCursor()
        self.assertEqual(self.everything.getPrev(),None)
    def testLen(self):
        self.assertEqual(self.everything.len(),0)
        
class SingleItemViewTestCase(DemocracyTestCase):
    def setUp(self):
        DemocracyTestCase.setUp(self)
        self.everything = database.defaultDatabase
        self.x = database.DDBObject()
    def testAdd(self):
        self.assertEqual(self.x.__class__, database.DDBObject)
    def testGetItem(self):
        a = self.everything[0]
        b = self.everything[1]
        self.assertEqual(a.__class__, database.DDBObject)
        self.assertEqual(b,None)
    def testNext(self):
        self.everything.resetCursor()
        a = self.everything.cur()
        b = self.everything.getNext()
        c = self.everything.cur()
        d = self.everything.getNext()
        assert ((a == None) and (b.__class__ == database.DDBObject) and
                (c == b) and (d == None))
    def testGetPrev(self):
        self.everything.resetCursor()
        self.assertEqual(self.everything.getPrev(),None)
    def testLen(self):
        self.assertEqual(self.everything.len(),1)

class AddBeforeViewTestCase(DemocracyTestCase):
    def setUp(self):
        DemocracyTestCase.setUp(self)
        self.x = database.DDBObject()
        self.y = database.DDBObject()
        database.resetDefaultDatabase()
        self.everything = database.defaultDatabase
        self.everything.addBeforeCursor(self.y)
        self.everything.resetCursor()
        self.everything.addBeforeCursor(self.x)
    def testUnique(self):
        self.assertNotEqual(self.x,self.y)
    def testUniqueID(self):
        self.assertNotEqual(self.x.getID(),self.y.getID())
    def testGetItem(self):
        a = self.everything[0]
        b = self.everything[1]
        c = self.everything[2]
        self.assertEqual(a.__class__,database.DDBObject)
        self.assertEqual(b.__class__,database.DDBObject)
        self.assertNotEqual(a,b)
        self.assertEqual(c,None)
    def testNextGetPrev(self):
        self.everything.resetCursor()
        a = self.everything.cur()
        b = self.everything.getNext()
        c = self.everything.cur()
        d = self.everything.getNext()
        e = self.everything.cur()
        f = self.everything.getNext()
        assert ((a == None) and (b.__class__ == database.DDBObject) and
                (c == b) and (d.__class__ == database.DDBObject) and
                (d != c) and (d == e) and (f == None))
    def testLen(self):
        self.assertEqual(self.everything.len(),2)

class AddAfterViewTestCase(DemocracyTestCase):
    def setUp(self):
        DemocracyTestCase.setUp(self)
        self.x = database.DDBObject()
        self.y = database.DDBObject()
        database.resetDefaultDatabase()
        self.everything = database.defaultDatabase
        self.everything.addAfterCursor(self.x)
        self.everything.resetCursor()
        self.everything.getNext()
        self.everything.addAfterCursor(self.y)
    def testUnique(self):
        self.assertNotEqual(self.x,self.y)
    def testUniqueID(self):
        self.assertNotEqual(self.x.getID(),self.y.getID())
    def testGetItem(self):
        a = self.everything[0]
        b = self.everything[1]
        c = self.everything[2]
        self.assertEqual(a.__class__,database.DDBObject)
        self.assertEqual(b.__class__,database.DDBObject)
        self.assertNotEqual(a,b)
        self.assertEqual(c,None)
    def testNextGetPrev(self):
        self.everything.resetCursor()
        a = self.everything.cur()
        b = self.everything.getNext()
        c = self.everything.cur()
        d = self.everything.getNext()
        e = self.everything.cur()
        f = self.everything.getNext()
        assert ((a == None) and (b.__class__ == database.DDBObject) and
                (c == b) and (d.__class__ == database.DDBObject) and
                (d != c) and (d == e) and (f == None))
    def testLen(self):
        self.assertEqual(self.everything.len(),2)

class DeletedItemViewTestCase(DemocracyTestCase):
    def setUp(self):
        DemocracyTestCase.setUp(self)
        self.everything = database.defaultDatabase
        self.x = database.DDBObject()
        self.y = database.DDBObject()
        self.x.remove()
    def testRemoveMissing(self):
        self.everything.resetCursor()
        self.assertRaises(database.ObjectNotFoundError,self.everything.remove)
    def testAdd(self):
        self.assertEqual(self.x.__class__,database.DDBObject)
    def testGetItem(self):
        a = self.everything[0]
        b = self.everything[1]
        self.assertEqual(a,self.y)
        self.assertEqual(b,None)
    def testNext(self):
        self.everything.resetCursor()
        a = self.everything.cur()
        b = self.everything.getNext()
        c = self.everything.cur()
        d = self.everything.getNext()
        assert ((a == None) and (b.__class__ == database.DDBObject) and
                (c == b) and (d == None))
    def testLen(self):
        self.assertEqual(self.everything.len(),1)

class FilterViewTestCase(DemocracyTestCase):
    def setUp(self):
        DemocracyTestCase.setUp(self)
        self.everything = database.defaultDatabase
        self.x = database.DDBObject()
        self.y = database.DDBObject()
        self.filtered = self.everything.filter(lambda q:q != self.x)
    def testGetItem(self):
        a = self.filtered[0]
        b = self.filtered[1]
        self.assertEqual(a,self.y)
        self.assertEqual(b,None)
    def testNext(self):
        self.filtered.resetCursor()
        a = self.filtered.cur()
        b = self.filtered.getNext()
        c = self.filtered.cur()
        d = self.filtered.getNext()
        assert ((a == None) and (b == self.y) and
                (c == b) and (d == None))
    def testGetPrev(self):
        self.filtered.resetCursor()
        self.assertEqual(self.filtered.getPrev(),None)
    def testLen(self):
        self.assertEqual(self.filtered.len(),1)

class RecomputeFilterViewTestCase(DemocracyTestCase):
    def setUp(self):
        DemocracyTestCase.setUp(self)
        self.everything = database.defaultDatabase
        self.x = database.DDBObject()
        self.y = database.DDBObject()
        self.accept = self.x.getID()
        self.filtered = self.everything.filter(self.changeFilt)
        self.accept = self.y.getID()
        self.everything.recomputeFilters()
    def changeFilt(self, obj):
        return obj.getID() == self.accept
    def testGetItem(self):
        a = self.filtered[0]
        b = self.filtered[1]
        self.assertEqual(a,self.y)
        self.assertEqual(b,None)
    def testNext(self):
        self.filtered.resetCursor()
        a = self.filtered.cur()
        b = self.filtered.getNext()
        c = self.filtered.cur()
        d = self.filtered.getNext()
        assert ((a == None) and (b == self.y) and
                (c == b) and (d == None))
    def testGetPrev(self):
        self.filtered.resetCursor()
        self.assertEqual(self.filtered.getPrev(),None)
    def testLen(self):
        self.assertEqual(self.filtered.len(),1)

class SortTestCase(DemocracyTestCase):
    def setUp(self):
        DemocracyTestCase.setUp(self)
        self.everything = database.defaultDatabase
        self.x = database.DDBObject()
        self.y = database.DDBObject()
        self.higher = self.x
        self.sorted = self.everything.sort(self.sortFunc)
    def sortFunc(self,x,y):
        if x is y:
            return 0
        elif x == self.higher:
            return 1
        else:
            return -1
    def testUnique(self):
        self.assertNotEqual(self.x,self.y)
    def testUniqueID(self):
        self.assertNotEqual(self.x.getID(),self.y.getID())
    def testGetItem(self):
        a = self.sorted[0]
        b = self.sorted[1]
        c = self.sorted[2]
        self.assertEqual(a,self.y)
        self.assertEqual(b,self.x)
        self.assertNotEqual(a,b)
        self.assertEqual(c,None)
    def testNextGetPrev(self):
        self.sorted.resetCursor()
        a = self.sorted.cur()
        b = self.sorted.getNext()
        c = self.sorted.cur()
        d = self.sorted.getNext()
        e = self.sorted.cur()
        f = self.sorted.getNext()
        assert ((a == None) and (b.__class__ == database.DDBObject) and
                (c == b) and (d.__class__ == database.DDBObject) and
                (d != c) and (d == e) and (f == None))
    def testLen(self):
        self.assertEqual(self.everything.len(),2)
    def testRecompute(self):
        self.higher = self.y
        self.everything.recomputeFilters()
        a = self.sorted[0]
        b = self.sorted[1]
        c = self.sorted[2]
        self.assertEqual(a,self.x)
        self.assertEqual(b,self.y)
        self.assertNotEqual(a,b)
        self.assertEqual(c,None)

class MapViewTestCase(DemocracyTestCase):
    def setUp(self):
        DemocracyTestCase.setUp(self)
        self.everything = database.defaultDatabase
        self.x = database.DDBObject()
        self.y = database.DDBObject()
        self.add = 0
        self.mapped = self.everything.map(self.mapFunc)
    def mapFunc(self,obj):
        return obj.getID()+self.add
    def testNotAltered(self):
        self.assertNotEqual(self.x,self.mapped[0])
        self.assertNotEqual(self.x,self.mapped[1])
        self.assertNotEqual(self.y,self.mapped[0])
        self.assertNotEqual(self.y,self.mapped[1])
    def testUnique(self):
        self.assertNotEqual(self.x,self.y)
    def testUniqueID(self):
        self.assertNotEqual(self.x.getID(),self.y.getID())
    def testGetItem(self):
        a = self.mapped[0]
        b = self.mapped[1]
        c = self.mapped[2]
        self.assertEqual(a,self.everything[0].getID())
        self.assertEqual(b,self.everything[1].getID())
        self.assertNotEqual(a,b)
        self.assertEqual(c,None)
    def testNextGetPrev(self):
        self.mapped.resetCursor()
        a = self.mapped.cur()
        b = self.mapped.getNext()
        c = self.mapped.cur()
        d = self.mapped.getNext()
        e = self.mapped.cur()
        f = self.mapped.getNext()
        assert ((a == None) and (b.__class__.__name__ == 'int') and
                (c == b) and (d.__class__.__name__ == 'int') and
                (d != c) and (d == e) and (f == None))
    def testLen(self):
        self.assertEqual(self.everything.len(),2)

class CallbackViewTestCase(DemocracyTestCase):
    def setUp(self):
        DemocracyTestCase.setUp(self)
        self.everything = database.defaultDatabase
        self.filtered = self.everything.filter(lambda x:True)
        self.mapped = self.everything.map(lambda x:x)
        self.sorted = self.everything.sort(lambda x,y:0)
        self.callcount = 0
    def call(self, obj, id):
        assert id == self.everything[0].getID()
        self.callcount+=1
    def removeCall(self, obj, id):
        self.callcount+=1
    def testAdd(self):
        self.everything.addAddCallback(self.call)
        self.x = database.DDBObject()
        self.assertEqual(self.callcount,1)
    def testChange(self):
        self.everything.addChangeCallback(self.call)
        self.x = database.DDBObject()
        self.x.beginChange()
        self.x.endChange()
        self.assertEqual(self.callcount,1)
    def testRemove(self):
        self.everything.addRemoveCallback(self.removeCall)
        self.x = database.DDBObject()
        self.x.remove()
        self.assertEqual(self.callcount,1)
    def TestFilterAdd(self):
        self.filtered.addAddCallBack(self.call)
        self.x = database.DDBObject()
        self.assertEqual(self.callcount,1)
    def TestFilterRemove(self):
        self.filtered.addRemoveCallBack(self.call)
        self.x = database.DDBObject()
        self.x.remove()
        self.assertEqual(self.callcount,1)
    def TestFilterChange(self):
        self.filtered.addChangeCallBack(self.call)
        self.x = database.DDBObject()
        self.x.change()
        self.assertEqual(self.callcount,1)
    def TestMapAdd(self):
        self.mapped.addAddCallBack(self.call)
        self.x = database.DDBObject()
        self.assertEqual(self.callcount,1)
    def TestMapRemove(self):
        self.mapped.addRemoveCallBack(self.call)
        self.x = database.DDBObject()
        self.x.remove()
        self.assertEqual(self.callcount,1)
    def TestMapChange(self):
        self.mapped.addChangeCallBack(self.call)
        self.x = database.DDBObject()
        self.x.change()
        self.assertEqual(self.callcount,1)
    def TestSortAdd(self):
        self.sorted.addAddCallBack(self.call)
        self.x = database.DDBObject()
        self.assertEqual(self.callcount,1)
    def TestSortRemove(self):
        self.sorted.addRemoveCallBack(self.call)
        self.x = database.DDBObject()
        self.x.remove()
        self.assertEqual(self.callcount,1)
    def TestSortChange(self):
        self.sorted.addChangeCallBack(self.call)
        self.x = database.DDBObject()
        self.x.change()
        self.assertEqual(self.callcount,1)

class SaveViewTestCase(DemocracyTestCase):
    def setUp(self):
        DemocracyTestCase.setUp(self)
        self.everything = database.defaultDatabase
        self.x = database.DDBObject()
        self.y = database.DDBObject()
    def testSaveRestore(self):
        storedatabase.saveDatabase(self.everything)
        self.z = database.DDBObject()
        self.zz = database.DDBObject()
        last = self.zz.getID()
        self.x.remove()
        storedatabase.restoreDatabase(self.everything)
        self.assertEqual(self.everything.len(),2)
        assert (self.everything[0].getID() == self.y.getID() or
                self.everything[0].getID() == self.x.getID())
        if self.everything[0].getID() == self.y.getID():
            self.assertEqual(self.everything[1].getID(),self.x.getID())
        if self.everything[0].getID() == self.x.getID():
            self.assertEqual(self.everything[1].getID(),self.y.getID())
        self.assertEqual(self.everything[2],None)
        assert database.DDBObject().getID() >= last
    def testLastID(self):
        last = self.y.getID()
        storedatabase.saveDatabase(self.everything)
        # Simulate restarting app
        self.y.remove()
        self.x.remove()
        database.DDBObject.lastID = 0 # This is implementation specific and
                             # needs to be updated when the
                             # implementation changes
        storedatabase.restoreDatabase(self.everything)
        assert database.DDBObject().getID() >= last
    def testBackup(self):
        storedatabase.saveDatabase(self.everything)
        storedatabase.backedUp = False
        storedatabase.saveDatabase(self.everything)
        remove(config.get(prefs.DB_PATHNAME))
        self.z = database.DDBObject()
        self.zz = database.DDBObject()
        last = self.zz.getID()
        self.x.remove()
        storedatabase.restoreDatabase(self.everything,config.get(prefs.DB_PATHNAME) + '.bak')
        self.assertEqual(self.everything.len(),2)
        assert (self.everything[0].getID() == self.y.getID() or
                self.everything[0].getID() == self.x.getID())
        if self.everything[0].getID() == self.y.getID():
            self.assertEqual(self.everything[1].getID(),self.x.getID())
        if self.everything[0].getID() == self.x.getID():
            self.assertEqual(self.everything[1].getID(),self.y.getID())
        self.assertEqual(self.everything[2],None)
        assert database.DDBObject().getID() >= last

class MapFilterRemoveViewTestCase(DemocracyTestCase):
    def setUp(self):
        DemocracyTestCase.setUp(self)
        self.everything = database.defaultDatabase
        self.objlist = []
        for x in range(0,10):
            database.DDBObject()
        self.add = 0
        self.mapped = self.everything.map(self.mapFunc)
        self.mapped = self.mapped.filter(lambda x:True)
    def mapFunc(self,obj):
        return obj.getID() % 2
    def testBasicMap(self):
        self.everything.resetCursor()
        self.mapped.resetCursor()
        for obj in self.everything:
            self.assertEqual(self.mapFunc(obj),self.mapped.getNext())
    def testOneOffBasicMap(self):
        self.everything.resetCursor()
        self.mapped.resetCursor()
        for x in range(1,6):
            obj = self.everything.getNext()
        obj.remove()
        self.everything.addBeforeCursor(database.DDBObject(add=False))
        self.everything.getPrev()
        self.everything.getPrev()
        self.everything.addAfterCursor(database.DDBObject(add=False))
        self.everything.resetCursor()
        for obj in self.everything:
            self.assertEqual(self.mapFunc(obj),self.mapped.getObjectByID(obj.getID()))

class FilterSortMapTestCase(DemocracyTestCase):
    def setUp(self):
        DemocracyTestCase.setUp(self)
        self.everything = database.defaultDatabase
        self.callbacks = 0
        self.objlist = []
        for x in range(0,10):
            self.objlist.append(database.DDBObject())
        self.myfiltFunc = lambda x:x.getID()%2 == 0
        self.filted = self.everything.filter(self.filterFunc)
        self.sorted = self.filted.sort(self.sortFunc)
        self.mapped = self.sorted.map(lambda x:x)
    def filterFunc(self, x):
        return self.myfiltFunc(x)
    def sortFunc(self, x, y):
        x = x.getID()
        y = y.getID()
        if x < y:
            return -1
        elif x > y:
            return 1
        else:
            return 0
    def call(self,obj, id):
        self.callbacks += 1
    def test(self):
        self.assertEqual(self.mapped.len(),5)
        self.mapped.addAddCallback(self.call)
        self.myfiltFunc = lambda x:x is self.objlist[1]
        self.everything.recomputeFilters()
        self.assertEqual(self.mapped.len(),1)
        self.myfiltFunc = lambda x:True
        self.everything.recomputeFilters()
        self.assertEqual(self.mapped.len(),10)
    def testTwoSets(self):
        self.callbacks2 = 0
        def call2(item,id):
            self.callbacks2 += 1
        filtFunc2 = lambda x:True
        filted2 = self.everything.filter(filtFunc2)
        sorted2 = filted2.sort(self.sortFunc)
        mapped2 = sorted2.map(lambda x:x)
        self.mapped.addChangeCallback(self.call)
        mapped2.addChangeCallback(call2)
        if self.myfiltFunc(self.objlist[0]):
            self.objlist[1].beginChange()
            self.objlist[1].endChange()
        else:
            self.objlist[0].beginChange()
            self.objlist[0].endChange()
        self.assertEqual(self.callbacks,0)
        self.assertEqual(self.callbacks2,1)
        self.everything.recomputeFilters()
        self.assertEqual(self.callbacks,0)
        self.assertEqual(self.callbacks2,1)
        self.everything.recomputeFilters()
        self.assertEqual(self.callbacks,0)
        self.assertEqual(self.callbacks2,1)
        if self.myfiltFunc(self.objlist[0]):
            self.objlist[0].beginChange()
            self.objlist[0].endChange()
        else:
            self.objlist[1].beginChange()
            self.objlist[1].endChange()
        self.assertEqual(self.callbacks,1)
        self.assertEqual(self.callbacks2,2)
        self.everything.recomputeFilters()
        self.assertEqual(self.callbacks,1)
        self.assertEqual(self.callbacks2,2)
        self.everything.recomputeFilters()
        self.assertEqual(self.callbacks,1)
        self.assertEqual(self.callbacks2,2)
    def testTwoSets2(self):
        self.callbacks2 = 0
        def call2(item, id):
            self.callbacks2 += 1
        filtFunc2 = lambda x:x.getID()%2 == 1
        filted2 = self.everything.filter(filtFunc2)
        sorted2 = filted2.sort(self.sortFunc)
        mapped2 = sorted2.map(lambda x:x)
        self.mapped.addChangeCallback(self.call)
        mapped2.addChangeCallback(call2)
        self.mapped.addAddCallback(self.call)
        mapped2.addAddCallback(call2)
        self.mapped.addRemoveCallback(self.call)
        mapped2.addRemoveCallback(call2)

        self.mapped.next()
        self.mapped.next()
        self.mapped.next()
        mapped2.next()
        self.assertEqual(self.mapped.cur(),self.mapped[2])
        self.assertEqual(mapped2.cur(),mapped2[0])
        if self.myfiltFunc(self.objlist[0]):
            self.objlist[1].beginChange()
            self.objlist[1].endChange()
        else:
            self.objlist[0].beginChange()
            self.objlist[0].endChange()
        self.assertEqual(self.mapped.cur(),self.mapped[2])
        self.assertEqual(mapped2.cur(),mapped2[0])
        self.assertEqual(self.callbacks,0)
        self.assertEqual(self.callbacks2,1)
        self.everything.recomputeFilters()
        self.assertEqual(self.mapped.cur(),self.mapped[2])
        self.assertEqual(mapped2.cur(),mapped2[0])
        self.assertEqual(self.callbacks,0)
        self.assertEqual(self.callbacks2,1)
        self.everything.recomputeFilters()
        self.assertEqual(self.mapped.cur(),self.mapped[2])
        self.assertEqual(mapped2.cur(),mapped2[0])
        self.assertEqual(self.callbacks,0)
        self.assertEqual(self.callbacks2,1)
        if self.myfiltFunc(self.objlist[0]):
            self.objlist[0].beginChange()
            self.objlist[0].endChange()
        else:
            self.objlist[1].beginChange()
            self.objlist[1].endChange()
        self.assertEqual(self.mapped.cur(),self.mapped[2])
        self.assertEqual(mapped2.cur(),mapped2[0])
        self.assertEqual(self.callbacks,1)
        self.assertEqual(self.callbacks2,1)
        self.everything.recomputeFilters()
        self.assertEqual(self.mapped.cur(),self.mapped[2])
        self.assertEqual(mapped2.cur(),mapped2[0])
        self.assertEqual(self.callbacks,1)
        self.assertEqual(self.callbacks2,1)
        self.everything.recomputeFilters()
        self.assertEqual(self.mapped.cur(),self.mapped[2])
        self.assertEqual(mapped2.cur(),mapped2[0])
        self.assertEqual(self.callbacks,1)
        self.assertEqual(self.callbacks2,1)
        if self.myfiltFunc(self.objlist[0]):
            self.objlist[0].remove()
        else:
            self.objlist[1].remove()
        self.assertEqual(self.mapped.cur(),self.mapped[1])
        self.assertEqual(mapped2.cur(),mapped2[0])
        self.assertEqual(self.callbacks,2)
        self.assertEqual(self.callbacks2,1)
        self.everything.recomputeFilters()
        self.assertEqual(self.mapped.cur(),self.mapped[1])
        self.assertEqual(mapped2.cur(),mapped2[0])
        self.assertEqual(self.callbacks,2)
        self.assertEqual(self.callbacks2,1)
        self.everything.recomputeFilters()
        self.assertEqual(self.mapped.cur(),self.mapped[1])
        self.assertEqual(mapped2.cur(),mapped2[0])
        self.assertEqual(self.callbacks,2)
        self.assertEqual(self.callbacks2,1)
        self.objlist.append(database.DDBObject(add = False))
        self.objlist.append(database.DDBObject(add = False))
        self.everything.resetCursor()
        self.everything.addAfterCursor(self.objlist[10])
        self.assertEqual(self.everything[0],self.objlist[10])
        self.assertEqual(self.everything.cur(),None)
        self.everything.addAfterCursor(self.objlist[11])
        self.assertEqual(self.everything[0],self.objlist[11])
        self.assertEqual(self.mapped.cur(),self.mapped[1])
        self.assertEqual(mapped2.cur(),mapped2[0])
        self.assertEqual(self.callbacks,3)
        self.assertEqual(self.callbacks2,2)
        self.objlist[10].beginChange()
        self.objlist[10].endChange()
        self.objlist[11].beginChange()
        self.objlist[11].endChange()
        self.assertEqual(self.mapped.cur(),self.mapped[1])
        self.assertEqual(mapped2.cur(),mapped2[0])
        self.assertEqual(self.callbacks,4)
        self.assertEqual(self.callbacks2,3)
        self.everything.recomputeFilters()
        self.assertEqual(self.callbacks,4)
        self.assertEqual(self.callbacks2,3)
        self.myfiltFunc = lambda x:x is self.objlist[2]
        self.everything.recomputeFilters()
        self.assertEqual(self.callbacks2,3)
        self.objlist[2].beginChange()
        self.objlist[2].endChange()
        self.objlist[3].beginChange()
        self.objlist[3].endChange()
        self.assertEqual(self.callbacks2,4)
        
class CursorTestCase(DemocracyTestCase):
    def setUp(self):
        DemocracyTestCase.setUp(self)
        self.everything = database.defaultDatabase
	self.origObjs = [database.DDBObject(), database.DDBObject(), database.DDBObject()]
	self.objs = self.everything.filter(lambda x: True).map(self.mapToObject).sort(self.sortOldID)
    def sortOldID(self,x,y):
	if x.oldID > y.oldID:
	    return 1
	elif x.oldID < y.oldID:
	    return -1
	else:
	    return 0
    def mapToObject(self, obj):
	temp = database.DDBObject(add = False)
	temp.oldID = obj.getID()
	return temp
    def test(self):
	self.assertEqual(self.objs.len(),3)
	self.assertEqual(self.objs.cur(),None)
	self.objs.getNext()
	self.objs.getNext()
	self.objs.getNext()
	self.assertEqual(self.objs.cur(),self.objs[2])
        self.origObjs[2].remove()
	self.assertEqual(self.objs.cur(),self.objs[1])
	self.objs.getPrev()
	self.assertEqual(self.objs.cur(),self.objs[0])
    def testStack(self):
        obj = self.everything.getNext()
        self.assertEqual(self.everything.cur(), obj)
        self.everything.saveCursor()
        while self.everything.getNext() is not None:
            pass
        self.everything.restoreCursor()
        self.assertEqual(self.everything.cur(), obj)

class RecomputeMapTestCase(DemocracyTestCase):
    def setUp(self):
        DemocracyTestCase.setUp(self)
        self.everything = database.defaultDatabase
	self.origObjs = [database.DDBObject(), database.DDBObject(), database.DDBObject()]
	self.objs = self.everything.filter(lambda x: True).map(self.mapToObject).sort(self.sortOldID).map(self.mapToObject)
	self.changeCalls = 0
    def sortOldID(self,x,y):
	if x.oldID > y.oldID:
	    return 1
	elif x.oldID < y.oldID:
	    return -1
	else:
	    return 0
    def mapToObject(self, obj):
	temp = database.DDBObject(add = False)
	temp.oldID = obj.getID()
	return temp
    def changeCall(self,item,id):
	self.changeCalls +=1
    def test(self):
	self.objs.addChangeCallback(self.changeCall)
	self.everything.recomputeFilters()
	self.everything.recomputeFilters()
	temp = self.everything.getNext()
	temp.beginChange()
	temp.endChange()
	self.assertEqual(self.changeCalls,1)

class FilterUpdateOnChange(DemocracyTestCase):
    def setUp(self):
        DemocracyTestCase.setUp(self)
        self.everything = database.defaultDatabase
	self.origObjs = [database.DDBObject(), database.DDBObject(), database.DDBObject()]
        self.origObjs[0].good = True
        self.origObjs[1].good = False
        self.origObjs[2].good = False
	self.objs = self.everything.filter(lambda x: x.good)
	self.changeCalls = 0
    def testLoss(self):
        self.assertEqual(self.objs.len(),1)
        self.origObjs[0].beginChange()
        self.origObjs[0].good = False
        self.origObjs[0].endChange()
        self.assertEqual(self.objs.len(),0)
    def testAdd(self):
        self.assertEqual(self.objs.len(),1)
        self.origObjs[1].beginChange()
        self.origObjs[1].good = True
        self.origObjs[1].endChange()
        self.assertEqual(self.objs.len(),2)

# Currently, we require that the database does NOT update maps on a change
class MapUpdateOnChange(DemocracyTestCase):
    def setUp(self):
        DemocracyTestCase.setUp(self)
        self.everything = database.defaultDatabase
	self.origObjs = [database.DDBObject(), database.DDBObject(), database.DDBObject()]
        self.origObjs[0].good = True
        self.origObjs[1].good = False
        self.origObjs[2].good = False
	self.objs = self.everything.map(self.mapToObject).filter(lambda x: x.good)
	self.changeCalls = 0
    def mapToObject(self, obj):
	temp = database.DDBObject(add = False)
	temp.good = obj.good
	return temp
    def testLoss(self):
        self.assertEqual(self.objs.len(),1)
        self.origObjs[0].beginChange()
        self.origObjs[0].good = False
        self.origObjs[0].endChange()
        self.assertEqual(self.objs.len(),1)
    def testAdd(self):
        self.assertEqual(self.objs.len(),1)
        self.origObjs[1].beginChange()
        self.origObjs[1].good = True
        self.origObjs[1].endChange()
        self.assertEqual(self.objs.len(),1)

class SortUpdateOnChange(DemocracyTestCase):
    def setUp(self):
        DemocracyTestCase.setUp(self)
        self.everything = database.defaultDatabase
	self.origObjs = [database.DDBObject(), database.DDBObject(), database.DDBObject()]
        self.origObjs[0].good = True
        self.origObjs[1].good = False
        self.origObjs[2].good = False
	self.objs = self.everything.sort(lambda x, y: 0).sort(lambda x, y: 0).filter(lambda x: x.good)
	self.changeCalls = 0
    def testLoss(self):
        self.assertEqual(self.objs.len(),1)
        self.origObjs[0].beginChange()
        self.origObjs[0].good = False
        self.origObjs[0].endChange()
        self.assertEqual(self.objs.len(),0)
    def testAdd(self):
        self.assertEqual(self.objs.len(),1)
        self.origObjs[1].beginChange()
        self.origObjs[1].good = True
        self.origObjs[1].endChange()
        self.assertEqual(self.objs.len(),2)

class IDBaseTraversal(DemocracyTestCase):
    def setUp(self):
        DemocracyTestCase.setUp(self)
        self.everything = database.defaultDatabase
	self.origObjs = [database.DDBObject(), database.DDBObject(), database.DDBObject()]
        self.sorted = self.everything.sort(self.sortID)
    def sortID(self,x,y):
	if x.getID() > y.getID():
	    return 1
	elif x.getID() < y.getID():
	    return -1
	else:
	    return 0
    def test(self):
        self.assertEqual(self.origObjs[0],
                         self.sorted.getObjectByID(self.origObjs[0].getID()))
        self.assertEqual(self.origObjs[1],
                         self.sorted.getObjectByID(self.origObjs[1].getID()))
        self.assertEqual(self.origObjs[2],
                         self.sorted.getObjectByID(self.origObjs[2].getID()))
        
        self.assertEqual(self.origObjs[1].getID(),
                         self.sorted.getNextID(self.origObjs[0].getID()))
        self.assertEqual(self.origObjs[2].getID(),
                         self.sorted.getNextID(self.origObjs[1].getID()))
        self.assertEqual(None,self.sorted.getNextID(self.origObjs[2].getID()))

        self.assertEqual(None,self.sorted.getPrevID(self.origObjs[0].getID()))
        self.assertEqual(self.origObjs[0].getID(),
                         self.sorted.getPrevID(self.origObjs[1].getID()))
        self.assertEqual(self.origObjs[1].getID(),
                         self.sorted.getPrevID(self.origObjs[2].getID()))
    
        self.sorted.resetCursor()
        self.sorted.getNext()
        self.assertEqual(self.origObjs[0].getID(), self.sorted.getCurrentID())
        self.sorted.getNext()
        self.assertEqual(self.origObjs[1].getID(), self.sorted.getCurrentID())
        self.sorted.getNext()
        self.assertEqual(self.origObjs[2].getID(), self.sorted.getCurrentID())

# class ThreadTest(DemocracyTestCase):
#     def setUp(self):
#         self.everything = database.defaultDatabase
#     def add100(self):
#         for x in range(0,100):
#             database.DDBObject()
#     def remove100(self):
#         for x in range(0,100):
#             self.everything[0].remove()
#     def testAddRemove(self):
#         self.add100()
#         thread = Thread(target = self.add100)
# 	thread.setDaemon(False)
# 	thread.start()
#         self.remove100()
#         thread.join()

class IndexFilterTest(DemocracyTestCase):
    def setUp(self):
        DemocracyTestCase.setUp(self)
        self.everything = database.defaultDatabase
        self.addCallbacks = 0
        self.removeCallbacks = 0
        self.changeCallbacks = 0
        self.shift = 0
    def mod10(self, x):
        return (x.getID() + self.shift) % 10
    def mod100(self, x):
        return (x.getID() + self.shift) % 100
    def sortFunc(self, x, y):
        x = x.getID()
        y = y.getID()
        if x < y:
            return -1
        elif x > y:
            return 1
        else:
            return 0
    def addCallback(self,value,id):
        self.addCallbacks += 1
    def removeCallback(self,value,id):
        self.removeCallbacks += 1
    def changeCallback(self,value,id):
        self.changeCallbacks += 1
    def testBasicIndexFilter(self):
        for x in range(0,100):
            database.DDBObject()
        self.everything.createIndex(self.mod10)
        for x in range(0,50):
            database.DDBObject()
        filtered = self.everything.filterWithIndex(self.mod10,0)
        filtered.addAddCallback(self.addCallback)
        filtered.addRemoveCallback(self.removeCallback)
        filtered.addChangeCallback(self.changeCallback)
        for x in range(0,50):
            database.DDBObject()

        self.assertEqual(self.addCallbacks,5)
        for obj in filtered:
            self.assertEqual(self.mod10(obj),0)
        filtered[0].remove()
        self.assertEqual(filtered.len(),19)
        self.assertEqual(self.removeCallbacks,1)
        filtered[0].beginChange()
        filtered[0].endChange()
        self.assertEqual(self.changeCallbacks,1)

        obj = filtered[0]
        self.everything.removeView(filtered)
        for x in range(0,50):
            database.DDBObject()
        self.assertEqual(self.addCallbacks,5)
        obj.beginChange()
        obj.endChange()
        self.assertEqual(self.changeCallbacks,1)
        obj.remove()
        self.assertEqual(self.removeCallbacks,1)
    def testRecomputeIndex(self):
        for x in range(0,100):
            database.DDBObject()
        self.everything.createIndex(self.mod10)
        for x in range(0,50):
            database.DDBObject()
        filtered = self.everything.filterWithIndex(self.mod10,0)
        for x in range(0,50):
            database.DDBObject()
        self.assertEqual(filtered.len(),20)
        filtered[0].remove()
        self.assertEqual(filtered.len(),19)
        self.shift = 1
        self.everything.recomputeFilters()
        self.assertEqual(filtered.len(),20)
    def testLargeSet(self):
        self.everything.createIndex(self.mod100)
        start = time.clock()
        for x in range(0,500):
            database.DDBObject()
            database.DDBObject()
            database.DDBObject()
            database.DDBObject()
            database.DDBObject()
            database.DDBObject()
            database.DDBObject()
            database.DDBObject()
            database.DDBObject()
            database.DDBObject()
        mid = time.clock()
        for x in range(0,500):
            database.DDBObject()
            database.DDBObject()
            database.DDBObject()
            database.DDBObject()
            database.DDBObject()
            database.DDBObject()
            database.DDBObject()
            database.DDBObject()
            database.DDBObject()
            database.DDBObject()
        end = time.clock()
        # Make sure insert time doesn't increase as the size of the
        # database increases
        assert ( (end-mid) - (mid-start) < (mid-start)/2)
        filtered = self.everything.filterWithIndex(self.mod100,0).sort(self.sortFunc)
        self.assertEqual(filtered.len(),100)


#FIXME: Add test for explicitly recomputing sorts
#FIXME: Add test for recomputing sorts on endChange()
#FIXME: Add test for unlink()

if __name__ == "__main__":
    unittest.main()
