/* prefops.cpp
 * Functions for initializing preferences
 *
 * for Denemo, a gtk+ frontend to GNU Lilypond
 * (c) 1999, 2000, 2001, 2002 Matthew Hiller
 */

#include <stdlib.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include "config.h"
#include <denemo/denemo.h>

enum scannertoks
{
  IMMEDIATEPLAYBACK = G_TOKEN_LAST,
  LILYPATH,
  MIDIPLAYER,
  AUDIOPLAYER,
  PLAYBACKOUTPUT
};

/* This checks to see if there's a .denemo/ directory in the user's
 * home directory, tries to create one if there isn't, and returns the
 * path to it
 *
 * .denemo/ is used for holding configuration files, templates, and so on.
 *
 * I suspect that this may have some Windows portability issues,
 * but I'm not sure on this. */

gchar *
locatedotdenemo ()
{
  static GString *dotdenemo = NULL;
  DIR *dir;

  if (!dotdenemo)
    {
      dotdenemo = g_string_new (getenv ("HOME"));
      g_string_append (dotdenemo, "/.denemo");
    }
  if ((dir = opendir (dotdenemo->str)))
    closedir (dir);
  else
    {
      if (errno == ENOTDIR || errno == ENOENT)
	{
	  if (errno == ENOTDIR)
	    if (unlink (dotdenemo->str) == -1)
	      g_warning (_("locatedotdenemo : error unlinking %s : %s"),
			 dotdenemo->str, g_strerror (errno));
#ifdef G_OS_WIN32
	  if (mkdir (dotdenemo->str) == -1)
#else
	  if (mkdir (dotdenemo->str, 0770) == -1)
#endif
	    g_warning (_("locatedotdenemo : error making directory %s : %s"),
		       dotdenemo->str, g_strerror (errno));
	}
      else
	g_warning (_("locatedotdenemo : error opening directory %s : %s"),
		   dotdenemo->str, g_strerror (errno));
    }
  return dotdenemo->str;
}

static GScannerConfig scanner_config_template = {
  (" \t\r\n") /* cset_skip_characters */ ,
  (G_CSET_a_2_z "_/" G_CSET_A_2_Z) /* cset_identifier_first */ ,
  (G_CSET_a_2_z
   "_0123456789/"
   G_CSET_A_2_Z G_CSET_LATINS G_CSET_LATINC) /* cset_identifier_nth */ ,
  ("#\n") /* cpair_comment_single */ ,

  FALSE /* case_sensitive */ ,

  TRUE /* skip_comment_multi */ ,
  TRUE /* skip_comment_single */ ,
  TRUE /* scan_comment_multi */ ,
  TRUE /* scan_identifier */ ,
  FALSE /* scan_identifier_1char */ ,
  FALSE /* scan_identifier_NULL */ ,
  TRUE /* scan_symbols */ ,
  FALSE /* scan_binary */ ,
  TRUE /* scan_octal */ ,
  TRUE /* scan_float */ ,
  TRUE /* scan_hex */ ,
  FALSE /* scan_hex_dollar */ ,
  TRUE /* scan_string_sq */ ,
  TRUE /* scan_string_dq */ ,
  TRUE /* numbers_2_int */ ,
  FALSE /* int_2_float */ ,
  FALSE /* identifier_2_string */ ,
  TRUE /* char_2_token */ ,
  TRUE /* symbol_2_token */ ,
  FALSE /* scope_0_fallback */ ,
};

/* Actually read through a file looking for configuration settings
 * and save them to prefs */

void
readpreffile (gchar * filename, struct prefinfo *prefs)
{
  gint fd;
  gint result;
  GScanner *scanner;

  if ((fd = open (filename, O_RDONLY)) == -1)
    g_warning (_("readpreffile : error opening %s : %s"), filename,
	       g_strerror (errno));
  else
    {
      scanner = g_scanner_new (&scanner_config_template);
      g_scanner_freeze_symbol_table (scanner);
      g_scanner_add_symbol (scanner, "immediateplayback",
			    GINT_TO_POINTER (IMMEDIATEPLAYBACK));
      g_scanner_add_symbol (scanner, "lilypath", GINT_TO_POINTER (LILYPATH));
      g_scanner_add_symbol (scanner, "midiplayer",
			    GINT_TO_POINTER (MIDIPLAYER));
      g_scanner_add_symbol (scanner, "audioplayer",
	  		    GINT_TO_POINTER(AUDIOPLAYER));
      g_scanner_add_symbol (scanner, "playbackoutput",
			    GINT_TO_POINTER (PLAYBACKOUTPUT));
      g_scanner_thaw_symbol_table (scanner);
      g_scanner_input_file (scanner, fd);
      while ((result = g_scanner_get_next_token (scanner)) != G_TOKEN_EOF)
	{
	  switch (result)
	    {
	    case IMMEDIATEPLAYBACK:
	      if (g_scanner_get_next_token (scanner) == G_TOKEN_EQUAL_SIGN
		  && g_scanner_get_next_token (scanner) == G_TOKEN_IDENTIFIER)
		{
		  prefs->immediateplayback =
		    (g_strcasecmp (scanner->value.v_identifier, "true") == 0);
		}
	    case LILYPATH:
	      if (g_scanner_get_next_token (scanner) == G_TOKEN_EQUAL_SIGN
		  && g_scanner_get_next_token (scanner) == G_TOKEN_IDENTIFIER)
		{
		  g_string_assign (prefs->lilypath,
				   scanner->value.v_identifier);
		}
	      break;
	    case MIDIPLAYER:
	      if (g_scanner_get_next_token (scanner) == G_TOKEN_EQUAL_SIGN
		  && g_scanner_get_next_token (scanner) == G_TOKEN_IDENTIFIER)
		{
		  g_string_assign (prefs->midiplayer,
				   scanner->value.v_identifier);
		}
	      break;
	    case AUDIOPLAYER:
              if (g_scanner_get_next_token (scanner) == G_TOKEN_EQUAL_SIGN
		  && g_scanner_get_next_token (scanner) == G_TOKEN_IDENTIFIER)
              {
		g_string_assign(prefs->audioplayer,
				scanner->value.v_identifier);
	      }
	      break;
	    case PLAYBACKOUTPUT:
	      if (g_scanner_get_next_token (scanner) == G_TOKEN_EQUAL_SIGN
		  && g_scanner_get_next_token (scanner) == G_TOKEN_IDENTIFIER)
		{
		  prefs->playbackoutput =
		    (g_strcasecmp (scanner->value.v_identifier, "true") == 0);
		}
	      break;
	    }
	}
      close (fd);
      g_scanner_destroy (scanner);
    }
}

struct prefinfo *
initprefs ()
{
  gchar *systemwide = PKGDATADIR "denemo.conf";
  struct prefinfo *ret = (struct prefinfo *)g_malloc (sizeof (struct prefinfo));
  gchar *localrc = g_strconcat (locatedotdenemo (), "/denemorc", NULL);

  /* Reasonable default values */

  ret->lilypath = g_string_new ("lilypond");
  ret->midiplayer = g_string_new ("playmidi");
  ret->audioplayer = g_string_new("play");
  ret->immediateplayback = TRUE;
  ret->playbackoutput = FALSE;
  ret->lilyentrystyle = FALSE;
  ret->createclones = FALSE;

  /* Read values from systemwide preferences file */

  readpreffile (systemwide, ret);

  /* Read values from personal preferences file */

  readpreffile (localrc, ret);

  g_free (localrc);
  return ret;
}
