/*
 This file is part of DepQBF.

 DepQBF, a solver for quantified boolean formulae (QBF).	
 Copyright 2010 Florian Lonsing, Johannes Kepler University, Linz, Austria.

 See also http://fmv.jku.at/depqbf for further information.

 DepQBF is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or (at
 your option) any later version.

 DepQBF is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with DepQBF.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include "qdpll_pcnf.h"
#include "qdpll_exit.h"
#include "qdpll_dep_man_generic.h"
#include "qdpll_dep_man_qdag.h"
#include "qdpll_dep_man_qdag_types.h"
#include "qdpll_config.h"
#include "qdpll.h"
#include "qdpll_stack.h"


#define QDPLL_ABORT_DEPMAN(cond,msg)					\
  do {									\
    if (cond)								\
      {									\
        fprintf (stderr, "[qdpll_depman] %s: %s\n", __func__, msg);	\
	fflush (stderr);						\
        abort();							\
      }									\
  } while (0)


struct QDPLLDepManQDAG
{
  /* DO NOT ADD MEMBERS BEFORE 'dmg'! */
  QDPLLDepManGeneric dmg;
  QDPLLMemMan *mm;
  QDPLLPCNF *pcnf;
  QDAGVarList candidates;
  struct
  {
    unsigned int init:1;
  } state;
};


#define VAR_LINK(vars, anchor, element, link)				\
  do {									\
    assert (!(element)->link.prev);					\
    assert (!(element)->link.next);					\
    if ((anchor).last)							\
      {									\
	assert (!VARID2VARPTR((vars), (anchor).last)->link.next);	\
	assert ((anchor).first);					\
	assert (!VARID2VARPTR((vars), (anchor).first)->link.prev);	\
	VARID2VARPTR((vars), (anchor).last)->link.next = (element)->id;	\
      }									\
    else								\
      {									\
	assert (!(anchor).first);					\
	(anchor).first = (element)->id;					\
      }									\
    (element)->link.prev = (anchor).last;				\
    (element)->link.next = 0;						\
    (anchor).last = (element)->id;					\
  } while (0)

#define VAR_UNLINK(vars, anchor, element, link)				\
  do {									\
    if ((element)->link.prev)						\
      {									\
	assert ((anchor).first);					\
	assert ((anchor).last);						\
	assert (VARID2VARPTR((vars), (element)->link.prev)->link.next == \
		(element)->id);						\
	VARID2VARPTR((vars), (element)->link.prev)->link.next =		\
	  (element)->link.next;						\
      }									\
    else								\
      {									\
	assert ((anchor).first == (element)->id);			\
	(anchor).first = (element)->link.next;				\
      }									\
    if ((element)->link.next)						\
      {									\
	assert ((anchor).first);					\
	assert ((anchor).last);						\
	assert (VARID2VARPTR((vars), (element)->link.next)->link.prev == \
		(element)->id);						\
	VARID2VARPTR((vars), (element)->link.next)->link.prev =		\
	  (element)->link.prev;						\
      }									\
    else								\
      {									\
	assert ((anchor).last == (element)->id);			\
	(anchor).last = (element)->link.prev;				\
      }									\
    (element)->link.prev = (element)->link.next = 0;			\
  } while (0)




/* -------------------- START: UNION-FIND -------------------- */
#define UF_IS_REP(var,ufoffset) \
  ((var)->qdag.uf[(ufoffset)].par == (var->id))
#define UF_IS_SINGLETON_SET(var,ufoffset)				\
  (UF_IS_REP(var,(ufoffset)) &&						\
   (var)->qdag.uf[(ufoffset)].members.list.first ==			\
   (var)->qdag.uf[(ufoffset)].members.list.last &&			\
   (var)->qdag.uf[(ufoffset)].members.list.last ==			\
   (var->id))

static void
uf_make_set (Var * v, const unsigned int ufoffset)
{
  assert (!v->qdag.uf[ufoffset].par);
  assert (!v->qdag.uf[ufoffset].members.list.first);
  assert (!v->qdag.uf[ufoffset].members.list.last);
  v->qdag.uf[ufoffset].par = v->id;
  v->qdag.uf[ufoffset].members.list.first =
    v->qdag.uf[ufoffset].members.list.last = v->id;
}

static Var *
uf_find (Var * vars, Var * v, const unsigned int ufoffset)
{
  Var *rep, *par;

  for (rep = v, par = VARID2VARPTR (vars, v->qdag.uf[ufoffset].par);
       rep != par;
       rep = par, par = VARID2VARPTR (vars, par->qdag.uf[ufoffset].par))
    ;

  Var *p;
  for (p = v, par = VARID2VARPTR (vars, v->qdag.uf[ufoffset].par); p != par;
       p = par, par = VARID2VARPTR (vars, par->qdag.uf[ufoffset].par))
    p->qdag.uf[ufoffset].par = rep->id;

  return rep;
}


#define UF_MERGE_MEMBERS_NN(vars, v1,v2,ufoffset)			\
  do {									\
    assert (!UF_IS_SINGLETON_SET((v1),(ufoffset)) ||			\
	    UF_IS_SINGLETON_SET((v2),(ufoffset)));			\
    assert (!VARID2VARPTR((vars),					\
			  (v1)->qdag.uf[ufoffset].members.list.last)->	\
	    qdag.uf[ufoffset].members.link.next);			\
    VARID2VARPTR((vars), (v1)->qdag.uf[ufoffset].members.list.last)->	\
      qdag.uf[ufoffset].members.link.next =				\
      (v2)->qdag.uf[ufoffset].members.list.first;			\
    assert (!VARID2VARPTR((vars),					\
			  (v2)->qdag.uf[ufoffset].members.list.first)->	\
	    qdag.uf[ufoffset].members.link.prev);			\
    VARID2VARPTR((vars), (v2)->qdag.uf[ufoffset].members.list.first)->	\
      qdag.uf[ufoffset].members.link.prev =				\
      (v1)->qdag.uf[ufoffset].members.list.last;			\
    (v1)->qdag.uf[ufoffset].members.list.last = (v2->id);		\
    assert (!VARID2VARPTR((vars),					\
			  (v2)->qdag.uf[ufoffset].members.list.last)->	\
	    qdag.uf[ufoffset].members.link.next);			\
    VARID2VARPTR((vars), (v2)->qdag.uf[ufoffset].members.list.last)->	\
      qdag.uf[ufoffset].members.link.next = (v2->id);			\
    (v2)->qdag.uf[ufoffset].members.link.prev =				\
      (v2)->qdag.uf[ufoffset].members.list.last;			\
    (v2)->qdag.uf[ufoffset].members.link.next = 0;			\
    assert ((v1)->qdag.uf[ufoffset].members.list.first);		\
    assert ((v1)->qdag.uf[ufoffset].members.list.last);			\
    assert (!(v2)->qdag.uf[ufoffset].members.link.next);		\
  } while (0)

#define UF_MERGE_MEMBERS_N1(vars,v1,v2,ufoffset)			\
  do {									\
    assert (!UF_IS_SINGLETON_SET((v1),(ufoffset)) ||			\
	    UF_IS_SINGLETON_SET((v2),(ufoffset)));			\
    assert ((v2)->qdag.uf[ufoffset].members.list.first == (v2->id) &&	\
	    (v2)->qdag.uf[ufoffset].members.list.last == (v2->id));	\
    VARID2VARPTR((vars), (v1)->qdag.uf[ufoffset].members.list.last)->	\
      qdag.uf[ufoffset].members.link.next = (v2->id);			\
    (v2)->qdag.uf[ufoffset].members.link.prev =				\
      (v1)->qdag.uf[ufoffset].members.list.last;			\
    (v2)->qdag.uf[ufoffset].members.link.next = 0;			\
    (v1)->qdag.uf[ufoffset].members.list.last = (v2->id);		\
    assert ((v1)->qdag.uf[ufoffset].members.list.first);		\
    assert ((v1)->qdag.uf[ufoffset].members.list.last);			\
    assert (!(v2)->qdag.uf[ufoffset].members.link.next);		\
  } while (0)

#define UF_MERGE_MEMBERS_11(vars, v1,v2,ufoffset)			\
  do {									\
    (v1)->qdag.uf[ufoffset].members.list.first =			\
      (v1)->qdag.uf[ufoffset].members.list.last = (v2)->id;		\
    (v2)->qdag.uf[ufoffset].members.link.prev =				\
      (v2)->qdag.uf[ufoffset].members.link.next = 0;			\
    assert ((v1)->qdag.uf[ufoffset].members.list.first);		\
    assert ((v1)->qdag.uf[ufoffset].members.list.last);			\
    assert (!(v2)->qdag.uf[ufoffset].members.link.next);		\
  } while (0)

static void
uf_link (Var * vars, Var * v1, Var * v2, const unsigned int ufoffset)
{
  assert (UF_IS_REP (v1, ufoffset));
  assert (UF_IS_REP (v2, ufoffset));
  assert (!UF_IS_SINGLETON_SET (v1, ufoffset)
	  || v1->qdag.uf[ufoffset].rank == 0);
  assert (!UF_IS_SINGLETON_SET (v2, ufoffset)
	  || v2->qdag.uf[ufoffset].rank == 0);
  assert (!UF_IS_SINGLETON_SET (v1, ufoffset)
	  || !UF_IS_SINGLETON_SET (v2, ufoffset)
	  || (v2->qdag.uf[ufoffset].rank == v1->qdag.uf[ufoffset].rank
	      && v1->qdag.uf[ufoffset].rank == 0));
  assert (UF_IS_SINGLETON_SET (v1, ufoffset)
	  || !UF_IS_SINGLETON_SET (v2, ufoffset)
	  || v2->qdag.uf[ufoffset].rank < v1->qdag.uf[ufoffset].rank);
  assert (!UF_IS_SINGLETON_SET (v1, ufoffset)
	  || UF_IS_SINGLETON_SET (v2, ufoffset)
	  || v2->qdag.uf[ufoffset].rank > v1->qdag.uf[ufoffset].rank);
  assert (v1->qdag.uf[ufoffset].class_link.prev
	  || v1->qdag.uf[ufoffset].class_link.next);
  assert (v2->qdag.uf[ufoffset].class_link.prev
	  || v2->qdag.uf[ufoffset].class_link.next);

  if (v1->qdag.uf[ufoffset].rank > v2->qdag.uf[ufoffset].rank)
    {
      assert (!UF_IS_SINGLETON_SET (v1, ufoffset));
      if (UF_IS_SINGLETON_SET (v2, ufoffset))
	UF_MERGE_MEMBERS_N1 (vars, v1, v2, ufoffset);
      else
	UF_MERGE_MEMBERS_NN (vars, v1, v2, ufoffset);
      v2->qdag.uf[ufoffset].par = v1->id;
      assert (!UF_IS_REP (v2, ufoffset));
      VAR_UNLINK (vars, v2->scope->classes[ufoffset], v2,
		  qdag.uf[ufoffset].class_link);
    }
  else
    {
      if (v1->qdag.uf[ufoffset].rank == v2->qdag.uf[ufoffset].rank)
	{
	  assert ((UF_IS_SINGLETON_SET (v1, ufoffset)
		   && UF_IS_SINGLETON_SET (v2, ufoffset))
		  || (!UF_IS_SINGLETON_SET (v1, ufoffset)
		      && !UF_IS_SINGLETON_SET (v2, ufoffset)));
	  assert (v1->qdag.uf[ufoffset].rank != 0
		  || (UF_IS_SINGLETON_SET (v1, ufoffset)
		      && UF_IS_SINGLETON_SET (v2, ufoffset)));
	  assert ((!UF_IS_SINGLETON_SET (v1, ufoffset)
		   || !UF_IS_SINGLETON_SET (v2, ufoffset))
		  || v1->qdag.uf[ufoffset].rank == 0);
	  v2->qdag.uf[ufoffset].rank++;
	  if (UF_IS_SINGLETON_SET (v1, ufoffset)
	      && UF_IS_SINGLETON_SET (v2, ufoffset))
	    UF_MERGE_MEMBERS_11 (vars, v2, v1, ufoffset);
	  else
	    UF_MERGE_MEMBERS_NN (vars, v2, v1, ufoffset);
	}
      else
	{
	  assert (!UF_IS_SINGLETON_SET (v2, ufoffset));
	  if (UF_IS_SINGLETON_SET (v1, ufoffset))
	    UF_MERGE_MEMBERS_N1 (vars, v2, v1, ufoffset);
	  else
	    UF_MERGE_MEMBERS_NN (vars, v2, v1, ufoffset);
	}
      v1->qdag.uf[ufoffset].par = v2->id;
      assert (!UF_IS_REP (v1, ufoffset));
      VAR_UNLINK (vars, v1->scope->classes[ufoffset], v1,
		  qdag.uf[ufoffset].class_link);
    }
}


static void
uf_unite (Var * vars, Var * v1, Var * v2, const unsigned int ufoffset)
{
  assert (v1->scope == v2->scope);
  assert (v1->qdag.uf[ufoffset].par);
  assert (v2->qdag.uf[ufoffset].par);

  Var *rep1, *rep2;
  rep1 = uf_find (vars, v1, ufoffset);
  rep2 = uf_find (vars, v2, ufoffset);

  if (rep1 != rep2)
    uf_link (vars, rep1, rep2, ufoffset);
}

/* -------------------- END: UNION-FIND -------------------- */



/* -------------------- START: EDGE PRIORITY-QUEUE -------------------- */

static void
pq_init (QDPLLMemMan * mm, EdgePriorityQueue * pq, unsigned int init_capacity)
{
  unsigned int bytes;

  if (init_capacity == 0)
    init_capacity = 1;

  bytes = init_capacity * sizeof (Edge *);
  pq->elems_start = pq->elems_top = (Edge **) qdpll_malloc (mm, bytes);
  pq->elems_end = pq->elems_start + init_capacity;
}


static unsigned int
pq_size (EdgePriorityQueue * pq)
{
  return pq->elems_end - pq->elems_start;
}


static void
pq_delete (QDPLLMemMan * mm, EdgePriorityQueue * pq)
{
  qdpll_free (mm, pq->elems_start, pq_size (pq) * sizeof (Edge *));
}


static unsigned int
pq_count (EdgePriorityQueue * pq)
{
  return pq->elems_top - pq->elems_start;
}


static unsigned int
get_left_child_pos (unsigned int cur_pos)
{
  return 2 * cur_pos + 1;
}


static unsigned int
get_right_child_pos (unsigned int cur_pos)
{
  return 2 * (cur_pos + 1);
}


static unsigned int
get_parent_pos (unsigned int cur_pos)
{
  unsigned int result;
  result = (cur_pos - 1) / 2;
  assert (cur_pos == get_right_child_pos (result) ||
	  cur_pos == get_left_child_pos (result));
  return result;
}


static int
compare (EdgePriorityQueue * pq, unsigned int pos_a, unsigned int pos_b)
{
  assert (pos_a < pq_count (pq));
  assert (pos_b < pq_count (pq));

  int result;

  Edge **start = pq->elems_start;

  Edge **elem_a = start + pos_a;
  Edge **elem_b = start + pos_b;

  unsigned int elem_a_priority = (*elem_a)->priority;
  unsigned int elem_b_priority = (*elem_b)->priority;

  if (elem_a_priority < elem_b_priority)
    result = -1;
  else if (elem_a_priority == elem_b_priority)
    result = 0;
  else
    result = 1;

  return result;
}


static void
swap (EdgePriorityQueue * pq, unsigned int pos_a, unsigned int pos_b)
{
  assert (pos_a != pos_b);
  assert (pos_a < pq_count (pq));
  assert (pos_b < pq_count (pq));

  Edge *tmp, **start;

  start = pq->elems_start;

  Edge **elem_a, **elem_b;

  elem_a = start + pos_a;
  tmp = *elem_a;
  elem_b = start + pos_b;

  assert ((*elem_b)->pos == pos_b);
  assert ((*elem_a)->pos == pos_a);
  assert (tmp->pos == pos_a);

  *elem_a = *elem_b;
  (*elem_a)->pos = pos_a;

  *elem_b = tmp;
  (*elem_b)->pos = pos_b;
}


static void
up_heap (EdgePriorityQueue * pq, unsigned int cur_pos)
{

#ifndef NDEBUG
  unsigned int count = pq_count (pq);
  assert (cur_pos < count);
#endif

  while (cur_pos > 0)
    {
      unsigned int parent_pos = get_parent_pos (cur_pos);

      if (compare (pq, cur_pos, parent_pos) >= 0)
	break;

      swap (pq, cur_pos, parent_pos);
      cur_pos = parent_pos;
    }				/* end: while top not reached */
}


static void
down_heap (EdgePriorityQueue * pq, unsigned int cur_pos)
{
  unsigned int child_pos, left_child_pos, right_child_pos;
  unsigned int count = pq_count (pq);

  for (;;)
    {
      left_child_pos = get_left_child_pos (cur_pos);

      if (left_child_pos >= count)
	break;			/* has no left child */

      right_child_pos = get_right_child_pos (cur_pos);

      if (right_child_pos < count &&
	  compare (pq, left_child_pos, right_child_pos) > 0)
	child_pos = right_child_pos;
      else
	child_pos = left_child_pos;

      if (compare (pq, cur_pos, child_pos) > 0)
	{
	  swap (pq, cur_pos, child_pos);
	  cur_pos = child_pos;
	}
      else
	break;
    }
}


static void
assert_pq_condition (EdgePriorityQueue * pq)
{
  Edge **start = pq->elems_start;

  unsigned int pos, no_children, left_child_pos, right_child_pos;
  no_children = pq_count (pq) / 2;

  for (pos = 0; pos < pq_count (pq); pos++)
    {
      Edge **cur, **left, **right;

      cur = start + pos;
      assert ((*cur)->pos == pos);

      left_child_pos = get_left_child_pos (pos);
      right_child_pos = get_right_child_pos (pos);

      if (pos < no_children)
	{
	  assert (left_child_pos < pq_count (pq));

	  left = start + left_child_pos;

	  if (right_child_pos < pq_count (pq))
	    right = start + right_child_pos;

	  assert ((*cur)->priority <= (*left)->priority);
	  assert (right_child_pos >= pq_count (pq) ||
		  (*cur)->priority <= (*right)->priority);
	}
      else			/* has no children */
	{
	  assert (right_child_pos >= pq_count (pq));
	  assert (left_child_pos >= pq_count (pq));
	}
    }

}


static void
pq_enlarge (QDPLLMemMan * mm, EdgePriorityQueue * pq)
{
  unsigned int old_size = pq_size (pq);
  unsigned int new_size = old_size ? 2 * old_size : 1;
  Edge **new = (Edge **) qdpll_malloc (mm, new_size * sizeof (Edge *));
  memcpy (new, pq->elems_start, old_size * sizeof (Edge *));
  qdpll_free (mm, pq->elems_start, old_size * sizeof (Edge *));
  pq->elems_start = new;
  pq->elems_top = pq->elems_start + old_size;
  pq->elems_end = pq->elems_start + new_size;
}


void
pq_insert (QDPLLMemMan * mm, EdgePriorityQueue * pq, Edge * edge,
	   unsigned int priority)
{
  if (pq->elems_top == pq->elems_end)
    pq_enlarge (mm, pq);

  *pq->elems_top++ = edge;
  edge->priority = priority;
  edge->pos = pq_count (pq) - 1;

  up_heap (pq, edge->pos);

#ifndef NDEBUG
#if QDAG_PQ_ASSERT_HEAP_CONDITION_INSERT
  assert_pq_condition (pq);
#endif
#endif
}


Edge *
pq_remove_min (EdgePriorityQueue * pq)
{
  Edge **start = pq->elems_start;
  Edge *result = 0;

  if (start == pq->elems_top)
    return 0;

  Edge **last;
  last = --(pq->elems_top);

  assert (start != pq->elems_top || pq_count (pq) == 0);

  result = *start;

  *start = *last;
  (*start)->pos = 0;

  down_heap (pq, 0);

#ifndef NDEBUG
#if QDAG_PQ_ASSERT_HEAP_CONDITION_REMOVE_MIN
  assert_pq_condition (pq);
#endif
#endif

  return result;
}


/* remove one element in constant time, destroying heap condition. */
Edge *
pq_remove_one (EdgePriorityQueue * pq)
{
  Edge **start = pq->elems_start;
  Edge *result = 0;

  if (start == pq->elems_top)
    return 0;

  Edge **last;
  last = --(pq->elems_top);

  assert (start != pq->elems_top || pq_count (pq) == 0);

  result = *start;

  *start = *last;
  (*start)->pos = 0;

  return result;
}


Edge *
pq_access_min (EdgePriorityQueue * pq)
{
  Edge **start = pq->elems_start;

  if (start == pq->elems_top)
    return 0;
  else
    return *start;
}


/* removes elem at index 'elem_pos' and maintains heap condition */
static Edge *
pq_remove_elem (EdgePriorityQueue * pq, int elem_pos)
{
  assert (elem_pos >= 0);
  assert ((unsigned int) elem_pos < pq_count (pq));

#ifndef NDEBUG
#if QDAG_PQ_ASSERT_HEAP_CONDITION_REMOVE_ELEM
  assert_pq_condition (pq);
#endif
#endif

  Edge *last, *del;
  Edge **pos = pq->elems_start + elem_pos;

  del = *pos;
  assert (del->pos == (unsigned int) elem_pos);
  del->pos = -1;

  --(pq->elems_top);
  last = *pq->elems_top;

  if (del != last)
    {
      *pos = last;
      last->pos = elem_pos;
      up_heap (pq, elem_pos);
      down_heap (pq, elem_pos);
    }

#ifndef NDEBUG
#if QDAG_PQ_ASSERT_HEAP_CONDITION_REMOVE_ELEM
  assert_pq_condition (pq);
#endif
#endif

  return del;
}

/* -------------------- END: EDGE PRIORITY-QUEUE -------------------- */


/* -------------------- START: EDGE-HASHTABLE -------------------- */
static void
et_init (QDPLLMemMan * mm, EdgeTable * et, unsigned int size)
{
  size = size == 0 ? 1 : size;
  size_t bytes = size * sizeof (Edge *);
  et->table = (Edge **) qdpll_malloc (mm, bytes);
  et->size = size;
}


static void
et_delete (QDPLLMemMan * mm, EdgeTable * et)
{
  qdpll_free (mm, et->table, et->size * sizeof (Edge *));
}


static unsigned int
et_hash (VarID var)
{
  return var * 1183477;
}


static Edge **
et_findpos (EdgeTable * t, VarID var)
{
  assert (t);
  unsigned int h = et_hash (var);
  h &= (t->size - 1);

  Edge **p, *d;
  for (p = t->table + h; (d = *p) && d->head_var != var; p = &d->chain_next)
    ;

  return p;
}


static Edge *
et_lookup (EdgeTable * t, VarID var)
{
  assert (t);
  return *et_findpos (t, var);
}


static void
et_enlarge (QDPLLMemMan * mm, EdgeTable * t)
{
  assert (t);
  unsigned int old_size = t->size;
  unsigned int new_size = old_size ? old_size * 2 : 1;
  size_t bytes = new_size * sizeof (Edge *);
  Edge **new_table = (Edge **) qdpll_malloc (mm, bytes);

  Edge *p, *n;
  unsigned int i;
  for (i = 0; i < old_size; i++)
    for (p = t->table[i]; p; p = n)
      {
	n = p->chain_next;
	unsigned int h = et_hash (p->head_var);
	h &= (new_size - 1);
	p->chain_next = new_table[h];
	new_table[h] = p;
      }

  qdpll_free (mm, t->table, old_size * sizeof (Edge *));
  t->table = new_table;
  t->size = new_size;
}


static void
et_insert (QDPLLMemMan * mm, EdgeTable * t, Edge * d)
{
  assert (t);
  assert (!d->chain_next);

  if (t->count == t->size)
    et_enlarge (mm, t);

  Edge **p = et_findpos (t, d->head_var);
  assert (!*p);

  *p = d;
  t->count++;
}


static Edge *
et_remove (EdgeTable * t, VarID var)
{
  assert (t);
  Edge **p = et_findpos (t, var);
  Edge *result = *p;
  assert (result);
  *p = result->chain_next;
  result->chain_next = 0;
  t->count--;
  return result;
}

/* -------------------- END: EDGE-HASHTABLE -------------------- */


/* -------------------- START: ASSERTION-ONLY CODE -------------------- */

static void
assert_graph_check_edges_pq (Var * vars, Var * v, EdgePriorityQueue * edge_pq)
{
  Edge **d, **end;
  end = edge_pq->elems_top;
  for (d = edge_pq->elems_start; d < end; d++)
    {
      assert ((*d)->tail_var);
      assert ((*d)->head_var);
      assert ((*d)->head_var == v->id);
      Var *tv = VARID2VARPTR (vars, (*d)->tail_var);
      assert (edge_pq != &(v->qdag.dedge_pq)
	      || tv->scope->type != v->scope->type);
      assert (edge_pq == &(v->qdag.dedge_pq)
	      || tv->scope->type == v->scope->type);
      assert (tv->scope->nesting < v->scope->nesting);
      assert ((*d)->priority == tv->scope->nesting);
    }
}

static void
assert_graph_check_edges (Var * vars, Var * v, EdgeTable * edge_table)
{
  unsigned int i, end = edge_table->size;
  Edge *d;
  for (i = 0; i < end; i++)
    for (d = edge_table->table[i]; d; d = d->chain_next)
      {
	assert (d->tail_var);
	assert (d->head_var);
	assert (d->tail_var == v->id);
	Var *hv = VARID2VARPTR (vars, d->head_var);
	assert (edge_table != &(v->qdag.dedges)
		|| hv->scope->type != v->scope->type);
	assert (edge_table == &(v->qdag.dedges)
		|| hv->scope->type == v->scope->type);
	assert (hv->scope->nesting > v->scope->nesting);
	assert (d->priority == v->scope->nesting);
      }
}


static void
assert_graph_integrity (QDPLLDepManQDAG * dm)
{
  Var *vars = dm->pcnf->vars;

  Scope *s;
  for (s = dm->pcnf->scopes.first; s; s = s->link.next)
    {
      VarID vid;
      Var *v;
      if (QDPLL_SCOPE_EXISTS (s))
	{
	  for (vid = s->classes[0].first; vid;
	       vid = v->qdag.uf[0].class_link.next)
	    {
	      v = VARID2VARPTR (vars, vid);

	      assert (UF_IS_REP (v, 0));

	      if (!UF_IS_SINGLETON_SET (v, 0))
		{
		  VarID mid;
		  Var *m;
		  for (mid = v->qdag.uf[0].members.list.first; mid;
		       mid = m->qdag.uf[0].members.link.next)
		    {
		      m = VARID2VARPTR (vars, mid);
		      assert (!UF_IS_REP (m, 0));
		      assert (!m->qdag.cedges.cpar);
		      assert (!m->qdag.cedges.cchilds.first
			      && !m->qdag.cedges.cchilds.last);
		      assert (!m->qdag.cedges.csibs.next
			      && !m->qdag.cedges.csibs.prev);
		      assert (pq_count (&(m->qdag.dedge_pq)) == 0);
		      assert (pq_count (&(m->qdag.sedge_pq)) == 0);
		      assert_graph_check_edges (vars, m, &(m->qdag.dedges));
		      assert_graph_check_edges (vars, m, &(m->qdag.sedges));
		    }
		}
	      else
		{
		  assert (v->qdag.uf[0].members.list.first ==
			  v->qdag.uf[0].members.list.last);
		  assert (v->qdag.uf[0].members.list.first == vid);
		}

	      assert (!v->qdag.cedges.cpar
		      ||
		      UF_IS_REP (VARID2VARPTR (vars, v->qdag.cedges.cpar),
				 0));
	      assert (!v->qdag.cedges.cpar
		      || (VARID2VARPTR (vars, v->qdag.cedges.cpar)->scope->
			  type == v->scope->type
			  && QDPLL_SCOPE_EXISTS (v->scope)));
	      assert (!v->qdag.cedges.cpar
		      || VARID2VARPTR (vars,
				       v->qdag.cedges.cpar)->scope->nesting <
		      v->scope->nesting);
	      assert (!v->qdag.cedges.cpar
		      || (VARID2VARPTR (vars, v->qdag.cedges.cpar)->qdag.
			  cedges.cchilds.first
			  && VARID2VARPTR (vars,
					   v->qdag.cedges.cpar)->qdag.cedges.
			  cchilds.last));

	      VarID cid;
	      Var *c;
	      for (cid = v->qdag.cedges.cchilds.first; cid;
		   cid = c->qdag.cedges.csibs.next)
		{
		  c = VARID2VARPTR (vars, cid);
		  assert (UF_IS_REP (c, 0));
		  assert (v->scope->type == c->scope->type
			  && QDPLL_SCOPE_EXISTS (v->scope));
		  assert (v->scope->nesting < c->scope->nesting);
		  assert (vid == c->qdag.cedges.cpar);
		}

	      assert_pq_condition (&(v->qdag.dedge_pq));

	      if (v->qdag.cedges.cpar && pq_access_min (&(v->qdag.dedge_pq)))
		{
		  assert (pq_access_min (&(v->qdag.dedge_pq))->priority >
			  VARID2VARPTR (vars,
					v->qdag.cedges.cpar)->scope->nesting);
		}

	      assert_pq_condition (&(v->qdag.sedge_pq));

	      if (v->qdag.cedges.cpar && pq_access_min (&(v->qdag.sedge_pq)))
		{
		  assert (pq_access_min (&(v->qdag.sedge_pq))->priority ==
			  VARID2VARPTR (vars,
					v->qdag.cedges.cpar)->scope->nesting);
		  Edge **d, **end = v->qdag.sedge_pq.elems_top;
		  for (d = v->qdag.sedge_pq.elems_start; d < end; d++)
		    assert ((*d)->priority ==
			    VARID2VARPTR (vars,
					  v->qdag.cedges.cpar)->
			    scope->nesting);
		}

	      assert_graph_check_edges (vars, v, &(v->qdag.dedges));
	      assert_graph_check_edges (vars, v, &(v->qdag.sedges));
	      assert_graph_check_edges_pq (vars, v, &(v->qdag.dedge_pq));
	      assert_graph_check_edges_pq (vars, v, &(v->qdag.sedge_pq));
	    }
	}
      else
	{
	  assert (QDPLL_SCOPE_FORALL (s));
	  for (vid = s->classes[0].first; vid;
	       vid = v->qdag.uf[0].class_link.next)
	    {
	      v = VARID2VARPTR (vars, vid);
	      assert (!v->qdag.cedges.cpar);
	      assert (!v->qdag.cedges.csibs.next
		      && !v->qdag.cedges.csibs.prev);
	      assert (!v->qdag.cedges.cchilds.first
		      && !v->qdag.cedges.cchilds.last);

	      assert (pq_count (&(v->qdag.sedge_pq)) == 0);
	      assert (v->qdag.sedges.count == 0);
	      assert_graph_check_edges (vars, v, &(v->qdag.dedges));
	      assert_graph_check_edges_pq (vars, v, &(v->qdag.dedge_pq));

	      if (!UF_IS_SINGLETON_SET (v, 0))
		{
		  VarID mid;
		  Var *m;
		  for (mid = v->qdag.uf[0].members.list.first; mid;
		       mid = m->qdag.uf[0].members.link.next)
		    {
		      m = VARID2VARPTR (vars, mid);
		      assert (!UF_IS_REP (m, 0));
		      assert (!m->qdag.cedges.cpar);
		      assert (!m->qdag.cedges.cchilds.first
			      && !m->qdag.cedges.cchilds.last);
		      assert (!m->qdag.cedges.csibs.next
			      && !m->qdag.cedges.csibs.prev);
		      assert (pq_count (&(m->qdag.dedge_pq)) == 0);
		      assert (pq_count (&(m->qdag.sedge_pq)) == 0);
		      assert (m->qdag.sedges.count == 0);
		      assert (m->qdag.dedges.count == 0);
		    }
		}
	      else
		{
		  assert (v->qdag.uf[0].members.list.first ==
			  v->qdag.uf[0].members.list.last);
		  assert (v->qdag.uf[0].members.list.first == vid);
		}
	    }
	}
    }
}


static void
assert_insert_c_edge_cchilds (Var * vars, Var * from, Var * to)
{
  assert (QDPLL_SCOPE_EXISTS (from->scope));
  assert (QDPLL_SCOPE_EXISTS (to->scope));
  Var *m;
  VarID mid;
  if (UF_IS_REP (from, 0))
    {
      assert (UF_IS_REP (to, 0) || !to->qdag.cedges.cpar);
      assert (UF_IS_REP (to, 0)
	      || (!to->qdag.cedges.cchilds.first
		  && !to->qdag.cedges.cchilds.last));

      if (!UF_IS_SINGLETON_SET (from, 0))
	for (mid = from->qdag.uf[0].members.list.first; mid;
	     mid = m->qdag.uf[0].members.link.next)
	  {
	    m = VARID2VARPTR (vars, mid);
	    assert (!UF_IS_REP (m, 0));
	    assert (!m->qdag.cedges.cpar);
	    assert (!m->qdag.cedges.cchilds.first
		    && !m->qdag.cedges.cchilds.last);
	  }
    }
  else
    {
      assert (!from->qdag.cedges.cpar);
      assert (!from->qdag.cedges.cchilds.first
	      && !from->qdag.cedges.cchilds.last);
    }

  if (UF_IS_REP (to, 0))
    {
      assert (UF_IS_REP (from, 0) || !from->qdag.cedges.cpar);
      assert (UF_IS_REP (from, 0)
	      || (!from->qdag.cedges.cchilds.first
		  && !from->qdag.cedges.cchilds.last));

      if (!UF_IS_SINGLETON_SET (to, 0))
	for (mid = to->qdag.uf[0].members.list.first; mid;
	     mid = m->qdag.uf[0].members.link.next)
	  {
	    m = VARID2VARPTR (vars, mid);
	    assert (!UF_IS_REP (m, 0));
	    assert (!m->qdag.cedges.cpar);
	    assert (!m->qdag.cedges.cchilds.first
		    && !m->qdag.cedges.cchilds.last);
	  }
    }
  else
    {
      assert (!to->qdag.cedges.cpar);
      assert (!to->qdag.cedges.cchilds.first
	      && !to->qdag.cedges.cchilds.last);
    }
}


static void
assert_c_edges_integrity (Var * vars, Var * start)
{
  assert (QDPLL_SCOPE_EXISTS (start->scope));
  Var *prev, *cur;
  VarID prev_id, cur_id;
  for (cur_id = prev_id = start->id; cur_id; cur_id = cur->qdag.cedges.cpar)
    {
      cur = VARID2VARPTR (vars, cur_id);
      prev = VARID2VARPTR (vars, prev_id);
      assert (UF_IS_REP (cur, 0));
      assert (!
	      (cur->qdag.cedges.cpar && pq_access_min (&(cur->qdag.dedge_pq)))
	      || pq_access_min (&(cur->qdag.dedge_pq))->priority >
	      VARID2VARPTR (vars, cur->qdag.cedges.cpar)->scope->nesting);
      assert (!
	      (cur->qdag.cedges.cpar && pq_access_min (&(cur->qdag.sedge_pq)))
	      || pq_access_min (&(cur->qdag.sedge_pq))->priority ==
	      VARID2VARPTR (vars, cur->qdag.cedges.cpar)->scope->nesting);
      assert (prev == cur || cur->scope->nesting < prev->scope->nesting);
      assert (cur->scope->type == prev->scope->type);
      prev = cur;
    }
}

static LitID *get_largest_active_literal_in_clause (Var * vars,
						    LitID * leftend,
						    LitID * rightend,
						    LitID * ptr,
						    unsigned int
						    ignore_dlevel);

static void
collect_deps_from_cnf_check_clause (QDPLLDepManQDAG * dm,
				    VarPtrStack * deps,
				    VarPtrStack * con,
				    QDPLLQuantifierType var_type,
				    unsigned int var_nesting, Constraint * c);

static void
collect_deps_from_cnf (QDPLLDepManQDAG * dm, VarPtrStack * deps, Var * v);

static void
collect_deps_from_graph_forall (QDPLLDepManQDAG * dm, VarPtrStack * deps,
				Var * v);

static void
collect_deps_from_graph_exists (QDPLLDepManQDAG * dm, VarPtrStack * deps,
				Var * v);

static void
collect_deps_from_graph (QDPLLDepManQDAG * dm, VarPtrStack * deps, Var * v);

static int qsort_compare_deps_by_id (const void *dep1, const void *dep2);

static void unmark_dependency_marks (VarPtrStack * deps);

/* Check whether dependencies collected from graph match dependencies 
   collected by trivial approach (searching clauses in QBF). */
static void
assert_xcheck_dependencies (QDPLLDepManQDAG * dm)
{
  VarPtrStack deps_from_cnf;
  VarPtrStack deps_from_graph;
  QDPLL_INIT_STACK (deps_from_cnf);
  QDPLL_INIT_STACK (deps_from_graph);
  Var *vars = dm->pcnf->vars;

  /* Check that qdag-marks are all set to 0. */
  Var *p, *e;
  for (p = vars, e = vars + dm->pcnf->size_vars; p < e; p++)
    {
      assert (!p->qdag.mark0);
      assert (!p->qdag.mark1);
    }

  Scope *s;
  for (s = dm->pcnf->scopes.first; s; s = s->link.next)
    {
      VarID *p, *e;
      for (p = s->vars.start, e = s->vars.top; p < e; p++)
	{
	  Var *v = VARID2VARPTR (vars, *p);

	  if (QDPLL_VAR_MARKED_PROPAGATED (v) && v->decision_level <= 0)
	    continue;

	  collect_deps_from_cnf (dm, &deps_from_cnf, v);
	  unmark_dependency_marks (&deps_from_cnf);
	  /* Note: connection marks have been reset already. */
	  qsort (deps_from_cnf.start, QDPLL_COUNT_STACK (deps_from_cnf),
		 sizeof (*deps_from_cnf.start), qsort_compare_deps_by_id);

	  collect_deps_from_graph (dm, &deps_from_graph, v);
	  /* Note: connection marks have been reset already. */
	  unmark_dependency_marks (&deps_from_graph);
	  qsort (deps_from_graph.start, QDPLL_COUNT_STACK (deps_from_graph),
		 sizeof (*deps_from_graph.start), qsort_compare_deps_by_id);

	  unsigned int cnt_deps_from_cnf = QDPLL_COUNT_STACK (deps_from_cnf);
	  unsigned int cnt_deps_from_graph =
	    QDPLL_COUNT_STACK (deps_from_graph);
	  assert (cnt_deps_from_cnf == cnt_deps_from_graph);

	  /* Traverse sorted stacks in parallel and compare collected dependencies. */
	  Var **p1, **prev1, **p2, **prev2, **e1, **e2;
	  for (p1 = prev1 = deps_from_cnf.start, e1 = deps_from_cnf.top,
	       p2 = prev2 = deps_from_graph.start, e2 = deps_from_graph.top;
	       p1 < e1; p1++, p2++)
	    {
	      assert (p2 < e2);
	      assert ((*prev1)->id <= (*p1)->id);
	      assert ((*prev2)->id <= (*p2)->id);
	      assert (*p1 == *p2);
	      prev1 = p1;
	      prev2 = p2;
	    }

	  QDPLL_RESET_STACK (deps_from_cnf);
	  QDPLL_RESET_STACK (deps_from_graph);
	}
    }

  QDPLL_DELETE_STACK (dm->mm, deps_from_cnf);
  QDPLL_DELETE_STACK (dm->mm, deps_from_graph);


  /* Check that qdag-marks are all set to 0. */
  for (p = vars, e = vars + dm->pcnf->size_vars; p < e; p++)
    {
      assert (!p->qdag.mark0);
      assert (!p->qdag.mark1);
    }
}


static void
assert_no_transitivities (QDPLLDepManQDAG * dm)
{

}


static void
assert_candidate_list (QDPLLDepManQDAG * dm)
{
  Var *vars = dm->pcnf->vars;
  Var *c;
  VarID cid;
  for (cid = dm->candidates.first; cid; cid = c->qdag.cand_link.next)
    {
      c = VARID2VARPTR (vars, cid);
      assert (c->mark_is_candidate);
    }
}


static int referenced_by_active_existential_var (Var * vars, Var * var);


#ifndef NDEBUG

/* Similar to 'mark_non_candidates_from_exists', but uses other mark. */
static void
assert_mark_non_candidates_from_exists (QDPLLDepManQDAG * dm, Scope * scope)
{
  assert (QDPLL_SCOPE_EXISTS (scope));
  QDPLLMemMan *mm = dm->mm;
  Var *vars = dm->pcnf->vars;
  VarPtrStack succ;
  QDPLL_INIT_STACK (succ);

  VarID *p, *e;
  for (p = scope->vars.start, e = scope->vars.top; p < e; p++)
    {
      Var *v = VARID2VARPTR (vars, *p);

      if (QDPLL_VAR_MARKED_PROPAGATED (v) && v->decision_level <= 0)
	continue;

      unsigned int i, end = v->qdag.sedges.size;
      Edge *s;
      for (i = 0; i < end; i++)
	for (s = v->qdag.sedges.table[i]; s; s = s->chain_next)
	  {
	    Var *svar = VARID2VARPTR (vars, s->head_var);
	    assert (s->tail_var == v->id);
	    assert (QDPLL_SCOPE_EXISTS (svar->scope));
	    assert (v->scope->nesting < svar->scope->nesting);
	    assert (UF_IS_REP (svar, 0));
	    QDPLL_PUSH_STACK (mm, succ, svar);
	  }

      while (!QDPLL_EMPTY_STACK (succ))
	{
	  Var *s = QDPLL_POP_STACK (succ);
	  assert (QDPLL_SCOPE_EXISTS (s->scope));
	  assert (UF_IS_REP (s, 0));

	  Edge **p, **e;
	  for (p = s->qdag.dedge_pq.elems_start, e =
	       s->qdag.dedge_pq.elems_top; p < e; p++)
	    {
	      Var *dvar = VARID2VARPTR (vars, (*p)->tail_var);
	      assert (QDPLL_SCOPE_FORALL (dvar->scope));
	      assert (v->scope->nesting < dvar->scope->nesting);
	      assert (UF_IS_REP (dvar, 0));

	      if (!dvar->qdag.mark_is_candidate_debug
		  && referenced_by_active_existential_var (vars, dvar))
		{
		  dvar->qdag.mark_is_candidate_debug = 1;

		  if (!UF_IS_SINGLETON_SET (dvar, 0))
		    {
		      Var *m;
		      VarID mid;
		      for (mid = dvar->qdag.uf[0].members.list.first; mid;
			   mid = m->qdag.uf[0].members.link.next)
			{
			  m = VARID2VARPTR (vars, mid);
			  assert (!m->qdag.mark_is_candidate_debug);
			  m->qdag.mark_is_candidate_debug = 1;
			}
		    }
		}
	    }

	  Var *c;
	  VarID cid;
	  for (cid = s->qdag.cedges.cchilds.first; cid;
	       cid = c->qdag.cedges.csibs.next)
	    {
	      c = VARID2VARPTR (vars, cid);
	      QDPLL_PUSH_STACK (mm, succ, c);
	    }
	}
    }

  QDPLL_DELETE_STACK (mm, succ);
}

static unsigned int count_non_propagated_in_class (Var * vars, Var * v);

/* Similar to 'mark_non_candidates_from_forall', but uses other mark. */
static void
assert_mark_non_candidates_from_forall (QDPLLDepManQDAG * dm, Scope * s)
{
  Var *vars = dm->pcnf->vars;
  QDPLLMemMan *mm = dm->mm;
  VarPtrStack forest_succ;
  QDPLL_INIT_STACK (forest_succ);
  assert (QDPLL_SCOPE_FORALL (s));

  Var *v;
  VarID vid;
  for (vid = s->classes[0].first; vid; vid = v->qdag.uf[0].class_link.next)
    {
      v = VARID2VARPTR (vars, vid);
      assert (UF_IS_REP (v, 0));
      assert (QDPLL_SCOPE_FORALL (v->scope));

      /* If class is fully propagated (and hence assigned), then skip. */
      if (count_non_propagated_in_class (vars, v) == 0)
	continue;

      /* Mark and push d-edges on traversal stack. */
      unsigned int i, end = v->qdag.dedges.size;
      Edge *d;
      for (i = 0; i < end; i++)
	for (d = v->qdag.dedges.table[i]; d; d = d->chain_next)
	  {
	    Var *dvar = VARID2VARPTR (vars, d->head_var);
	    assert (QDPLL_SCOPE_EXISTS (dvar->scope));
	    assert (UF_IS_REP (dvar, 0));
	    assert (v->scope->nesting < dvar->scope->nesting);

	    if (!dvar->qdag.mark_is_candidate_debug)
	      {
		dvar->qdag.mark_is_candidate_debug = 1;
		QDPLL_PUSH_STACK (mm, forest_succ, dvar);
	      }
	  }

      while (!QDPLL_EMPTY_STACK (forest_succ))
	{
	  Var *succ = QDPLL_POP_STACK (forest_succ);
	  assert (QDPLL_SCOPE_EXISTS (succ->scope));
	  assert (v->scope->nesting < succ->scope->nesting);
	  assert (UF_IS_REP (succ, 0));
	  assert (succ->qdag.mark_is_candidate_debug);

	  if (!UF_IS_SINGLETON_SET (succ, 0))
	    {
	      /* Mark class members. */
	      Var *m;
	      VarID mid;
	      for (mid = succ->qdag.uf[0].members.list.first; mid;
		   mid = m->qdag.uf[0].members.link.next)
		{
		  m = VARID2VARPTR (vars, mid);
		  m->qdag.mark_is_candidate_debug = 1;
		}
	    }

	  /* Mark and push all c-children not yet marked. */
	  Var *c;
	  VarID cid;
	  for (cid = succ->qdag.cedges.cchilds.first; cid;
	       cid = c->qdag.cedges.csibs.next)
	    {
	      c = VARID2VARPTR (vars, cid);
	      if (!c->qdag.mark_is_candidate_debug)
		{
		  c->qdag.mark_is_candidate_debug = 1;
		  QDPLL_PUSH_STACK (mm, forest_succ, c);
		}
	    }
	}
    }

  QDPLL_DELETE_STACK (mm, forest_succ);
}


static void
assert_candidate_marks_by_remarking (QDPLLDepManQDAG * dm)
{
  assert (dm->state.init);
  Var *vars = dm->pcnf->vars;

  Var *p, *e;
  for (p = vars, e = vars + dm->pcnf->size_vars; p < e; p++)
    assert (!p->qdag.mark_is_candidate_debug);

  Scope *s;
  for (s = dm->pcnf->scopes.first; s; s = s->link.next)
    {
      if (QDPLL_SCOPE_EXISTS (s))
	assert_mark_non_candidates_from_exists (dm, s);
      else
	{
	  assert (QDPLL_SCOPE_FORALL (s));
	  assert_mark_non_candidates_from_forall (dm, s);
	}
    }

  /* Compare marks for all variables. */
  for (p = vars, e = vars + dm->pcnf->size_vars; p < e; p++)
    {
      if (p->id)
	{
	  if (QDPLL_VAR_MARKED_PROPAGATED (p) && p->decision_level <= 0)
	    {
	      assert (!p->qdag.mark_is_candidate_debug);
	      continue;
	    }
	  assert (!p->qdag.mark_is_candidate_debug || !p->mark_is_candidate);
	  assert (p->mark_is_candidate || p->qdag.mark_is_candidate_debug);
	  p->qdag.mark_is_candidate_debug = 0;
	}
    }
}

static unsigned int count_direct_active_refs_by_sedge (Var * vars, Var * var);

static void
assert_inactive_sedge_frontier_by_ancestors (QDPLLDepManQDAG * dm, Var * var)
{
  assert (var);
  assert (QDPLL_SCOPE_EXISTS (var->scope));
  assert (UF_IS_REP (var, 0));
  Var *vars = dm->pcnf->vars;

  VarID parid = var->id;
  Var *par;
  while (parid)
    {
      par = VARID2VARPTR (vars, parid);
      assert (QDPLL_SCOPE_EXISTS (par->scope));
      assert (UF_IS_REP (par, 0));
      assert (par->qdag.cnt.exists.inactive_sedge_frontier);
      assert (par->qdag.cnt.exists.active_direct_refs_by_sedge == 0);
      assert (count_direct_active_refs_by_sedge (vars, par) == 0);

      Edge **p, **e;
      for (p = par->qdag.dedge_pq.elems_start,
	   e = par->qdag.dedge_pq.elems_top; p < e; p++)
	{
	  Var *u = VARID2VARPTR (vars, (*p)->tail_var);
	  assert (QDPLL_SCOPE_FORALL (u->scope));
	  assert (UF_IS_REP (u, 0));
	}

      parid = par->qdag.cedges.cpar;
    }
}

static int qdpll_dep_man_depends (QDPLLDepManGeneric * dmg, VarID x, VarID y);

static void
assert_check_deps_by_functions_in_clause (QDPLLDepManQDAG * dm,
					  Constraint * c)
{
  assert (!c->is_cube);
  Var *vars = dm->pcnf->vars;

  LitID *p, *tmp, *e = c->lits + c->num_lits - 1;
  e = get_largest_active_literal_in_clause (vars, c->lits, e, e, 0);
  for (p = c->lits; p <= e; p++)
    {
      LitID lit = *p;
      Var *var = LIT2VARPTR (vars, lit);
      assert (!
	      (QDPLL_VAR_MARKED_PROPAGATED (var)
	       && var->decision_level <= 0)
	      || (((QDPLL_LIT_NEG (lit) && QDPLL_VAR_ASSIGNED_TRUE (var)) ||
		   (QDPLL_LIT_POS (lit) && QDPLL_VAR_ASSIGNED_FALSE (var)))));
      if (QDPLL_VAR_MARKED_PROPAGATED (var) && var->decision_level <= 0)
	continue;
      for (tmp = c->lits; tmp <= e; tmp++)
	{
	  LitID lit_tmp = *tmp;
	  Var *var_tmp = LIT2VARPTR (vars, lit_tmp);
	  assert (!
		  (QDPLL_VAR_MARKED_PROPAGATED (var_tmp)
		   && var_tmp->decision_level <= 0)
		  || (((QDPLL_LIT_NEG (lit_tmp)
			&& QDPLL_VAR_ASSIGNED_TRUE (var_tmp))
		       || (QDPLL_LIT_POS (lit_tmp)
			   && QDPLL_VAR_ASSIGNED_FALSE (var_tmp)))));
	  if (QDPLL_VAR_MARKED_PROPAGATED (var_tmp)
	      && var_tmp->decision_level <= 0)
	    continue;
	  if (tmp <= p || var->scope->type == var_tmp->scope->type)
	    assert (!qdpll_dep_man_depends
		    ((QDPLLDepManGeneric *) dm, var->id, var_tmp->id));
	  else
	    assert (qdpll_dep_man_depends
		    ((QDPLLDepManGeneric *) dm, var->id, var_tmp->id));
	}
    }
}


static void
assert_check_dependencies_by_functions (QDPLLDepManQDAG * dm)
{
  Constraint *c;
  for (c = dm->pcnf->clauses.first; c; c = c->link.next)
    {
      assert (!c->is_cube);
      assert (!c->learnt);
      if (!c->disabled)
	assert_check_deps_by_functions_in_clause (dm, c);
    }
}

#endif


/* -------------------- END: ASSERTION-ONLY CODE -------------------- */


/* -------------------- START: INTERNAL FUNCTIONS -------------------- */

static void
delete_edge (QDPLLMemMan * mm, Edge * d)
{
  qdpll_free (mm, d, sizeof (Edge));
}


static Edge *
create_edge (QDPLLMemMan * mm)
{
  Edge *result;
  size_t bytes = sizeof (Edge);
  result = (Edge *) qdpll_malloc (mm, bytes);
  return result;
}

#define HAS_EDGE(from, to, table) (et_lookup (&((from)->table), to))

/* Re-link all c-children of 'oldpar' to 'newpar', set pointers
   appropriately.  This is required when existential classes are
   merged during insertion of c-edges. */
static void
fix_cchilds (Var * vars, Var * oldpar, Var * newpar, Var * oldparpar)
{
  Var *c, *n;
  VarID cid, nid;
  for (cid = (oldpar)->qdag.cedges.cchilds.first; cid; cid = nid)
    {
      c = VARID2VARPTR (vars, cid);
      nid = c->qdag.cedges.csibs.next;
      VAR_UNLINK (vars, oldpar->qdag.cedges.cchilds, c, qdag.cedges.csibs);
      VAR_LINK (vars, newpar->qdag.cedges.cchilds, c, qdag.cedges.csibs);
      c->qdag.cedges.cpar = newpar->id;
    }
  if (oldparpar)
    {
      VAR_UNLINK (vars, oldparpar->qdag.cedges.cchilds, oldpar,
		  qdag.cedges.csibs);
      oldpar->qdag.cedges.cpar = 0;
    }
}


#define SWAP(v1, v2)				\
  do {						\
    Var *tmp;					\
    tmp = v1;					\
    v1 = v2;					\
    v2 = tmp;					\
  } while (0)


#define FIX_EDGE_PQS(vars, from, to, table, pq)				\
  do {									\
    assert (from->id != to);						\
    Edge *e;								\
    Var *tail_var;							\
    while ((e = pq_remove_one (&(from)->pq)))				\
      {									\
	tail_var = VARID2VARPTR(vars, e->tail_var);			\
	et_remove (&(tail_var->table), from->id);			\
	assert (tail_var->scope->nesting == e->priority);		\
    	assert (e->head_var == from->id);				\
	assert (e->priority < from->scope->nesting);			\
	assert (e->priority < VARID2VARPTR(vars,			\
					   (to))->scope->nesting);	\
									\
	if (!HAS_EDGE(tail_var, to, table))				\
	  {								\
	    e->head_var = to;						\
	    pq_insert (mm, &(VARID2VARPTR(vars, (to))->pq), e,		\
		       e->priority);					\
	    et_insert(mm, &(tail_var->table), e);			\
	  }								\
	else								\
	  {								\
	    delete_edge(mm, e);						\
	  }								\
      }									\
  } while (0)


#define UPSHIFT_EDGES(vars, v, table, pq)				\
  do {									\
    assert (UF_IS_REP (v,0));						\
    Edge *e;								\
    Var *cpar, *cur, *prev, *edge_from, *v_tmp = v;			\
    VarID cparid;							\
    unsigned int en;							\
    while ((cparid = v_tmp->qdag.cedges.cpar) &&			\
	   (cpar = VARID2VARPTR(vars, cparid)))				\
      {									\
	assert (!(e = pq_access_min(&(v_tmp->pq)))			\
		|| (en = VARID2VARPTR(vars,				\
				      e->tail_var)->scope->nesting) ==	\
		e->priority);						\
									\
	while ((e = pq_access_min (&(v_tmp->pq))) &&			\
	       (en = e->priority) < cpar->scope->nesting)		\
	  {								\
	    e = pq_remove_min (&(v_tmp->pq));				\
	    edge_from = VARID2VARPTR(vars, e->tail_var);		\
	    assert (v_tmp == VARID2VARPTR(vars, e->head_var));		\
	    assert (edge_from->scope->nesting == en);			\
	    et_remove (&(edge_from->table), v_tmp->id);			\
									\
	    cur = prev = cpar;						\
									\
	    while ((cparid = cur->qdag.cedges.cpar) &&			\
		   (cur = VARID2VARPTR(vars, cparid)) &&		\
		   en < cur->scope->nesting)				\
	      {								\
		prev = cur;						\
	      }								\
									\
	    if (!HAS_EDGE (edge_from, prev->id, table))			\
	      {								\
		e->head_var = prev->id;					\
		pq_insert (mm, &(prev->pq), e, en);			\
		et_insert (mm, &(edge_from->table), e);			\
	      }								\
	    else							\
	      {								\
		/* delete d-edge (is already removed from pq) */	\
		delete_edge (mm, e);					\
	      }								\
	  }								\
	v_tmp = cpar;							\
      }									\
  } while (0)


static Var *
balance (Var * vars, Var * v1, Var ** v2)
{
  Var *p, *b, *prev;
  prev = b = *v2;
  const unsigned int nesting = v1->scope->nesting;
  assert (nesting <= b->scope->nesting);
  VarID cpar;

  while ((cpar = b->qdag.cedges.cpar) && (p = VARID2VARPTR (vars, cpar)) &&
	 p->scope->nesting >= nesting)
    {
      prev = b;
      b = p;
    }

  *v2 = b;

  assert (v1);
  assert (*v2);
  assert ((v1)->scope->nesting <= (*v2)->scope->nesting);
  assert (!(*v2)->qdag.cedges.cpar ||
	  VARID2VARPTR (vars,
			(*v2)->qdag.cedges.cpar)->scope->nesting <
	  (v1)->scope->nesting);

  assert (prev == *v2 || prev->scope->nesting > (*v2)->scope->nesting);

  return prev;
}


/* Insertion of either d-edge or s-edge from variable 'from' to
   variable 'to'.  Edges in c-forest are handled in function
   'insert_c_edge'.*/
static void
insert_edge (QDPLLDepManQDAG * dm, Var * from, Var * to)
{
  Var *vars = dm->pcnf->vars;
  QDPLLMemMan *mm = dm->mm;
  assert (QDPLL_SCOPE_EXISTS (to->scope));
#ifndef NDEBUG
  int inserting_dedge = QDPLL_SCOPE_FORALL (from->scope);
#endif
  assert (!inserting_dedge || QDPLL_SCOPE_FORALL (from->scope));
  assert (inserting_dedge || QDPLL_SCOPE_EXISTS (from->scope));
  assert (UF_IS_REP (to, 0));
  assert (!inserting_dedge
	  || (UF_IS_REP (from, 0) && UF_IS_SINGLETON_SET (from, 0)));
  assert (!inserting_dedge || from->scope->type != to->scope->type);
  assert (from->scope->nesting < to->scope->nesting);

  if (QDPLL_SCOPE_FORALL (from->scope))
    balance (vars, from, &to);
  else
    to = balance (vars, from, &to);

  assert (!inserting_dedge || from->scope->type != to->scope->type);
  assert (from->scope->nesting < to->scope->nesting);

  EdgeTable *et;
  EdgePriorityQueue *pq;
  if (QDPLL_SCOPE_FORALL (from->scope))
    {
      /* Inserting a d-edge. */
      et = &(from->qdag.dedges);
      pq = &(to->qdag.dedge_pq);
    }
  else
    {
      /* Inserting an s-edge. */
      et = &(from->qdag.sedges);
      pq = &(to->qdag.sedge_pq);
    }
  assert (!inserting_dedge || et == &(from->qdag.dedges));
  assert (!inserting_dedge || pq == &(to->qdag.dedge_pq));
  assert (inserting_dedge || et == &(from->qdag.sedges));
  assert (inserting_dedge || pq == &(to->qdag.sedge_pq));

  if (!et_lookup (et, to->id))
    {
      Edge *edge;
      edge = create_edge (mm);
      edge->tail_var = from->id;
      edge->head_var = to->id;
      pq_insert (mm, pq, edge, from->scope->nesting);
      et_insert (mm, et, edge);
    }
}


static void
insert_c_edge (QDPLLMemMan * mm, Var * vars, Var * from, Var * to)
{
#ifndef NDEBUG
  assert (UF_IS_REP (from, 0));
  assert (UF_IS_REP (to, 0));
  assert (QDPLL_SCOPE_EXISTS (from->scope));
  assert (QDPLL_SCOPE_EXISTS (to->scope));
  assert (from->scope->type == to->scope->type);
  assert (from->scope->nesting <= to->scope->nesting);
  Var *old_to = to;
#if QDAG_ASSERT_INSERT_C_EDGE_INTEGRITY_BEFORE
  assert_c_edges_integrity (vars, to);
  assert_c_edges_integrity (vars, from);
#endif
#endif

  Var *start_to;

  /* check if 'from' is predecessor of 'to' */
  balance (vars, from, &to);
  if (from == to)
    return;

  start_to = to;

  VarID next_to_id, tmp1_id;
  Var *next_to, *tmp1;
  while (from)
    {
      assert (from->scope->nesting <= to->scope->nesting);
      assert (UF_IS_REP (from, 0));
      assert (UF_IS_REP (to, 0));
      assert (from->scope->type == to->scope->type);

      balance (vars, from, &to);
      assert (from->scope->nesting <= to->scope->nesting);

      if (from == to)
	break;

#ifndef NDEBUG
#if QDAG_ASSERT_INSERT_C_EDGE_CCHILDS
      assert_insert_c_edge_cchilds (vars, from, to);
#endif
#endif

      next_to = (next_to_id = to->qdag.cedges.cpar) ?
	VARID2VARPTR (vars, next_to_id) : 0;

      if (from->scope == to->scope)
	{
	  tmp1 = (tmp1_id = from->qdag.cedges.cpar) ?
	    VARID2VARPTR (vars, tmp1_id) : 0;

	  uf_unite (vars, from, to, 0);
	  assert ((!UF_IS_REP (from, 0) && UF_IS_REP (to, 0)) ||
		  (UF_IS_REP (from, 0) && !UF_IS_REP (to, 0)));

	  if (UF_IS_REP (from, 0))
	    {
	      fix_cchilds (vars, to, from, next_to);
	      FIX_EDGE_PQS (vars, to, from->id, qdag.dedges, qdag.dedge_pq);
	      FIX_EDGE_PQS (vars, to, from->id, qdag.sedges, qdag.sedge_pq);
	    }
	  else
	    {
	      assert (UF_IS_REP (to, 0));
	      fix_cchilds (vars, from, to, tmp1);
	      FIX_EDGE_PQS (vars, from, to->id, qdag.dedges, qdag.dedge_pq);
	      FIX_EDGE_PQS (vars, from, to->id, qdag.sedges, qdag.sedge_pq);

	      next_to = tmp1;
	      SWAP (from, to);
	    }
	}
      else
	{
	  assert (from->scope->nesting < to->scope->nesting);
	  if (next_to)
	    {
	      assert (VARID2VARPTR (vars, to->qdag.cedges.cpar) == next_to);
	      VAR_UNLINK (vars, next_to->qdag.cedges.cchilds, to,
			  qdag.cedges.csibs);
	    }
	  else
	    assert (!to->qdag.cedges.csibs.next
		    && !to->qdag.cedges.csibs.prev);
	  to->qdag.cedges.cpar = from->id;
	  VAR_LINK (vars, from->qdag.cedges.cchilds, to, qdag.cedges.csibs);
	}

#ifndef NDEBUG
#if QDAG_ASSERT_INSERT_C_EDGE_CCHILDS
      assert_insert_c_edge_cchilds (vars, from, to);
#endif
#endif

      to = from;
      from = next_to;
    }

  assert (start_to->qdag.uf[0].par);
  start_to = uf_find (vars, start_to, 0);
  UPSHIFT_EDGES (vars, start_to, qdag.dedges, qdag.dedge_pq);
  UPSHIFT_EDGES (vars, start_to, qdag.sedges, qdag.sedge_pq);
#ifndef NDEBUG
#if QDAG_ASSERT_INSERT_C_EDGE_INTEGRITY_AFTER
  assert_c_edges_integrity (vars, uf_find (vars, old_to, 0));
#endif
#endif
}

static LitID *
find_next_lit_from_other_scope (Var * vars, LitID * lits, LitID * lit,
				unsigned int ignore_dlevel)
{
  assert (lit);
  assert (lits <= lit);

  Var *var = 0;
  Scope *s = LIT2VARPTR (vars, (*lit))->scope;
  for (lit = lit - 1;
       lits <= lit &&
       ((var = LIT2VARPTR (vars, (*lit)))->scope == s ||
	(QDPLL_VAR_MARKED_PROPAGATED (var)
	 && var->decision_level <= ignore_dlevel)); lit--);

  return lit;
}


static LitID *
find_next_unpropagated_lit (Var * vars, int move_left, LitID * border,
			    LitID * cur_p, unsigned int ignore_dlevel)
{
  /* Must skip propagated variables here. */
  assert ((move_left && border <= cur_p) || (!move_left && cur_p <= border));
  Var *cur_var = LIT2VARPTR (vars, *cur_p);
  while (QDPLL_VAR_MARKED_PROPAGATED (cur_var)
	 && cur_var->decision_level <= ignore_dlevel)
    {
      /* Any propagated literal must be assigned false as unit or pure
	 at top-level. */
      assert (((QDPLL_LIT_NEG (*cur_p) && QDPLL_VAR_ASSIGNED_TRUE (cur_var))
	       || (QDPLL_LIT_POS (*cur_p)
		   && QDPLL_VAR_ASSIGNED_FALSE (cur_var))));
      if (move_left)
	cur_p--;
      else
	cur_p++;
      if ((move_left && cur_p < border) || (!move_left && border < cur_p))
	return cur_p;
      cur_var = LIT2VARPTR (vars, *cur_p);
    }
  return cur_p;
}


static void
extract_dependencies_from_clause (QDPLLDepManQDAG * dm, Constraint * c)
{
  assert (!c->is_cube);
  assert (c->num_lits);
  assert (!c->disabled);
  QDPLLMemMan *mm = dm->mm;
  Var *vars = dm->pcnf->vars;

  /* Unit clauses do not generate dependencies, can return immediately. */
  if (c->num_lits == 1)
    return;

  unsigned int ignore_dlevel = 0;
  LitID *leftlit_p, *rightlit_p, *tmp_p, *last_e_p = 0, *prev_p = 0, *lits =
    c->lits;

  /* Start at last (biggest) literal of clause. */
  rightlit_p = lits + c->num_lits - 1;
  assert (lits <= rightlit_p);
  assert (QDPLL_SCOPE_EXISTS (LIT2VARPTR (vars, *rightlit_p)->scope));
  rightlit_p = get_largest_active_literal_in_clause (vars, lits,
						     rightlit_p, rightlit_p,
						     ignore_dlevel);
  assert (lits <= rightlit_p);
  assert (QDPLL_SCOPE_EXISTS (LIT2VARPTR (vars, *rightlit_p)->scope));
  assert (!(QDPLL_VAR_MARKED_PROPAGATED (LIT2VARPTR (vars, *rightlit_p)) &&
	    LIT2VARPTR (vars, *rightlit_p)->decision_level <= ignore_dlevel));

  /* Extraction loop. */
  while (lits <= rightlit_p)
    {
      leftlit_p =
	find_next_lit_from_other_scope (vars, lits, rightlit_p,
					ignore_dlevel);
      assert (leftlit_p < rightlit_p);
      assert (lits <= leftlit_p || leftlit_p == lits - 1);
      assert (leftlit_p < lits
	      || (LIT2VARPTR (vars, *leftlit_p)->scope->nesting <
		  LIT2VARPTR (vars, *rightlit_p)->scope->nesting &&
		  !(QDPLL_VAR_MARKED_PROPAGATED
		    (LIT2VARPTR (vars, *leftlit_p))
		    && LIT2VARPTR (vars,
				   *leftlit_p)->decision_level <=
		    ignore_dlevel)));

      if (QDPLL_SCOPE_EXISTS (LIT2VARPTR (vars, *rightlit_p)->scope))
	{
	  tmp_p = rightlit_p - 1;
	  if (lits <= tmp_p)
	    tmp_p =
	      find_next_unpropagated_lit (vars, 1, lits, tmp_p,
					  ignore_dlevel);
	  /* Unite adjacent existential literals from same scope. */
	  while (leftlit_p < tmp_p)
	    {
	      assert (lits <= tmp_p);
	      assert (!
		      (QDPLL_VAR_MARKED_PROPAGATED (LIT2VARPTR (vars, *tmp_p))
		       && LIT2VARPTR (vars,
				      *tmp_p)->decision_level <=
		       ignore_dlevel));
	      assert (!
		      (QDPLL_VAR_MARKED_PROPAGATED
		       (LIT2VARPTR (vars, *rightlit_p))
		       && LIT2VARPTR (vars,
				      *rightlit_p)->decision_level <=
		       ignore_dlevel));
	      Var *rep_tmp = uf_find (vars, LIT2VARPTR (vars, *tmp_p), 0);
	      Var *rep_right =
		uf_find (vars, LIT2VARPTR (vars, *rightlit_p), 0);
	      assert (!
		      (QDPLL_VAR_MARKED_PROPAGATED (rep_tmp)
		       && rep_tmp->decision_level <= ignore_dlevel));
	      assert (!
		      (QDPLL_VAR_MARKED_PROPAGATED (rep_right)
		       && rep_right->decision_level <= ignore_dlevel));

#ifndef NDEBUG
#if QDAG_ASSERT_EXTRACT_DEPS_INSERT_C_EDGE_BEFORE
	      assert_insert_c_edge_cchilds (vars, LIT2VARPTR (vars, *tmp_p),
					    LIT2VARPTR (vars, *rightlit_p));
	      assert_insert_c_edge_cchilds (vars, rep_tmp, rep_right);
#endif
#endif
	      insert_c_edge (mm, vars, rep_tmp, rep_right);

#ifndef NDEBUG
#if QDAG_ASSERT_EXTRACT_DEPS_INSERT_C_EDGE_AFTER
	      assert_insert_c_edge_cchilds (vars, uf_find (vars, rep_tmp, 0),
					    uf_find (vars, rep_right, 0));
#endif
#endif
	      rightlit_p = tmp_p--;
	      if (lits <= tmp_p)
		tmp_p =
		  find_next_unpropagated_lit (vars, 1, lits, tmp_p,
					      ignore_dlevel);
	    }
	}
      else
	{
	  rightlit_p = leftlit_p + 1;
	  rightlit_p =
	    find_next_unpropagated_lit (vars, 0, c->lits + c->num_lits - 1,
					rightlit_p, ignore_dlevel);
	}
      assert (!last_e_p
	      || !(QDPLL_VAR_MARKED_PROPAGATED (LIT2VARPTR (vars, *last_e_p))
		   && LIT2VARPTR (vars,
				  *last_e_p)->decision_level <=
		   ignore_dlevel));
      assert (!last_e_p
	      ||
	      !(QDPLL_VAR_MARKED_PROPAGATED
		(uf_find (vars, LIT2VARPTR (vars, *last_e_p), 0))
		&& uf_find (vars,
			    LIT2VARPTR (vars,
					*last_e_p), 0)->decision_level <=
		ignore_dlevel));

      if (QDPLL_SCOPE_EXISTS (LIT2VARPTR (vars, *rightlit_p)->scope))
	{
	  assert (!
		  (QDPLL_VAR_MARKED_PROPAGATED
		   (LIT2VARPTR (vars, *rightlit_p))
		   && LIT2VARPTR (vars,
				  *rightlit_p)->decision_level <=
		   ignore_dlevel));
	  if (last_e_p)
	    {
	      assert (!
		      (QDPLL_VAR_MARKED_PROPAGATED
		       (uf_find (vars, LIT2VARPTR (vars, *rightlit_p), 0))
		       && uf_find (vars,
				   LIT2VARPTR (vars,
					       *rightlit_p), 0)->
		       decision_level <= ignore_dlevel));
	      insert_c_edge (mm, vars,
			     uf_find (vars, LIT2VARPTR (vars, *rightlit_p),
				      0), uf_find (vars, LIT2VARPTR (vars,
								     *last_e_p),
						   0));

	      tmp_p = rightlit_p;
	      assert (tmp_p < prev_p);
	      do
		{
		  assert (!
			  (QDPLL_VAR_MARKED_PROPAGATED
			   (LIT2VARPTR (vars, *tmp_p))
			   && LIT2VARPTR (vars,
					  *tmp_p)->decision_level <=
			   ignore_dlevel));
		  insert_edge (dm, LIT2VARPTR (vars, *tmp_p),
			       uf_find (vars, LIT2VARPTR (vars, *last_e_p),
					0));
		  tmp_p++;
		  tmp_p =
		    find_next_unpropagated_lit (vars, 0, prev_p, tmp_p,
						ignore_dlevel);
		}
	      while (tmp_p < prev_p);

	    }
	  last_e_p = rightlit_p;
	}
      else
	{
	  assert (QDPLL_SCOPE_FORALL (LIT2VARPTR (vars, *rightlit_p)->scope));
	  assert (last_e_p);
	  rightlit_p =
	    find_next_unpropagated_lit (vars, 0, prev_p, rightlit_p,
					ignore_dlevel);
	  tmp_p = rightlit_p;
	  assert (tmp_p < prev_p);
	  do
	    {
	      assert (!
		      (QDPLL_VAR_MARKED_PROPAGATED (LIT2VARPTR (vars, *tmp_p))
		       && LIT2VARPTR (vars,
				      *tmp_p)->decision_level <=
		       ignore_dlevel));
	      insert_edge (dm, LIT2VARPTR (vars, *tmp_p),
			   uf_find (vars, LIT2VARPTR (vars, *last_e_p), 0));
	      tmp_p++;
	      tmp_p =
		find_next_unpropagated_lit (vars, 0, prev_p, tmp_p,
					    ignore_dlevel);
	    }
	  while (tmp_p < prev_p);
	}

      prev_p = rightlit_p;
      rightlit_p = leftlit_p;
    }
}


static int
remove_transitivities (QDPLLDepManQDAG * dm)
{
  Var *vars = dm->pcnf->vars;
  QDPLLMemMan *mm = dm->mm;
  int removed_edges = 0;

  VarPtrStack stack;
  VarPtrStack marks;
  QDPLL_INIT_STACK (stack);
  QDPLL_INIT_STACK (marks);

#ifndef NDEBUG
#if QDAG_ASSERT_REMOVE_TRANSITIVITIES_VARS_UNMARKED
  /* Check that qdag-marks are all set to 0. */
  do
    {
      Var *p, *e;
      for (p = vars, e = vars + dm->pcnf->size_vars; p < e; p++)
	{
	  assert (!p->qdag.mark0);
	  assert (!p->qdag.mark1);
	}
    }
  while (0);
#endif
#endif

  Scope *s;
  for (s = dm->pcnf->scopes.first; s; s = s->link.next)
    {
      if (QDPLL_SCOPE_FORALL (s))
	continue;
      assert (QDPLL_SCOPE_EXISTS (s));

      /* Check all existential variables. */
      VarID *vidp, *vide, vid;
      for (vidp = s->vars.start, vide = s->vars.top; vidp < vide; vidp++)
	{
	  vid = *vidp;
	  Var *v = VARID2VARPTR (vars, vid);
	  assert (QDPLL_SCOPE_EXISTS (v->scope));
	  if (QDPLL_VAR_MARKED_PROPAGATED (v) && v->decision_level <= 0)
	    continue;

	  /* Check if dependency 'vtd_tmp_var' of variable 'v' is
	     reachable from other dependencies of 'v' by following
	     dependency pointers.  If so, then remove 'vtd_tmp_var',
	     which is transitive in this case. */
	  Edge *vtd_tmp, *vtd_tmp_next;
	  Var *vtd_tmp_var;
	  VarID vtd_tmp_varid;
	  unsigned int i, end;
	  end = v->qdag.dedges.size;
	  for (i = 0; i < end; i++)
	    for (vtd_tmp = v->qdag.dedges.table[i]; vtd_tmp;
		 vtd_tmp = vtd_tmp_next)
	      {
		vtd_tmp_next = vtd_tmp->chain_next;
		assert (VARID2VARPTR (vars, vtd_tmp->tail_var) == v);
		vtd_tmp_varid = vtd_tmp->head_var;
		vtd_tmp_var = VARID2VARPTR (vars, vtd_tmp_varid);
		unsigned int vtd_tmp_var_nesting =
		  vtd_tmp_var->scope->nesting;
		assert (UF_IS_REP (vtd_tmp_var, 0));
		assert (QDPLL_SCOPE_FORALL (vtd_tmp_var->scope));

		/* Push other dependencies of 'v' on traversal stack. */
		Edge *d;
		unsigned int j;
		for (j = 0; j < end; j++)
		  for (d = v->qdag.dedges.table[j]; d; d = d->chain_next)
		    {
		      if (d == vtd_tmp)
			continue;

		      Var *dvar = VARID2VARPTR (vars, d->head_var);
		      assert (QDPLL_SCOPE_FORALL (dvar->scope));
		      assert (UF_IS_REP (dvar, 0));
		      /* Ignore variables larger than 'vtd_tmp_var', 
		         can not reach 'vtd_tmp_var' from such variables. */
		      if (dvar->scope->nesting < vtd_tmp_var_nesting)
			{
			  QDPLL_PUSH_STACK (mm, stack, dvar);
			  assert (!dvar->qdag.mark0);
			  dvar->qdag.mark0 = 1;
			  QDPLL_PUSH_STACK (mm, marks, dvar);
			}
		    }

		/* Follow dependency pointers on stack, 
		   check if 'vtd_tmp_var' is reachable. */
		while (!QDPLL_EMPTY_STACK (stack))
		  {
		    Var *tmp = QDPLL_POP_STACK (stack);
		    assert (tmp->qdag.mark0);
		    assert (QDPLL_SCOPE_EXISTS (tmp->scope)
			    || UF_IS_REP (tmp, 0));
		    assert (tmp->scope->nesting < vtd_tmp_var_nesting);

		    if (QDPLL_SCOPE_EXISTS (tmp->scope)
			&& HAS_EDGE (tmp, vtd_tmp_varid, qdag.dedges))
		      {
			/* Remove edge from 'v' to 'vtd_tmp_var'. */
#ifndef NDEBUG
			Edge *del =
			  pq_remove_elem (&(vtd_tmp_var->qdag.dedge_pq),
					  vtd_tmp->pos);
			assert (del == vtd_tmp);
			assert (del->tail_var == vid);
			assert (del->head_var == vtd_tmp_varid);
#else
			pq_remove_elem (&(vtd_tmp_var->qdag.dedge_pq),
					vtd_tmp->pos);
#endif
			et_remove (&(v->qdag.dedges), vtd_tmp_varid);
			delete_edge (mm, vtd_tmp);
			removed_edges++;

			QDPLL_RESET_STACK (stack);
			/* While-loop will break subsequently. */
		      }
		    else
		      {
			/* Push and mark dependency pointers of 'tmp'. Must also 
			   handle existential class members. */
			Edge *d;
			unsigned int i, end = tmp->qdag.dedges.size;
			for (i = 0; i < end; i++)
			  for (d = tmp->qdag.dedges.table[i]; d;
			       d = d->chain_next)
			    {
			      Var *dvar = VARID2VARPTR (vars, d->head_var);
			      assert (UF_IS_REP (dvar, 0));
			      assert (dvar != vtd_tmp_var);

			      /* Ignore marked variables and such that
			         are larger than 'vtd_tmp_var'. */
			      if (dvar->qdag.mark0
				  || dvar->scope->nesting >=
				  vtd_tmp_var_nesting)
				continue;

			      dvar->qdag.mark0 = 1;
			      QDPLL_PUSH_STACK (mm, marks, dvar);
			      QDPLL_PUSH_STACK (mm, stack, dvar);

			      /* Push dependencies of existential
			         class members. */
			      if (QDPLL_SCOPE_EXISTS (dvar->scope)
				  && !UF_IS_SINGLETON_SET (dvar, 0))
				{
				  VarID mid;
				  Var *m;
				  for (mid =
				       dvar->qdag.uf[0].members.list.first;
				       mid;
				       mid = m->qdag.uf[0].members.link.next)
				    {
				      m = VARID2VARPTR (vars, mid);
				      assert (!m->qdag.mark0);
				      assert (m->scope->nesting <
					      vtd_tmp_var_nesting);
				      m->qdag.mark0 = 1;
				      QDPLL_PUSH_STACK (mm, marks, m);
				      QDPLL_PUSH_STACK (mm, stack, m);
				    }
				}
			    }
		      }
		  }
		/* Unmark variables, reset mark stack. */
		while (!QDPLL_EMPTY_STACK (marks))
		  {
		    Var *tmp = QDPLL_POP_STACK (marks);
		    assert (tmp->qdag.mark0);
		    tmp->qdag.mark0 = 0;
		  }
		assert (marks.top == marks.start);
		assert (stack.top == stack.start);
	      }
	}
    }


  QDPLL_DELETE_STACK (mm, stack);
  QDPLL_DELETE_STACK (mm, marks);

#ifndef NDEBUG
#if QDAG_ASSERT_REMOVE_TRANSITIVITIES_VARS_UNMARKED
  /* Check that qdag-marks are all set to 0. */
  do
    {
      Var *p, *e;
      for (p = vars, e = vars + dm->pcnf->size_vars; p < e; p++)
	{
	  assert (!p->qdag.mark0);
	  assert (!p->qdag.mark1);
	}
    }
  while (0);
#endif
#endif

  return removed_edges;
}


static int
var_edges_subseteq (Var * v1, Var * v2)
{
  assert (v1->scope->type == v2->scope->type);
  assert (v1->scope->nesting == v2->scope->nesting);
  assert (QDPLL_SCOPE_EXISTS (v1->scope) || UF_IS_REP (v1, 0));
  assert (QDPLL_SCOPE_EXISTS (v2->scope) || UF_IS_REP (v2, 0));
  EdgeTable *t1, *t2;

  if (QDPLL_SCOPE_FORALL (v1->scope))
    {
      t1 = &(v1->qdag.dedges);
      t2 = &(v2->qdag.dedges);
    }
  else
    {
      assert (QDPLL_SCOPE_EXISTS (v1->scope));
      t1 = &(v1->qdag.sedges);
      t2 = &(v2->qdag.sedges);
    }

  if (t1->count > t2->count)
    return 0;

  int result, i, end = t1->size;
  Edge *d;
  for (i = 0; i < end; i++)
    for (d = t1->table[i]; d; d = d->chain_next)
      {
	if (!et_lookup (t2, d->head_var))
	  return 0;
      }

  return 1;
}


#ifndef NDEBUG
static void
assert_merged_univ_vars (Var * vars, Scope * s)
{
  assert (QDPLL_SCOPE_FORALL (s));
  Var *v1, *v2;
  VarID vid1, vid2;
  for (vid1 = s->classes[0].first; vid1;
       vid1 = v1->qdag.uf[0].class_link.next)
    {
      v1 = VARID2VARPTR (vars, vid1);
      for (vid2 = v1->qdag.uf[0].class_link.next; vid2;
	   vid2 = v2->qdag.uf[0].class_link.next)
	{
	  v2 = VARID2VARPTR (vars, vid2);
	  assert (!var_edges_subseteq (v1, v2)
		  || !var_edges_subseteq (v2, v1));
	}
      if (!UF_IS_SINGLETON_SET (v1, 0))
	{
	  Var *m;
	  VarID mid;
	  for (mid = v1->qdag.uf[0].members.list.first; mid;
	       mid = m->qdag.uf[0].members.link.next)
	    {
	      m = VARID2VARPTR (vars, mid);
	      assert (m->qdag.dedges.count == 0);
	    }
	}
    }
}


static void
assert_merged_exist_vars (Var * vars, Scope * s)
{
  assert (QDPLL_SCOPE_EXISTS (s));
  Var *v1, *v2;
  VarID vid1, vid2;
  for (vid1 = s->classes[1].first; vid1;
       vid1 = v1->qdag.uf[1].class_link.next)
    {
      v1 = VARID2VARPTR (vars, vid1);
      for (vid2 = v1->qdag.uf[1].class_link.next; vid2;
	   vid2 = v2->qdag.uf[1].class_link.next)
	{
	  v2 = VARID2VARPTR (vars, vid2);
	  assert (!var_edges_subseteq (v1, v2)
		  || !var_edges_subseteq (v2, v1));
	}
      if (!UF_IS_SINGLETON_SET (v1, 1))
	{
	  Var *m;
	  VarID mid;
	  for (mid = v1->qdag.uf[1].members.list.first; mid;
	       mid = m->qdag.uf[1].members.link.next)
	    {
	      m = VARID2VARPTR (vars, mid);
	    }
	}
    }
}

#endif


static void
cleanup_non_rep_sedges (QDPLLMemMan * mm, Var * vars, Var * v)
{
  assert (!UF_IS_REP (v, 1));
  assert (QDPLL_SCOPE_EXISTS (v->scope));
  int i, end = v->qdag.sedges.size;
  Edge *d, *n;
  for (i = 0; i < end; i++)
    {
      for (d = v->qdag.sedges.table[i]; d; d = n)
	{
	  n = d->chain_next;
	  Var *h = VARID2VARPTR (vars, d->head_var);
	  assert (VARID2VARPTR (vars, d->tail_var) == v);
	  assert (QDPLL_SCOPE_EXISTS (h->scope));
	  assert (UF_IS_REP (h, 1));
	  assert (d->priority == v->scope->nesting);
#ifndef NDEBUG
	  Edge *tmp = pq_remove_elem (&(h->qdag.sedge_pq), d->pos);
	  assert (tmp == d);
#else
	  pq_remove_elem (&(h->qdag.sedge_pq), d->pos);
#endif
	  delete_edge (mm, d);
	}
      v->qdag.sedges.table[i] = 0;
    }

  v->qdag.sedges.count = 0;
}


static void
cleanup_non_rep_dedges (QDPLLMemMan * mm, Var * vars, Var * v)
{
  assert (!UF_IS_REP (v, 0));
  assert (QDPLL_SCOPE_FORALL (v->scope));
  int i, end = v->qdag.dedges.size;
  Edge *d, *n;
  for (i = 0; i < end; i++)
    {
      for (d = v->qdag.dedges.table[i]; d; d = n)
	{
	  n = d->chain_next;
	  Var *h = VARID2VARPTR (vars, d->head_var);
	  assert (VARID2VARPTR (vars, d->tail_var) == v);
	  assert (QDPLL_SCOPE_EXISTS (h->scope));
	  assert (UF_IS_REP (h, 0));
	  assert (d->priority == v->scope->nesting);
#ifndef NDEBUG
	  Edge *tmp = pq_remove_elem (&(h->qdag.dedge_pq), d->pos);
	  assert (tmp == d);
#else
	  pq_remove_elem (&(h->qdag.dedge_pq), d->pos);
#endif
	  delete_edge (mm, d);
	}
      v->qdag.dedges.table[i] = 0;
    }

  v->qdag.dedges.count = 0;
}


static void
merge_exist_scope_vars_by_sedges (QDPLLMemMan * mm, Var * vars, Scope * s)
{
  assert (QDPLL_SCOPE_EXISTS (s));
  Var *v1, *v2, *n1, *n2;
  VarID vid1, vidn1, vid2, vidn2;
  for (vid1 = s->classes[1].first; vid1; vid1 = vidn1)
    {
      v1 = VARID2VARPTR (vars, vid1);
      assert (UF_IS_REP (v1, 1));
      vidn1 = v1->qdag.uf[1].class_link.next;
      n1 = vidn1 ? VARID2VARPTR (vars, vidn1) : 0;
      for (vid2 = v1->qdag.uf[1].class_link.next; vid2; vid2 = vidn2)
	{
	  v2 = VARID2VARPTR (vars, vid2);
	  assert (UF_IS_REP (v2, 1));
	  vidn2 = v2->qdag.uf[1].class_link.next;
	  n2 = vidn2 ? VARID2VARPTR (vars, vidn2) : 0;
	  if (var_edges_subseteq (v1, v2) && var_edges_subseteq (v2, v1))
	    {
	      if (v2 == n1)
		{
		  vidn1 = vidn2;
		  n1 = n2;
		}
	      uf_unite (vars, v1, v2, 1);

	      if (UF_IS_REP (v1, 1))
		{

		}
	      else
		{
		  v1 = v2;
		}
	    }
	}
    }

#ifndef NDEBUG
#if QDAG_ASSERT_MERGED_UNIV_VARS
  assert_merged_exist_vars (vars, s);
#endif
#endif
}


static void
merge_univ_scope_vars_by_dedges (QDPLLMemMan * mm, Var * vars, Scope * s)
{
  assert (QDPLL_SCOPE_FORALL (s));
  Var *v1, *v2, *n1, *n2;
  VarID vid1, vidn1, vid2, vidn2;
  for (vid1 = s->classes[0].first; vid1; vid1 = vidn1)
    {
      v1 = VARID2VARPTR (vars, vid1);
      assert (UF_IS_REP (v1, 0));
      vidn1 = v1->qdag.uf[0].class_link.next;
      n1 = vidn1 ? VARID2VARPTR (vars, vidn1) : 0;
      for (vid2 = v1->qdag.uf[0].class_link.next; vid2; vid2 = vidn2)
	{
	  v2 = VARID2VARPTR (vars, vid2);
	  assert (UF_IS_REP (v2, 0));
	  vidn2 = v2->qdag.uf[0].class_link.next;
	  n2 = vidn2 ? VARID2VARPTR (vars, vidn2) : 0;
	  if (var_edges_subseteq (v1, v2) && var_edges_subseteq (v2, v1))
	    {
	      if (v2 == n1)
		{
		  vidn1 = vidn2;
		  n1 = n2;
		}
	      uf_unite (vars, v1, v2, 0);

	      if (UF_IS_REP (v1, 0))
		{
		  cleanup_non_rep_dedges (mm, vars, v2);
		}
	      else
		{
		  cleanup_non_rep_dedges (mm, vars, v1);
		  v1 = v2;
		}
	    }
	}
    }

#ifndef NDEBUG
#if QDAG_ASSERT_MERGED_UNIV_VARS
  assert_merged_univ_vars (vars, s);
#endif
#endif
}


static void
insert_existential_deps (QDPLLDepManQDAG * dm)
{
  Var *vars = dm->pcnf->vars;
  QDPLLMemMan *mm = dm->mm;

  Scope *s;
  for (s = dm->pcnf->scopes.last; s; s = s->link.prev)
    {
      if (QDPLL_SCOPE_EXISTS (s))
	{
	  VarID vid;
	  Var *v, *c, *insert;
	  for (vid = s->classes[0].first; vid;
	       vid = v->qdag.uf[0].class_link.next)
	    {
	      v = VARID2VARPTR (vars, vid);
	      insert = v;
	      VarID cid = v->id;
	      /* Check 'v' and all c-forest ancestors 'c' of 'v'. */
	      while (cid)
		{
		  c = VARID2VARPTR (vars, cid);

		  /* By moving the insert position, edges are 'chained' 
		     -> heuristically tries to avoid transitive edges. */
		  if (pq_count (&(c->qdag.dedge_pq)) != 0)
		    insert = c;

		  /* Check all s-edges of current c-forest ancestor 'c'. */
		  Edge *s, **cur, **end;
		  end = c->qdag.sedge_pq.elems_top;
		  cur = c->qdag.sedge_pq.elems_start;
		  for (; cur < end; cur++)
		    {
		      s = *cur;
		      Var *s_tailvar = VARID2VARPTR (vars, s->tail_var);
		      Edge *d, **d_cur, **d_end;
		      d_end = insert->qdag.dedge_pq.elems_top;
		      d_cur = insert->qdag.dedge_pq.elems_start;
		      for (; d_cur < d_end; d_cur++)
			{
			  d = *d_cur;
			  Var *d_tailvar = VARID2VARPTR (vars, d->tail_var);
			  /* Only representative of universal class
			     has d-edges, but not members. */
			  assert (UF_IS_REP (d_tailvar, 0));
			  /* Insert edge. */
			  if (!HAS_EDGE
			      (s_tailvar, d_tailvar->id, qdag.dedges))
			    {
			      Edge *edge;
			      edge = create_edge (mm);
			      edge->tail_var = s_tailvar->id;
			      edge->head_var = d_tailvar->id;
			      pq_insert (mm, &(d_tailvar->qdag.dedge_pq),
					 edge, s_tailvar->scope->nesting);
			      et_insert (mm, &(s_tailvar->qdag.dedges), edge);
			    }
			}
		    }
		  cid = c->qdag.cedges.cpar;
		}
	    }
	}
    }

  remove_transitivities (dm);
#ifndef NDEBUG
#if QDAG_ASSERT_NO_TRANSITIVITIES
  assert_no_transitivities (dm);
#endif
#endif
}


static void
extract_dependencies (QDPLLDepManQDAG * dm)
{
  Constraint *c;
  for (c = dm->pcnf->clauses.first; c; c = c->link.next)
    {
      assert (!c->is_cube);
      assert (!c->learnt);
      if (!c->disabled)
	extract_dependencies_from_clause (dm, c);
    }

#ifndef NDEBUG
#if QDAG_ASSERT_GRAPH_INTEGRITY
  assert_graph_integrity (dm);
#endif
#endif

  Var *vars = dm->pcnf->vars;
  QDPLLMemMan *mm = dm->mm;

  Scope *s;
  for (s = dm->pcnf->scopes.first; s; s = s->link.next)
    {
      if (QDPLL_SCOPE_FORALL (s))
	merge_univ_scope_vars_by_dedges (mm, vars, s);
      else
	merge_exist_scope_vars_by_sedges (mm, vars, s);
    }

#ifndef NDEBUG
#if QDAG_ASSERT_GRAPH_INTEGRITY
  assert_graph_integrity (dm);
#endif
#endif

  insert_existential_deps (dm);

#ifndef NDEBUG
#if QDAG_ASSERT_GRAPH_INTEGRITY
  assert_graph_integrity (dm);
#endif
#if QDAG_ASSERT_XCHECK_DEPENDENCIES
  assert_xcheck_dependencies (dm);
#endif
#if QDAG_ASSERT_CHECK_DEPENDENCIES_BY_FUNCTIONS
  assert_check_dependencies_by_functions (dm);
#endif
#endif

}


static LitID *
get_largest_active_literal_in_clause (Var * vars, LitID * leftend,
				      LitID * rightend, LitID * ptr,
				      unsigned int ignore_dlevel)
{
  assert (leftend <= ptr);
  assert (ptr <= rightend);
  Var *v = LIT2VARPTR (vars, *ptr);
  while ((QDPLL_VAR_MARKED_PROPAGATED (v)
	  && v->decision_level <= ignore_dlevel)
	 || !QDPLL_SCOPE_EXISTS (v->scope))
    {
      /* Any propagated literal must be assigned false as unit or pure
         at top-level. */
      assert (!
	      (QDPLL_VAR_MARKED_PROPAGATED (v)
	       && v->decision_level <= ignore_dlevel) ||
	      ((QDPLL_LIT_NEG (*ptr) && QDPLL_VAR_ASSIGNED_TRUE (v)) ||
	       (QDPLL_LIT_POS (*ptr) && QDPLL_VAR_ASSIGNED_FALSE (v))));
      ptr--;
      assert (leftend <= ptr);
      v = LIT2VARPTR (vars, *ptr);
    }
  assert (leftend <= ptr);
  assert (ptr <= rightend);
  assert (!
	  (QDPLL_VAR_MARKED_PROPAGATED (v)
	   && v->decision_level <= ignore_dlevel)
	  && QDPLL_SCOPE_EXISTS (v->scope));
  return ptr;
}


static void
collect_deps_from_cnf_check_clause (QDPLLDepManQDAG * dm,
				    VarPtrStack * deps,
				    VarPtrStack * con,
				    QDPLLQuantifierType var_type,
				    unsigned int var_nesting, Constraint * c)
{
  assert (!c->is_cube);
  if (c->disabled || c->learnt)
    return;

  unsigned int ignore_dlevel = 0;
  QDPLLMemMan *mm = dm->mm;
  Var *vars = dm->pcnf->vars;
  LitID *litp, *lite = c->lits + c->num_lits - 1;
  lite =
    get_largest_active_literal_in_clause (vars, c->lits, lite, lite,
					  ignore_dlevel);
  for (litp = c->lits; litp <= lite; litp++)
    {
      LitID lit = *litp;
      Var *d = LIT2VARPTR (vars, lit);
      assert (!
	      (QDPLL_VAR_MARKED_PROPAGATED (d)
	       && d->decision_level <= ignore_dlevel)
	      || (((QDPLL_LIT_NEG (lit) && QDPLL_VAR_ASSIGNED_TRUE (d)) ||
		   (QDPLL_LIT_POS (lit) && QDPLL_VAR_ASSIGNED_FALSE (d)))));
      if (QDPLL_VAR_MARKED_PROPAGATED (d)
	  && d->decision_level <= ignore_dlevel)
	continue;

      Scope *d_scope = d->scope;

      if (var_nesting < d_scope->nesting)
	{
	  /* Assumes that clauses are universally reduced. */
	  if (var_type != d_scope->type)
	    {
	      /* Collect and mark dependency. */
	      if (!d->qdag.mark0)
		{
		  d->qdag.mark0 = 1;
		  QDPLL_PUSH_STACK (mm, *deps, d);
		}
	    }
	  if (QDPLL_SCOPE_EXISTS (d_scope))
	    {
	      /* Collect and mark connecting variable. */
	      if (!d->qdag.mark1)
		{
		  d->qdag.mark1 = 1;
		  QDPLL_PUSH_STACK (mm, *con, d);
		}
	    }
	}
    }
}


static void
collect_deps_from_cnf (QDPLLDepManQDAG * dm, VarPtrStack * deps, Var * v)
{
  QDPLLQuantifierType var_type = v->scope->type;
  unsigned int var_nesting = v->scope->nesting;
  QDPLLMemMan *mm = dm->mm;
  VarPtrStack con;
  QDPLL_INIT_STACK (con);

  Var *vars = dm->pcnf->vars;

  assert (!v->qdag.mark0);	/* 'mark0' marks collected dependencies. */
  assert (!v->qdag.mark1);	/* 'mark1' marks collected connections. */
  /* Mark as connecting variable. */
  v->qdag.mark1 = 1;
  QDPLL_PUSH_STACK (mm, con, v);

  while (!QDPLL_EMPTY_STACK (con))
    {
      Var *t = QDPLL_POP_STACK (con);
      assert (!(QDPLL_VAR_MARKED_PROPAGATED (t) && t->decision_level <= 0));
      assert (t->qdag.mark1);	/* Must be marked as collected. */
      assert (t == v || v->scope->nesting < t->scope->nesting);
      assert (t == v || t->scope->type == QDPLL_QTYPE_EXISTS);

      /* Check all clauses containing literals of 'v'. */
      LitID lit = t->id;
      OccListIterator it;
      OLITER_INIT (it, t->pos_occ_clauses.first, t->pos_occ_clauses.foffset,
		   lit);
      while (OLITER_CUR (it))
	{
	  collect_deps_from_cnf_check_clause (dm, deps, &(con), var_type,
					      var_nesting, OLITER_CUR (it));
	  OLITER_NEXT (it, lit);
	}
      lit = -t->id;
      OLITER_INIT (it, t->neg_occ_clauses.first, t->neg_occ_clauses.foffset,
		   lit);
      while (OLITER_CUR (it))
	{
	  collect_deps_from_cnf_check_clause (dm, deps, &(con), var_type,
					      var_nesting, OLITER_CUR (it));
	  OLITER_NEXT (it, lit);
	}
    }

  QDPLL_DELETE_STACK (mm, con);

  Var *p, *e;
  for (p = vars, e = vars + dm->pcnf->size_vars; p < e; p++)
    p->qdag.mark1 = 0;

  /* Marks for collected dependencies will be reset later after
     cross-checking. */
}


/* Collect dependencies of universal variable by collecting all
c-forest successors. */
static void
collect_deps_from_graph_forall (QDPLLDepManQDAG * dm, VarPtrStack * deps,
				Var * v)
{
  assert (QDPLL_SCOPE_FORALL (v->scope));
  QDPLLMemMan *mm = dm->mm;
  Var *vars = dm->pcnf->vars;
  VarPtrStack forest_succ;
  QDPLL_INIT_STACK (forest_succ);

  v = uf_find (dm->pcnf->vars, v, 0);
  assert (UF_IS_REP (v, 0));

  unsigned int i, end = v->qdag.dedges.size;
  Edge *d;
  for (i = 0; i < end; i++)
    for (d = v->qdag.dedges.table[i]; d; d = d->chain_next)
      {
	Var *dvar = VARID2VARPTR (vars, d->head_var);
	assert (UF_IS_REP (dvar, 0));
	assert (VARID2VARPTR (vars, d->tail_var) == v);
	assert (v->scope->nesting < dvar->scope->nesting);
	assert (QDPLL_SCOPE_EXISTS (dvar->scope));
	QDPLL_PUSH_STACK (mm, forest_succ, dvar);
      }

  while (!QDPLL_EMPTY_STACK (forest_succ))
    {
      Var *succ = QDPLL_POP_STACK (forest_succ);
      assert (UF_IS_REP (succ, 0));
      assert (QDPLL_SCOPE_EXISTS (succ->scope));
      assert (v->scope->nesting < succ->scope->nesting);

      /* Actually, need not mark collected dependencies/connections
         because of tree property. Setting mark for assertion checking
         only. */
      assert (!succ->qdag.mark0);
      succ->qdag.mark0 = 1;
      QDPLL_PUSH_STACK (mm, *deps, succ);

      /* Must also collect class members, if any. */
      if (!UF_IS_SINGLETON_SET (succ, 0))
	{
	  VarID mid;
	  Var *m;
	  for (mid = succ->qdag.uf[0].members.list.first; mid;
	       mid = m->qdag.uf[0].members.link.next)
	    {
	      m = VARID2VARPTR (vars, mid);
	      assert (!UF_IS_REP (m, 0));
	      assert (QDPLL_SCOPE_EXISTS (m->scope));
	      assert (v->scope->nesting < m->scope->nesting);
	      assert (!m->qdag.mark0);
	      m->qdag.mark0 = 1;
	      QDPLL_PUSH_STACK (mm, *deps, m);
	    }
	}

      /* Push c-children on successor stack for further traversal. */
      VarID cid;
      Var *c;
      for (cid = succ->qdag.cedges.cchilds.first; cid;
	   cid = c->qdag.cedges.csibs.next)
	{
	  c = VARID2VARPTR (vars, cid);
	  assert (!c->qdag.mark0);
	  assert (!c->qdag.mark1);
	  QDPLL_PUSH_STACK (mm, forest_succ, c);
	}
    }

  QDPLL_DELETE_STACK (mm, forest_succ);
}


static void
collect_deps_from_graph_exists (QDPLLDepManQDAG * dm, VarPtrStack * deps,
				Var * v)
{
  assert (QDPLL_SCOPE_EXISTS (v->scope));
  QDPLLMemMan *mm = dm->mm;
  Var *vars = dm->pcnf->vars;
  VarPtrStack forest_succ;
  QDPLL_INIT_STACK (forest_succ);

  /* Push all s-edges. */
  unsigned int i, end = v->qdag.sedges.size;
  Edge *s;
  for (i = 0; i < end; i++)
    for (s = v->qdag.sedges.table[i]; s; s = s->chain_next)
      {
	Var *svar = VARID2VARPTR (vars, s->head_var);
	assert (VARID2VARPTR (vars, s->tail_var) == v);
	assert (QDPLL_SCOPE_EXISTS (svar->scope));
	assert (UF_IS_REP (svar, 0));
	assert (v->scope->nesting < svar->scope->nesting);
	QDPLL_PUSH_STACK (mm, forest_succ, svar);
      }

  while (!QDPLL_EMPTY_STACK (forest_succ))
    {
      Var *succ = QDPLL_POP_STACK (forest_succ);
      assert (UF_IS_REP (succ, 0));
      assert (QDPLL_SCOPE_EXISTS (succ->scope));
      assert (v->scope->nesting < succ->scope->nesting);

      /* Collect all universal variables which have d-edge to 'succ'. */
      Edge **p, **e;
      for (p = succ->qdag.dedge_pq.elems_start,
	   e = succ->qdag.dedge_pq.elems_top; p < e; p++)
	{
	  Edge *d = *p;
	  Var *dvar = VARID2VARPTR (vars, (*p)->tail_var);
	  assert (VARID2VARPTR (vars, (*p)->head_var) == succ);
	  assert (v->scope->nesting < dvar->scope->nesting);
	  assert (dvar->scope->nesting < succ->scope->nesting);
	  assert (QDPLL_SCOPE_FORALL (dvar->scope));
	  assert (UF_IS_REP (dvar, 0));

	  if (!dvar->qdag.mark0)
	    {
	      dvar->qdag.mark0 = 1;
	      QDPLL_PUSH_STACK (mm, *deps, dvar);

	      if (!UF_IS_SINGLETON_SET (dvar, 0))
		{
		  VarID mid;
		  Var *m;
		  for (mid = dvar->qdag.uf[0].members.list.first; mid;
		       mid = m->qdag.uf[0].members.link.next)
		    {
		      m = VARID2VARPTR (vars, mid);
		      assert (!m->qdag.mark0);
		      m->qdag.mark0 = 1;
		      QDPLL_PUSH_STACK (mm, *deps, m);
		    }
		}
	    }
	}

      /* Push c-children on successor stack for further traversal. */
      VarID cid;
      Var *c;
      for (cid = succ->qdag.cedges.cchilds.first; cid;
	   cid = c->qdag.cedges.csibs.next)
	{
	  c = VARID2VARPTR (vars, cid);
	  assert (!c->qdag.mark0);
	  assert (!c->qdag.mark1);
	  QDPLL_PUSH_STACK (mm, forest_succ, c);
	}
    }

  QDPLL_DELETE_STACK (mm, forest_succ);
}

static void
collect_deps_from_graph (QDPLLDepManQDAG * dm, VarPtrStack * deps, Var * v)
{
  if (QDPLL_SCOPE_FORALL (v->scope))
    collect_deps_from_graph_forall (dm, deps, v);
  else
    {
      assert (QDPLL_SCOPE_EXISTS (v->scope));
      collect_deps_from_graph_exists (dm, deps, v);
    }
}


static int
qsort_compare_deps_by_id (const void *dep1, const void *dep2)
{
  Var *v1 = *((Var **) dep1);
  Var *v2 = *((Var **) dep2);

  if (v1->id < v2->id)
    return -1;
  else if (v1->id > v2->id)
    return 1;
  else
    return 0;
}


static void
unmark_dependency_marks (VarPtrStack * deps)
{
  Var **p, **e;
  for (p = deps->start, e = deps->top; p < e; p++)
    {
      Var *v = *p;
      assert (v->qdag.mark0);
      v->qdag.mark0 = 0;
    }
}


/* Mark all universal variables (i.e. classes!) pointed to by d-edges
   from existential variables (NOT classes!) as depending. */
static void
mark_non_candidates_from_exists (QDPLLDepManQDAG * dm, Scope * s)
{
  Var *vars = dm->pcnf->vars;
  assert (QDPLL_SCOPE_EXISTS (s));

  VarID *p, *e;
  for (p = s->vars.start, e = s->vars.top; p < e; p++)
    {
      Var *v = VARID2VARPTR (vars, *p);
      unsigned int i, e = v->qdag.dedges.size;
      Edge *d;
      for (i = 0; i < e; i++)
	for (d = v->qdag.dedges.table[i]; d; d = d->chain_next)
	  {
	    Var *dvar = VARID2VARPTR (vars, d->head_var);
	    assert (QDPLL_SCOPE_FORALL (dvar->scope));
	    assert (UF_IS_REP (dvar, 0));
	    assert (v->scope->nesting < dvar->scope->nesting);
	    dvar->qdag.mark0 = 1;
	  }
    }
}


/* Mark all forest-successors (i.e. classes!) of universal variables as
   depending. Such variables can not be candidates since they depend
   on at least one universal variable. */
static void
mark_non_candidates_from_forall (QDPLLDepManQDAG * dm, Scope * s)
{
  Var *vars = dm->pcnf->vars;
  QDPLLMemMan *mm = dm->mm;
  VarPtrStack forest_succ;
  QDPLL_INIT_STACK (forest_succ);
  assert (QDPLL_SCOPE_FORALL (s));

  Var *v;
  VarID vid;
  for (vid = s->classes[0].first; vid; vid = v->qdag.uf[0].class_link.next)
    {
      v = VARID2VARPTR (vars, vid);
      assert (UF_IS_REP (v, 0));
      assert (QDPLL_SCOPE_FORALL (v->scope));

      /* Mark and push d-edges on traversal stack. */
      unsigned int i, end = v->qdag.dedges.size;
      Edge *d;
      for (i = 0; i < end; i++)
	for (d = v->qdag.dedges.table[i]; d; d = d->chain_next)
	  {
	    Var *dvar = VARID2VARPTR (vars, d->head_var);
	    assert (QDPLL_SCOPE_EXISTS (dvar->scope));
	    assert (UF_IS_REP (dvar, 0));
	    assert (v->scope->nesting < dvar->scope->nesting);

	    if (!dvar->qdag.mark0)
	      {
		dvar->qdag.mark0 = 1;
		QDPLL_PUSH_STACK (mm, forest_succ, dvar);
	      }
	  }

      while (!QDPLL_EMPTY_STACK (forest_succ))
	{
	  Var *succ = QDPLL_POP_STACK (forest_succ);
	  assert (QDPLL_SCOPE_EXISTS (succ->scope));
	  assert (v->scope->nesting < succ->scope->nesting);
	  assert (UF_IS_REP (succ, 0));
	  assert (succ->qdag.mark0);

	  /* Mark and push all c-children not yet marked. */
	  Var *c;
	  VarID cid;
	  for (cid = succ->qdag.cedges.cchilds.first; cid;
	       cid = c->qdag.cedges.csibs.next)
	    {
	      c = VARID2VARPTR (vars, cid);
	      if (!c->qdag.mark0)
		{
		  c->qdag.mark0 = 1;
		  QDPLL_PUSH_STACK (mm, forest_succ, c);
		}
	    }
	}
    }

  QDPLL_DELETE_STACK (mm, forest_succ);
}


/* Returns the number of variables not yet marked as propagated in the
   class of universal 'v', i.e. the number of active variables. */
static unsigned int
count_non_propagated_in_class (Var * vars, Var * v)
{
  assert (QDPLL_SCOPE_FORALL (v->scope));
  assert (UF_IS_REP (v, 0));
  unsigned int result = 0;

  if (!QDPLL_VAR_MARKED_PROPAGATED (v))
    result++;
#if SKIP_DELAYED_NOTIF_ASSERT
  else if (!v->qdag.mark_notified_inactive)
    result++;
#endif

  if (!UF_IS_SINGLETON_SET (v, 0))
    {
      Var *m;
      VarID mid;
      for (mid = v->qdag.uf[0].members.list.first; mid;
	   mid = m->qdag.uf[0].members.link.next)
	{
	  m = VARID2VARPTR (vars, mid);
	  if (!QDPLL_VAR_MARKED_PROPAGATED (m))
	    result++;
#if SKIP_DELAYED_NOTIF_ASSERT
	  else if (!m->qdag.mark_notified_inactive)
	    result++;
#endif
	}
    }

  return result;
}


/* Returns the number of direct references from universal classes to
   existential 'v' which are not yet marked as candidate. */
static unsigned int
count_direct_non_candidate_refs_by_univ_class (Var * vars, Var * v)
{
  assert (QDPLL_SCOPE_EXISTS (v->scope));
  assert (UF_IS_REP (v, 0));
  unsigned int result = 0;

  Edge **p, **e;
  for (p = v->qdag.dedge_pq.elems_start,
       e = v->qdag.dedge_pq.elems_top; p < e; p++)
    {
      Edge *u = *p;
      Var *uvar = VARID2VARPTR (vars, u->tail_var);
      assert (QDPLL_SCOPE_FORALL (uvar->scope));
      assert (uvar->scope->nesting < v->scope->nesting);
      assert (UF_IS_REP (uvar, 0));
      /*'uvar' references 'v'. */
      if (!uvar->mark_is_candidate)
	result++;
    }

  return result;
}


/* Returns the number of direct references from existential variables to
   universal 'v' which are not yet marked as propagated i.e. the number
   of active references. */
static unsigned int
count_direct_active_refs_by_exist_var (Var * vars, Var * v)
{
  assert (QDPLL_SCOPE_FORALL (v->scope));
  assert (UF_IS_REP (v, 0));
  unsigned int result = 0;

  Edge **p, **e;
  for (p = v->qdag.dedge_pq.elems_start,
       e = v->qdag.dedge_pq.elems_top; p < e; p++)
    {
      Edge *u = *p;
      Var *uvar = VARID2VARPTR (vars, u->tail_var);
      assert (QDPLL_SCOPE_EXISTS (uvar->scope));
      assert (uvar->scope->nesting < v->scope->nesting);
      /* 'uvar' references 'v'. */
      if (!QDPLL_VAR_MARKED_PROPAGATED (uvar))
	result++;
#if SKIP_DELAYED_NOTIF_ASSERT
      else if (!uvar->qdag.mark_notified_inactive)
	result++;
#endif
    }

  return result;
}


/* Returns the number of direct references by s-edges from existential
   variables to existential classes 'v' which are not yet marked as
   propagated i.e. the number of active references. */
static unsigned int
count_direct_active_refs_by_sedge (Var * vars, Var * v)
{
  assert (QDPLL_SCOPE_EXISTS (v->scope));
  assert (UF_IS_REP (v, 0));
  unsigned int result = 0;

  Edge **p, **e;
  for (p = v->qdag.sedge_pq.elems_start,
       e = v->qdag.sedge_pq.elems_top; p < e; p++)
    {
      Edge *u = *p;
      Var *uvar = VARID2VARPTR (vars, u->tail_var);
      assert (QDPLL_SCOPE_EXISTS (uvar->scope));
      assert (uvar->scope->nesting < v->scope->nesting);
      /* 'uvar' references 'v'. */
      if (!QDPLL_VAR_MARKED_PROPAGATED (uvar))
	result++;
#if SKIP_DELAYED_NOTIF_ASSERT
      else if (!uvar->qdag.mark_notified_inactive)
	result++;
#endif
    }

  return result;
}


/* Returns the number of direct references from universal classes to
   existential 'v' which are not yet marked as propagated i.e. the number
   of active references. */
static unsigned int
count_direct_active_refs_by_univ_class (Var * vars, Var * v)
{
  assert (QDPLL_SCOPE_EXISTS (v->scope));
  assert (UF_IS_REP (v, 0));
  unsigned int result = 0;

  Edge **p, **e;
  for (p = v->qdag.dedge_pq.elems_start,
       e = v->qdag.dedge_pq.elems_top; p < e; p++)
    {
      Edge *u = *p;
      Var *uvar = VARID2VARPTR (vars, u->tail_var);
      assert (QDPLL_SCOPE_FORALL (uvar->scope));
      assert (uvar->scope->nesting < v->scope->nesting);
      assert (UF_IS_REP (uvar, 0));
      /*'uvar' references 'v'. */
      if (count_non_propagated_in_class (vars, uvar) != 0)
	result++;
    }

  return result;
}


static unsigned int
count_class_members (Var * vars, Var * v)
{
  assert (UF_IS_REP (v, 0));
  unsigned int result = 1;
  if (!UF_IS_SINGLETON_SET (v, 0))
    {
      Var *m;
      VarID mid;
      for (mid = v->qdag.uf[0].members.list.first; mid;
	   mid = m->qdag.uf[0].members.link.next)
	{
	  m = VARID2VARPTR (vars, mid);
	  result++;
	}
    }
  return result;
}


/* Find all variables 'x' such that D^{-1}(x) is empty, that is,
   variables which do not depend on any other variable. */
static void
fill_candidates (QDPLLDepManQDAG * dm)
{
  assert (dm->state.init);
  Var *vars = dm->pcnf->vars;

#ifndef NDEBUG
#if QDAG_ASSERT_FILL_CANDIDATES_VARS
  do
    {
      Var *p, *e;
      for (p = vars, e = vars + dm->pcnf->size_vars; p < e; p++)
	{
	  assert (!p->qdag.mark0);
	  assert (!p->qdag.mark1);
	}
    }
  while (0);
#endif
#endif

  Scope *s;
  for (s = dm->pcnf->scopes.first; s; s = s->link.next)
    {
      if (QDPLL_SCOPE_EXISTS (s))
	mark_non_candidates_from_exists (dm, s);
      else
	{
	  assert (QDPLL_SCOPE_FORALL (s));
	  mark_non_candidates_from_forall (dm, s);
	}
    }

  /* Traverse all classes of variables and collect unmarked variables
     as candidates. Unmark depending variables on-the-fly. */
  for (s = dm->pcnf->scopes.first; s; s = s->link.next)
    {
      unsigned int cnt_members;
      Var *v;
      VarID vid;
      for (vid = s->classes[0].first; vid;
	   vid = v->qdag.uf[0].class_link.next)
	{
	  cnt_members = 0;
	  v = VARID2VARPTR (vars, vid);
	  assert (UF_IS_REP (v, 0));

	  /* Mark forest roots as as beginning of
	     inactive-sedge-frontier. */
	  if (QDPLL_SCOPE_EXISTS (s) && !v->qdag.cedges.cpar)
	    v->qdag.cnt.exists.inactive_sedge_frontier = 1;

	  if (!v->qdag.mark0)
	    {
	      /* Link variable and all class members to candidate list. */
	      assert (!v->mark_is_candidate);
	      v->mark_is_candidate = 1;
	      VAR_LINK (vars, dm->candidates, v, qdag.cand_link);
	      cnt_members++;
	      if (!UF_IS_SINGLETON_SET (v, 0))
		{
		  Var *m;
		  VarID mid;
		  for (mid = v->qdag.uf[0].members.list.first; mid;
		       mid = m->qdag.uf[0].members.link.next)
		    {
		      m = VARID2VARPTR (vars, mid);
		      assert (!m->qdag.mark0);
		      assert (!m->mark_is_candidate);
		      m->mark_is_candidate = 1;
		      VAR_LINK (vars, dm->candidates, m, qdag.cand_link);
		      cnt_members++;
		    }
		}
	    }
	  else
	    v->qdag.mark0 = 0;
	  /* Set up counters for incremental candidate maintenance in
	     variables. */
	  if (QDPLL_SCOPE_FORALL (s))
	    {
	      if (!cnt_members)
		cnt_members = count_class_members (vars, v);
	      assert (cnt_members == count_non_propagated_in_class (vars, v));
	      v->qdag.cnt.forall.non_propagated_in_class = cnt_members;
	      assert (pq_count (&(v->qdag.dedge_pq)) ==
		      count_direct_active_refs_by_exist_var (vars, v));
	      v->qdag.cnt.forall.active_direct_refs_by_exist_var =
		pq_count (&(v->qdag.dedge_pq));
	    }
	  else
	    {
	      assert (QDPLL_SCOPE_EXISTS (s));
	      assert (pq_count (&(v->qdag.dedge_pq)) ==
		      count_direct_active_refs_by_univ_class (vars, v));
	      v->qdag.cnt.exists.active_direct_refs_by_univ_class =
		pq_count (&(v->qdag.dedge_pq));
	      assert (pq_count (&(v->qdag.sedge_pq)) ==
		      count_direct_active_refs_by_sedge (vars, v));
	      v->qdag.cnt.exists.active_direct_refs_by_sedge =
		pq_count (&(v->qdag.sedge_pq));

	    }
	}
    }

#ifndef NDEBUG
#if QDAG_ASSERT_FILL_CANDIDATES_VARS
  do
    {
      Var *p, *e;
      for (p = vars, e = vars + dm->pcnf->size_vars; p < e; p++)
	{
	  assert (!p->qdag.mark0);
	  assert (!p->qdag.mark1);
	  if (p->id && UF_IS_REP (p, 0))
	    {
	      if (QDPLL_SCOPE_EXISTS (p->scope))
		{
		  assert (p->qdag.cnt.
			  exists.active_direct_refs_by_univ_class ==
			  count_direct_active_refs_by_univ_class (vars, p));
		  assert (p->qdag.cnt.exists.active_direct_refs_by_sedge ==
			  count_direct_active_refs_by_sedge (vars, p));
		}
	      else
		{
		  assert (QDPLL_SCOPE_FORALL (p->scope));
		  assert (p->qdag.cnt.forall.
			  active_direct_refs_by_exist_var ==
			  count_direct_active_refs_by_exist_var (vars, p));
		  assert (p->qdag.cnt.forall.non_propagated_in_class ==
			  count_non_propagated_in_class (vars, p));
		}
	    }
	}
    }
  while (0);
#endif
#if QDAG_ASSERT_CANDIDATE_LIST
  assert_candidate_list (dm);
#endif
#endif
}


static int
referenced_by_active_existential_var (Var * vars, Var * v)
{
  assert (QDPLL_SCOPE_FORALL (v->scope));
  assert (UF_IS_REP (v, 0));
  assert (count_direct_active_refs_by_exist_var (vars, v) ==
	  v->qdag.cnt.forall.active_direct_refs_by_exist_var);

  if (v->qdag.cnt.forall.active_direct_refs_by_exist_var != 0)
    return 1;

  /* Check if 'v' is indirectly referenced by removed transitive edge. */
  unsigned int i, end = v->qdag.dedges.size;
  Edge *d;
  for (i = 0; i < end; i++)
    for (d = v->qdag.dedges.table[i]; d; d = d->chain_next)
      {
	Var *dvar = VARID2VARPTR (vars, d->head_var);
	assert (QDPLL_SCOPE_EXISTS (dvar->scope));
	assert (UF_IS_REP (dvar, 0));

	Var *par;
	VarID parid = dvar->id;
	while (parid)
	  {
	    par = VARID2VARPTR (vars, parid);
	    assert (QDPLL_SCOPE_EXISTS (dvar->scope));
	    assert (UF_IS_REP (dvar, 0));

	    /* Check s-edges of 'par'. */
	    Edge **p, **e;
	    for (p = par->qdag.sedge_pq.elems_start,
		 e = par->qdag.sedge_pq.elems_top; p < e; p++)
	      {
		Edge *s = *p;
		Var *svar = VARID2VARPTR (vars, s->tail_var);
		assert (QDPLL_SCOPE_EXISTS (svar->scope));

		if (!QDPLL_VAR_MARKED_PROPAGATED (svar))
		  return 1;
#if SKIP_DELAYED_NOTIF_ASSERT
		else if (!svar->qdag.mark_notified_inactive)
		  return 1;
#endif
	      }

	    parid = par->qdag.cedges.cpar;
	  }
      }

  return 0;
}


static void
collect_class_as_candidate (QDPLLDepManQDAG * dm, Var * vars, Var * class)
{
  assert (UF_IS_REP (class, 0));
  assert (!class->mark_is_candidate);
  class->mark_is_candidate = 1;
  VAR_LINK (vars, dm->candidates, class, qdag.cand_link);
  if (!UF_IS_SINGLETON_SET (class, 0))
    {
      VarID mid;
      Var *m;
      for (mid = class->qdag.uf[0].members.list.first;
	   mid; mid = m->qdag.uf[0].members.link.next)
	{
	  m = VARID2VARPTR (vars, mid);
	  assert (!m->mark_is_candidate);
	  m->mark_is_candidate = 1;
	  VAR_LINK (vars, dm->candidates, m, qdag.cand_link);
	}
    }
}


static void
undo_collect_class_as_candidate (QDPLLDepManQDAG * dm, Var * vars,
				 Var * class)
{
  assert (UF_IS_REP (class, 0));
  assert (class->mark_is_candidate);
  class->mark_is_candidate = 0;

  if (class->qdag.cand_link.prev || class->qdag.cand_link.next
      || class->id == dm->candidates.first)
    VAR_UNLINK (vars, dm->candidates, class, qdag.cand_link);

  if (!UF_IS_SINGLETON_SET (class, 0))
    {
      VarID mid;
      Var *m;
      for (mid = class->qdag.uf[0].members.list.first;
	   mid; mid = m->qdag.uf[0].members.link.next)
	{
	  m = VARID2VARPTR (vars, mid);
	  assert (m->mark_is_candidate);
	  m->mark_is_candidate = 0;

	  if (m->qdag.cand_link.prev || m->qdag.cand_link.next
	      || mid == dm->candidates.first)
	    VAR_UNLINK (vars, dm->candidates, m, qdag.cand_link);
	}
    }
}


static int
is_candidate_by_frontier (Var * vars, Var * u)
{
  assert (QDPLL_SCOPE_FORALL (u->scope));
  assert (UF_IS_REP (u, 0));

  unsigned int i, end = u->qdag.dedges.size;
  Edge *d;
  for (i = 0; i < end; i++)
    for (d = u->qdag.dedges.table[i]; d; d = d->chain_next)
      {
	Var *dvar = VARID2VARPTR (vars, d->head_var);
	assert (QDPLL_SCOPE_EXISTS (dvar->scope));
	assert (UF_IS_REP (dvar, 0));

	if (!dvar->qdag.cnt.exists.inactive_sedge_frontier)
	  {
	    return 0;
	  }
	else
	  assert (dvar->qdag.cnt.exists.active_direct_refs_by_sedge == 0);
      }

  assert (!referenced_by_active_existential_var (vars, u));
  return 1;
}


static void
get_universal_candidates_top_down (QDPLLDepManQDAG * dm, Var * var)
{
  assert (QDPLL_SCOPE_EXISTS (var->scope));
  assert (UF_IS_REP (var, 0));
  Var *vars = dm->pcnf->vars;
  assert (var->qdag.cnt.exists.active_direct_refs_by_sedge == 0);
  assert (var->qdag.cnt.exists.active_direct_refs_by_sedge ==
	  count_direct_active_refs_by_sedge (vars, var));
#ifndef NDEBUG
#if QDAG_ASSERT_INACTIVE_SEDGE_FRONTIER
  if (var->qdag.cedges.cpar)
    assert_inactive_sedge_frontier_by_ancestors (dm,
						 VARID2VARPTR (vars,
							       var->
							       qdag.cedges.
							       cpar));
#endif
#endif
  QDPLLMemMan *mm = dm->mm;
  VarPtrStack succ;
  QDPLL_INIT_STACK (succ);

  QDPLL_PUSH_STACK (mm, succ, var);

  /* Propagate inactive s-edge frontier downwards. */
  while (!QDPLL_EMPTY_STACK (succ))
    {
      Var *s = QDPLL_POP_STACK (succ);
      assert (QDPLL_SCOPE_EXISTS (s->scope));
      assert (UF_IS_REP (s, 0));
      assert (s->qdag.cnt.exists.active_direct_refs_by_sedge == 0);
      assert (s->qdag.cnt.exists.active_direct_refs_by_sedge ==
	      count_direct_active_refs_by_sedge (vars, s));

      assert (!s->qdag.cnt.exists.inactive_sedge_frontier);
      s->qdag.cnt.exists.inactive_sedge_frontier = 1;

      /* Collect all universal candidates. */
      Edge **p, **e;
      for (p = s->qdag.dedge_pq.elems_start,
	   e = s->qdag.dedge_pq.elems_top; p < e; p++)
	{
	  Var *d = VARID2VARPTR (vars, (*p)->tail_var);
	  assert (QDPLL_SCOPE_FORALL (d->scope));
	  assert (UF_IS_REP (d, 0));

	  if (!d->mark_is_candidate && is_candidate_by_frontier (vars, d))
	    collect_class_as_candidate (dm, vars, d);
#ifndef NDEBUG
	  else
	    {

	    }
#endif
	}

      /* Push all c-children which are not referenced by active s-edge. */
      VarID cid;
      Var *c;
      for (cid = s->qdag.cedges.cchilds.first; cid;
	   cid = c->qdag.cedges.csibs.next)
	{
	  c = VARID2VARPTR (vars, cid);
	  if (c->qdag.cnt.exists.active_direct_refs_by_sedge == 0)
	    QDPLL_PUSH_STACK (mm, succ, c);
#ifndef NDEBUG
	  else
	    {

	    }
#endif
	}
    }

  QDPLL_DELETE_STACK (mm, succ);
}


static void
notify_inactive_exists (QDPLLDepManQDAG * dm, Var * inactive_var)
{
  assert (QDPLL_SCOPE_EXISTS (inactive_var->scope));
  assert (QDPLL_VAR_MARKED_PROPAGATED (inactive_var));
  Var *vars = dm->pcnf->vars;
  Var *inactive_var_rep = uf_find (vars, inactive_var, 0);

  unsigned int i, end = inactive_var->qdag.dedges.size;
  Edge *s;

  /* Decrement active reference counter in universal classes. */
  for (i = 0; i < end; i++)
    for (s = inactive_var->qdag.dedges.table[i]; s; s = s->chain_next)
      {
	Var *svar = VARID2VARPTR (vars, s->head_var);
	assert (s->tail_var == inactive_var->id);
	assert (QDPLL_SCOPE_FORALL (svar->scope));
	assert (inactive_var->scope->nesting < svar->scope->nesting);
	assert (UF_IS_REP (svar, 0));
	assert (svar->qdag.cnt.forall.active_direct_refs_by_exist_var > 0);
	svar->qdag.cnt.forall.active_direct_refs_by_exist_var--;
	assert (svar->qdag.cnt.forall.active_direct_refs_by_exist_var ==
		count_direct_active_refs_by_exist_var (vars, svar));
      }

  /* Update s-edges counter and, if needed, update inactive s-edge
     frontier by traversal. */
  end = inactive_var->qdag.sedges.size;
  for (i = 0; i < end; i++)
    for (s = inactive_var->qdag.sedges.table[i]; s; s = s->chain_next)
      {
	Var *svar = VARID2VARPTR (vars, s->head_var);
	assert (s->tail_var == inactive_var->id);
	assert (QDPLL_SCOPE_EXISTS (svar->scope));
	assert (inactive_var->scope->nesting < svar->scope->nesting);
	assert (UF_IS_REP (svar, 0));
	assert (svar->qdag.cedges.cpar == inactive_var_rep->id);

	assert (svar->qdag.cnt.exists.active_direct_refs_by_sedge > 0);
	svar->qdag.cnt.exists.active_direct_refs_by_sedge--;
	assert (svar->qdag.cnt.exists.active_direct_refs_by_sedge ==
		count_direct_active_refs_by_sedge (vars, svar));

	if (inactive_var_rep->qdag.cnt.exists.inactive_sedge_frontier &&
	    svar->qdag.cnt.exists.active_direct_refs_by_sedge == 0)
	  get_universal_candidates_top_down (dm, svar);
      }
}


static int
referenced_by_active_universal_class (Var * vars, Var * v)
{
  assert (QDPLL_SCOPE_EXISTS (v->scope));
  assert (UF_IS_REP (v, 0));
  assert (count_direct_active_refs_by_univ_class (vars, v) ==
	  v->qdag.cnt.exists.active_direct_refs_by_univ_class);
  return (v->qdag.cnt.exists.active_direct_refs_by_univ_class != 0);
}


static void
get_candidate_and_successors (QDPLLDepManQDAG * dm, Var * dvar)
{
  assert (QDPLL_SCOPE_EXISTS (dvar->scope));
  assert (UF_IS_REP (dvar, 0));
  QDPLLMemMan *mm = dm->mm;
  Var *vars = dm->pcnf->vars;
  VarPtrStack succ;
  QDPLL_INIT_STACK (succ);

  assert (dvar->qdag.cnt.exists.active_direct_refs_by_univ_class == 0);
  assert (!referenced_by_active_universal_class (vars, dvar));
  assert (!dvar->mark_is_candidate);
  QDPLL_PUSH_STACK (mm, succ, dvar);

  while (!QDPLL_EMPTY_STACK (succ))
    {
      Var *s = QDPLL_POP_STACK (succ);
      assert (QDPLL_SCOPE_EXISTS (s->scope));
      assert (UF_IS_REP (s, 0));
      assert (!referenced_by_active_universal_class (vars, s));
      assert (s->qdag.cnt.exists.active_direct_refs_by_univ_class == 0);

      assert (!s->mark_is_candidate);
      collect_class_as_candidate (dm, vars, s);

      /* Push c-children not referenced by an active universal class. */
      Var *c;
      VarID cid;
      for (cid = s->qdag.cedges.cchilds.first; cid;
	   cid = c->qdag.cedges.csibs.next)
	{
	  c = VARID2VARPTR (vars, cid);
	  if (c->qdag.cnt.exists.active_direct_refs_by_univ_class == 0)
	    QDPLL_PUSH_STACK (mm, succ, c);
	  else
	    {
	      assert (!c->mark_is_candidate);
	      assert (referenced_by_active_universal_class (vars, c));
#ifndef NDEBUG
#if QDAG_ASSERT_GET_EXIST_CANDIDATES_MEMBERS
	      if (!UF_IS_SINGLETON_SET (c, 0))
		{
		  Var *m;
		  VarID mid;
		  for (mid = c->qdag.uf[0].members.list.first; mid;
		       mid = m->qdag.uf[0].members.link.next)
		    {
		      m = VARID2VARPTR (vars, mid);
		      assert (!m->mark_is_candidate);
		    }
		}
#endif
#endif
	    }
	  /* If 's' is referenced by active universal class, then 
	     so are all c-successors. Must stop traversal at this point. */
	}
    }

  QDPLL_DELETE_STACK (mm, succ);
}


/* A universal class 'inactive_class' has become inactive. Check if
   there are variables (classes) 'y' in D^{*}(inactive_var) where 'y'
   does not depend on any other universal variable. This can be done
   by checking forest ancestors of d-edges of 'inactive_var'. Collect
   any candidates. */
static void
notify_inactive_forall (QDPLLDepManQDAG * dm, Var * inactive_class)
{
  Var *vars = dm->pcnf->vars;
  assert (count_non_propagated_in_class (vars, inactive_class) == 0);
  assert (inactive_class->qdag.cnt.forall.non_propagated_in_class == 0);
  assert (QDPLL_SCOPE_FORALL (inactive_class->scope));
  assert (UF_IS_REP (inactive_class, 0));

  /* All references from universal class 'inactive_class' are inactive
     now. Decrement counter in existential head-variables of references;
     if any counter goes down to zero, then this variable is not
     referenced actively any more. */
  unsigned int i, end = inactive_class->qdag.dedges.size;
  Edge *d;
  for (i = 0; i < end; i++)
    for (d = inactive_class->qdag.dedges.table[i]; d; d = d->chain_next)
      {
	Var *dvar = VARID2VARPTR (vars, d->head_var);
	assert (!dvar->mark_is_candidate);
	assert (d->tail_var == inactive_class->id);
	assert (QDPLL_SCOPE_EXISTS (dvar->scope));
	assert (inactive_class->scope->nesting < dvar->scope->nesting);
	assert (UF_IS_REP (dvar, 0));

	assert (dvar->qdag.cnt.exists.active_direct_refs_by_univ_class != 0);
	dvar->qdag.cnt.exists.active_direct_refs_by_univ_class--;
	assert (dvar->qdag.cnt.exists.active_direct_refs_by_univ_class ==
		count_direct_active_refs_by_univ_class (vars, dvar));

	if (dvar->qdag.cnt.exists.active_direct_refs_by_univ_class == 0)
	  {
	    assert (!referenced_by_active_universal_class (vars, dvar));
	    /* Here, 'dvar' is not referenced by an active universal
	       variable. Must also check if parent is candidate
	       already. If so, then can start traversal to collect new
	       candidates. */
	    VarID parid = dvar->qdag.cedges.cpar;
	    Var *par = parid ? VARID2VARPTR (vars, parid) : 0;
	    if (!par || par->mark_is_candidate)
	      get_candidate_and_successors (dm, dvar);
	  }
      }
}


static void
undo_get_universal_candidates_top_down (QDPLLDepManQDAG * dm, Var * var)
{
  assert (QDPLL_SCOPE_EXISTS (var->scope));
  assert (UF_IS_REP (var, 0));
  Var *vars = dm->pcnf->vars;
  assert (var->qdag.cnt.exists.inactive_sedge_frontier);
  assert (var->qdag.cnt.exists.active_direct_refs_by_sedge == 1);
  assert (var->qdag.cnt.exists.active_direct_refs_by_sedge ==
	  count_direct_active_refs_by_sedge (vars, var));
#ifndef NDEBUG
#if QDAG_ASSERT_INACTIVE_SEDGE_FRONTIER
  if (var->qdag.cedges.cpar)
    assert_inactive_sedge_frontier_by_ancestors (dm,
						 VARID2VARPTR (vars,
							       var->
							       qdag.cedges.
							       cpar));
#endif
#endif
  QDPLLMemMan *mm = dm->mm;
  VarPtrStack succ;
  QDPLL_INIT_STACK (succ);

  QDPLL_PUSH_STACK (mm, succ, var);

  /* Propagate inactive s-edge frontier downwards. */
  while (!QDPLL_EMPTY_STACK (succ))
    {
      Var *s = QDPLL_POP_STACK (succ);
      assert (QDPLL_SCOPE_EXISTS (s->scope));
      assert (UF_IS_REP (s, 0));
      assert (s == var
	      || s->qdag.cnt.exists.active_direct_refs_by_sedge == 0);
      assert (s->qdag.cnt.exists.active_direct_refs_by_sedge ==
	      count_direct_active_refs_by_sedge (vars, s));

      assert (s->qdag.cnt.exists.inactive_sedge_frontier);
      s->qdag.cnt.exists.inactive_sedge_frontier = 0;

      /* "Reset" all universal candidates. */
      Edge **p, **e;
      for (p = s->qdag.dedge_pq.elems_start,
	   e = s->qdag.dedge_pq.elems_top; p < e; p++)
	{
	  Var *d = VARID2VARPTR (vars, (*p)->tail_var);
	  assert (QDPLL_SCOPE_FORALL (d->scope));
	  assert (UF_IS_REP (d, 0));
	  assert (referenced_by_active_existential_var (vars, d));

	  if (d->mark_is_candidate)
	    {
	      undo_collect_class_as_candidate (dm, vars, d);
	    }
	}

      /* Push all c-children which are not directly referenced by
         active s-edge. Such ones are now indirectly referenced by active
         variable. */
      VarID cid;
      Var *c;
      for (cid = s->qdag.cedges.cchilds.first; cid;
	   cid = c->qdag.cedges.csibs.next)
	{
	  c = VARID2VARPTR (vars, cid);
	  if (c->qdag.cnt.exists.active_direct_refs_by_sedge == 0)
	    QDPLL_PUSH_STACK (mm, succ, c);
#ifndef NDEBUG
	  else
	    {
	      assert (!c->qdag.cnt.exists.inactive_sedge_frontier);
	    }
#endif
	}
    }

  QDPLL_DELETE_STACK (mm, succ);
}


/* This is very similar to 'notify-inactive-exists'. */
static void
notify_active_exists (QDPLLDepManQDAG * dm, Var * active_var)
{
  assert (QDPLL_SCOPE_EXISTS (active_var->scope));
  assert (!QDPLL_VAR_MARKED_PROPAGATED (active_var));
  Var *vars = dm->pcnf->vars;
  Var *active_var_rep = uf_find (vars, active_var, 0);

  unsigned int i, end = active_var->qdag.dedges.size;
  Edge *s;

  /* Increment active reference counter in universal classes. */
  for (i = 0; i < end; i++)
    for (s = active_var->qdag.dedges.table[i]; s; s = s->chain_next)
      {
	Var *svar = VARID2VARPTR (vars, s->head_var);
	assert (s->tail_var == active_var->id);
	assert (QDPLL_SCOPE_FORALL (svar->scope));
	assert (active_var->scope->nesting < svar->scope->nesting);
	assert (UF_IS_REP (svar, 0));

	svar->qdag.cnt.forall.active_direct_refs_by_exist_var++;
	assert (svar->qdag.cnt.forall.active_direct_refs_by_exist_var ==
		count_direct_active_refs_by_exist_var (vars, svar));
      }

  /* Update s-edges counter and, if needed, update inactive s-edge
     frontier by traversal. */
  end = active_var->qdag.sedges.size;
  for (i = 0; i < end; i++)
    for (s = active_var->qdag.sedges.table[i]; s; s = s->chain_next)
      {
	Var *svar = VARID2VARPTR (vars, s->head_var);
	assert (s->tail_var == active_var->id);
	assert (QDPLL_SCOPE_EXISTS (svar->scope));
	assert (active_var->scope->nesting < svar->scope->nesting);
	assert (UF_IS_REP (svar, 0));
	assert (svar->qdag.cedges.cpar == active_var_rep->id);

	svar->qdag.cnt.exists.active_direct_refs_by_sedge++;
	assert (svar->qdag.cnt.exists.active_direct_refs_by_sedge ==
		count_direct_active_refs_by_sedge (vars, svar));

	if (active_var_rep->qdag.cnt.exists.inactive_sedge_frontier &&
	    svar->qdag.cnt.exists.active_direct_refs_by_sedge == 1)
	  undo_get_universal_candidates_top_down (dm, svar);
      }
}


static void
undo_get_candidate_and_successors (QDPLLDepManQDAG * dm, Var * dvar)
{
  assert (QDPLL_SCOPE_EXISTS (dvar->scope));
  assert (UF_IS_REP (dvar, 0));
  QDPLLMemMan *mm = dm->mm;
  Var *vars = dm->pcnf->vars;
  VarPtrStack succ;
  QDPLL_INIT_STACK (succ);

  assert (dvar->mark_is_candidate);
  QDPLL_PUSH_STACK (mm, succ, dvar);

  while (!QDPLL_EMPTY_STACK (succ))
    {
      Var *s = QDPLL_POP_STACK (succ);
      assert (QDPLL_SCOPE_EXISTS (s->scope));
      assert (UF_IS_REP (s, 0));
      assert (s->mark_is_candidate);

      undo_collect_class_as_candidate (dm, vars, s);

      Var *c;
      VarID cid;
      for (cid = s->qdag.cedges.cchilds.first; cid;
	   cid = c->qdag.cedges.csibs.next)
	{
	  c = VARID2VARPTR (vars, cid);
	  if (c->mark_is_candidate)
	    QDPLL_PUSH_STACK (mm, succ, c);
	}
    }

  QDPLL_DELETE_STACK (mm, succ);
}


static void
notify_active_forall (QDPLLDepManQDAG * dm, Var * active_class)
{
  Var *vars = dm->pcnf->vars;
  assert (UF_IS_REP (active_class, 0));
  assert (QDPLL_SCOPE_FORALL (active_class->scope));
  assert (active_class->qdag.cnt.forall.non_propagated_in_class == 1);
  assert (count_non_propagated_in_class (vars, active_class) ==
	  active_class->qdag.cnt.forall.non_propagated_in_class);

  unsigned int i, end = active_class->qdag.dedges.size;
  Edge *d;
  for (i = 0; i < end; i++)
    for (d = active_class->qdag.dedges.table[i]; d; d = d->chain_next)
      {
	Var *dvar = VARID2VARPTR (vars, d->head_var);
	assert (d->tail_var == active_class->id);
	assert (QDPLL_SCOPE_EXISTS (dvar->scope));
	assert (active_class->scope->nesting < dvar->scope->nesting);
	assert (UF_IS_REP (dvar, 0));

	dvar->qdag.cnt.exists.active_direct_refs_by_univ_class++;
	assert (dvar->qdag.cnt.exists.active_direct_refs_by_univ_class ==
		count_direct_active_refs_by_univ_class (vars, dvar));

	if (dvar->mark_is_candidate)
	  {
	    /* Unmark 'dvar' and all of its c-successors. */
	    undo_get_candidate_and_successors (dm, dvar);
	  }
      }
}


static void
unlink_all_non_candidates (QDPLLDepManQDAG * dm)
{
  Var *vars = dm->pcnf->vars;
  Var *c;
  VarID cid, cidn;
  for (cid = dm->candidates.first; cid; cid = cidn)
    {
      c = VARID2VARPTR (vars, cid);
      cidn = c->qdag.cand_link.next;
      if (!c->mark_is_candidate)
	{
	  VAR_UNLINK (vars, dm->candidates, c, qdag.cand_link);
	}
    }
}


static void
print_candidate_list (QDPLLDepManQDAG * dm)
{
  Var *vars = dm->pcnf->vars;
  Var *c;
  VarID cid, cidn;
  for (cid = dm->candidates.first; cid; cid = c->qdag.cand_link.next)
    {
      c = VARID2VARPTR (vars, cid);
      fprintf (stderr, "%d ", c->id);
    }
  fprintf (stderr, "\n");
}


static void
print_marked_candidates (QDPLLDepManQDAG * dm)
{
  Var *p, *e;
  for (p = dm->pcnf->vars, e = p + dm->pcnf->size_vars; p < e; p++)
    {
      if (p->id && p->mark_is_candidate)
	{
	  fprintf (stderr, "%d ", p->id);
	}
    }
  fprintf (stderr, "\n");
}


static Var *
depends_get_ancestor (Var * vars, Var * start, unsigned int stop_nesting)
{
  assert (QDPLL_SCOPE_EXISTS (start->scope));
  assert (UF_IS_REP (start, 0));
  assert (stop_nesting <= start->scope->nesting);

  Var *pvar, *prev;
  VarID pid;

  prev = pvar = start;

  while (stop_nesting < pvar->scope->nesting)
    {
      assert (QDPLL_SCOPE_EXISTS (pvar->scope));
      assert (UF_IS_REP (pvar, 0));
      prev = pvar;
      pid = pvar->qdag.cedges.cpar;
      /* Abort if has no ancestor. */
      if (pid == 0)
	break;
      pvar = VARID2VARPTR (vars, pid);
    }
  assert (stop_nesting < prev->scope->nesting);
  return prev;
}


/* Check if 'y' is reachable from root classes of 'x'. This can be
   done by bottom-up travering y's forest ancestors as far as possible 
   and then checking x's root class set. */
static int
depends_forall_exists (QDPLLDepManQDAG * dm, Var * x, Var * y)
{
  assert (QDPLL_SCOPE_FORALL (x->scope));
  assert (QDPLL_SCOPE_EXISTS (y->scope));
  assert (x->scope->nesting < y->scope->nesting);
  Var *vars = dm->pcnf->vars;

  Var *rep_x = uf_find (vars, x, 0);
  Var *rep_y = uf_find (vars, y, 0);

  Var *ancestor = depends_get_ancestor (vars, rep_y, x->scope->nesting);

  /* Check if (representative of) 'x' has d-edge to 'pvar', which is
     ancestor of 'y'. if so, then 'x' is connected to 'y' and there is
     dependency. */
  if (HAS_EDGE (rep_x, ancestor->id, qdag.dedges))
    return 1;
  else
    return 0;
}


/* Similar to above function for the other case, but must now do
   upward traversal for each root class of universal variable. */
static int
depends_exists_forall (QDPLLDepManQDAG * dm, Var * x, Var * y)
{
  assert (QDPLL_SCOPE_EXISTS (x->scope));
  assert (QDPLL_SCOPE_FORALL (y->scope));
  assert (x->scope->nesting < y->scope->nesting);
  Var *vars = dm->pcnf->vars;
  unsigned int stop_nesting = x->scope->nesting;

  Var *rep_y = uf_find (vars, y, 0);
  unsigned int i, end = rep_y->qdag.dedges.size;
  Edge *d;
  for (i = 0; i < end; i++)
    for (d = rep_y->qdag.dedges.table[i]; d; d = d->chain_next)
      {
	assert (d->tail_var == rep_y->id);
	VarID head_varid = d->head_var;
	Var *head_var = VARID2VARPTR (vars, head_varid);
	assert (QDPLL_SCOPE_EXISTS (head_var->scope));
	assert (UF_IS_REP (head_var, 0));
	Var *ancestor = depends_get_ancestor (vars, head_var, stop_nesting);
	if (HAS_EDGE (x, ancestor->id, qdag.sedges))
	  return 1;
      }
  return 0;
}


/* Given a stack of universal variables/classes, mark all those
   collected existential variables/classes where one of the universals
   depends on. */
static void
qdpll_dep_man_mark_inverse_depending_exists_forall (QDPLLMemMan * mm,
						    Var * vars,
						    VarPtrStack * var_stack)
{
  VarPtrStack marks;
  QDPLL_INIT_STACK (marks);

  Var **stack_p, **stack_e;
  for (stack_p = var_stack->start, stack_e = var_stack->top;
       stack_p < stack_e; stack_p++)
    {
      Var *stack_var = *stack_p;
      assert (QDPLL_SCOPE_FORALL (stack_var->scope));
      assert (UF_IS_REP (stack_var, 0));

      unsigned int i, end = stack_var->qdag.dedges.size;
      Edge *d;
      for (i = 0; i < end; i++)
	for (d = stack_var->qdag.dedges.table[i]; d; d = d->chain_next)
	  {
	    assert (d->tail_var == stack_var->id);
	    VarID cur_id = d->head_var;
	    assert (cur_id);
	    while (cur_id)
	      {
		Var *cur_var = VARID2VARPTR (vars, cur_id);
		if (QDPLL_VAR_NEG_MARKED (cur_var))
		  break;
		QDPLL_VAR_NEG_MARK (cur_var);
		QDPLL_PUSH_STACK (mm, marks, cur_var);
		assert (QDPLL_SCOPE_EXISTS (cur_var->scope));
		assert (UF_IS_REP (cur_var, 0));
		Edge **p, **e;
		for (p = cur_var->qdag.sedge_pq.elems_start,
		     e = cur_var->qdag.sedge_pq.elems_top; p < e; p++)
		  {
		    Edge *s = *p;
		    VarID owner_id = s->tail_var;
		    assert (owner_id);
		    Var *owner = VARID2VARPTR (vars, owner_id);
		    assert (QDPLL_SCOPE_EXISTS (owner->scope));
		    Var *owner_rep =
		      uf_find (vars, VARID2VARPTR (vars, owner_id), 1);
		    /* Check if 'owner_rep' occurs on stack then
		       unmark -> will be skipped then. */
		    if (QDPLL_VAR_POS_MARKED (owner_rep))
		      QDPLL_VAR_POS_UNMARK (owner_rep);
		  }
		cur_id = cur_var->qdag.cedges.cpar;
	      }
	  }
    }

  for (stack_p = marks.start, stack_e = marks.top;
       stack_p < stack_e; stack_p++)
    {
      assert (QDPLL_VAR_NEG_MARKED (*stack_p));
      QDPLL_VAR_NEG_UNMARK (*stack_p);
    }
  QDPLL_DELETE_STACK (mm, marks);
}


/* Given a stack of existential variables/classes, mark all those
   collected universal variables/classes where one of the existentials
   depends on. */
static void
qdpll_dep_man_mark_inverse_depending_forall_exists (QDPLLMemMan * mm,
						    Var * vars,
						    VarPtrStack * var_stack)
{
  VarPtrStack marks;
  QDPLL_INIT_STACK (marks);

  Var **stack_p, **stack_e;
  for (stack_p = var_stack->start, stack_e = var_stack->top;
       stack_p < stack_e; stack_p++)
    {
      Var *stack_var = *stack_p;
      VarID cur_id = stack_var->id;
      assert (cur_id);
      while (cur_id)
	{
	  Var *cur_var = VARID2VARPTR (vars, cur_id);
	  assert (QDPLL_SCOPE_EXISTS (cur_var->scope));
	  assert (UF_IS_REP (cur_var, 0));
	  if (QDPLL_VAR_NEG_MARKED (cur_var))
	    break;
	  QDPLL_VAR_NEG_MARK (cur_var);
	  QDPLL_PUSH_STACK (mm, marks, cur_var);
	  Edge **p, **e;
	  for (p = cur_var->qdag.dedge_pq.elems_start,
	       e = cur_var->qdag.dedge_pq.elems_top; p < e; p++)
	    {
	      Edge *s = *p;
	      VarID owner_id = s->tail_var;
	      assert (owner_id);
	      Var *owner = VARID2VARPTR (vars, owner_id);
	      assert (QDPLL_SCOPE_FORALL (owner->scope));
	      assert (UF_IS_REP (owner, 0));
	      /* Check if 'owner_rep' occurs on stack then unmark ->
	         will be skipped then. */
	      if (QDPLL_VAR_POS_MARKED (owner))
		QDPLL_VAR_POS_UNMARK (owner);
	    }
	  cur_id = cur_var->qdag.cedges.cpar;
	}
    }

  for (stack_p = marks.start, stack_e = marks.top;
       stack_p < stack_e; stack_p++)
    {
      assert (QDPLL_VAR_NEG_MARKED (*stack_p));
      QDPLL_VAR_NEG_UNMARK (*stack_p);
    }
  QDPLL_DELETE_STACK (mm, marks);
}


static unsigned int
count_non_learnt_occs (Var * var)
{
  unsigned int result = 0;
  OccListIterator it;
  LitID lit = -var->id;
  OLITER_INIT (it, var->neg_occ_clauses.first, var->neg_occ_clauses.foffset,
	       lit);
  while (OLITER_CUR (it))
    {
      assert (!OLITER_CUR (it)->learnt);
      if (!OLITER_CUR (it)->learnt && !OLITER_CUR (it)->counted)
	{
	  OLITER_CUR (it)->counted = 1;
	  result++;
	}
      OLITER_NEXT (it, lit);
    }
  lit = var->id;
  OLITER_INIT (it, var->pos_occ_clauses.first, var->pos_occ_clauses.foffset,
	       lit);
  while (OLITER_CUR (it))
    {
      assert (!OLITER_CUR (it)->learnt);
      if (!OLITER_CUR (it)->learnt && !OLITER_CUR (it)->counted)
	{
	  OLITER_CUR (it)->counted = 1;
	  result++;
	}
      OLITER_NEXT (it, lit);
    }
  return result;
}


/* -------------------- END: INTERNAL FUNCTIONS -------------------- */


/* -------------------- START: CORE FUNCTIONS -------------------- */
static void
qdpll_dep_man_init (QDPLLDepManGeneric * dmg)
{
  QDPLLDepManQDAG *dm = (QDPLLDepManQDAG *) dmg;
  assert (!dm->state.init);
  QDPLL_ABORT_DEPMAN (dm->state.init,
		      "dependency manager already initialized.");
  dm->state.init = 1;

  /* Traverse all scopes, make-set on all variables, link singleton
     classes to scope's list of equivalence classes. */
  Scope *s;
  for (s = dm->pcnf->scopes.first; s; s = s->link.next)
    {
      VarID *p, *e;
      Var *vars = dm->pcnf->vars;
      for (p = s->vars.start, e = s->vars.top; p < e; p++)
	{
	  Var *var = VARID2VARPTR (vars, *p);
	  if (QDPLL_VAR_MARKED_PROPAGATED (var) && var->decision_level <= 0)
	    continue;
	  assert (!var->qdag.uf[0].par);
	  assert (!var->qdag.uf[1].par);
	  uf_make_set (var, 0);
	  VAR_LINK (vars, s->classes[0], var, qdag.uf[0].class_link);
	  if (QDPLL_SCOPE_EXISTS (s))
	    {
	      uf_make_set (var, 1);
	      VAR_LINK (vars, s->classes[1], var, qdag.uf[1].class_link);
	    }
	}
    }

  extract_dependencies (dm);
  fill_candidates (dm);

#ifndef NDEBUG
#if QDAG_ASSERT_CANDIDATE_MARKS_BY_REMARKING
  assert_candidate_marks_by_remarking (dm);
#endif
#endif
}


/* Allocate memory for edge tables and priority queues. This can be redundant 
   for some variables, especially from innermost scope. */
static void
qdpll_dep_man_notify_init_variable (QDPLLDepManGeneric * dmg, VarID id)
{
  QDPLLDepManQDAG *dm = (QDPLLDepManQDAG *) dmg;
  assert (id > 0 && id < dm->pcnf->size_vars);
  QDPLLMemMan *mm = dm->mm;
  Var *v = VARID2VARPTR (dm->pcnf->vars, id);
  assert (!v->qdag.dedges.table);
  et_init (mm, &(v->qdag.dedges), DEFAULT_EDGE_TABLE_SIZE);
  assert (!v->qdag.sedges.table);
  et_init (mm, &(v->qdag.sedges), DEFAULT_EDGE_TABLE_SIZE);
  assert (!v->qdag.dedge_pq.elems_start);
  pq_init (mm, &(v->qdag.dedge_pq), DEFAULT_EDGE_PQUEUE_SIZE);
  assert (!v->qdag.sedge_pq.elems_start);
  pq_init (mm, &(v->qdag.sedge_pq), DEFAULT_EDGE_PQUEUE_SIZE);
}


/* De-allocate memory for edge tables and priority queues for given variable. */
static void
qdpll_dep_man_notify_reset_variable (QDPLLDepManGeneric * dmg, VarID id)
{
  QDPLLDepManQDAG *dm = (QDPLLDepManQDAG *) dmg;
  assert (id > 0 && id < dm->pcnf->size_vars);
  QDPLLMemMan *mm = dm->mm;
  Var *v = VARID2VARPTR (dm->pcnf->vars, id);
  unsigned int i, end;
  Edge *d, *n;

  if (v->qdag.dedges.table)
    {
      assert (v->qdag.dedges.size > 0);
      /* Delete edges in table. */
      end = v->qdag.dedges.size;
      for (i = 0; i < end; i++)
	for (d = v->qdag.dedges.table[i]; d; d = n)
	  {
	    n = d->chain_next;
	    delete_edge (mm, d);
	  }
      et_delete (mm, &(v->qdag.dedges));
    }

  if (v->qdag.sedges.table)
    {
      assert (v->qdag.dedges.size > 0);
      /* Delete edges in table. */
      end = v->qdag.sedges.size;
      for (i = 0; i < end; i++)
	for (d = v->qdag.sedges.table[i]; d; d = n)
	  {
	    n = d->chain_next;
	    delete_edge (mm, d);
	  }
      et_delete (mm, &(v->qdag.sedges));
    }

  /* Edges already deleted, can free pqueues immediately. */
  if (v->qdag.dedge_pq.elems_start)
    pq_delete (mm, &(v->qdag.dedge_pq));
  if (v->qdag.sedge_pq.elems_start)
    pq_delete (mm, &(v->qdag.sedge_pq));
}


static void
qdpll_dep_man_reset (QDPLLDepManGeneric * dmg)
{
  QDPLLDepManQDAG *dm = (QDPLLDepManQDAG *) dmg;
  QDPLLMemMan *mm = dm->mm;

  Var *p, *e;
  for (p = dm->pcnf->vars, e = p + dm->pcnf->size_vars; p < e; p++)
    {
      if (p->id)
	{
	  /* Remove variable from priority queue. This is done right
	     here although queue is implemented outside (this avoids
	     traversing the variable list again). */
	  p->priority_pos = QDPLL_INVALID_PQUEUE_POS;
	  qdpll_dep_man_notify_reset_variable (dmg, p->id);
	  memset (&(p->qdag), 0, sizeof (QDAG));
	  qdpll_dep_man_notify_init_variable (dmg, p->id);
	  p->mark_is_candidate = 0;
	}
    }

  /* Reset data in dep-man. */
  dm->state.init = 0;
  dm->candidates.first = dm->candidates.last = 0;

  /* Reset class links in scopes. */
  Scope *s;
  for (s = dm->pcnf->scopes.first; s; s = s->link.next)
    {
      s->classes[0].first = s->classes[0].last = 0;
      s->classes[1].first = s->classes[1].last = 0;
    }
}


static VarID
qdpll_dep_man_get_candidate (QDPLLDepManGeneric * dmg)
{
  QDPLLDepManQDAG *dm = (QDPLLDepManQDAG *) dmg;
  assert (dm->state.init);

  VarID result;
  if ((result = dm->candidates.first))
    {
      Var *vars = dm->pcnf->vars;
      Var *r = VARID2VARPTR (vars, result);
      VAR_UNLINK (vars, dm->candidates, r, qdag.cand_link);
    }

  return result;
}


static void
qdpll_dep_man_notify_inactive (QDPLLDepManGeneric * dmg, VarID id)
{
  QDPLLDepManQDAG *dm = (QDPLLDepManQDAG *) dmg;
  assert (id > 0 && id < dm->pcnf->size_vars);
  assert (dm->state.init);
  Var *vars = dm->pcnf->vars;
  Var *inactive_var = VARID2VARPTR (vars, id);
  assert (QDPLL_VAR_MARKED_PROPAGATED (inactive_var));
#ifndef NDEBUG
  assert (!inactive_var->qdag.mark_notified_inactive);
  inactive_var->qdag.mark_notified_inactive = 1;
#endif

  if (QDPLL_SCOPE_EXISTS (inactive_var->scope))
    notify_inactive_exists (dm, inactive_var);
  else
    {
      assert (QDPLL_SCOPE_FORALL (inactive_var->scope));
      Var *rep = uf_find (vars, inactive_var, 0);
      assert (rep->qdag.cnt.forall.non_propagated_in_class);
      rep->qdag.cnt.forall.non_propagated_in_class--;
      assert (rep->qdag.cnt.forall.non_propagated_in_class ==
	      count_non_propagated_in_class (vars, rep));

      if (rep->qdag.cnt.forall.non_propagated_in_class == 0)
	notify_inactive_forall (dm, rep);
    }

#ifndef NDEBUG
#if QDAG_ASSERT_CANDIDATE_LIST
  assert_candidate_list (dm);
#endif
#if QDAG_ASSERT_CANDIDATE_MARKS_BY_REMARKING
  assert_candidate_marks_by_remarking (dm);
#endif
#endif
}


static void
qdpll_dep_man_notify_active (QDPLLDepManGeneric * dmg, VarID id)
{
  QDPLLDepManQDAG *dm = (QDPLLDepManQDAG *) dmg;
  assert (id > 0 && id < dm->pcnf->size_vars);
  assert (dm->state.init);
  Var *vars = dm->pcnf->vars;
  Var *active_var = VARID2VARPTR (vars, id);
  assert (!QDPLL_VAR_MARKED_PROPAGATED (active_var));
#ifndef NDEBUG
  assert (active_var->qdag.mark_notified_inactive);
  active_var->qdag.mark_notified_inactive = 0;
#endif

  if (QDPLL_SCOPE_EXISTS (active_var->scope))
    notify_active_exists (dm, active_var);
  else
    {
      assert (QDPLL_SCOPE_FORALL (active_var->scope));
      Var *rep = uf_find (vars, active_var, 0);
      rep->qdag.cnt.forall.non_propagated_in_class++;
      assert (count_non_propagated_in_class (vars,
					     rep) ==
	      rep->qdag.cnt.forall.non_propagated_in_class);

      if (rep->qdag.cnt.forall.non_propagated_in_class == 1)
	notify_active_forall (dm, rep);
    }

  /* Link active-var to candidates again, if is candidate. Check, if
     not already in list. */
  if (active_var->mark_is_candidate)
    {
      if (!active_var->qdag.cand_link.prev && !active_var->qdag.cand_link.next
	  && id != dm->candidates.first)
	VAR_LINK (vars, dm->candidates, active_var, qdag.cand_link);
    }

#ifndef NDEBUG
#if QDAG_ASSERT_CANDIDATE_LIST
  assert_candidate_list (dm);
#endif
#if QDAG_ASSERT_CANDIDATE_MARKS_BY_REMARKING
  assert_candidate_marks_by_remarking (dm);
#endif
#endif
}


static int
qdpll_dep_man_is_candidate (QDPLLDepManGeneric * dmg, VarID id)
{
  QDPLLDepManQDAG *dm = (QDPLLDepManQDAG *) dmg;
  assert (id > 0 && id < dm->pcnf->size_vars);
  assert (dm->state.init);
  Var *v = VARID2VARPTR (dm->pcnf->vars, id);
  return v->mark_is_candidate;
}


static int
qdpll_dep_man_is_init (QDPLLDepManGeneric * dmg)
{
  QDPLLDepManQDAG *dm = (QDPLLDepManQDAG *) dmg;
  return dm->state.init;
}


/* Checks if 'y' depends on 'x', i.e. whether it is the case that 
   'y \in D^{std}(x)'. */
static int
qdpll_dep_man_depends (QDPLLDepManGeneric * dmg, VarID x, VarID y)
{
  QDPLLDepManQDAG *dm = (QDPLLDepManQDAG *) dmg;
  assert (dm->state.init);
  Var *vars = dm->pcnf->vars;

  Var *var_x = VARID2VARPTR (vars, x);
  Var *var_y = VARID2VARPTR (vars, y);
  Scope *scope_x = var_x->scope;
  Scope *scope_y = var_y->scope;

  /* Do early check. */
  if (scope_x->nesting >= scope_y->nesting || scope_x->type == scope_y->type)
    return 0;
  else if (QDPLL_SCOPE_FORALL (scope_x))
    {
      assert (QDPLL_SCOPE_EXISTS (scope_y));
      return depends_forall_exists (dm, var_x, var_y);
    }
  else
    {
      assert (QDPLL_SCOPE_EXISTS (scope_x));
      assert (QDPLL_SCOPE_FORALL (scope_y));
      return depends_exists_forall (dm, var_x, var_y);
    }
}


static int
qdpll_dep_man_con_subseteq (QDPLLDepManGeneric * dmg, VarID x, VarID y)
{
  QDPLLDepManQDAG *dm = (QDPLLDepManQDAG *) dmg;
  assert (dm->state.init);
  Var *vars = dm->pcnf->vars;

  Var *var_x = VARID2VARPTR (vars, x);
  Var *var_y = VARID2VARPTR (vars, y);
  Scope *scope_x = var_x->scope;
  Scope *scope_y = var_y->scope;

  /* Do early check. */
  if (scope_x->nesting != scope_y->nesting)
    return 0;
  else
    {
      assert (scope_x->type == scope_y->type);
      if (QDPLL_SCOPE_FORALL (scope_x))
	{
	  /* Must get representatives for universal variables because
	     of merged universal classes. */
	  assert (QDPLL_SCOPE_FORALL (scope_y));
	  var_x = uf_find (vars, var_x, 0);
	  var_y = uf_find (vars, var_y, 0);
	}
      return var_edges_subseteq (var_x, var_y);
    }
}


/* Returns x's class representative wrt. d-edges. */
static VarID
qdpll_dep_man_get_class_rep (QDPLLDepManGeneric * dmg, VarID x,
			     const unsigned int ufoffset)
{
  QDPLLDepManQDAG *dm = (QDPLLDepManQDAG *) dmg;
  assert (dm->state.init);
  Var *vars = dm->pcnf->vars;
  Var *var_x = VARID2VARPTR (vars, x);
  assert (ufoffset <= 1);
  assert (QDPLL_SCOPE_EXISTS (var_x->scope) || ufoffset == 0);
  return uf_find (vars, var_x, ufoffset)->id;
}


static void
qdpll_dep_man_mark_inverse_depending (QDPLLDepManGeneric * dmg,
				      VarPtrStack * var_stack,
				      const QDPLLQuantifierType type)
{
  QDPLLDepManQDAG *dm = (QDPLLDepManQDAG *) dmg;
  assert (dm->state.init);
  Var *vars = dm->pcnf->vars;
  if (type == QDPLL_QTYPE_EXISTS)
    qdpll_dep_man_mark_inverse_depending_exists_forall (dm->mm, vars,
							var_stack);
  else
    qdpll_dep_man_mark_inverse_depending_forall_exists (dm->mm, vars,
							var_stack);
}

/* -------------------- END: CORE FUNCTIONS -------------------- */


/* -------------------- START: PUBLIC FUNCTIONS -------------------- */

QDPLLDepManQDAG *
qdpll_qdag_dep_man_create (QDPLLMemMan * mm, QDPLLPCNF * pcnf, QDPLL * qdpll)
{
  QDPLLDepManQDAG *dm =
    (QDPLLDepManQDAG *) qdpll_malloc (mm, sizeof (QDPLLDepManQDAG));
  dm->mm = mm;
  dm->pcnf = pcnf;

  dm->dmg.qdpll = qdpll;

  /* Set fptrs */
  dm->dmg.init = qdpll_dep_man_init;
  dm->dmg.reset = qdpll_dep_man_reset;
  dm->dmg.get_candidate = qdpll_dep_man_get_candidate;
  dm->dmg.notify_inactive = qdpll_dep_man_notify_inactive;
  dm->dmg.notify_active = qdpll_dep_man_notify_active;
  dm->dmg.is_candidate = qdpll_dep_man_is_candidate;
  dm->dmg.notify_init_variable = qdpll_dep_man_notify_init_variable;
  dm->dmg.notify_reset_variable = qdpll_dep_man_notify_reset_variable;
  dm->dmg.is_init = qdpll_dep_man_is_init;

  dm->dmg.depends = qdpll_dep_man_depends;
  dm->dmg.con_subseteq = qdpll_dep_man_con_subseteq;
  dm->dmg.get_class_rep = qdpll_dep_man_get_class_rep;
  dm->dmg.mark_inverse_depending = qdpll_dep_man_mark_inverse_depending;
  return dm;
}


void
qdpll_qdag_dep_man_delete (QDPLLDepManQDAG * dm)
{
  qdpll_free (dm->mm, dm, sizeof (QDPLLDepManQDAG));
}

/* -------------------- END: PUBLIC FUNCTIONS -------------------- */
