/* dhcp_options.h
 * 
 *                Dynamic Host Configuration Options utilities
 * 
 *  Copyright (c) Jason Vas Dias <jvdias@redhat.com>, Red Hat Inc., 2005
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation at 
 *           http://www.fsf.org/licensing/licenses/gpl.txt
 *  and included in this software distribution as the "LICENSE" file.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

/* Loosely based on (and partly copied from) ISC DHCP "tables.c":
 * tables.c
 * 2004/09/01 17:06:35 dhankins Exp
 * Copyright (c) 2004 by Internet Systems Consortium, Inc. ("ISC")
 * Copyright (c) 1995-2003 by Internet Software Consortium
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * DHCP Option names, formats and codes, from RFC1533.
 */
 
#ifndef DHC_OPTIONS_H
#define DHC_OPTIONS_H

/* '#define DHC_DEFINE_OPTION_TABLES'
 * to define the option tables when this file is included
 */  

/*
 *  Type Codes:
 */
typedef
enum dhc_ot_e
{   /* These are the same codes as used by dhcpd / dhclient,
     * and which are written by dhclient with '-x' option to 
     * new option information variables named x._universe_.x:
     */
    /* "atomic" types: */
    DHC_T_IP_ADDRESS        = 'I', /*    I - IP address */
    DHC_T_HEX_STRING        = 'X', /*    X - either an ASCII string or binary data.   On output, the string is */
                                   /*        scanned to see if it's printable ASCII and, if so, output as a */
                                   /*        quoted string.   If not, it's output as colon-seperated hex.   On */
                                   /*        input, the option can be specified either as a quoted string or as */
                                   /*        a colon-seperated hex list. */
    DHC_T_DOMAIN_NAME       = 'd', /*    d - Domain name (i.e., FOO or FOO.BAR). */
    DHC_T_INT32             = 'l', /*    l - 32-bit signed integer */
    DHC_T_UINT32            = 'L', /*    L - 32-bit unsigned integer */
    DHC_T_INT16             = 's', /*    s - 16-bit signed integer */
    DHC_T_UINT16            = 'S', /*    S - 16-bit unsigned integer */
    DHC_T_CHAR              = 'b', /*    b - 8-bit signed integer */
    DHC_T_UCHAR             = 'B', /*    B - 8-bit unsigned integer */
    DHC_T_TEXT              = 't', /*    t - ASCII text */
    DHC_T_BOOL              = 'f', /*    f - flag (true or false) */
    DHC_T_IMPLICIT_BOOL     = 'F', /*    F - implicit flag - the presence of the option indicates that the */

    /* compound types: */
    DHC_T_ARRAY             = 'A', /*    A - array of whatever precedes (e.g., IA means array of IP addresses) */
    DHC_T_LIST              = 'a', /*    a - array of the preceding character (e.g., IIa means two or more IP */
                                   /*        addresses) */
    /* there are NO enumeration type dhcp options defined 
     * and no means of specifying user defined enumeration types -
     * the 'parameters.state' option we deal with as an integer or text
     */
    DHC_T_ENUMERATION       = 'N', /*    N - enumeration.   N is followed by a text string containing */
                                   /*        the name of the set of enumeration values to parse or emit, */
                                   /*        followed by a '.'.   The width of the data is specified in the */
                                   /*        named enumeration.   Named enumerations are tracked in parse.c. */
    /* these are types internal to DHCP and can  safely be ignored .
     * ( they are not emitted by dhclient in environment -
     *   except "vendor-option-space", which we deal with by use of
     *   '._universe_.' information.
     * )
     */
    DHC_T_OPTION_SPACE      = 'U', /*    U - name of an option space (universe) */

                                   /*        flag is true. */
    DHC_T_OPTIONAL          = 'o', /*    o - the preceding value is optional. */
    DHC_T_ENCAPSULATION     = 'E', /*    E - encapsulation, string or colon-seperated hex list (the latter */
                                   /*        two for parsing).   E is followed by a text string containing */
                                   /*        the name of the option space to encapsulate, followed by a '.'. */
                                   /*        If the E is immediately followed by '.', the applicable vendor */
                                   /*        option space is used if one is defined. */
    DHC_T_ENCAPSULATION_NO_DIRECTIVE= 'e',
                                   /*    e - If an encapsulation directive is not the first thing in the string, */
                                   /*        the option scanner requires an efficient way to find the encapsulation. */
                                   /*        This is done by placing a 'e' at the beginning of the option.   The */
                                   /*        'e' has no other purpose, and is not required if 'E' is the first */
                                   /*        thing in the option. */
    DHC_T_NONE               = '\0'
} DHCP_Option_Type;

typedef
enum
{
    DHC_M_NONE,
    DHC_M_IP_ADDRESS,
    DHC_M_HEX_STRING,
    DHC_M_TEXT,
    DHC_M_DOMAIN_NAME,
    DHC_M_INT32,
    DHC_M_UINT32,
    DHC_M_INT16,
    DHC_M_UINT16,
    DHC_M_CHAR,
    DHC_M_UCHAR,
    DHC_M_BOOL,
    DHC_M_IMPLICIT_BOOL,
    DHC_M_ENUMERATION,
    DHC_N_ATOMIC_TYPES
}   DHCP_Option_Type_Map_Code;

typedef
uint8_t DHCP_Option_Type_Map_Table[  DHC_N_ATOMIC_TYPES ];

typedef
struct dhco_m_s *  DHCP_Option_Type_Mapping;

/* DHCP Code Spaces:
 *
 * Option Codes have unique values in one of these "DHCP_Universe" 
 * code spaces:
 */

#define DHCP_OPTION_MAX   256
/*  DHCP can define no more than 256 options, 0-255, in a "universe" */

#define DHCP_UNIVERSE_MAX 10
/*  DHCP currently supports no more than 10 universes (a hardcoded constant in tables.c).
 *  It defines 5 of its own and supports up to 5 user defined option spaces.
 */

typedef 
enum dhcp_u_e
{
    DHC_P_Universe=0,              /* Non-Option DHCP Parameters (eg. ip-address, interface).... */
    DHC_O_Universe=1,              /* Proper DHCP Options as defined in RFC 1533 */
    DHC_RAI_Universe=2,            /* Relay Agent Information sub-option encapsulation */
    DHC_NWIP_Universe=3,           /* NetWare/IP sub-option encapsulation */
    DHC_FQDN_Universe=4,           /* Fully Qualified Domain Name sub-option encapsulation */
    DHC_UNIVERSE_N=5
} DHCP_Universe;


/*  DHC "Parameter" Codes: 
 *
 *  These are for fields in the DHCP packet that we present in the same way
 *  as options.
 */
typedef 
enum dhcp_c_e
{
    DHCP_PAD                            =       0, /* Unused */
    DHCP_IP_ADDRESS                     =       1, 
    DHCP_INTERFACE                      =       2,
    DHCP_DHCP_STATE                     =       3,
    DHCP_PID                            =       4,
    DHCP_EXPIRY                         =       5,
    DHCP_NETWORK_NUMBER                 =       6,
    DHCP_KEEP_OLD_IP                    =       7,
    DHCP_END                            =       7,
    DHCP_N                              =       8 /* Unused */
} DHCP_Parameter_Codes;

/*  DHC Option Codes: 
 */
typedef
enum dhcp_oc_e
{
    DHCO_PAD				=	0, /* Unused */
    DHCO_SUBNET_MASK			=	1,
    DHCO_TIME_OFFSET			=	2,
    DHCO_ROUTERS			=	3,
    DHCO_TIME_SERVERS			=	4,
    DHCO_NAME_SERVERS			=	5,
    DHCO_DOMAIN_NAME_SERVERS		=	6,
    DHCO_LOG_SERVERS			=	7,
    DHCO_COOKIE_SERVERS			=	8,
    DHCO_LPR_SERVERS			=	9,
    DHCO_IMPRESS_SERVERS		=	10,
    DHCO_RESOURCE_LOCATION_SERVERS	=	11,
    DHCO_HOST_NAME			=	12,
    DHCO_BOOT_SIZE			=	13,
    DHCO_MERIT_DUMP			=	14,
    DHCO_DOMAIN_NAME			=	15,
    DHCO_SWAP_SERVER			=	16,
    DHCO_ROOT_PATH			=	17,
    DHCO_EXTENSIONS_PATH		=	18,
    DHCO_IP_FORWARDING			=	19,
    DHCO_NON_LOCAL_SOURCE_ROUTING	=	20,
    DHCO_POLICY_FILTER			=	21,
    DHCO_MAX_DGRAM_REASSEMBLY		=	22,
    DHCO_DEFAULT_IP_TTL			=	23,
    DHCO_PATH_MTU_AGING_TIMEOUT		=	24,
    DHCO_PATH_MTU_PLATEAU_TABLE		=	25,
    DHCO_INTERFACE_MTU			=	26,
    DHCO_ALL_SUBNETS_LOCAL		=	27,
    DHCO_BROADCAST_ADDRESS		=	28,
    DHCO_PERFORM_MASK_DISCOVERY		=	29,
    DHCO_MASK_SUPPLIER			=	30,
    DHCO_ROUTER_DISCOVERY		=	31,
    DHCO_ROUTER_SOLICITATION_ADDRESS	=	32,
    DHCO_STATIC_ROUTES			=	33,
    DHCO_TRAILER_ENCAPSULATION		=	34,
    DHCO_ARP_CACHE_TIMEOUT		=	35,
    DHCO_IEEE802_3_ENCAPSULATION	=	36,
    DHCO_DEFAULT_TCP_TTL		=	37,
    DHCO_TCP_KEEPALIVE_INTERVAL		=	38,
    DHCO_TCP_KEEPALIVE_GARBAGE		=	39,
    DHCO_NIS_DOMAIN			=	40,
    DHCO_NIS_SERVERS			=	41,
    DHCO_NTP_SERVERS			=	42,
    DHCO_VENDOR_ENCAPSULATED_OPTIONS	=	43,
    DHCO_NETBIOS_NAME_SERVERS		=	44,
    DHCO_NETBIOS_DD_SERVER		=	45,
    DHCO_NETBIOS_NODE_TYPE		=	46,
    DHCO_NETBIOS_SCOPE			=	47,
    DHCO_FONT_SERVERS			=	48,
    DHCO_X_DISPLAY_MANAGER		=	49,
    DHCO_DHCP_REQUESTED_ADDRESS		=	50,
    DHCO_DHCP_LEASE_TIME		=	51,
    DHCO_DHCP_OPTION_OVERLOAD		=	52,
    DHCO_DHCP_MESSAGE_TYPE		=	53,
    DHCO_DHCP_SERVER_IDENTIFIER		=	54,
    DHCO_DHCP_PARAMETER_REQUEST_LIST	=	55,
    DHCO_DHCP_MESSAGE			=	56,
    DHCO_DHCP_MAX_MESSAGE_SIZE		=	57,
    DHCO_DHCP_RENEWAL_TIME		=	58,
    DHCO_DHCP_REBINDING_TIME		=	59,
    DHCO_VENDOR_CLASS_IDENTIFIER	=	60,
    DHCO_DHCP_CLIENT_IDENTIFIER		=	61,
    DHCO_NWIP_DOMAIN_NAME		=	62,
    DHCO_NWIP_SUBOPTIONS		=	63,
    DHCO_NISPLUS_DOMAIN                 =       64,
    DHCO_NISPLUS_SERVERS                =       65,
    DHCO_TFTP_SERVER_NAME               =       66,
    DHCO_BOOTFILE_NAME                  =       67,
    DHCO_MOBILE_IP_HOME_AGENT           =       68,
    DHCO_SMTP_SERVER                    =       69,
    DHCO_POP_SERVER                     =       70,
    DHCO_NNTP_SERVER                    =       71,
    DHCO_WWW_SERVER                     =       72,
    DHCO_FINGER_SERVER                  =       73,
    DHCO_IRC_SERVER                     =       74,
    DHCO_STREETTALK_SERVER              =       75,
    DHCO_STREETTALK_DIRECTORY_ASSISTANCE_SERVER=76,    
    DHCO_USER_CLASS			=	77,
    DHCO_SLP_DIRECTORY_AGENT            =       78,
    DHCO_SLP_SERVICE_SCOPE              =       79,
    DHCO_FQDN				=	81,
    DHCO_DHCP_AGENT_OPTIONS		=	82,
    DHCO_NDS_SERVERS                    =       85,
    DHCO_NDS_TREE_NAME                  =       86,
    DHCO_NDS_CONTEXT                    =       87,
    DHCO_UAP_SERVERS                    =       98,     
    DHCO_SUBNET_SELECTION		=	118,
    DHCO_AUTHENTICATE			=	210,
    DHCO_END                            =       210,
    DHCO_N                              =       211
}   DHCP_Option_Code;

/* DHCP Relay Agent Information Sub-Options:
 */
typedef
enum dho_rai_c_e
{
    DHCO_RAI_PAD                         =       0,
    DHCO_RAI_CIRCUIT_ID	                 =       1,
    DHCO_RAI_REMOTE_ID	                 =       2,
    DHCO_RAI_AGENT_ID	                 =       3,
    DHCO_RAI_END                         =       3,
    DHCO_RAI_N                           =       4
} DHCP_RAI_Options;

/* DHCP NetWare/IP Sub-Options:
 */
typedef
enum dho_nwip_c_e
{
    DHCO_NWIP_PAD                        =       0,
    DHCO_NWIP_ILL1                       =       1,
    DHCO_NWIP_ILL2                       =       2,
    DHCO_NWIP_ILL3                       =       3,
    DHCO_NWIP_ILL4                       =       4,
    DHCO_NWIP_NSQ_BROADCAST              =       5,
    DHCO_NWIP_PREFERRED_DSS              =       6,
    DHCO_NWIP_NEAREST_NWIP_SERVER        =       7,
    DHCO_NWIP_AUTORETRIES                =       8,
    DHCO_NWIP_AUTORETRY_SECS             =       9,
    DHCO_NWIP_1_1                        =      10,
    DHCO_NWIP_PRIMARY_DSS                =      11,
    DHCO_NWIP_END                        =      11,
    DHCO_NWIP_N                          =      12
} DHCP_NWIP_Options;

/* DHCP Fully Qualified Domain Name Sub-Options:
 */
typedef
enum dho_fqdn_c_e
{
    DHCO_FQDN_PAD                        =       0,
    DHCO_FQDN_NO_CLIENT_UPDATE		 =       1,
    DHCO_FQDN_SERVER_UPDATE		 =       2,
    DHCO_FQDN_ENCODED			 =       3,
    DHCO_FQDN_RCODE1			 =       4,
    DHCO_FQDN_RCODE2			 =       5,
    DHCO_FQDN_HOSTNAME			 =       6,
    DHCO_FQDN_DOMAINNAME		 =       7,
    DHCO_FQDN_FQDN			 =       8,
    DHCO_FQDN_END                        =       8,
    DHCO_FQDN_N                          =	 9
} DHCP_FQDN_Options;

typedef 
enum  dhc_of_e
{
    DHC_F_DEFINED,
    DHC_F_NEW
} DHC_Option_Flags;

/* DHC Option Structure:
 */
typedef 
struct dhc_o_s
{
    char *name;
    char *type;
    DHCP_Universe universe;
    DHCP_Option_Code code ;
    uint32_t         flags;
} DHC_Option;

/* DHC Universe (Option Space) Structure:
 */
typedef struct dhcp_u_s
{   char *name;
    DHCP_Universe code;
    unsigned n;
    DHC_Option* universe;
}  DHC_Universe;

/* Structures for Enumeration types:
 */
typedef
struct dhc_ev_s
{
    char *name;
    uint32_t number;
} DHCP_EnumValue; 

typedef
struct dhc_e_s
{
    char *name;
    DHCP_Universe universe;
    int  n;    
    DHCP_EnumValue *values;
    void      *ts;
    void      *tn;
} DHCP_Enumeration; 

typedef 
struct dhc_a_s
{
    char *name;
    DHC_Option *opt;
} DHC_Alias;

#define DHCP_UNIVERSE_SEPARATOR '.'

/* Global Options Structures:
 */
#ifndef  DHC_DEFINE_OPTION_TABLES

/* Declarations:
 */

extern DHC_Universe
dhcp_universes [ DHCP_UNIVERSE_MAX ];

extern DHC_Option 
dhcp_parameters   [DHCP_OPTION_MAX];

extern DHC_Option 
dhcp_options      [DHCP_OPTION_MAX];

extern DHC_Option
dhcp_rai_options  [DHCP_OPTION_MAX];

extern DHC_Option
dhcp_fqdn_options [DHCP_OPTION_MAX];

extern DHC_Option 
dhcp_nwip_options [DHCP_OPTION_MAX];

extern DHCP_Enumeration dhcp_enums[];
extern int n_dhcp_enums;

extern DHC_Alias
dhcp_aliases[];
extern int n_dhcp_aliases;


#else /* DHC_DEFINE_OPTION_TABLES */

/* Definitions:
 */

DHC_Option
dhcp_parameters [ DHCP_N ] =
{
    { "pad", "",                                         DHC_P_Universe,        0, 0 },
    { "ip-address", "I",			       	 DHC_P_Universe,        1, 0 },
    { "interface",  "t",                                 DHC_P_Universe,        2, 0 },
    { "reason",     "Nstate",                          	 DHC_P_Universe,	3, 0 },
    { "pid",        "L",                                 DHC_P_Universe,        4, 0 },
    { "expiry",     "L",                                 DHC_P_Universe,        5, 0 }, 
    { "network-number", "I",                             DHC_P_Universe,        6, 0 },
    { "keep-old-ip","f",                                 DHC_P_Universe,        7, 0 }
};
/* These are extra "options" sent by dhclient .
 * 
 * Luckily, these names do not conflict with any other universe...
 *
 * And you cannot add new options to "parameters" universe ...
 */

DHC_Option
dhcp_options [DHCP_OPTION_MAX] = 
{
    { "pad", "",					 DHC_O_Universe,	0, 0 }, 
    { "subnet-mask", "I",				 DHC_O_Universe,	1, 0 },
    { "time-offset", "l",				 DHC_O_Universe,	2, 0 },
    { "routers", "IA",				 	 DHC_O_Universe,	3, 0 },
    { "time-servers", "IA",				 DHC_O_Universe,	4, 0 },
    { "ien116-name-servers", "IA",			 DHC_O_Universe,	5, 0 },
    { "domain-name-servers", "IA",			 DHC_O_Universe,	6, 0 },
    { "log-servers", "IA",				 DHC_O_Universe,	7, 0 },
    { "cookie-servers", "IA",			 	 DHC_O_Universe,	8, 0 },
    { "lpr-servers", "IA",				 DHC_O_Universe,	9, 0 },
    { "impress-servers", "IA",			 	 DHC_O_Universe,	10, 0 },
    { "resource-location-servers", "IA",		 DHC_O_Universe,	11, 0 },
    { "host-name", "X",				 	 DHC_O_Universe,	12, 0 },
    { "boot-size", "S",				 	 DHC_O_Universe,	13, 0 },
    { "merit-dump", "t",				 DHC_O_Universe,	14, 0 },
    { "domain-name", "t",				 DHC_O_Universe,	15, 0 },
    { "swap-server", "I",				 DHC_O_Universe,	16, 0 },
    { "root-path", "t",				 	 DHC_O_Universe,	17, 0 },
    { "extensions-path", "t",			 	 DHC_O_Universe,	18, 0 },
    { "ip-forwarding", "f",				 DHC_O_Universe,	19, 0 },
    { "non-local-source-routing", "f",		 	 DHC_O_Universe,	20, 0 },
    { "policy-filter", "IIA",			 	 DHC_O_Universe,	21, 0 },
    { "max-dgram-reassembly", "S",			 DHC_O_Universe,	22, 0 },
    { "default-ip-ttl", "B",			 	 DHC_O_Universe,	23, 0 },
    { "path-mtu-aging-timeout", "L",		 	 DHC_O_Universe,	24, 0 },
    { "path-mtu-plateau-table", "SA",		 	 DHC_O_Universe,	25, 0 },
    { "interface-mtu", "S",				 DHC_O_Universe,	26, 0 },
    { "all-subnets-local", "f",			 	 DHC_O_Universe,	27, 0 },
    { "broadcast-address", "I",			 	 DHC_O_Universe,	28, 0 },
    { "perform-mask-discovery", "f",		 	 DHC_O_Universe,	29, 0 },
    { "mask-supplier", "f",				 DHC_O_Universe,	30, 0 },
    { "router-discovery", "f",			 	 DHC_O_Universe,	31, 0 },
    { "router-solicitation-address", "I",		 DHC_O_Universe,	32, 0 },
    { "static-routes", "IIA",			 	 DHC_O_Universe,	33, 0 },
    { "trailer-encapsulation", "f",			 DHC_O_Universe,	34, 0 },
    { "arp-cache-timeout", "L",			 	 DHC_O_Universe,	35, 0 },
    { "ieee802-3-encapsulation", "f",		 	 DHC_O_Universe,	36, 0 },
    { "default-tcp-ttl", "B",			 	 DHC_O_Universe,	37, 0 },
    { "tcp-keepalive-interval", "L",		 	 DHC_O_Universe,	38, 0 },
    { "tcp-keepalive-garbage", "f",			 DHC_O_Universe,	39, 0 },
    { "nis-domain", "t",				 DHC_O_Universe,	40, 0 },
    { "nis-servers", "IA",				 DHC_O_Universe,	41, 0 },
    { "ntp-servers", "IA",				 DHC_O_Universe,	42, 0 },
    { "vendor-encapsulated-options", "E.",		 DHC_O_Universe,	43, 0 },
    { "netbios-name-servers", "IA",			 DHC_O_Universe,	44, 0 },
    { "netbios-dd-server", "IA",			 DHC_O_Universe,	45, 0 },
    { "netbios-node-type", "B",			 	 DHC_O_Universe,	46, 0 },
    { "netbios-scope", "t",				 DHC_O_Universe,	47, 0 },
    { "font-servers", "IA",				 DHC_O_Universe,	48, 0 },
    { "x-display-manager", "IA",			 DHC_O_Universe,	49, 0 },
    { "dhcp-requested-address", "I",		 	 DHC_O_Universe,	50, 0 },
    { "dhcp-lease-time", "L",			 	 DHC_O_Universe,	51, 0 },
    { "dhcp-option-overload", "B",			 DHC_O_Universe,	52, 0 },
    { "dhcp-message-type", "B",			 	 DHC_O_Universe,	53, 0 },
    { "dhcp-server-identifier", "I",		 	 DHC_O_Universe,	54, 0 },
    { "dhcp-parameter-request-list", "BA",		 DHC_O_Universe,	55, 0 },
    { "dhcp-message", "t",				 DHC_O_Universe,	56, 0 },
    { "dhcp-max-message-size", "S",			 DHC_O_Universe,	57, 0 },
    { "dhcp-renewal-time", "L",			 	 DHC_O_Universe,	58, 0 },
    { "dhcp-rebinding-time", "L",			 DHC_O_Universe,	59, 0 },
    { "vendor-class-identifier", "X",		 	 DHC_O_Universe,	60, 0 },
    { "dhcp-client-identifier", "X",		 	 DHC_O_Universe,	61, 0 },
    { "nwip-domain", "X",				 DHC_O_Universe,	62, 0 },
    { "nwip-suboptions", "Enwip.",			 DHC_O_Universe,	63, 0 },
    { "nisplus-domain", "t",			 	 DHC_O_Universe,	64, 0 },
    { "nisplus-servers", "IA",			 	 DHC_O_Universe,	65, 0 },
    { "tftp-server-name", "t",			 	 DHC_O_Universe,	66, 0 },
    { "bootfile-name", "t",				 DHC_O_Universe,	67, 0 },
    { "mobile-ip-home-agent", "IA",			 DHC_O_Universe,	68, 0 },
    { "smtp-server", "IA",				 DHC_O_Universe,	69, 0 },
    { "pop-server", "IA",				 DHC_O_Universe,	70, 0 },
    { "nntp-server", "IA",				 DHC_O_Universe,	71, 0 },
    { "www-server", "IA",				 DHC_O_Universe,	72, 0 },
    { "finger-server", "IA",			 	 DHC_O_Universe,	73, 0 },
    { "irc-server", "IA",				 DHC_O_Universe,	74, 0 },
    { "streettalk-server", "IA",			 DHC_O_Universe,	75, 0 },
    { "streettalk-directory-assistance-server", "IA",	 DHC_O_Universe,	76, 0 },
    { "user-class", "t",				 DHC_O_Universe,	77, 0 },
    { "slp-directory-agent", "fIa",			 DHC_O_Universe,	78, 0 },
    { "slp-service-scope", "fto",			 DHC_O_Universe,	79, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { "fqdn", "Efqdn.",				 	 DHC_O_Universe,	81, 0 },
    { "relay-agent-information", "Eagent.",		 DHC_O_Universe,	82, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { "nds-servers", "IA",				 DHC_O_Universe,	85, 0 },
    { "nds-tree-name", "X",				 DHC_O_Universe,	86, 0 },
    { "nds-context", "X",				 DHC_O_Universe,	87, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { "uap-servers", "t",				 DHC_O_Universe,	98, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { "subnet-selection", "X",			 	 DHC_O_Universe,	118, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { 0L,	0L,				 0,	0, 0 },
    { "authenticate", "X",				 DHC_O_Universe,	210, 0 }
};

DHC_Option
dhcp_rai_options [ DHCP_OPTION_MAX ] =
{
    { "pad",          "",				 DHC_RAI_Universe,	0, 0 },
    { "circuit-id",   "t",	                         DHC_RAI_Universe,      1, 0 },
    { "remote-id",    "t",	                         DHC_RAI_Universe,      2, 0 },
    { "agent-id",     "t",	                         DHC_RAI_Universe,      3, 0 }
};

DHC_Option 
dhcp_nwip_options [DHCP_OPTION_MAX] =
{
    { "pad", "",					 DHC_NWIP_Universe, 0, 0 },
    { "illegal-1", "",					 DHC_NWIP_Universe, 1, 0 },
    { "illegal-2", "",					 DHC_NWIP_Universe, 2, 0 },
    { "illegal-3", "",					 DHC_NWIP_Universe, 3, 0 },
    { "illegal-4", "",					 DHC_NWIP_Universe, 4, 0 },
    { "nsq-broadcast", "f",				 DHC_NWIP_Universe, 5, 0 },
    { "preferred-dss", "IA",				 DHC_NWIP_Universe, 6, 0 },
    { "nearest-nwip-server", "IA",			 DHC_NWIP_Universe, 7, 0 },
    { "autoretries", "B",				 DHC_NWIP_Universe, 8, 0 },
    { "autoretry-secs", "B",				 DHC_NWIP_Universe, 9, 0 },
    { "nwip-1-1", "f",					 DHC_NWIP_Universe, 10, 0 },
    { "primary-dss", "I",				 DHC_NWIP_Universe, 11, 0 }
};

DHC_Option 
dhcp_fqdn_options [DHCO_FQDN_N] = 
{
    { "pad", "",					 DHC_FQDN_Universe, 0, 0 },
    { "no-client-update", "f",				 DHC_FQDN_Universe, 1, 0 },
    { "server-update", "f",				 DHC_FQDN_Universe, 2, 0 },
    { "encoded", "f",					 DHC_FQDN_Universe, 3, 0 },
    { "rcode1", "B",					 DHC_FQDN_Universe, 4, 0 },
    { "rcode2", "B",					 DHC_FQDN_Universe, 5, 0 },
    { "hostname", "t",					 DHC_FQDN_Universe, 6, 0 },
    { "domainname", "t",				 DHC_FQDN_Universe, 7, 0 },
    { "fqdn", "t",					 DHC_FQDN_Universe, 8, 0 }
};

DHC_Universe
dhcp_universes [ DHC_UNIVERSE_N ] =
{
    {   "parameters",   /* name not used in lookups */
	DHC_P_Universe,
	DHCP_N,
	&dhcp_parameters[0]
    },
    {	"dhcp",
	DHC_O_Universe,
	DHCO_N,
	&dhcp_options[0]
    },
    {   "agent",
	DHC_RAI_Universe,
	DHCO_RAI_N,
        &dhcp_rai_options[0]
    },
    {   "nwip",
	DHC_NWIP_Universe,
	DHCO_NWIP_N,
	&dhcp_nwip_options[0]
    },
    {   "fqdn",
	DHC_FQDN_Universe,
	DHCO_FQDN_N,
	&dhcp_fqdn_options[0]
    }
};


DHCP_EnumValue
dhcp_state_enumeration[]=
{  { "NBI",        0 }, /* no broadcast interfaces */
   { "PREINIT",    1 }, /* configuration started   */
   { "BOUND",      2 }, /* lease obtained          */
   { "RENEW",      3 }, /* lease renewed           */
   { "REBOOT",     4 }, /* have valid lease, but now obtained a different one */
   { "REBIND",     5 }, /* new, different lease */
   { "STOP"  ,     6 }, /* remove old lease */
   { "MEDIUM",     7 }, /* media selection begun */
   { "TIMEOUT",    8 }, /* timed out contacting DHCP server */
   { "FAIL",       9 }, /* all attempts to contact server timed out, sleeping */
   { "EXPIRE",    10 }, /* lease has expired, renewing */
   { "RELEASE",   11 }, /* releasing lease */
   { "START",     12 }, /* dhclient started sucessfully  */
   { "ABEND",     13 }, /* dhclient exited abnormally    */
   { "END",       14 }, /* dhclient exited normally      */
   { "END_OPTIONS",15}  /* last option sent for state change */
};

DHCP_Enumeration dhcp_enumerations[]=
{
    { "state", DHC_P_Universe, 14, &(dhcp_state_enumeration[0]), 0L, 0L }
};

int n_dhcp_enumerations=1;

DHC_Alias
dhcp_aliases[]=
{ {"server_name", &dhcp_options[ DHCO_TFTP_SERVER_NAME ]},
  {"filename",    &dhcp_options[ DHCO_BOOTFILE_NAME ]}
};
int n_dhcp_aliases = 2;

#endif /* DHC_DEFINE_OPTION_TABLE */

/* DHCP Option utility functions */

typedef 
struct dhco_s *DHCO;
/* DHCP Option handle type */

extern DHCO 
dhco_new( int (*optional_error_handler)(const char *fmt,...));
/* return new handle 
 * optional_error_handler can be null - if not, will be called
 * to log error messages generated by any dhc* function.
 */

extern void 
dhco_free( DHCO );
/* free handle */

extern DHC_Option *
dhco_find_by_name(DHCO, const char *name );
/* returns the option pointer corresponding to name 
 * If name is not qualified by a sub-option universe name,
 * searches "parameters", followed by "option" .
 */

extern DHC_Option *
dhco_find_by_dhclient_name(DHCO, const char *name, char **prefix );
/* returns the option pointer corresponding to name 
 * Where "name" is as produced by dhclient's "pretty_print_option" / client_envadd -
 * it may be prefixed by "old_" or "new_", and will have '-'s replaced by '_'s .
 * If a non-null match is returned, and prefix is non-null, on return 
 * *prefix will point to any prefix string (without the '_') .
 * If *prefix is not null on entry, and a prefix is found,
 * it must point to at least 4 bytes of storage; else if it is 0, 
 * and a prefix is found, it will be allocated.
 */

extern DHC_Option *
dhco_find_by_code(DHCO, uint8_t code, DHCP_Universe );
/* returns the option pointer corresponding to code in optional universe;
 * If universe is <0 or >=DHC_UNIVERSE_N, returns DNC_O_UNIVERSE option.
 */

extern const char *
dhco_type( DHC_Option *o );
/* returns type string from option */

extern DHCP_Option_Code
dhco_code( DHC_Option *o );
/* returns code from option */

extern DHC_Universe *
dhco_find_universe_by_name(DHCO, const char *name );
/* returns universe with name or 0L */

extern DHC_Universe *
dhco_find_universe_by_dhclient_name(DHCO, const char *name, char **prefix );
/* looks up universe by dhclient name as described above */

extern DHC_Universe *
dhco_find_universe_by_code(DHCO, uint8_t code );
/* returns universe with code or 0L */

extern DHC_Universe *
dhco_new_universe( DHCO, const char *name );
/* creates and returns new universe with name or 0L if there are DHCP_MAX_UNIVERSEs
 * or universe with name exists
 */

extern DHC_Option *
dhco_new_option(DHCO, const char *name, uint8_t code, char *type);
/* creates a new option with name .
 * If option with name already exists, replaces that option code & type.
 * (this is allowed by DHCP!).
 * If code is 0, then allocates the next available code .
 */

typedef void (*dhco_OptionHandler)(DHCO, DHC_Option*, void *value, int length, char *prefix, void *handler_param);
extern unsigned
dhco_parse_option_settings(DHCO, char *settings, dhco_OptionHandler handler, void *handler_param);
/* parses option setting statement text as produced in the environment of
 * dhclient's '-s' child, of the forms :
 *     [ universe_name '.' ] option_name '=' value
 * for existing options (universe is emitted if 'dhcp.' or 'parameters.')
 * or :  universe_name '._universe_.' option_name '=' code_number ':' type_string 
 * to create new universes or new options from setting statements.
 * If new option creation is to be enabled, dhclient must be run in '-x' mode to produce
 * universe qualified option statements of the second form .
 * Setting statements must be separated by the '\n' character, which is not allowed
 * (unencoded) in dhcp option text string values.
 * The "handler" will be called for each option "value" of "length" parsed, with
 * "prefix" set to any dhclient prefix ("new" or "old") found and with "handler_param" arg.
 * Returns number of options parsed.
 */

extern unsigned 
dhco_value_from_text(DHCO, DHCP_Option_Type type, char *input, void **value, int len, unsigned n_preceding_bytes);
/* Converts atomic (non-compound) type input text into binary *value.
 * returns > 0 (size of value) if input text successfully parsed into *value,
 * or 0 if input could not be parsed to a value of type. 
 * If *value is NULL, a new value will be allocated and its length returned. 
 * else if len is > 0:
 *   no more than len bytes will be stored;
 * else
 *   nothing will be stored, 
 *   n_preceding_bytes must hold the number of bytes that precede this value,
 *   and the length required to be allocated, including padding for alignment
 *   constraints given n_preceding_bytes, will be returned.
 */

extern unsigned 
dhco_text_from_value(DHCO, DHCP_Option_Type type, void **value, char **output, unsigned len, unsigned vlen, unsigned plen );
/* Converts binary value of type into output text.
 * returns length of successfully output bytes stored in *output or 0 if value could not
 * be represented as text. 
 * *value points to the value to be converted. On return, *value will be incremented 
 * by any pad bytes necessary to store a value of type preceded by plen bytes, plus
 * the bytes consumed by the value .
 * vlen must be the total length of the whole value.
 * plen must be the number of bytes that precede this value.
 * If *output is non-null,
 *    len must be > 0 and no more than len bytes are stored;
 * else if *output is null
 *    *output will be allocated to contain returned length;
 * else if output!=0 and len < length required,
 *    the length required to hold output is returned  .
 */

extern unsigned
dhco_input_option(DHCO, DHC_Option *, char *input, void **value, unsigned len );
/* parses complete input text for option (compound or not) into value
 * and returns  total length. If *value is 0, it will be allocated;
 * else len must be !=0 and no more than len bytes will be stored; 
 * returns total length stored on success or 0 on failure.
 */

extern unsigned 
dhco_output_option(DHCO, DHC_Option*, void *input, unsigned input_length, char **output, unsigned output_length ); 
/* converts complete option value (compound or not) of total size input_length
 * into *output text and returns total length. 
 * If *output is 0, output_length can be 0 and *output will be allocated; 
 * else output_length must be > 0 and no more than output_length bytes will
 * be stored .
 * If *output != 0, and output_length is 0 or less that the number of bytes
 * required, no value is stored.
 * returns total length stored on success or 0 on failure.
 */

extern DHCP_Option_Type_Mapping
dhco_new_mapping(DHCO, DHCP_Option_Type_Map_Table );

extern unsigned
dhco_map_option(DHCO, DHC_Option *opt, void *value, unsigned length,
		DHCP_Option_Type_Mapping mapping, 
		void (*map)(uint8_t, uint8_t, void*, unsigned, void *),
		void *arg
               );
/* for each member of the DHCP Option *opt *value, will call the *map function
 * with the uint8_t mapped type code, dhcp type code, void * address of value of 
 * unsigned length and optional void *arg .
 */
 
#endif /* DHC_OPTIONS_H */
