/* dia-canvas-editable.c
 * Copyright (C) 2003 Arjan Molenaar
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "dia-canvas-editable.h"
#include "diamarshal.h"
#include "../config.h"

enum
{
	START_EDITING,
	EDITING_DONE,
	TEXT_CHANGED,
	LAST_SIGNAL
};

static void dia_canvas_editable_base_init (gpointer klass);

static guint editable_signals[LAST_SIGNAL] = { 0 };

GType
dia_canvas_editable_get_type (void)
{
	static GType editable_type = 0;

	if (!editable_type) {
		static const GTypeInfo editable_info =
		{
			sizeof (DiaCanvasEditableIface), /* class_size */
			dia_canvas_editable_base_init,   /* base_init */
			NULL,           /* base_finalize */
			NULL,
			NULL,           /* class_finalize */
			NULL,           /* class_data */
			0,
			0,              /* n_preallocs */
			NULL
		};

		editable_type = g_type_register_static (G_TYPE_INTERFACE,
							 "DiaCanvasEditable",
							 &editable_info, 0);
		g_type_interface_add_prerequisite (editable_type,
						   DIA_TYPE_CANVAS_ITEM);
	}

	return editable_type;
}

static void
dia_canvas_editable_base_init (gpointer iface)
{
	static gboolean initialized = FALSE;

	if (!initialized) {
		  editable_signals[START_EDITING] =
			  g_signal_new ("start_editing",
					G_TYPE_FROM_INTERFACE (iface),
					G_SIGNAL_RUN_LAST,
					G_STRUCT_OFFSET (DiaCanvasEditableIface, start_editing),
					NULL, NULL,
					dia_marshal_VOID__POINTER,
					G_TYPE_NONE, 1,
					G_TYPE_POINTER);
		  editable_signals[EDITING_DONE] =
			  g_signal_new ("editing_done",
					G_TYPE_FROM_INTERFACE (iface),
					G_SIGNAL_RUN_LAST,
					G_STRUCT_OFFSET (DiaCanvasEditableIface, editing_done),
					NULL, NULL,
					dia_marshal_VOID__POINTER_STRING,
					G_TYPE_NONE, 2,
					G_TYPE_POINTER, G_TYPE_STRING);
		  editable_signals[TEXT_CHANGED] =
			  g_signal_new ("text_changed",
					G_TYPE_FROM_INTERFACE (iface),
					G_SIGNAL_RUN_LAST,
					G_STRUCT_OFFSET (DiaCanvasEditableIface, text_changed),
					NULL, NULL,
					dia_marshal_VOID__POINTER_STRING,
					G_TYPE_NONE, 2,
					G_TYPE_POINTER, G_TYPE_STRING);
		  initialized = TRUE;
	}
}


void
dia_canvas_editable_start_editing (DiaCanvasEditable *editable,
				   DiaShapeText *text_shape)
{
	g_return_if_fail (DIA_IS_CANVAS_EDITABLE (editable));
	g_return_if_fail (text_shape != NULL);
	g_return_if_fail (((DiaShape*) text_shape)->type == DIA_SHAPE_TEXT);

	g_signal_emit (editable, editable_signals[START_EDITING], 0,
		       text_shape);
}

void
dia_canvas_editable_editing_done (DiaCanvasEditable *editable,
				  DiaShapeText *text_shape,
				  const gchar *new_text)
{
	g_return_if_fail (DIA_IS_CANVAS_EDITABLE (editable));
	g_return_if_fail (text_shape != NULL);
	g_return_if_fail (((DiaShape*) text_shape)->type == DIA_SHAPE_TEXT);
	g_return_if_fail (new_text != NULL);

	g_signal_emit (editable, editable_signals[EDITING_DONE], 0,
		       text_shape, new_text);
}

void
dia_canvas_editable_text_changed (DiaCanvasEditable *editable,
				  DiaShapeText *text_shape,
				  const gchar *new_text)
{
	g_return_if_fail (DIA_IS_CANVAS_EDITABLE (editable));
	g_return_if_fail (text_shape != NULL);
	g_return_if_fail (((DiaShape*) text_shape)->type == DIA_SHAPE_TEXT);
	g_return_if_fail (new_text != NULL);

	g_signal_emit (editable, editable_signals[TEXT_CHANGED], 0,
		       text_shape, new_text);
}

