/* dia-item-tool.c
 * Copyright (C) 2004  Arjan Molenaar
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "dia-item-tool.h"
#include "dia-canvas-i18n.h"
#include "dia-canvas-text.h"
#include "dia-canvas-editable.h"

static void dia_item_tool_class_init		(DiaToolClass *class);
static void dia_item_tool_init			(DiaItemTool *tool);
static void dia_item_tool_dispose		(GObject *object);

static gboolean dia_item_tool_button_press	(DiaTool *tool,
						 DiaCanvasView *view,
						 GdkEventButton *event);
static gboolean dia_item_tool_button_release	(DiaTool *tool,
						 DiaCanvasView *view,
						 GdkEventButton *event);
static gboolean dia_item_tool_motion_notify	(DiaTool *tool,
						 DiaCanvasView *view,
						 GdkEventMotion *event);
static gboolean dia_item_tool_key_press		(DiaTool *tool,
						 DiaCanvasView *view,
						 GdkEventKey *event);
static gboolean dia_item_tool_key_release	(DiaTool *tool,
						 DiaCanvasView *view,
						 GdkEventKey *event);

static DiaToolClass *parent_class = NULL;

GType
dia_item_tool_get_type (void)
{
	static GtkType object_type = 0;

	if (!object_type) {
		static const GTypeInfo object_info = {
			sizeof (DiaItemToolClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) dia_item_tool_class_init,
			(GClassFinalizeFunc) NULL,
			(gconstpointer) NULL, /* class_data */
			sizeof (DiaItemTool),
			(guint16) 0, /* n_preallocs */
			(GInstanceInitFunc) dia_item_tool_init,
		};

		object_type = g_type_register_static (DIA_TYPE_TOOL,
						      "DiaItemTool",
						      &object_info, 0);
	}

	return object_type;
}

static void
dia_item_tool_class_init (DiaToolClass *klass)
{
	GObjectClass *object_class = (GObjectClass*) klass;

	parent_class = g_type_class_peek_parent (klass);

	object_class->dispose = dia_item_tool_dispose;
	klass->button_press_event = dia_item_tool_button_press;
	klass->button_release_event = dia_item_tool_button_release;
	klass->motion_notify_event = dia_item_tool_motion_notify;
	klass->key_press_event = dia_item_tool_key_press;
	klass->key_release_event = dia_item_tool_key_release;
}

static void
dia_item_tool_init (DiaItemTool *tool)
{
}

static void
dia_item_tool_dispose (GObject *object)
{
	DiaItemTool *tool = (DiaItemTool*) object;

	G_OBJECT_CLASS (parent_class)->dispose (object);
}

static gboolean
dia_item_tool_button_press (DiaTool *tool, DiaCanvasView *view,
			         GdkEventButton *event)
{
	DiaItemTool *itool = (DiaItemTool*) tool;
	DiaCanvasViewItem *view_item;
	DiaCanvasItem *item = NULL;
	gboolean result = FALSE;

	//g_message(G_STRLOC);

	itool->current_item = NULL;

	view_item = (DiaCanvasViewItem*) gnome_canvas_get_item_at (GNOME_CANVAS (view), event->x, event->y);

	if (!view_item)
		return FALSE;

	item = view_item->item;

	if (!item || !DIA_CANVAS_ITEM_INTERACTIVE (item))
		return FALSE;

	if ((event->state & GDK_CONTROL_MASK)
	    && (dia_canvas_view_item_is_selected (view_item))) {
		/* Unselect a selected object when CTRL is pressed. */
		dia_canvas_view_unselect (view, view_item);
		dia_canvas_view_focus (view, NULL);
		dia_canvas_item_request_update (item);
		return TRUE;
	} else {
		/* Unselect all entries if no modifiers are pressed and
		 * the object is not already selected. */
		if (!(event->state & (GDK_SHIFT_MASK | GDK_CONTROL_MASK))
		    && !dia_canvas_view_item_is_selected (view_item)) {
			dia_canvas_view_unselect_all (view);
		}
		dia_canvas_view_focus (view, view_item);
		itool->current_item = view_item;
	}

	switch (event->type) {
	case GDK_BUTTON_PRESS:
		if (event->button == 1) {
			itool->grabbed_item = view_item;
			itool->old_x = event->x;
			itool->old_y = event->y;
			dia_undo_manager_begin_transaction (dia_canvas_get_undo_manager (item->canvas));
			dia_canvas_item_request_update (item);
			result = TRUE;
		}
		break;

	case GDK_2BUTTON_PRESS:
		if (DIA_IS_CANVAS_EDITABLE (item)) {
			if (dia_canvas_editable_is_editable(DIA_CANVAS_EDITABLE (item))) {
				dia_canvas_view_start_editing (view, view_item,
							       event->x,
							       event->y);
				result = TRUE;
			}
		}
		break;

	default:
		break;
	}
	return result;
}

static gboolean
dia_item_tool_button_release (DiaTool *tool, DiaCanvasView *view,
				   GdkEventButton *event)
{
	DiaItemTool *itool = DIA_ITEM_TOOL (tool);

	dia_undo_manager_commit_transaction (dia_canvas_get_undo_manager (view->canvas));
	itool->grabbed_item = NULL;

	return FALSE;
}

static gboolean
dia_item_tool_motion_notify (DiaTool *tool, DiaCanvasView *view,
				  GdkEventMotion *event)
{
	DiaItemTool *itool = DIA_ITEM_TOOL (tool);

	if (itool->grabbed_item && (event->state & GDK_BUTTON_PRESS_MASK)) {
		dia_canvas_view_move (view,
				      event->x - itool->old_x,
				      event->y - itool->old_y,
				      NULL);
		itool->old_x = event->x;
		itool->old_y = event->y;
		return TRUE;
	}
	return FALSE;
}

static gboolean
dia_item_tool_key_press (DiaTool *tool, DiaCanvasView *view,
			      GdkEventKey *event)
{
	DiaItemTool *itool = DIA_ITEM_TOOL (tool);

	return FALSE;
}

static gboolean
dia_item_tool_key_release (DiaTool *tool, DiaCanvasView *view,
			        GdkEventKey *event)
{
	DiaItemTool *itool = DIA_ITEM_TOOL (tool);

	return FALSE;
}

DiaTool*
dia_item_tool_new (void)
{
	return g_object_new (DIA_TYPE_ITEM_TOOL, NULL);
}
