/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/util/diff_generator.t.cpp
 *
 * @brief [LEVEL: alpha] testing @ref diagnostics::internal::Diff_Generator
 *
 * $Id: diff_generator.t.cpp 786 2007-03-01 21:48:44Z tautschn $
 *
 * @author Piotr Esden-Tempski
 */

#include <diagnostics/unittest.hpp>

#include <diagnostics/memory.hpp>

#include <diagnostics/unittest/std_test_case.hpp>

// used components
#include <diagnostics/util/diff_generator.hpp>

#define TEST_COMPONENT_NAME diff_generator
#define TEST_COMPONENT_NAMESPACE diagnostics::internal

DIAGNOSTICS_NAMESPACE_BEGIN;
INTERNAL_NAMESPACE_BEGIN;
TEST_NAMESPACE_BEGIN;
TEST_COMPONENT_TEST_NAMESPACE_BEGIN;
using namespace unittest;

/**
 * @brief [PRO] fct/normal: testing what happens when we do not initialize
 * text[a|b] but want a result
 */
void no_init(Test_Data & test_data)
{
    
    // color
    {
	Diff_Generator dg;
	::std::string result;
	dg.diff_text(result, true);
	TEST_ASSERT(test_data.compare("color",result));
    }
    
    // no color
    {
	Diff_Generator dg;
	::std::string result;
	dg.diff_text(result, false);
	TEST_ASSERT(test_data.compare("no color",result));
    }
}

/**
 * @brief [PRO] fct/normal: testing what happens when we initialize text[a|b]
 * with empty strings
 */
void init_empty(Test_Data & test_data)
{
    
    // color
    {
	Diff_Generator dg;
	::std::string result;
	
	dg.texta("");
	dg.textb("");
	dg.diff_text(result, true);
	
	TEST_ASSERT(test_data.compare("color",result));
    }

    // no color
    {
	Diff_Generator dg;
	::std::string result;
	
	dg.texta("");
	dg.textb("");
	dg.diff_text(result, false);
	
	TEST_ASSERT(test_data.compare("no color",result));
    }
}

/**
 * @brief [PRO] fct/normal: testing one line diffs
 */
void one_line(Test_Data & test_data)
{
    
    // match
    
    // color
    {
	Diff_Generator dg;
	::std::string result;
	
	dg.texta("This is a very intelligent text!");
	dg.textb("This is a very intelligent text!");
	dg.diff_text(result, true);
	
	TEST_ASSERT(test_data.compare("match color",result));
    }
    
    // no color
    {
	Diff_Generator dg;
	::std::string result;
	
	dg.texta("This is a very intelligent text!");
	dg.textb("This is a very intelligent text!");
	dg.diff_text(result, false);
	
	TEST_ASSERT(test_data.compare("match no color",result));
    }
    
    // delete
    
    // color
    {
	Diff_Generator dg;
	::std::string result;
	
	dg.texta("This is a very intelligent text!");
	dg.textb("");
	dg.diff_text(result, true);
	
	TEST_ASSERT(test_data.compare("delete color",result));
    }
    
    // no color
    {
	Diff_Generator dg;
	::std::string result;
	
	dg.texta("This is a very intelligent text!");
	dg.textb("");
	dg.diff_text(result, false);
	
	TEST_ASSERT(test_data.compare("delete no color",result));
    }
    
    // add
    
    // color
    {
	Diff_Generator dg;
	::std::string result;
	
	dg.texta("");
	dg.textb("This is a very intelligent text!");
	dg.diff_text(result, true);
	
	TEST_ASSERT(test_data.compare("add color",result));
    }
    
    // no color
    {
	Diff_Generator dg;
	::std::string result;
	
	dg.texta("");
	dg.textb("This is a very intelligent text!");
	dg.diff_text(result, false);
	
	TEST_ASSERT(test_data.compare("add no color",result));
    }
}

/**
 * @brief [PRO] fct/normal: testing multiple calls of the same diff class
 * instance
 */
void multi_call(Test_Data & test_data)
{
    Diff_Generator dg;
    ::std::string result;
    
    dg.texta("");
    dg.textb("This is a very intelligent text!");
    dg.diff_text(result, true);
    
    TEST_ASSERT(test_data.compare("color",result));
    
    dg.diff_text(result, false);
    
    TEST_ASSERT(test_data.compare("no color",result));
    
    dg.texta("This is a very intelligent text!");
    dg.textb("");
    
    dg.diff_text(result, true);
    
    TEST_ASSERT(test_data.compare("changed color",result));
    
    dg.diff_text(result, false);
    
    TEST_ASSERT(test_data.compare("changed no color",result));
}

/**
 * @brief [PRO] fct/normal: testing mult line diffs
 */
void multi_line(Test_Data & test_data)
{
    // TEST_MEMORY_LOG_ENTER;
    TEST_MEMORY_BALANCE_ENTER(TYPE_TRACE);
    
    Diff_Generator dg;
    ::std::string result;
    
    // equal
    dg.texta("Row1\nRow2\nRow3\nRow4");
    dg.textb("Row1\nRow2\nRow3\nRow4");
    dg.diff_text(result, true);
	
    TEST_ASSERT(test_data.compare("match",result));
    
    // added
    dg.texta("Row1\nRow2\nRow3\nRow4");
    dg.textb("RowX\nRowX\nRow1\nRow2\nRow3\nRow4");
    dg.diff_text(result, true);
    
    TEST_ASSERT(test_data.compare("block added beginning",result));
    
    dg.texta("Row1\nRow2\nRow3\nRow4");
    dg.textb("Row1\nRow2\nRowX\nRowX\nRow3\nRow4");
    dg.diff_text(result, true);
    
    TEST_ASSERT(test_data.compare("block added center",result));
    
    dg.texta("Row1\nRow2\nRow3\nRow4");
    dg.textb("Row1\nRow2\nRow3\nRow4\nRowX\nRowX");
    dg.diff_text(result, true);
    
    TEST_ASSERT(test_data.compare("block added end",result));
    
    // deleted
    dg.texta("RowX\nRowX\nRow1\nRow2\nRow3\nRow4");
    dg.textb("Row1\nRow2\nRow3\nRow4");
    dg.diff_text(result, true);
    
    TEST_ASSERT(test_data.compare("block deleted beginning",result));
	
    dg.texta("Row1\nRow2\nRowX\nRowX\nRow3\nRow4");
    dg.textb("Row1\nRow2\nRow3\nRow4");
    dg.diff_text(result, true);
    
    TEST_ASSERT(test_data.compare("block deleted center",result));
    
    dg.texta("Row1\nRow2\nRow3\nRow4\nRowX\nRowX");
    dg.textb("Row1\nRow2\nRow3\nRow4");
    dg.diff_text(result, true);
    
    TEST_ASSERT(test_data.compare("block deleted end",result));
    
    // changed
    dg.texta("RowX\nRowX\nRow1\nRow2\nRow3\nRow4");
    dg.textb("RowY\nRowY\nRow1\nRow2\nRow3\nRow4");
    dg.diff_text(result, true);
    
    TEST_ASSERT(test_data.compare("block changed beginning",result));
    
    dg.texta("Row1\nRow2\nRowX\nRowX\nRow3\nRow4");
    dg.textb("Row1\nRow2\nRowY\nRowY\nRow3\nRow4");
    dg.diff_text(result, true);
	
    TEST_ASSERT(test_data.compare("block changed center",result));
	
    dg.texta("Row1\nRow2\nRow3\nRow4\nRowX\nRowX");
    dg.textb("Row1\nRow2\nRow3\nRow4\nRowY\nRowY");
    dg.diff_text(result, true);
	
    TEST_ASSERT(test_data.compare("block changed end",result));

    // changed+added
    dg.texta("RowX\nRowX\nRow1\nRow2\nRow3\nRow4");
    dg.textb("RowY\nRowY\nRowY\nRow1\nRow2\nRow3\nRow4");
    dg.diff_text(result, true);

    TEST_ASSERT(test_data.compare("block changed+added beginning",result));
	
    dg.texta("Row1\nRow2\nRowX\nRowX\nRow3\nRow4");
    dg.textb("Row1\nRow2\nRowY\nRowY\nRowY\nRow3\nRow4");
    dg.diff_text(result, true);
	
    TEST_ASSERT(test_data.compare("block changed+added center",result));
	
    dg.texta("Row1\nRow2\nRow3\nRow4\nRowX\nRowX");
    dg.textb("Row1\nRow2\nRow3\nRow4\nRowY\nRowY\nRowY");
    dg.diff_text(result, true);
	
    TEST_ASSERT(test_data.compare("block changed+added end",result));

    // changed+deleted
    dg.texta("RowX\nRowX\nRowX\nRow1\nRow2\nRow3\nRow4");
    dg.textb("RowY\nRowY\nRow1\nRow2\nRow3\nRow4");
    dg.diff_text(result, true);

    TEST_ASSERT(test_data.compare("block changed+deleted beginning",result));
	
    dg.texta("Row1\nRow2\nRowX\nRowX\nRowX\nRow3\nRow4");
    dg.textb("Row1\nRow2\nRowY\nRowY\nRow3\nRow4");
    dg.diff_text(result, true);
	
    TEST_ASSERT(test_data.compare("block changed+deleted center",result));
	
    dg.texta("Row1\nRow2\nRow3\nRow4\nRowX\nRowX\nRowX");
    dg.textb("Row1\nRow2\nRow3\nRow4\nRowY\nRowY");
    dg.diff_text(result, true);
	
    TEST_ASSERT(test_data.compare("block changed+deleted end",result));
    
    TEST_MEMORY_BALANCE_EXIT;
    // TEST_MEMORY_LOG_EXIT;
}

/**
 * @brief [PRO] fct/normal: testing big diffs, simulating real diff
 */
void big_data(Test_Data & test_data)
{
    // TEST_MEMORY_LOG_ENTER; 
    TEST_MEMORY_BALANCE_ENTER(TYPE_TRACE); 

    int i;
    Diff_Generator dg;
    ::std::string inputa;
    ::std::string inputb;
    ::std::string result;

    for(i=0; i<1000; i++)inputa += "A line\n";

    inputb = inputa;

    dg.texta(inputa);
    dg.textb(inputb);
    dg.diff_text(result, true);

    TEST_ASSERT(test_data.compare("equal",result));

    inputb = "Blubbering ;)\n";
    for(i=0; i<99; i++) inputb += "A line\n";

    for(i=0; i<100; i++) inputb += "A different line\n";

    for(i=0; i<100; i++) inputb += "A line\n";

    for(i=0; i<300; i++) inputb += "A another different line\n";

    for(i=0; i<150; i++) inputb += "A line\n";

    inputb += "A single line added\n";
    for(i=0; i<150; i++) inputb += "A line\n";

    for(i=0; i<100; i++) inputb += "another lines changed\n";
	
    dg.texta(inputa);
    dg.textb(inputb);
    dg.diff_text(result, true);

    TEST_ASSERT(test_data.compare("differ",result));
	
    TEST_MEMORY_BALANCE_EXIT;
    // TEST_MEMORY_LOG_EXIT;
}

TEST_COMPONENT_TEST_NAMESPACE_END;
TEST_NAMESPACE_END;
INTERNAL_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;


TEST_SUITE_BEGIN;
TEST_NORMAL_CASE(&no_init,LEVEL_PROD);
TEST_NORMAL_CASE(&init_empty,LEVEL_PROD);
TEST_NORMAL_CASE(&one_line,LEVEL_PROD);
TEST_NORMAL_CASE(&multi_call,LEVEL_PROD);
TEST_NORMAL_CASE(&multi_line,LEVEL_PROD);
TEST_NORMAL_CASE(&big_data,LEVEL_PROD);
TEST_SUITE_END;

STREAM_TEST_SYSTEM_MAIN;
// vim:ts=4:sw=4
