/*
 *   This file is part of Dianara
 *   Copyright 2012-2014  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "avatarbutton.h"

AvatarButton::AvatarButton(ASPerson *person,
                           PumpController *pumpController,
                           QSize avatarSize,
                           QWidget *parent) : QToolButton(parent)
{
    this->pController = pumpController;

    this->setPopupMode(QToolButton::InstantPopup);
    this->setStyleSheet("QToolButton { border: none }"
                        "QToolButton:hover { border: none; "
                        "                    background-color: palette(highlight) }");
    this->setIconSize(avatarSize);
    this->iconWidth = avatarSize.width();

    this->setToolTip(person->getTooltipInfo());

    // Get local file name for avatar, which is stored in base64 hash form
    QString avatarFile = MiscHelpers::getCachedAvatarFilename(person->getAvatar());
    if (QFile::exists(avatarFile))
    {
        this->updateAvatarIcon(avatarFile);
    }
    else
    {
        this->setGenericAvatarIcon();

        qDebug() << "AvatarButton() Using placeholder, downloading real avatar now";

        pController->enqueueAvatarForDownload(person->getAvatar());
        connect(pController, SIGNAL(avatarStored(QString,QString)),
                this, SLOT(redrawAvatar(QString,QString)));
    }


    this->authorId = person->getId();
    this->authorName = person->getName();
    // In case there's no name to show, for some reason, use ID
    if (authorName.isEmpty()) // This happens sometimes with shared objects
    {
        authorName = authorId;
    }
    authorUrl = person->getUrl();
    authorAvatarUrl = person->getAvatar();


    // Get "following" status before creating avatar menu
    this->authorFollowed = this->pController->userInFollowing(authorId);

    createAvatarMenu();
    if (!authorId.isEmpty())  // Don't add the menu for invalid users
    {
        this->setMenu(avatarMenu);
    }


    qDebug() << "AvatarButton created";
}


AvatarButton::~AvatarButton()
{
    qDebug() << "AvatarButton destroyed";
}




void AvatarButton::setGenericAvatarIcon()
{
    this->setIcon(QIcon::fromTheme("user-identity",
                                   QIcon(":/images/no-avatar.png")));

}


void AvatarButton::updateAvatarIcon(QString filename)
{
    QPixmap avatarPixmap = QPixmap(filename)
                           .scaledToWidth(this->iconWidth,
                                          Qt::SmoothTransformation);

    if (!avatarPixmap.isNull())
    {
        this->setIcon(QIcon(avatarPixmap));
    }
    else
    {
        qDebug() << "AvatarButton() avatar pixmap is null, using generic";
        this->setGenericAvatarIcon();
    }
}


/*
 * Create the menu shown when clicking the avatar
 *
 */
void AvatarButton::createAvatarMenu()
{
    bool userIsAuthor = false;
    if (authorId == pController->currentUserId())
    {
        userIsAuthor = true; // The post or comment is ours
    }

    this->avatarMenu = new QMenu(this);
    avatarMenu->setSeparatorsCollapsible(false);

    this->avatarMenuIdAction = new QAction(QIcon::fromTheme("user-identity",
                                                            QIcon(":/images/no-avatar.png")),
                                           this->authorId, this);
    avatarMenuIdAction->setSeparator(true); // Make it nicer and not clickable
    avatarMenu->addAction(avatarMenuIdAction);


    QString openProfileString = tr("Open %1's profile in web browser")
                                .arg(this->authorName);
    if (userIsAuthor)
    {
        openProfileString = tr("Open your profile in web browser");
    }
    this->avatarMenuProfileAction = new QAction(QIcon::fromTheme("internet-web-browser"),
                                                 openProfileString, this);
    connect(avatarMenuProfileAction, SIGNAL(triggered()),
            this, SLOT(openAuthorProfileInBrowser()));
    if (authorUrl.isEmpty()) // Disable if there isn't actually an URL
    {
        avatarMenuProfileAction->setDisabled(true);
    }
    avatarMenu->addAction(avatarMenuProfileAction);


    this->avatarMenuFollowAction = new QAction("*follow/unfollow*", this);
    // Only add "follow/unfollow" option, if we're not the author
    if (!userIsAuthor)
    {
        avatarMenu->addAction(avatarMenuFollowAction);
        this->setFollowUnfollow();
    }

    // More options can be added from outside, via addActionToMenu()
}


/*
 * Set the icon and text of the follow/unfollow option of the avatar menu
 * according to whether we're following that user or not
 *
 */
void AvatarButton::setFollowUnfollow()
{
    if (this->authorFollowed)
    {
        this->avatarMenuFollowAction->setIcon(QIcon::fromTheme("list-remove-user"));
        this->avatarMenuFollowAction->setText(tr("Stop following"));
        connect(avatarMenuFollowAction, SIGNAL(triggered()),
                this, SLOT(unfollowUser()));
        disconnect(avatarMenuFollowAction, SIGNAL(triggered()),
                   this, SLOT(followUser()));
        qDebug() << "post author followed, connecting to UNFOLLOW()" << this->authorId;
    }
    else
    {
        this->avatarMenuFollowAction->setIcon(QIcon::fromTheme("list-add-user"));
        this->avatarMenuFollowAction->setText(tr("Follow"));
        connect(avatarMenuFollowAction, SIGNAL(triggered()),
                this, SLOT(followUser()));
        disconnect(avatarMenuFollowAction, SIGNAL(triggered()),
                   this, SLOT(unfollowUser()));
        qDebug() << "post author not followed, connecting to FOLLOW()" << this->authorId;
    }
}

void AvatarButton::addSeparatorToMenu()
{
    avatarMenu->addSeparator();
}

void AvatarButton::addActionToMenu(QAction *action)
{
    avatarMenu->addAction(action);
}



////////////////////////////////////////////////////////////////////////////
////////////////////////////////// SLOTS ///////////////////////////////////
////////////////////////////////////////////////////////////////////////////


void AvatarButton::openAuthorProfileInBrowser()
{
    QDesktopServices::openUrl(this->authorUrl);
}


void AvatarButton::followUser()
{
    this->pController->followContact(this->authorId);

    authorFollowed = true; // FIXME: should wait until confirmed by server...
    this->setFollowUnfollow();
}


void AvatarButton::unfollowUser()
{
    int confirmation = QMessageBox::question(this, tr("Stop following?"),
                                             tr("Are you sure you want to stop following %1?")
                                             .arg(this->authorId),
                                             tr("&Yes, stop following"), tr("&No"), "", 1, 1);

    if (confirmation == 0)
    {
        this->pController->unfollowContact(this->authorId);

        authorFollowed = false; // FIXME: should wait until confirmed by server...
        this->setFollowUnfollow();
    }
}


/*
 * Redraw avatar after receiving it
 *
 */
void AvatarButton::redrawAvatar(QString avatarUrl, QString avatarFilename)
{
    if (avatarUrl == this->authorAvatarUrl)
    {
        this->updateAvatarIcon(avatarFilename);

        disconnect(pController, SIGNAL(avatarStored(QString,QString)),
                   this, SLOT(redrawAvatar(QString,QString)));
    }
}
