/**
 * ContextSensitive: a library for generating context-sensitive
 * content on HTML elements. It will take over the
 * click/oncontextmenu functions for the document, and works
 * only where these are possible to override.  It allows contextmenus to be
 * created via both a left and right mouse click.
 *
 * The classes rely on prototype.js (http://www.prototypejs.org/) and
 * are released under the same terms as the prototype library.
 *
 * Original code by Havard Eide (http://eide.org/) released under the MIT
 * license.
 *
 * $Horde: dimp/js/src/ContextSensitive.js,v 1.45.2.5 2008/05/14 04:35:19 slusarz Exp $
 *
 * @author Chuck Hagenbuch <chuck@horde.org>
 */

var ContextSensitive = Class.create({

    initialize: function()
    {
        this.current = this.target = null;
        this.elements = [];

        document.observe('contextmenu', this.rightClickHandler.bindAsEventListener(this));
        document.observe('click', this.leftClickHandler.bindAsEventListener(this));
    },

    /**
     * Elements are of type ContextSensitive.Element.
     */
    addElement: function(id, target, opts)
    {
        if (id && !this.validElement(id, opts.left || false)) {
            this.elements.push(new ContextSensitive.Element(id, target, opts));
        }
    },

    /**
     * Remove a registered element.
     */
    removeElement: function(id)
    {
        this.elements = this.elements.findAll(function(s) {
            return s.id != id;
        });
    },

    /**
     * Hide the current element.
     */
    close: function()
    {
        if (this.current) {
            this.current.hide();
        }
    },

    /**
     * Get the element that triggered the current context menu (if any).
     */
    element: function()
    {
        return this.target;
    },

    /**
     * Returns the current displayed menu element, if any.
     */
    currentmenu: function()
    {
        if (this.current && this.current.visible()) {
            return this.current;
        }
    },

    /**
     * Get a valid element (the ones that can be right-clicked) based
     * on a element ID.
     */
    validElement: function(id, left)
    {
        left = left || false;
        return this.elements.find(function(s) {
            return (s.id == id && s.opts.left == left) ? s : false;
        });
    },

    /**
     * Set the disabled flag of an event.
     */
    disable: function(id, left, disable)
    {
        this.elements.find(function(s) {
            if (s.id == id && s.opts.left == left) {
                s.disable = disable;
                return true;
            }
        });
    },

    /**
     * Called when a left click event occurs. Will return before the
     * element is closed if we click on an element inside of it.
     */
    leftClickHandler: function(e)
    {
        if (!DimpCore.isLeftClick(e)) {
            return true;
        }

        if (this.currentmenu()) {
            this.close();
            e.stop();
        }

        // Check if the mouseclick is registered to an element now.
        return this.rightClickHandler(e);
    },

    /**
     * Called when a rightclick event occurs.
     */
    rightClickHandler: function(e)
    {
        // Get the element based on the ID of the element we just clicked.
        var result = this.trigger(e.element(), DimpCore.isLeftClick(e), e.pointerX(), e.pointerY());

        if (!result) {
            // Stop event to prevent browser context menu from displaying.
            e.stop();
        }

        return result;
    },

    /**
     * Display context menu if valid element has been activated.
     */
    trigger: function(target, leftclick, x, y)
    {
        // Make sure we hide any other ContextObjects.
        this.close();

        target = $(target);
        var ctx, el, offset, offsets, size, v;
        [ target ].concat(target.ancestors()).find(function(n) {
            ctx = this.validElement(n.id, leftclick);
            return ctx;
        }.bind(this));

        // Return if event not found or event is disabled.
        if (!ctx || ctx.disable) {
            return true;
        }

        // Try to retrieve the context-sensitive element we want to
        // display. If we can't find it we just return.
        el = $(ctx.ctx);
        if (!el) {
            return true;
        }

        // Register the current element that will be shown and the
        // element that was clicked on.
        this.current = el;
        this.target = $(ctx.id);

        // Get the base element positions.
        offset = ctx.opts.offset;
        if (!offset && (Object.isUndefined(x) || Object.isUndefined(y))) {
            offset = target.id;
        }
        offset = $(offset);

        if (offset) {
            switch (ctx.opts.offsettype) {
            case 'cumulative':
                offsets = offset.cumulativeOffset();
                break;

            case 'real':
                offsets = offset.cumulativeScrollOffset();
                break;

            case 'positioned':
            default:
                offsets = offset.positionedOffset();
                break;
            }
            x = offsets[0];
            y = offsets[1] + offset.getHeight();
        }

        // Get window/element dimensions
        v = document.viewport.getDimensions();
        size = el.getDimensions();

        // Give a little space below element
        y += 2;

        // Make sure context window is entirely on screen
        if ((y + size.height) > v.height) {
            y = v.height - size.height - 10;
        }
        if ((x + size.width) > v.width) {
            x = v.width - size.width - 10;
        }

        if (ctx.opts.onShow) {
            ctx.opts.onShow(ctx);
        }

        el.setStyle({ display: 'block', left: x + 'px', top: y + 'px' });

        return false;
    }
});

ContextSensitive.Element = Class.create({

    // opts: 'left' -> monitor left click; 'offset' -> id of element used to
    //       determine offset placement
    initialize: function(id, target, opts)
    {
        this.id = id;
        this.ctx = target;
        this.opts = opts;
        this.opts.left = opts.left || false;
        this.disable = false;
    }

});
