/**
 * DraggableMail.js - Extensions to the stock Scriptaculous Draggable library
 * which allows us to do click-selections, move multiple items, DIMP-specific
 * update graphics, etc.
 *
 * $Horde: dimp/js/src/DraggableMail.js,v 1.40.2.8 2008/05/14 04:09:16 slusarz Exp $
 *
 * Copyright 2005-2008 The Horde Project (http://www.horde.org/)
 *
 * See the enclosed file COPYING for license information (GPL). If you
 * did not receive this file, see http://www.fsf.org/copyleft/gpl.html.
 */

// For DIMP, we need to take into account the scroll bar since the folder list
// may exceed the available size and cause the drop zones to move offscreen.
// Requires the parameter 'scroll' to be true when passed into add(). Also,
// will ignore Draggable elements that have a non-empty 'nodrop' property.
Droppables.isAffected = function(point, element, drop)
{
    element = $(element);
    return (!element.readAttribute('nodrop') &&
            (drop.element != element) &&
            ((!drop._containers) || this.isContained(element, drop)) &&
            ((!drop.accept) || (element.classNames().detect(function(v) { return drop.accept.include(v) } ) )) &&
            ((drop.scroll)
               ? Position.withinIncludingScrolloffsets(drop.element, point[0], point[1])
               : Position.within(drop.element, point[0], point[1]))
           );
};

var DraggableDimp = Class.create(Draggable, {

    // Need to redefine to ignore scrolling unless mouse is directly above
    // or below the defined scrolling element.
    startScrolling: function($super, speed)
    {
        // Added code to determine if current mouse is directly above scrolling
        // element.
        var p = this.options.scroll.viewportOffset(),
            leftx = p[0] + this.options.scroll.scrollLeft + Position.deltaX;
        if (Draggables._lastPointer[0] > (leftx + this.options.scroll.offsetWidth) ||
            Draggables._lastPointer[0] < leftx) {
            return;
        }

        $super(speed);
    }

});

var DraggableMail = Class.create(DraggableDimp, {

    // Overrides the builtin Draggable effects.
    draw: function(e)
    {
        var d_pos, m = $('mousedrag'), m_pos;
        if (this.element.id == m.id ||
            ++this.movethreshold < 3) {
            return;
        }

        // left = e[0]; top = e[1]
        e[0] += 15;
        e[1] += 10;
        d_pos = document.viewport.getDimensions();
        m_pos = m.getDimensions();

        // Don't show if it would appear beyond the bounds of the screen.
        if ((e[0] + m_pos.width > d_pos.width) ||
            (e[1] + m_pos.height > d_pos.height)) {
            m.hide();
        } else {
            m.setStyle({ left: e[0] + 'px', top: e[1] + 'px' }).show();
            $('dragcount').update(DimpBase.selectedCount());
        }
    },

    // Need to redefine this because the original will stop the Event (which
    // we don't want until after onStart), won't call onStart (which we do
    // want), ignores right clicks, and won't init the move threshold.
    initDrag: function(e)
    {
        if (!Object.isUndefined(Draggable._dragging[this.element]) &&
            Draggable._dragging[this.element]) {
            return;
        }

        // abort on form elements, fixes a Firefox issue
        var pointer, pos, tag_name = e.element().tagName.toUpperCase();
        if (tag_name &&
            (tag_name == 'INPUT' ||
             tag_name == 'SELECT' ||
             tag_name == 'OPTION' ||
             tag_name == 'BUTTON' ||
             tag_name == 'TEXTAREA')) {
             return;
        }

        pointer = [e.pointerX(), e.pointerY()];
        pos = this.element.cumulativeOffset();
        this.offset = [0,1].map( function(i) { return (pointer[i] - pos[i]) });

        Draggables.activate(this);
        Draggables.notify('onStart', this, e);
        e.stop();
        this.movethreshold = 0;
    },

    // Need to redefine or else onEnd won't be triggered when we click &
    // immediately release.
    endDrag: function(e)
    {
        this.stopScrolling();
        this.finishDrag(e, true);
        e.stop();
    }

});

var DraggableMailOpts = {
    endeffect: false,
    scroll: 'foldersSidebar',
    scrollSensitivity: 15,

    onStart: function(d, e)
    {
        // We only care about 'mousedown' events in this code.
        if (e.type != 'mousedown') {
            return;
        }

        var args = { right: e.isRightClick() },
            id = d.element.id;

        // Handle selection first.
        if (e.ctrlKey || e.metaKey) {
            DimpBase.msgSelect(id, $H({ ctrl: true }).merge(args).toObject());
        } else if (e.shiftKey) {
            DimpBase.msgSelect(id, $H({ shift: true }).merge(args).toObject());
        } else {
            if (DimpBase.isSelected('domid', id)) {
                d.selectIfNoDrag = true;
                d.wasDragged = false;
            } else {
                DimpBase.msgSelect(id, args);
            }
        }

        if (!DimpCore.isLeftClick(e)) {
            d.selectIfNoDrag = d.wasDragged = false;
        }
    },

    onDrag: function(d, e)
    {
        d.wasDragged = true;
    },

    onEnd: function(d, e)
    {
        if (d.selectIfNoDrag && !d.wasDragged) {
            DimpBase.msgSelect(d.element.id, { right: e.isRightClick() });
        }

        d.selectIfNoDrag = d.wasDragged = false;
        $('mousedrag').hide();
    }
};
