/* dirmngr.c - LDAP access
 *	Copyright (C) 2002 Klarlvdalens Datakonsult AB
 *      Copyright (C) 2003, 2004 g10 Code GmbH
 *
 * This file is part of DirMngr.
 *
 * DirMngr is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DirMngr is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <config.h>

#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>
#include <assert.h>
#include <time.h>
#include <fcntl.h>
#include <unistd.h>
#include <signal.h>

#include <gcrypt.h>
#include <ksba.h>
#include <assuan.h> /* needed for the malloc hooks */

#define JNLIB_NEED_LOG_LOGV
#include "dirmngr.h"
#include "crlcache.h"
#include "crlfetch.h"
#include "misc.h"

enum cmd_and_opt_values 
{ aNull = 0,
  oCsh		  = 'c',
  oQuiet	  = 'q',
  oSh		  = 's',
  oVerbose	  = 'v',
  oNoVerbose = 500,
  
  aServer,
  aListCRLs,
  aLoadCRL,
  aFetchCRL,
  aShutdown,
  aFlush,
  aGPGConfList,

  oOptions,
  oDebug,
  oDebugAll,
  oDebugWait,
  oDebugLevel,
  oNoGreeting,
  oNoOptions,
  oHomedir,
  oNoDetach,
  oLogFile,
  oBatch,
  oLDAPFile,
  oLDAPTimeout,
  oLDAPAddServers,
  oOCSPResponder,
  oOCSPSigner,
  oMaxReplies,
  oFakedSystemTime,
  oForce,
  oAllowOCSP,
aTest };



static ARGPARSE_OPTS opts[] = {
  
  { 300, NULL, 0, N_("@Commands:\n ") },

  { aServer,   "server",    256, N_("run in server mode") },
  { aListCRLs, "list-crls", 256, N_("list the contents of the CRL cache")},
  { aLoadCRL,  "load-crl",  256, N_("|FILE|load CRL from FILE into cache")},
  { aFetchCRL, "fetch-crl", 256, N_("|URL|fetch a CRL from URL")},
  { aShutdown, "shutdown",  256, N_("shutdown the dirmngr")},
  { aFlush,    "flush",     256, N_("flush the cache")},
  { aGPGConfList, "gpgconf-list", 256, "@" },

  { 301, NULL, 0, N_("@\nOptions:\n ") },

  { oVerbose,  "verbose",   0, N_("verbose") },
  { oQuiet,    "quiet",     0, N_("be somewhat more quiet") },
  { oSh,       "sh",        0, N_("sh-style command output") },
  { oCsh,      "csh",       0, N_("csh-style command output") },
  { oOptions,  "options"  , 2, N_("|FILE|read options from FILE")},
  { oDebugLevel, "debug-level" ,2, N_("|NAME|set debugging level to NAME")},
  { oNoDetach, "no-detach" ,0, N_("do not detach from the console")},
  { oLogFile,  "log-file"  ,2, N_("|FILE|write logs to FILE")},
  { oBatch   , "batch"     ,0, N_("run without asking a user")},
  { oForce,    "force"     ,0, N_("force loading of outdated CRLs")},
  { oAllowOCSP, "allow-ocsp",0,N_("allow sending OCSP requests")},

  { oLDAPFile, "ldapserverlist-file", 2,
    N_("|FILE|read LDAP server list from FILE")},
  { oLDAPAddServers, "add-servers", 0,
    N_("add new servers discovered in CRL distribution points to serverlist")},
  { oLDAPTimeout, "ldaptimeout", 1,
    N_("|N|set LDAP timeout to N seconds")},

  { oOCSPResponder, "ocsp-responder", 2, N_("|URL|use OCSP responder at URL")},
  { oOCSPSigner, "ocsp-signer", 2, N_("|FPR|OCSP response signed by FPR")}, 

  { oMaxReplies, "max-replies", 1,
    N_("|N|do not return more than N items in one query")},

  { oFakedSystemTime, "faked-system-time", 4, "@" }, /* (epoch time) */
  { oDebug,    "debug"     ,4|16, "@"},
  { oDebugAll, "debug-all" ,0,    "@"},
  { oDebugWait, "debug-wait", 1, "@"},
  { oNoGreeting, "no-greeting", 0, "@"},
  {0}
};

#define DEFAULT_MAX_REPLIES 10
#define DEFAULT_LDAP_TIMEOUT 100 /* arbitrary large timeout */

static void cleanup (void);
static ldap_server_t parse_ldapserver_file (const char* filename);
static void free_ldapservers_list (ldap_server_t servers);


static const char *
my_strusage( int level )
{
    const char *p;
    switch( level ) {
      case 11: p = "dirmngr";
	break;
      case 13: p = VERSION; break;
      case 14: p = "Copyright (C) 2004 g10 Code GmbH"; break;
      case 17: p = PRINTABLE_OS_NAME; break;
      case 19: p =
	    _("Please report bugs to <gpa-dev@gnupg.org>.\n");
	break;
      case 1:
      case 40:	p =
	    _("Usage: dirmngr [options] (-h for help)");
	break;
      case 41:	p =
	    _("Syntax: dirmngr [options] [command [args]]\n"
	      "LDAP and OCSP access for GnuPG\n");
	break;

      default:	p = NULL;
    }
    return p;
}


/* Used by gcry for logging. */
static void
my_gcry_logger (void *dummy, int level, const char *fmt, va_list arg_ptr)
{
  /* translate the log levels */
  switch (level)
    {
    case GCRY_LOG_CONT: level = JNLIB_LOG_CONT; break;
    case GCRY_LOG_INFO: level = JNLIB_LOG_INFO; break;
    case GCRY_LOG_WARN: level = JNLIB_LOG_WARN; break;
    case GCRY_LOG_ERROR:level = JNLIB_LOG_ERROR; break;
    case GCRY_LOG_FATAL:level = JNLIB_LOG_FATAL; break;
    case GCRY_LOG_BUG:  level = JNLIB_LOG_BUG; break;
    case GCRY_LOG_DEBUG:level = JNLIB_LOG_DEBUG; break;
    default:            level = JNLIB_LOG_ERROR; break;  
    }
  log_logv (level, fmt, arg_ptr);
}


/* Callback from libksba to hash a provided buffer.  Our current
   implementation does only allow SHA-1 for hashing. This may be
   extended by mapping the name, testing for algorithm availibility
   and adjust the length checks accordingly. */
static gpg_error_t 
my_ksba_hash_buffer (void *arg, const char *oid,
                     const void *buffer, size_t length, size_t resultsize,
                     unsigned char *result, size_t *resultlen)
{
  if (oid && strcmp (oid, "1.3.14.3.2.26")) 
    return gpg_error (GPG_ERR_NOT_SUPPORTED); 
  if (resultsize < 20)
    return gpg_error (GPG_ERR_BUFFER_TOO_SHORT);
  gcry_md_hash_buffer (2, result, buffer, length); 
  *resultlen = 20;
  return 0;
}


/* Setup the debugging.  With a LEVEL of NULL only the active debug
   flags are propagated to the subsystems.  With LEVEL set, a specific
   set of debug flags is set; thus overriding all flags already
   set. */
static void
set_debug (const char *level)
{
  if (!level)
    ;
  else if (!strcmp (level, "none"))
    opt.debug = 0;
  else if (!strcmp (level, "basic"))
    opt.debug = DBG_ASSUAN_VALUE;
  else if (!strcmp (level, "advanced"))
    opt.debug = DBG_ASSUAN_VALUE|DBG_X509_VALUE|DBG_LOOKUP_VALUE;
  else if (!strcmp (level, "expert"))
    opt.debug = (DBG_ASSUAN_VALUE|DBG_X509_VALUE|DBG_LOOKUP_VALUE
                 |DBG_CACHE_VALUE|DBG_CRYPTO_VALUE);
  else if (!strcmp (level, "guru"))
    opt.debug = ~0;
  else
    {
      log_error (_("invalid debug-level `%s' given\n"), level);
      log_info (_("valid debug levels are: %s\n"),
                "none, basic, advanced, expert, guru");
      dirmngr_exit (2);
    }


  if (opt.debug && !opt.verbose)
    {
      opt.verbose = 1;
      gcry_control (GCRYCTL_SET_VERBOSITY, (int)opt.verbose);
    }
  if (opt.debug && opt.quiet)
    opt.quiet = 0;

  if (opt.debug & DBG_CRYPTO_VALUE )
    gcry_control (GCRYCTL_SET_DEBUG_FLAGS, 1);
}
 

static void
i18n_init(void)
{
#ifdef USE_SIMPLE_GETTEXT
  set_gettext_file (PACKAGE);
#else
# ifdef ENABLE_NLS
#  ifdef HAVE_LC_MESSAGES
  setlocale (LC_TIME, "");
  setlocale (LC_MESSAGES, "");
#  else
  setlocale (LC_ALL, "" );
#  endif
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);
# endif
#endif
}


static void
wrong_args (const char *text)
{
  fputs (_("usage: dirmngr [options] "), stderr);
  fputs (text, stderr);
  putc ('\n', stderr);
  dirmngr_exit (2);
}



int
main (int argc, char **argv )
{
  enum cmd_and_opt_values cmd = 0;
  ARGPARSE_ARGS pargs;
  int orig_argc;
  char **orig_argv;
  FILE *configfp = NULL;
  char *configname = NULL;
  const char *shell;
  unsigned configlineno;
  int parse_debug = 0;
  int default_config =1;
  int greeting = 0;
  int nogreeting = 0;
  const char *debug_level = NULL;
  int nodetach = 0;
  int csh_style = 0;
  char *logfile = NULL;
  char *ldapfile = NULL;
  int debug_wait = 0;
  int rc;

  set_strusage (my_strusage);
  log_set_prefix ("dirmngr", 1|4); 
  
  /* Check that the libraries are suitable.  Do it here because
     the option parsing may need services of the libraries. */

  /* Init Libgcrypt. */
  gcry_control (GCRYCTL_DISABLE_SECMEM, 0);
  if (!gcry_check_version (NEED_LIBGCRYPT_VERSION) )
    {
      log_fatal (_("%s is too old (need %s, have %s)\n"),
                 "libgcrypt",
                 NEED_LIBGCRYPT_VERSION, gcry_check_version (NULL) );
    }
  gcry_set_log_handler (my_gcry_logger, NULL);
  gcry_control (GCRYCTL_INITIALIZATION_FINISHED);

  /* Init KSBA.  */
  if (!ksba_check_version (NEED_KSBA_VERSION) )
    {
      log_fatal( _("%s is too old (need %s, have %s)\n"),
                 "libksba",
                 NEED_KSBA_VERSION, ksba_check_version (NULL) );
    }
  ksba_set_malloc_hooks (gcry_malloc, gcry_realloc, gcry_free );
  ksba_set_hash_buffer_function (my_ksba_hash_buffer, NULL);


  /* Init Assuan. */
  assuan_set_malloc_hooks (gcry_malloc, gcry_realloc, gcry_free);
  assuan_set_assuan_log_stream (log_get_stream ());
  assuan_set_assuan_log_prefix (log_get_prefix (NULL));

  /* Setup I18N. */
  i18n_init();

  /* Setup defaults. */
  shell = getenv ("SHELL");
  if (shell && strlen (shell) >= 3 && !strcmp (shell+strlen (shell)-3, "csh") )
    csh_style = 1;
  
  opt.homedir = getenv("GNUPGHOME");
  if (!opt.homedir || !*opt.homedir)
    {
#ifdef HAVE_DRIVE_LETTERS
      opt.homedir = "c:/gnupg";
#else
      opt.homedir = "~/.gnupg";
#endif
    }

  /* LDAP defaults */
  opt.add_new_ldapservers = 0;
  opt.ldaptimeout.tv_sec = DEFAULT_LDAP_TIMEOUT;
  opt.ldaptimeout.tv_usec = 0;

  /* Default max replies */
  opt.max_replies = DEFAULT_MAX_REPLIES;

  /* Check whether we have a config file given on the commandline */
  orig_argc = argc;
  orig_argv = argv;
  pargs.argc = &argc;
  pargs.argv = &argv;
  pargs.flags= 1|(1<<6);  /* do not remove the args, ignore version */
  while (arg_parse( &pargs, opts))
    {
      if (pargs.r_opt == oDebug || pargs.r_opt == oDebugAll)
        parse_debug++;
      else if (pargs.r_opt == oOptions)
        { /* Yes there is one, so we do not try the default one, but
	     read the option file when it is encountered at the
	     commandline */
          default_config = 0;
	}
	else if (pargs.r_opt == oNoOptions)
          default_config = 0; /* --no-options */
	else if (pargs.r_opt == oHomedir)
          opt.homedir = pargs.r.ret_str;
    }

  if (default_config)
    configname = make_filename (opt.homedir, "dirmngr.conf", NULL );
  
  argc = orig_argc;
  argv = orig_argv;
  pargs.argc = &argc;
  pargs.argv = &argv;
  pargs.flags= 1;  /* do not remove the args */
 next_pass:
  if (configname)
    {
      configlineno = 0;
      configfp = fopen (configname, "r");
      if (!configfp)
        {
          if (default_config)
            {
              if( parse_debug )
                log_info (_("NOTE: no default option file `%s'\n"),
                          configname );
	    }
          else
            {
              log_error (_("option file `%s': %s\n"),
                         configname, strerror(errno) );
              exit(2);
	    }
          xfree (configname); 
          configname = NULL;
	}
      if (parse_debug && configname )
        log_info (_("reading options from `%s'\n"), configname );
      default_config = 0;
    }

  while (optfile_parse( configfp, configname, &configlineno, &pargs, opts) )
    {
      switch (pargs.r_opt)
        {
        case aServer: 
        case aShutdown: 
        case aFlush: 
	case aListCRLs: 
	case aLoadCRL: 
        case aFetchCRL:
	case aGPGConfList:
          cmd = pargs.r_opt;
          break;

        case oQuiet: opt.quiet = 1; break;
        case oVerbose: opt.verbose++; break;
        case oBatch: opt.batch=1; break;

        case oDebug: opt.debug |= pargs.r.ret_ulong; break;
        case oDebugAll: opt.debug = ~0; break;
        case oDebugLevel: debug_level = pargs.r.ret_str; break;
        case oDebugWait: debug_wait = pargs.r.ret_int; break;

        case oOptions:
          /* Config files may not be nested (silently ignore them) */
          if (!configfp)
            {
		xfree(configname);
		configname = xstrdup(pargs.r.ret_str);
		goto next_pass;
	    }
          break;
        case oNoGreeting: nogreeting = 1; break;
        case oNoVerbose: opt.verbose = 0; break;
        case oNoOptions: break; /* no-options */
        case oHomedir: /* Ignore this option here. */; break;
        case oNoDetach: nodetach = 1; break;
        case oLogFile: logfile = pargs.r.ret_str; break;
        case oCsh: csh_style = 1; break;
        case oSh: csh_style = 0; break;
	case oLDAPFile: ldapfile = pargs.r.ret_str; break;
	case oLDAPAddServers: opt.add_new_ldapservers = 1; break;
	case oLDAPTimeout: 
	  opt.ldaptimeout.tv_sec = pargs.r.ret_int; 
	  opt.ldaptimeout.tv_usec = 0;
	  break;
	case oMaxReplies: opt.max_replies = pargs.r.ret_int; break;

        case oAllowOCSP: opt.allow_ocsp = 1; break;
        case oOCSPResponder: opt.ocsp_responder = pargs.r.ret_str; break;
        case oOCSPSigner:    opt.ocsp_signer = pargs.r.ret_str; break;

        case oFakedSystemTime:
          set_time ( (time_t)pargs.r.ret_ulong, 0);
          break;

        case oForce: opt.force = 1; break;

        default : pargs.err = configfp? 1:2; break;
	}
    }
  if (configfp)
    {
      fclose( configfp );
      configfp = NULL;
      xfree(configname);
      configname = NULL;
      goto next_pass;
    }
  xfree (configname);
  configname = NULL;
  if (log_get_errorcount(0))
    exit(2);
  if (nogreeting )
    greeting = 0;

  if (greeting)
    {
      fprintf (stderr, "%s %s; %s\n",
               strusage(11), strusage(13), strusage(14) );
      fprintf (stderr, "%s\n", strusage(15) );
    }

#ifdef IS_DEVELOPMENT_VERSION
  log_info ("NOTE: this is a development version!\n");
#endif

  if (faked_time_p ())
    {
      dirmngr_isotime_t tbuf;
      get_isotime (tbuf);
      log_info (_("WARNING: running with faked system time %s\n"), tbuf);
    }

  set_debug (debug_level);

  /* Get LDAP server list from file. */
  if (!ldapfile) 
    {
      ldapfile = make_filename (opt.homedir, "dirmngr_ldapservers.conf", NULL);
      opt.ldapservers = parse_ldapserver_file (ldapfile);
      xfree (ldapfile);
    }
  else
      opt.ldapservers = parse_ldapserver_file (ldapfile);

  /* Ready.  Now to our duties. */
  if (!cmd)
    cmd = aServer;
  rc = 0;
  if (cmd == aServer)
    {
      if (argc)
        wrong_args ("--server");

      /* We need to ignore the PIPE signal because the we might log to
         a socket and that code handles EPIPE properly.  Assuan would
         set this signal to ignore anyway.*/
      signal (SIGPIPE, SIG_IGN);

      if (logfile)
        {
          log_set_file (logfile);
          log_set_prefix (NULL, 2|4);
        }
       
      if (debug_wait)
        {
          log_debug ("waiting for debugger - my pid is %u .....\n",
                     (unsigned int)getpid());
          sleep (debug_wait);
          log_debug ("... okay\n");
        }
      
      start_command_handler();
    }
  else if (cmd == aListCRLs)
    {
      /* Just list the CRL cache and exit. */
      crl_cache_t cache;

      if (argc)
        wrong_args ("--list-crls");
      cache = crl_cache_init ();
      crl_cache_list (cache, stdout);
    }
  else if (cmd == aLoadCRL)
    {
      crl_cache_t c;

      c = crl_cache_init ();
      if (!argc)
        rc = crl_cache_load (NULL, c, NULL);
      else
        {
          for (; !rc && argc; argc--, argv++)
            rc = crl_cache_load (NULL, c, *argv);
        }
    }
  else if (cmd == aFetchCRL)
    {
      crl_cache_t c;
      FILE *fp;

      if (argc != 1)
        wrong_args ("--fetch-crl URL");

      c = crl_cache_init ();
      rc = crl_fetch (argv[0], &fp);
      if (rc)
        log_error (_("fetching CRL from `%s' failed: %s\n"),
                     argv[0], gpg_strerror (rc));
      else
        {
          rc = crl_cache_insert (NULL, c,  argv[0], fp); 
          if (rc)
            log_error (_("processing CRL from `%s' failed: %s\n"),
                       argv[0], gpg_strerror (rc));
          fclose (fp);
        }
    }
  else if (cmd == aFlush)
    {
      /* Delete cache and exit. */
      if (argc)
        wrong_args ("--flush");
      rc = crl_cache_flush();
    }
  else if (cmd == aGPGConfList)
    {
      char *filename;

      /* List options and default values in the GPG Conf format.  */

/* The following list is taken from gnupg/tools/gpgconf-comp.c.  */
/* Option flags.  YOU MUST NOT CHANGE THE NUMBERS OF THE EXISTING
   FLAGS, AS THEY ARE PART OF THE EXTERNAL INTERFACE.  */
#define GC_OPT_FLAG_NONE	0UL
/* The DEFAULT flag for an option indicates that the option has a
   default value.  */
#define GC_OPT_FLAG_DEFAULT	(1UL << 4)
/* The DEF_DESC flag for an option indicates that the option has a
   default, which is described by the value of the default field.  */
#define GC_OPT_FLAG_DEF_DESC	(1UL << 5)
/* The NO_ARG_DESC flag for an option indicates that the argument has
   a default, which is described by the value of the ARGDEF field.  */
#define GC_OPT_FLAG_NO_ARG_DESC	(1UL << 6)

      /* First the configuration file.  This is not an option, but it
	 is vital information for GPG Conf.  */
      filename = make_filename (opt.homedir, "dirmngr.conf", NULL);
      printf ("gpgconf-dirmngr.conf:%lu:\"%s\n",
              GC_OPT_FLAG_DEFAULT, filename);
      xfree (filename);

      printf ("verbose:%lu:\n", GC_OPT_FLAG_NONE);
      printf ("quiet:%lu:\n", GC_OPT_FLAG_NONE);
      printf ("debug-level:%lu:\"none\n", GC_OPT_FLAG_DEFAULT);
      printf ("log-file:%lu:\n", GC_OPT_FLAG_NONE);
      printf ("force:%lu:\n", GC_OPT_FLAG_NONE);

      /* --csh and --sh are mutually exclusive, something we can not
         express in GPG Conf.  --options is only usable from the
         command line, really.  --debug-all interacts with --debug,
         and having both of them is thus problematic.  --no-detach is
         also only usable on the command line.  --batch is unused.  */

      filename = make_filename (opt.homedir, "dirmngr_ldapservers.conf", NULL);
      printf ("ldapserverlist-file:%lu:\"%s\n", GC_OPT_FLAG_DEFAULT, filename);
      xfree (filename);

      printf ("ldaptimeout:%lu:%u\n",
              GC_OPT_FLAG_DEFAULT, DEFAULT_LDAP_TIMEOUT);
      printf ("max-replies:%lu:%u\n",
              GC_OPT_FLAG_DEFAULT, DEFAULT_MAX_REPLIES);
      printf ("allow-ocsp:%lu:\n", GC_OPT_FLAG_NONE);
      printf ("ocsp-responder:%lu:\n", GC_OPT_FLAG_NONE);
      printf ("ocsp-signer:%lu:\n", GC_OPT_FLAG_NONE);

      printf ("faked-system-time:%lu:\n", GC_OPT_FLAG_NONE);
      printf ("no-greeting:%lu:\n", GC_OPT_FLAG_NONE);
    }
  cleanup ();
  return !!rc;
}


static void
cleanup (void)
{
  free_ldapservers_list (opt.ldapservers);
  opt.ldapservers = NULL;
}


void 
dirmngr_exit (int rc)
{
  cleanup ();
  exit (rc);
}


void
dirmngr_init_default_ctrl (ctrl_t ctrl)
{
  /* Nothing for now. */
}


/* Create a list of LDAP servers from the file FILENAME. Returns the
   list or NULL in case of errors. 

   The format fo such a file is line oriented where empty lines and
   lines starting with a hash mark are ignored.  All other lines are
   assumed to be colon seprated with these fields:

   1. field: Hostname
   2. field: Portnumber
   3. field: Username 
   4. field: Password
   5. field: Base DN

*/
static ldap_server_t
parse_ldapserver_file (const char* filename)
{
  char buffer[1024];
  char *p, *endp;
  ldap_server_t server, serverstart, *serverend;
  int c;
  unsigned int lineno = 0;
  FILE *fp;
  int fieldno;

  fp = fopen (filename, "r");
  if (!fp)
    {
      log_error (_("error opening `%s': %s\n"), filename, strerror (errno));
      return NULL;
    }

  serverstart = NULL;
  serverend = &serverstart;
  while (fgets (buffer, sizeof buffer, fp))
    {
      int fail = 0;

      lineno++;
      if (!*buffer || buffer[strlen(buffer)-1] != '\n')
        {
          if (*buffer && feof (fp))
            ; /* Last line not terminated - continue. */
          else
            {
              log_error (_("%s:%u: line too long - skipped\n"),
                         filename, lineno);
              while ( (c=fgetc (fp)) != EOF && c != '\n')
                ; /* Skip until end of line. */
              continue;
            }
        }
      /* Skip empty and comment lines.*/
      for (p=buffer; spacep (p); p++)
        ;
      if (!*p || *p == '\n' || *p == '#')
        continue;

      /* Parse the colon separated fields. */
      server = xcalloc (1, sizeof *server);
      for (fieldno=1, p = buffer; p; p = endp, fieldno++ )
        {
          endp = strchr (p, ':');
          if (endp)
            *endp++ = '\0';
          trim_spaces (p);
          switch (fieldno)
            {
            case 1:
              if (*p)
                server->host = xstrdup (p);
              else
                {
                  log_error (_("%s:%u: no hostname given\n"),
                             filename, lineno);
                  fail = 1;
                }
              break;
                
            case 2:
              if (*p)
                server->port = atoi (p);
              break;

            case 3:
              if (*p)
                server->user = xstrdup (p);
              break;

            case 4:
              if (*p && !server->user)
                {
                  log_error (_("%s:%u: password given without user\n"), 
                             filename, lineno);
                  fail = 1;
                }
              else if (*p)
                server->pass = xstrdup (p);
              break;

            case 5:
              if (*p)
                server->base = xstrdup (p);
              break;

            default:
              /* (We silently ignore extra fields.) */
              break;
            }
        }
          
      if (fail)
        {
          log_info (_("%s:%u: skipping this line\n"), filename, lineno);
          free_ldapservers_list (server);
        }
      else
        {
          *serverend = server;
          serverend = &server->next;
        }
    } 
  
  if (ferror (fp))
    log_error (_("error reading `%s': %s\n"), filename, strerror (errno));
  fclose (fp);

  return serverstart;
}


/* Release the list of SERVERS. As usual it is okay to call this
   fucntion with SERVERS passed as NULL. */
static void 
free_ldapservers_list (ldap_server_t servers)
{
  while (servers)
    {
      ldap_server_t tmp = servers->next;
      xfree (servers->host);
      xfree (servers->user);
      if (servers->pass)
        memset (servers->pass, 0, strlen (servers->pass));
      xfree (servers->pass);
      xfree (servers->base);
      xfree (servers);
      servers = tmp;
    }
}
