##
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
##

import gtk
import Constants
import DSApp

class DlgDiskEdit:
    """This class contains the implementation of the Edit disk dialog
    (for the selected disk)."""
    
    def __init__ (self):
        """Constructor of DlgDiskEdit class."""

        # get the name of the selected disk
        selection = DSApp.DSApp.view.list_disks.get_selection ()        
        selected_tuple = selection.get_selected ()
        selected_tree_iter = selected_tuple[1]
        if selected_tree_iter == None:
            DSApp.DSApp.view.display_message_dialog (gtk.MESSAGE_ERROR, _("You need to select a disk first!"))
            return
            
        # find disk and print content to buffer
        selected_disk_name = DSApp.DSApp.view.list_disks.get_model ().get_value (selected_tree_iter, 1)
        selected_disk_name = DSApp.DSApp.view.convert_from_unicode (selected_disk_name)

        self.edited_disk = None
        for disk in DSApp.DSApp.doc.diskList.content:
            if (disk.name == selected_disk_name):
                self.edited_disk = disk
                break

        # display name, type and comment in dialog
        dlg_content_widget_tree = gtk.glade.XML (Constants.RESOURCE_DIR + "/disksearch.glade", "dlg_edit")
        self.dlg_edit = dlg_content_widget_tree.get_widget ("dlg_edit")
        self.tf_name = dlg_content_widget_tree.get_widget ("tf_name")
        self.tf_name.set_text (DSApp.DSApp.view.convert_to_unicode (self.edited_disk.name))
        self.tf_remark = dlg_content_widget_tree.get_widget ("tf_remark")
        self.tf_remark.set_text (DSApp.DSApp.view.convert_to_unicode (self.edited_disk.remark))
        self.om_type = dlg_content_widget_tree.get_widget ("om_type")
        DSApp.DSApp.view.fill_media_type_optionmenu (self.om_type, self.edited_disk.type)
        
        # display content in treeview
        self.tv_content = dlg_content_widget_tree.get_widget ("tv_content")
        self.display_content ()
        
        dlg_content_widget_tree.signal_connect ("on_cancel_clicked", self.edit_disk_cancel)
        dlg_content_widget_tree.signal_connect ("on_ok_clicked", self.edit_disk_ok)

        self.dlg_edit.set_transient_for (DSApp.DSApp.view.app_disksearch)
        self.dlg_edit.set_icon (DSApp.DSApp.view.pxb_appicon)
        self.dlg_edit.show ()
    
    def display_content (self):
        """This method initializes the treeview and displays the content in it."""
        
        # init and add columns for name and size
        # (the cell renderer of the size column get's the xalign value from the third column)
        column1 = gtk.TreeViewColumn (_("Name"), gtk.CellRendererText (), text = 0)
        cellRendererSize = gtk.CellRendererText ()
        column2 = gtk.TreeViewColumn (_("Size"), cellRendererSize, text = 1)
        column2.add_attribute (cellRendererSize, "xalign", 2)
        column1.set_resizable (True)
        column1.set_expand (True)
        
        self.tv_content.append_column (column1)
        self.tv_content.append_column (column2)
        self.tv_content.get_selection ().set_mode (gtk.SELECTION_NONE)

        # build the treestore and fill it hierarchically with all dirs and files
        store = gtk.TreeStore (str, str, float)        
        for dir in self.edited_disk.content:
            
            # append directory name (no size)
            iterDir = store.append (None)
            store.set (iterDir, 0, DSApp.DSApp.view.convert_to_unicode (dir[0]))
            store.set (iterDir, 1, "")
            store.set (iterDir, 2, 1.0)
                        
            for file in dir[1]:
                # append file name
                iterFile = store.append (iterDir)
                store.set (iterFile, 0, DSApp.DSApp.view.convert_to_unicode (file[0]))

                # append file size (optional)
                if file[1] != None:
                    store.set (iterFile, 1, DSApp.DSApp.view.convert_to_unicode (str (file[1])))
                else:
                    store.set (iterFile, 1, "")
                store.set (iterFile, 2, 1.0)
        
        # set model and expand all directories
        self.tv_content.set_model (store)
        self.tv_content.expand_all ()
            
    def edit_disk_cancel (self, *args):
        """Closes the edit disk dialog."""
        
        self.edited_disk = None
        self.dlg_edit.destroy ()
        
    def edit_disk_ok (self, *args):
        """Stores the edited disk data in list."""
        
        # get disk name (required) and remark from dialog
        name = self.tf_name.get_text ().strip ()
        name = DSApp.DSApp.view.convert_from_unicode (name)
        remark = self.tf_remark.get_text ().strip ()
        remark = DSApp.DSApp.view.convert_from_unicode (remark)        
        type = DSApp.DSApp.doc.diskList.media_types[self.om_type.get_history ()][0]

        if len (name) == 0:
            self.tf_name.select_region (0, -1)
            DSApp.DSApp.view.display_message_dialog_with_parrent (self.dlg_edit, gtk.MESSAGE_ERROR, _("You need to enter a valid name!"))
            self.tf_name.grab_focus ()
            return

        # was a new name entered ? => make sure that new name is not in use yet
        if name != self.edited_disk.name:
            for disk in DSApp.DSApp.doc.diskList.content:
                if name == disk.name:
                    self.tf_name.select_region (0, -1)
                    DSApp.DSApp.view.display_message_dialog_with_parrent (self.dlg_edit, gtk.MESSAGE_ERROR, _("This name is allready in use, please enter another one!"))
                    self.tf_name.grab_focus ()
                    return
        
        # store disk data and close dialog
        self.edited_disk.name = name
        self.edited_disk.type = type
        self.edited_disk.remark = remark
        
        # remove and re-add the disk in list to keep alphabetical order
        # (the name might be changed)
        DSApp.DSApp.doc.diskList.remove_disk (self.edited_disk.name)
        DSApp.DSApp.doc.diskList.add_scanned_disk (self.edited_disk)

        self.edited_disk = None
        self.dlg_edit.destroy ()
        DSApp.DSApp.view.display_disk_list (name)
