#
# commit.py - DITrack 'commit' command
#
# Copyright (c) 2006-2007 The DITrack Project, www.ditrack.org.
#
# $Id: commit.py 1144 2007-01-23 07:07:37Z gli $
# $HeadURL: https://127.0.0.1/ditrack/src/tags/0.5/DITrack/Command/commit.py $
#
# Redistribution and use in source and binary forms, with or without 
# modification, are permitted provided that the following conditions are met:
#
#  * Redistributions of source code must retain the above copyright notice, 
# this list of conditions and the following disclaimer.
#  * Redistributions in binary form must reproduce the above copyright notice, 
# this list of conditions and the following disclaimer in the documentation 
# and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
# ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
# POSSIBILITY OF SUCH DAMAGE.
#

import string
import sys

# DITrack modules
import DITrack.Command.generic

class Handler(DITrack.Command.generic.Handler):
    canonical_name = "commit"
    description = """Commit local changes to the repository.
usage: %s [ID...]""" % canonical_name

    def run(self, opts, globals):

        self.check_options(opts)

        db = DITrack.Util.common.open_db(globals, opts)

        if len(opts.fixed) > 1:

            # XXX: we should sort the entities to be committed
            ids = []

            # We are told explicitly what to commit
            for id in opts.fixed[1:]:

                try:
                    issue_id, comment_id = \
                        DITrack.DB.entity_id_split(id.upper())
                except ValueError:
                    sys.stderr.write(
                        "Invalid entity name '%s', ignored\n" % id
                        )
                    continue

                assert(issue_id is not None)

                if comment_id is None:

                    invalid = False
                    for x in issue_id:
                        if x in string.digits:
                            invalid = True
                            break
                    
                    if invalid:
                        sys.stderr.write(
                            "Non-local issue id '%s', ignored\n" % id
                            )
                        continue

                else:

                    invalid = False
                    for x in comment_id:
                        if x in string.digits:
                            invalid = True
                            break
                    
                    if invalid:
                        sys.stderr.write(
                            "Non-local comment id in '%s', ignored\n" % id
                            )
                        continue


                ids.append((issue_id, comment_id))

        else:
            # Commit all local issues (not comments!)
            # See i#108: for now, let's just commit new issues since handling
            # comments is much more complex.
            ids = map(lambda x: (x[0], None), db.issues(from_wc=False))

        for issue_id, comment_id in ids:

            if comment_id is None:
                firm_id = db.commit_issue(issue_id)

                id = "%s" % issue_id

            else:
                assert issue_id

                firm_id = db.commit_comment(issue_id, comment_id)

                firm_id = "%s.%s" % (issue_id, firm_id)
                id = "%s.%s" % (issue_id, comment_id)

            assert firm_id is not None

            # XXX: print 'Local i#ABC committed as i#123 in r234'.
            sys.stdout.write("Local i#%s committed as i#%s.\n" % (id, firm_id))
