#
# list.py - DITrack 'list' command
#
# Copyright (c) 2006-2007 The DITrack Project, www.ditrack.org.
#
# $Id: list.py 2190 2007-10-15 22:31:17Z vss $
# $HeadURL: https://127.0.0.1/ditrack/src/tags/0.7/DITrack/Command/list.py $
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#  * Redistributions of source code must retain the above copyright notice,
# this list of conditions and the following disclaimer.
#  * Redistributions in binary form must reproduce the above copyright notice,
# this list of conditions and the following disclaimer in the documentation
# and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
#

import os
import sys

# DITrack modules
import DITrack.DB.Common
import DITrack.DB.Exceptions
import DITrack.Client
import DITrack.Command.generic
import DITrack.XML

class _FallbackDict:
    """
    A wrapper (fallback) dictionary-like class allowing to fetch info header 
    fields and fall back to default values if there is no such header present.
    """
    def __getitem__(self, key):
        if key == "id":
            return "%s" % self.id
        elif key in self.info:
            return self.info[key]
        else:
            return "-";

    def __init__(self, id, info):
        self.id = id
        self.info = info

class Handler(DITrack.Command.generic.Handler):
    canonical_name = "list"
    description = """List issues, possibly filtering.
usage: %s [FILTER...]""" % canonical_name

    def _xml_output(self, globals, issues, filters):

        input = [
            ("user",    globals.username)
        ]

        for f in filters:
            input.append(("filter", str(f)))

        xo = DITrack.XML.Output("list", input) 
            
        for id, issue in issues:
            xo.writer.opentag("issue", { "id": str(id) }, nl=True)

            for k, v in issue.info.items():
                xo.writer.tag_enclose(k, {}, v, nl=True, indent=2)

            xo.writer.closetag("issue")
            xo.writer.text("\n\n")

        xo.finish()


    def run(self, opts, globals):
        self.check_options(opts)

        db = DITrack.Util.common.open_db(globals, opts)

        if opts.var.has_key("listing_format_list") \
                and opts.var["listing_format_list"]:
            format_list = db.cfg.listing_format.items.keys()
            format_list.sort()
            for f in format_list:
                sys.stdout.write("%s\n" % f)
            return

        xml = ("xml" in opts.var) and opts.var["xml"]

        if xml and ("listing_format" in opts.var):
            DITrack.Util.common.err(
                "Listing output format can't be specified when performing "
                "output in XML."
            )

        client = DITrack.Client.Client(db)

        filters = []

        for e in opts.fixed[1:]:
            try:
                f = DITrack.DB.Common.Filter(e, globals.username)
            except DITrack.DB.Exceptions.FilterIsPredefinedError:
                if e in db.cfg.filters:
                    f = db.cfg.filters[e]
                else:
                    DITrack.Util.common.err(
                        "ERROR: '%s' is not a predefined filter" % e
                    )

            except DITrack.DB.Exceptions.FilterExpressionError:
                DITrack.Util.common.err(
                    "ERROR: '%s' contains a syntax error" % e
                )

            filters.append(f)
       
        try:
            if opts.var.has_key("listing_format"):
                format = db.cfg.listing_format[opts.var["listing_format"]]
            else:
                format = db.cfg.listing_format["default"]
        except DITrack.DB.Exceptions.InvalidListingFormatError, key:
            DITrack.Util.common.err(
                "ERROR: Unknown listing format '%s'" % key
            )

        issues = client.issues(filters)
        if xml:
            self._xml_output(globals, issues, filters)
        else:
            for id, issue in issues:
                print format % _FallbackDict(id, issue.info)
