#
# XML.py - DITrack XML output support
#
# Copyright (c) 2007 The DITrack Project, www.ditrack.org.
#
# $Id: list.py 2182 2007-10-09 21:23:50Z vss $
# $HeadURL: https://127.0.0.1/ditrack/src/trunk/DITrack/Command/list.py $
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#  * Redistributions of source code must retain the above copyright notice,
# this list of conditions and the following disclaimer.
#  * Redistributions in binary form must reproduce the above copyright notice,
# this list of conditions and the following disclaimer in the documentation
# and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
#

# XXX: this module is dt-specific, so it should be moved to an appropriate
# place.

import sys
from xml.sax.saxutils import XMLGenerator
from xml.sax.xmlreader import AttributesImpl

import DITrack.dt.globals

# XML format version
XML_FORMAT = "1"

class Writer:
    """
    Low level XML output primitives.
    """
    def __init__(self, out=sys.stdout):
        self._xml = XMLGenerator(out, "utf-8")
        self._xml.startDocument()

    def closetag(self, tag, nl=False, indent=0):
        assert indent >= 0

        self.text(" " * indent)

        self._xml.endElement(tag)
        if nl:
            self.text("\n")

    def finish(self):
        self._xml.endDocument()

    def opentag(self, tag, attrs={}, nl=False, indent=0):
        assert indent >= 0

        self.text(" " * indent)

        self._xml.startElement(tag, AttributesImpl(attrs))
        if nl:
            self.text("\n")

    def tag_enclose(self, tag, attrs, text, nl=False, indent=0):
        assert indent >= 0

        self.text(" " * indent)

        self._xml.startElement(tag, attrs)
        self._xml.characters(text)
        self._xml.endElement(tag)

        if nl:
            self.text("\n")

    def text(self, s):
        self._xml.characters(s)


class Output:
    """
    DITrack-specific XML output producer.
    """

    def __init__(self, command, input):
        """
        Writes the dt XML output header to stdout. COMMAND is a command name
        and INPUT is a list of tuples (TAG, VALUE) to be output as input
        parameters to the command (e.g. enclosed in an '<input>' block).
        """

        self.writer = Writer()
        self.writer.opentag("dt",
            {
                "format":       XML_FORMAT,
                "dt-version":   DITrack.dt.globals.VERSION,
            }
        )
        self.writer.text("\n\n")

        self.writer.opentag("input", { "command": command })

        for k, v in input:
            self.writer.text("\n    ")
            self.writer.tag_enclose(k, {}, v)

        self.writer.text("\n")

        self.writer.closetag("input")
        self.writer.text("\n\n")

        self.writer.opentag("output")
        self.writer.text("\n\n")

    def finish(self):
        self.writer.closetag("output")
        self.writer.text("\n")
        self.writer.closetag("dt")
        self.writer.text("\n")
        self.writer.finish()

