/*
 *  Remote Network application system for DNAS -- replay-multi routine
 *  Copyright (C) 2002 Junichi Uekawa
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: rnas-replay-multi.c,v 1.4 2002/10/30 11:01:47 dancer Exp $
 */
#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <pthread.h>
#include "config.h"
#include "dmachinemon/dmachinemon-debug.h"
#include "dmachinemon/dmachinemon.h"
#include "dmachinemon/libsocket.h"
#include "dmachinemon/dmachinemon-libdatabase.h"
#include "dmachinemon/dmachinemon-commandlineparser.h"


pthread_mutex_t database_lock = PTHREAD_MUTEX_INITIALIZER;

static void
rnas_printhelp(void)
{
  fprintf(stderr, 
	  "RNAS application, Copyright (c)2002 Junichi Uekawa.\n\n"
	  "\trnas-replay-multi [DNAS options] -- [filenames to replay]\n"
	  "Please specify files to replay\n\n\n"
	  );
  dmachinemon_print_commandline_help();      
}

/* load the file into memory */
static void
load_file(const char * filename)
{
  FILE* f=fopen (filename, "r");

  pthread_mutex_lock(&database_lock);
  printf ("Load file %s\n", filename);
  
  if (!f)
    {
      fprintf (stderr, PROGRAMNAME ": cannot open specified file\n");
      exit(1);
    }

  if (dm_process_nodes_input (f, &maininfo, NULL, NULL))
    {
      fprintf (stderr, PROGRAMNAME ": could not handle input...\n");
      exit (1);
    }  

  pthread_mutex_unlock(&database_lock);
  fclose(f);
}

/* send the info out to the file descriptor. */
static void
send_info(FILE*f)
{
  dm_machinelist * m;
  dm_pairstring * s;
  

  pthread_mutex_lock(&database_lock);
  for (m=maininfo.machinedb; m; m=m->next)
    {
      fprintf(f, ":%s:\n", m->machinename);
      for (s=m->data; s; s=s->next)
	{
	  fprintf(f, "%s: %s\n", s->stringtitle, s->stringdata);
	}
    }
  pthread_mutex_unlock(&database_lock);
}

/* code nicked from -servent */
static int 
handle_clients(void * data)
{				/* access from clients. */
  int t = ( (struct dm_handle_incoming_params*) data )->t;
  FILE*f = (t!=-1)?fdopen(t, "w"):NULL;
  if (!f) 
    {
      fprintf (stderr, "Error: could not open incoming connection ? \n");
      return 0;
    }

  send_info(f);
  
  fclose(f);
  dm_tcp_free_incoming_params(data);
  return 0;
}

/*
  Creating a server interface for clients, giving out info if accessed to this port.
*/
static void
clientserver (void * parameter)
{
  dm_tcp_host_setup("rnas-reply-single", (char*)parameter,
		 (void*) handle_clients);
  exit (1);
}

int
main(int ac, char ** av)
{
  int i;
  dm_commandoption cdat;
  pthread_t thread;
  
  dmachinemon_parse_options(ac, av, &cdat);

  if (!av[optind])
    {
      rnas_printhelp();
      exit(1);
    }
  dm_daemonize(&cdat);
  
  pthread_create(&thread, NULL, (void*)clientserver, cdat.port_client);
  
  while (1)
    {
      for (i=optind; av[i]; ++i)
	{
	  load_file (av[i]);
	  usleep(1000000L * cdat.sleeptime);
	}
    }
  exit(0);
}


/*
What follows is a m4 script!
MANUAL([.TH "rnas-replay-multi" 1 "2002 Oct 10" "DNAS" "RNAS"
.SH NAME
rnas-replay-multi \- RNAS replay daemon
.SH SYNOPSIS
.BI "rnas-reply-multi [" "DNAS-options" "] -- [" "filename-to-load" "]*"
.SH DESCRIPTION
Replay RNAS output files to fool other RNAS application.
.SH "DNAS OPTIONS"
] m4_include([dmachinemon-commandlineparser.inc]) COMMAND [
.SH "AUTHOR"
Junichi Uekawa <dancer@mikilab.doshisha.ac.jp> <dancer@debian.org> <dancer@netfort.gr.jp>

This manual page is autogenerated from rnas-replay-single.c and 
dmachinemon-commandlineparser.c
.SH "SEE ALSO"
.RI "rnas-dump (" 1 "), "
.RI "rnas-dump-xml (" 1 "), "
.RI "rnas-replay-single (" 1 "), "
.RI "rnas-monitor-xml (" 1 ")"])
*/

