/* dnotify.c - Execute a command every time the contents of a directory changes
 *
 * Copyright (C) 2002  Oskar Liljeblad
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <stdarg.h>
#include <errno.h>
#include <getopt.h>
#include <dirent.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include "dnotify.h"

#define VERSION_MESSAGE \
	"%s %s\n" \
	"Written by %s.\n" \
	"\n" \
	"Copyright (C) 2002 %s.\n" \
	"This is free software; see the source for copying conditions.  There is NO\n" \
	"warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n"
#define HELP_MESSAGE \
	"Usage: %s [OPTION]... DIRECTORY... [-e COMMAND...]\n" \
	"Execute a command every time the contents of a directory change.\n" \
	"\n" \
	"Events:\n" \
	"  -A, --access             trigger when a file in the directory was accessed\n" \
	"  -M, --modify             trigger when a file in the directory was modified\n" \
	"  -C, --create             trigger when a file was created in the directory\n" \
	"  -D, --delete             trigger whan a file was unlinked from the directory\n" \
	"  -R, --rename             trigger when a file in the directory was renamed\n" \
	"  -B, --attrib             trigger when the directory had its attributes\n" \
	"                           changed (after chmod, chown)\n" \
	"  -a, --all                all of the above\n" \
	"\n" \
	"General:\n" \
	"  -e, --execute=COMMAND..  command to execute when an event is triggered\n" \
	"                           (all remaining args are treated as command args)\n" \
	"  -p, --processes=COUNT    max number of commands to run at a time\n" \
	"  -q, --queue=DEPTH        max depth of queue holding commands to be run\n" \
	"  -t, --times=COUNT        exit after running the command COUNT times\n" \
	"  -o, --once               same as `--times 1'\n" \
	"  -r, --recursive          monitor subdirectories too (recursively)\n" \
	"  -s, --silent             don't print warnings about non-zero child exits\n" \
	"      --quiet              same as `--silent'\n" \
	"      --help               display this help and exit\n" \
	"      --version            output version information and exit\n" \
	"\n" \
	"Unless events have been specified, the events are set to create and delete.\n" \
	"\n" \
	"Specifying -1 for queue means unlimited depth. The default value for --queue\n" \
	"is -1, and 1 for --processes.\n" \
	"\n" \
	"The default command (unless specified with -e) is to print the name of the\n" \
	"of the directory updated. The string `{}' in the command specification is\n" \
	"replaced by the name of the directory that was updated.\n" \
	"\n" \
	"Report bugs to <" PACKAGE_BUGREPORT ">.\n"

typedef struct {
	int fd;
	char *name;
} MonitorFD;

typedef struct {
	void *next;
	int fd;
} QueueFD;

static QueueFD *first_queue_fd = NULL;
static QueueFD *last_queue_fd = NULL;

static const char *const short_opts = "-AMCDRBaep:q:rt:os";
static struct option long_opts[] = {
	{ "access",             no_argument,        NULL, 'A' },
	{ "modify",             no_argument,        NULL, 'M' },
	{ "create",             no_argument,        NULL, 'C' },
	{ "delete",             no_argument,        NULL, 'D' },
	{ "rename",             no_argument,        NULL, 'R' },
	{ "attrib",             no_argument,        NULL, 'B' },
   	{ "all",                no_argument,        NULL, 'a' },
	{ "execute",            no_argument,        NULL, 'e' },
	{ "processes",          required_argument,  NULL, 'p' },
	{ "queue",              required_argument,  NULL, 'q' },
	{ "times",              required_argument,  NULL, 't' },
	{ "once",               no_argument,        NULL, 'o' },
	{ "recursive",          no_argument,        NULL, 'r' },
	{ "silent",             no_argument,        NULL, 's' },
	{ "quiet",              no_argument,        NULL, 's' },
	{ "version",            no_argument,        NULL, 'V' },
	{ "help",               no_argument,        NULL, 'H' },
	{ 0,                    0,                  0,    0,  },
};

static void
sigrt_handler(int sig, siginfo_t *si, void *data)
{
	/* No operation. */
}

static void
sigchld_handler(int sig)
{
	/* No operation. */
}

static void
add_directory(char *dirname, MonitorFD **monv, int *monc, int *monvsize, int recursive)
{
	int fd;

	fd = open(dirname, O_RDONLY);
	if (fd < 0)
		die("cannot open directory `%s' - %s", dirname, errstr);

	if ((*monc)+1 >= *monvsize) {
		*monvsize = (*monvsize == 0 ? 8 : (*monvsize) * 2);
		*monv = realloc(*monv, (*monvsize) * sizeof(MonitorFD));
	}
	(*monv)[*monc].fd = fd;
	(*monv)[*monc].name = dirname;
	(*monc)++;

	if (recursive) {
		DIR *dir;
		struct dirent *ep;

		dir = opendir(dirname);
		if (dir == NULL)
			die("cannot open directory `%s' - %s", dirname, errstr);

		while ((ep = readdir(dir)) != NULL) {
			if ((strcmp(ep->d_name, ".") == 0 || strcmp(ep->d_name, "..") == 0))
				continue;

			if (ep->d_type == DT_UNKNOWN) {
				char *subdirname;
				struct stat statbuf;

				/* I agree this is slow but modern libc's don't return DT_UNKNOWN */
				subdirname = concat_dirname(dirname, ep->d_name);

				if (lstat(subdirname, &statbuf) < 0)
					die("cannot stat file `%s' - %s", subdirname, errstr);
				if (S_ISDIR(statbuf.st_mode))
					add_directory(subdirname, monv, monc, monvsize, recursive);
				else
					free(subdirname);
			}
			else if (ep->d_type == DT_DIR) {
				char *subdirname;

				subdirname = concat_dirname(dirname, ep->d_name);
				add_directory(subdirname, monv, monc, monvsize, recursive);
			}
			/* subdirname will be freed when the MonitorFD is freed! */
		}

		if (errno != 0)	/* assume readdir sets errno to 0 if all is ok */
			die("cannot read directory `%s' - %s", dirname, errstr);
		if (closedir(dir) < 0)
			die("cannot read directory `%s' - %s", dirname, errstr);
	}
}

/* This function returns non-zero if a child had terminated. */
static int
wait_process(void)
{
	int status;
	int rc;

	rc = waitpid(-1, &status, WNOHANG);
	if (rc < 0)
		die("waitpid on child failed - %s", errstr);
	if (rc > 0) {
		if (!WIFEXITED(status))
			warn("child terminated abnormally");
		if (WEXITSTATUS(status) != 0)
			warn("child terminated with non-zero status");
		return 1;
	}

	return 0;
}

static void
run_child(int fd, char **argv, int argcount, MonitorFD *monv, int monc)
{
	pid_t child;

	child = fork();
	if (child < 0)
		die("cannot fork child - %s", errstr);
	if (child == 0) {
		char *child_argv[argcount == 0 ? 3 : argcount+1];
		char *trigfile = NULL;
		int c;

		/* Try a smart guess to find monitor filename, otherwise do O(n) search */
		c = fd-monv[0].fd;
		if (c >= 0 && c < monc && monv[c].fd == fd) {
			trigfile = monv[c].name;
		} else {
			for (c = 0; c < monc; c++) {
				if (monv[c].fd == fd) {
					trigfile = monv[c].name;
					break;
				}
			}
			if (trigfile == NULL)
				exit(1);
		}

		if (argcount == 0) {
			puts(trigfile);
			exit(0);
		}

		for (c = 0; c < monc; c++)
			close(monv[c].fd); /* ignore errors */

		child_argv[0] = argv[0];
		for (c = 1; c < argcount; c++)
			child_argv[c] = expand_braces(argv[c], trigfile);
		child_argv[argcount] = NULL;

		if (execvp(child_argv[0], child_argv) < 0)
			die("cannot execute `%s' - %s", child_argv[0], errstr);
		exit(1);
	}
}

static int
unqueue_fd(void)
{
	QueueFD *qfd = first_queue_fd;
	
	first_queue_fd = qfd->next;
	if (qfd->next == NULL)
		last_queue_fd = NULL;

	return qfd->fd;
}

static void
queue_fd(int fd)
{
	QueueFD *qfd = malloc(sizeof(QueueFD));
	if (qfd == NULL)
		die("out of memory");
	qfd->fd = fd;

	if (last_queue_fd != NULL)
		last_queue_fd->next = qfd;
	qfd->next = NULL;
	last_queue_fd = qfd;
	if (first_queue_fd == NULL)
		first_queue_fd = qfd;
}

int main(int argc, char **argv)
{
	MonitorFD *monv = NULL;
	int monc = 0;
	int monvsize = 0;
	char *dirv[argc-1];
	int dirc = 0;
	struct sigaction act;
	long notifyarg = 0;
	int c;
	int recursive = 0;			/* corresponds to --recursive */
	int32_t times = -1;			/* corresponds to --times */
	int32_t max_processes = 1;	/* corresponds to --processes */
	int32_t max_queue = -1;		/* corresponds to --queue */
	int32_t cur_processes = 0;
	int32_t cur_queue = 0;
	sigset_t signalset;

	while (1) {
		c = getopt_long(argc, argv, short_opts, long_opts, NULL);
		if (c == -1 || c == 'e')
			break;

		switch (c) {
		case 1:	/* non-option argument */
			dirv[dirc] = strdup(optarg);
			if (dirv[dirc] == NULL)
				die("out of memory");
			dirc++;
			break;
		case 'A':
			notifyarg |= DN_ACCESS;
			break;
		case 'M':
			notifyarg |= DN_MODIFY;
			break;
		case 'C':
			notifyarg |= DN_CREATE;
			break;
		case 'D':
			notifyarg |= DN_DELETE;
			break;
		case 'R':
			notifyarg |= DN_RENAME;
			break;
		case 'B':
			notifyarg |= DN_ATTRIB;
			break;
		case 'a':
			notifyarg |= DN_ACCESS|DN_MODIFY|DN_CREATE|DN_DELETE|DN_RENAME|DN_ATTRIB;
			break;
		case 's':
			quiet = 1;
			break;
		case 'r':
			recursive = 1;
			break;
		case 'p':
			if (!parse_int32(optarg, &max_processes))
				die("invalid processes value: %s", optarg);
			if (max_processes <= 0)
				max_processes = -1;
			break;
		case 'q':
			if (!parse_int32(optarg, &max_queue))
				die("invalid queue value: %s", optarg);
			break;
		case 't':
			if (!parse_int32(optarg, &times))
				die("invalid times value: %s", optarg);
			if (times <= 0)
				times = -1;
			break;
		case 'o':
			times = 1;
			break;
		case 'V': /* --version */
			printf(VERSION_MESSAGE, PACKAGE, VERSION, AUTHOR, AUTHOR);
			exit(0);
			break;
		case 'H': /* --help */
			printf(HELP_MESSAGE, argv[0]);
			exit(0);
			break;
		case '?':
			exit(1);
		}
	}

	if (dirc == 0)
		die("missing directory argument\nTry `%s --help' for more information.", argv[0]);

	if (notifyarg == 0)
		notifyarg = DN_CREATE|DN_DELETE;

	sigemptyset(&signalset);
	sigaddset(&signalset, SIGCHLD);
	sigaddset(&signalset, SIGRTMIN);

	/* Work around lpthread bug (libc/4927). */
	if (sigprocmask(SIG_UNBLOCK, &signalset, NULL) < 0)
		die("sigprocmask failed - %s", errstr);

	/* Register SIGRTMIN and SIGCHLD signal handlers. */
	act.sa_sigaction = sigrt_handler;
	sigemptyset(&act.sa_mask);
	act.sa_flags = SA_SIGINFO | SA_RESTART | SA_NODEFER;
	if (sigaction(SIGRTMIN, &act, NULL) < 0)
		die("sigaction failed - %s", errstr);

	act.sa_sigaction = (void (*)(int, siginfo_t *, void *)) sigchld_handler;
	act.sa_flags = SA_RESTART | SA_NODEFER | SA_NOCLDSTOP;
	if (sigaction(SIGCHLD, &act, NULL) < 0)
		die("sigaction failed - %s", errstr);

	/* Block SIGRTMIN and SIGCHLD. */
	if (sigprocmask(SIG_BLOCK, &signalset, NULL) < 0)
		die("sigprocmask failed - %s", errstr);

	for (c = 0; c < dirc; c++)
		add_directory(dirv[c], &monv, &monc, &monvsize, recursive);

	for (c = 0; c < monc; c++) {
		if (fcntl(monv[c].fd, F_SETSIG, SIGRTMIN) < 0)
			die("fcntl F_SETSIG on `%s' failed - %s", monv[c].name, errstr);
		if (fcntl(monv[c].fd, F_NOTIFY, notifyarg|DN_MULTISHOT) < 0)
			die("fcntl F_NOTIFY on `%s' failed - %s", monv[c].name, errstr);
	}

	for (;;) {
	    siginfo_t signalinfo;
	    int signal;

	    signal = TEMP_FAILURE_RETRY(sigwaitinfo(&signalset, &signalinfo));
	    if (signal < 0)
	    	die("sigwaitinfo failed - %s", errstr);

	    if (signal == SIGCHLD) {
		    while (cur_processes > 0 && wait_process()) {
			    cur_processes--;
			    if (times == 0 && cur_processes == 0)
				    break; /* Terminate program */
			    if (cur_queue > 0) {
				    cur_queue--;
				    run_child(unqueue_fd(), argv+optind, argc-optind, monv, monc);
				    cur_processes++;
				    if (times > 0)
					    times--;
			    }
		    }
	    }
	    else if (signal == SIGRTMIN && (times < 0 || times > 0)) {
			int fd = signalinfo.si_fd;

		    if (max_processes < 0 || cur_processes < max_processes) {
			    run_child(fd, argv+optind, argc-optind, monv, monc);
			    cur_processes++;
			    if (times > 0)
				    times--;
		    } else if (max_queue < 0 || cur_queue < max_queue) {
			    queue_fd(fd);
			    cur_queue++;
		    }
	    }
	}

	for (c = 0; c < monc; c++) {
		close(monv[c].fd); /* ignore errors */
		free(monv[c].name);
	}
	free(monv);

	exit(0);
}
