#ifndef ExceptionDocumentationAnalyzer_h
#include "ExceptionDocumentationAnalyzer.h"
#endif

#ifndef AST_h
#include "AST.h"
#endif

#ifndef SpellChecker_h
#include "SpellChecker.h"
#endif

#ifndef JavadocTags_h
#include "JavadocTags.h"
#endif

#ifndef DocumentationErrors_h
#include "DocumentationErrors.h"
#endif

#ifndef std_algorithm
#define std_algorithm
#include <algorithm>
#endif

using namespace doctorj;

static const int MAX_EDIT_DISTANCE = 4;

ExceptionDocumentationAnalyzer::ExceptionDocumentationAnalyzer(Reporter* const reporter,
                                                               AstNoncode* const noncode,
                                                               AstModifierList* const modifiers,
                                                               AstThrowsNameList* const throwsList) :
        JavadocAnalyzer(reporter, modifiers), 
     noncode_(noncode),
     throwsList_(throwsList),
     lastDocThrow_(NULL)
{
}

ExceptionDocumentationAnalyzer::~ExceptionDocumentationAnalyzer()
{
    if (throwsList_) {
        DocumentedThrowsListType::const_iterator stop = throwsDocumented_.end();
        DocumentedThrowsListType::const_iterator it   = throwsDocumented_.begin();
        while (it != stop) {
            AstTaggedDescribedComment* tdc = it->first;
            ++it;
        }
    }
}

void ExceptionDocumentationAnalyzer::checkThrows(AstTaggedComment* const tc, const string& tag) 
{
    if (throwsList_) {
        AstTaggedDescribedComment* tdc = dynamic_cast<AstTaggedDescribedComment*>(tc);
        if (tdc) {
            int nTargets = tc->countTargets();
            if (nTargets == 0) {
                ErrorExceptionWithoutClassName err(reporter(), tc, tag);
                err.process();
            }
            else if (nTargets == 1) {
                ErrorExceptionWithoutDescription err(reporter(), tc, tag);
                err.process();
            }

            if (codeThrowExists(tdc, tag)) {
                string docThrow = tdc->target();
                if (lastDocThrow_ && docThrow < lastDocThrow_->target()) {
                    AstStringLeaf tlf = tdc->targetLeaf();
                    ErrorExceptionOrderNotAlphabetical err(reporter(), tdc, &tlf, tag, docThrow, lastDocThrow_);
                    err.process();
                }
                throwDocumented(tdc, tdc->target());
            }
            else if (!docThrowMisspelled(tdc, tag)) {
                AstStringLeaf tlf = tc->tagLeaf();
                ErrorExceptionMismatchWithCode err(reporter(), &tlf, tag);
                err.process();
            }
            lastDocThrow_ = tdc;
        }
        else {
            // AstStringLeaf tlf = tc->tagLeaf();
            ErrorExceptionWithoutClassName err(reporter(), tc, tag);
            err.process();
        }
    }
}

void ExceptionDocumentationAnalyzer::checkException(AstTaggedComment* const tc) 
{
    // I thought I once read that @throws is preferred, but I haven't been able
    // to verify this in the standard:
    // complain(tc, 777, "@throws is preferred to @exception");
    
    checkThrows(tc, JavadocTags::EXCEPTION);
}

bool ExceptionDocumentationAnalyzer::docThrowMisspelled(AstTaggedDescribedComment* const tdc, const string& tag) 
{
    string docThrow = tdc->target();
    const char* const cDocThrow = docThrow.c_str();
    int docThrowLen = docThrow.length();
    
    AstNameList* codeNames = throwsList_->getNameList();
    int nCodeThrows = codeNames->getNameCount();
    SpellChecker cchk;

    for (int i = 0; i < nCodeThrows; ++i) {
        string codeThrow = codeNames->getName(i)->fullName();
        int edist = cchk.editDistance(codeThrow.c_str(), codeThrow.length(), cDocThrow, docThrowLen);
        if (edist >= 0 && edist <= MAX_EDIT_DISTANCE) {
            throwDocumented(tdc, codeThrow);
            // throwDocumented(codeThrow);
            AstStringLeaf tlf = tdc->targetLeaf();
            ErrorExceptionMisspelled err(reporter(), &tlf, tag, docThrow, codeThrow);
            err.process();
            return true;
        }
    }
    return false;
}

void ExceptionDocumentationAnalyzer::checkUndocumentedThrows()
{
    if (throwsList_) {
        AstNameList* names = throwsList_->getNameList();
        int nThrows = names->getNameCount();
        for (int ti = 0; ti < nThrows; ++ti) {
            AstName* name = names->getName(ti);
            string codeThrow = name->fullName();
            bool found = false;
            DocumentedThrowsListType::const_iterator stop = throwsDocumented_.end();
            DocumentedThrowsListType::const_iterator it   = throwsDocumented_.begin();
            while (!found && it != stop) {
                AstTaggedDescribedComment* tdc = it->first;
                string name = it->second;
                if (tdc->target() == codeThrow || name == codeThrow) {
                    found = true;
                }
                ++it;
            }
            
            if (!found) {
                ErrorExceptionUndocumented err(reporter(), noncode_, name, codeThrow);
                err.process();
            }
        }
    }
}

void ExceptionDocumentationAnalyzer::throwDocumented(AstTaggedDescribedComment* const thr,
                                                     const string& name)
{
    throwsDocumented_.push_back(make_pair(thr, name));
}

bool ExceptionDocumentationAnalyzer::codeThrowExists(AstTaggedDescribedComment* const tdc,
                                                     const string& tag)
{
    if (throwsList_) {
        string docThrow = tdc->target();
        AstNameList* codeNames = throwsList_->getNameList();
        int nCodeThrows = codeNames->getNameCount();
        for (int i = 0; i < nCodeThrows; ++i) {
            string codeThrow = codeNames->getName(i)->fullName();
            if (codeThrow == docThrow) {
                return true;
            }
        }
    }
    return false;
}
