#ifndef MethodCtorDocumentationAnalyzer_h
#include "MethodCtorDocumentationAnalyzer.h"
#endif

#ifndef File_h
#include "File.h"
#endif

#ifndef DocumentationErrors_h
#include "DocumentationErrors.h"
#endif

#ifndef AST_h
#include "AST.h"
#endif

#ifndef StringUtilities_h
#include "StringUtilities.h"
#endif

#ifndef JavadocTags_h
#include "JavadocTags.h"
#endif

#ifndef ExceptionDocumentationAnalyzer_h
#include "ExceptionDocumentationAnalyzer.h"
#endif

using namespace doctorj;

// some exceptions are OK not to be declared, specifically, those that are
// RuntimeExceptions, which do not have to be in the throws declaration of the
// method/constructor
static char* VALID_UNDECLARED_EXCEPTIONS[] = {
    "ArithmeticException",
    "ArrayStoreException",
    "BufferOverflowException",
    "BufferUnderflowException",
    "CMMException",
    "CannotRedoException",
    "CannotUndoException",
    "ClassCastException",
    "ConcurrentModificationException",
    "DOMException",
    "EmptyStackException",
    "IllegalArgumentException",
    "IllegalMonitorStateException",
    "IllegalPathStateException",
    "IllegalStateException",
    "ImagingOpException",
    "IndexOutOfBoundsException",
    "MissingResourceException",
    "NegativeArraySizeException",
    "NoSuchElementException",
    "NullPointerException",
    "ProfileDataException",
    "ProviderException",
    "RasterFormatException",
    "SecurityException",
    "SystemException",
    "UndeclaredThrowableException",
    "UnmodifiableSetException",
    "UnsupportedOperationException"
};
static int NUM_VALID_UNDECLARED_EXCEPTIONS = sizeof(VALID_UNDECLARED_EXCEPTIONS) / sizeof(VALID_UNDECLARED_EXCEPTIONS[0]);

/**
 * Returns whether the tagged comment documents an exception that is valid not
 * to be declared in the throws clause of a method or constructor. Eventually,
 * at run-time we will determine the superclass of the given exception.
 */
static bool isValidUndeclaredException(AstTaggedComment* const tc)
{
    AstTaggedDescribedComment* tdc = dynamic_cast<AstTaggedDescribedComment*>(tc);
    if (tdc) {
        string namedException = tdc->target();
        const string JAVA_LANG = "java.lang.";
        if (StringUtilities::startsWith(namedException, JAVA_LANG)) {
            namedException = namedException.substr(JAVA_LANG.length());
        }
        for (int i = 0; i < NUM_VALID_UNDECLARED_EXCEPTIONS; ++i) {
            if (string(VALID_UNDECLARED_EXCEPTIONS[i]) == namedException) {
                return true;
            }
        }
    }
    return false;
}

MethodCtorDocumentationAnalyzer::MethodCtorDocumentationAnalyzer(Reporter* const reporter,
                                                                 AstModifierList* const modifiers,
                                                                 AstFormalParameterList* const parameters,
                                                                 AstThrowsNameList* const throwsList) :
        ItemDocumentationAnalyzer(reporter, modifiers),
     throwsList_(throwsList),
     exceptionAnalyzer_(NULL),
     parameterAnalyzer_(reporter, modifiers, parameters)
{
}

MethodCtorDocumentationAnalyzer::~MethodCtorDocumentationAnalyzer()
{
    if (exceptionAnalyzer_) {
        delete exceptionAnalyzer_;
    }
}

bool MethodCtorDocumentationAnalyzer::handleTag(AstTaggedComment* const tc) 
{
    string tag = tc->tag();
    if (tag == JavadocTags::PARAM) {
        checkParameter(tc);
        return true;
    }
    else if (tag == JavadocTags::EXCEPTION) {
        checkException(tc);
        return true;
    }
    else if (tag == JavadocTags::THROWS) {
        checkThrows(tc);
        return true;
    }
    else if (tag == JavadocTags::RETURN) {
        checkReturn(tc);
        return true;
    }
    else if (tag == JavadocTags::SERIALDATA) {
        checkSerialData(tc);
        return true;
    }
    else {
        return ItemDocumentationAnalyzer::handleTag(tc);
    }
}

void MethodCtorDocumentationAnalyzer::checkUndocumented()
{
    checkUndocumentedReturn();
    checkUndocumentedParams();
    checkUndocumentedThrows();
}

void MethodCtorDocumentationAnalyzer::checkSerialData(AstTaggedComment* const tc) 
{
    int nTargets = tc->countTargets();
    if (nTargets <= 0) {
        ErrorSerialDataWithoutDescription err(reporter(), tc);
        err.process();
    }
}

void MethodCtorDocumentationAnalyzer::checkThrows(AstTaggedComment* const tc) 
{
    if (throwsList_) {
        getExceptionAnalyzer()->checkThrows(tc, JavadocTags::THROWS);
    }
    else if (!isValidUndeclaredException(tc)) {
        AstStringLeaf tlf = tc->tagLeaf();
        ErrorExceptionNotInCode err(reporter(), &tlf, getSubject()->text());
        err.process();
    }
}

void MethodCtorDocumentationAnalyzer::checkException(AstTaggedComment* const tc) 
{
    if (throwsList_) {
        getExceptionAnalyzer()->checkException(tc);
    }
    else if (!isValidUndeclaredException(tc)) {
        AstStringLeaf tlf = tc->tagLeaf();
        ErrorExceptionNotInCode err(reporter(), &tlf, getSubject()->text());
        err.process();
    }
}

void MethodCtorDocumentationAnalyzer::checkParameter(AstTaggedComment* const tc) 
{
    // check the parameter against the current argument (note that in the
    // documentation, we're calling it a 'parameter'; in the method (code)
    // itself, its an 'argument'.

    // save the current parameter now, since getParameterStatus may advance it.
    AstFormalParameter* cp = parameterAnalyzer_.currentParameter();
    int status = parameterAnalyzer_.getParameterStatus(tc);

    if (status & DOC_PARAMS_NOT_IN_CODE) {
        AstStringLeaf tlf = tc->tagLeaf();
        ErrorParametersDocumentedButNotInCode err(reporter(), &tlf, type());
        err.process();
    }
    else if (status & DOC_PARAM_NAME_MISSING) {
        AstStringLeaf tlf = tc->tagLeaf();
        ErrorNoParameterName err(reporter(), &tlf, cp->name());
        err.process();
    }
    else if (AstTaggedDescribedComment* tdc = dynamic_cast<AstTaggedDescribedComment*>(tc)) {
        // it must be described
        string docName = tdc->target();
        char* tgtSt = tdc->targetStart();
        char* tgtEnd = tdc->targetEnd();

        if (status & DOC_PARAM_NAME_MISSING) {
            ErrorNoParameterName err(reporter(), tdc, cp->name());
            err.process();
        }
        else if (status & DOC_PARAM_DESCRIPTION_MISSING) {
            ErrorNoParameterDescription err(reporter(), tdc);
            err.process();
        }
        
        if (status & DOC_PARAM_OK) {
            // nothing, but this bypasses all the following checks.
        }
        else if (status & DOC_PARAM_USES_DATA_TYPE) {
            AstStringLeaf tlf = tdc->targetLeaf();
            ErrorParameterDataTypeUsed err(reporter(), &tlf, docName, cp->dataType(), cp->name());
            err.process();        
        }
        else if (status & DOC_PARAM_MISSPELLED) {
            AstStringLeaf tlf = tdc->targetLeaf();
            ErrorParameterMisspelled err(reporter(), &tlf, docName, cp->name());
            err.process();
        }
        else if (status & DOC_PARAM_REPEATED) {
            AstStringLeaf tlf = tdc->targetLeaf();
            ErrorParameterRepeatedInDocumentation err(reporter(), &tlf, docName);
            err.process();
        }
        else if (status & DOC_PARAM_MISORDERED) {
            AstStringLeaf tlf = tdc->targetLeaf();
            ErrorParameterOutOfOrder err(reporter(), &tlf, docName);
            err.process();
        }
        else if (status & DOC_PARAM_MISORDERED_AND_MISSPELLED) {
            AstStringLeaf tlf = tdc->targetLeaf();
            ErrorParameterOutOfOrderAndMispelled err(reporter(), &tlf, docName, cp->name());
            err.process();
        }
        else if (status & DOC_PARAM_NOT_IN_CODE) {
            AstStringLeaf tlf = tdc->targetLeaf();
            ErrorParameterNotInCode err(reporter(), &tlf, docName);
            err.process();
        }
        else {
            /* escond */ cout << "wtf? status of " << docName << " = " << status << endl;
        }

    }
    else {
        /* escond */ cout << "wtf? this shouldn't happen. tag = " << tc << " status = " << status;
    }
}

void MethodCtorDocumentationAnalyzer::checkUndocumentedParams()
{
    vector<AstFormalParameter*> undocumented = parameterAnalyzer_.getUndocumentedParams();
    vector<AstFormalParameter*>::const_iterator it   = undocumented.begin();
    vector<AstFormalParameter*>::const_iterator stop = undocumented.end();
    while (it != stop) {
        AstFormalParameter* param = *it;
        ErrorParameterUndocumented err(reporter(), param, param->name());
        err.process();
        ++it;
    }
}

void MethodCtorDocumentationAnalyzer::checkUndocumentedThrows()
{
    if (throwsList_) {
        getExceptionAnalyzer()->checkUndocumentedThrows();
    }
}

ExceptionDocumentationAnalyzer* MethodCtorDocumentationAnalyzer::getExceptionAnalyzer()
{
    if (!exceptionAnalyzer_) {
        exceptionAnalyzer_ = new ExceptionDocumentationAnalyzer(reporter(), 
                                                                leadingNoncode(),
                                                                modifiers(), 
                                                                throwsList_);
    }
    return exceptionAnalyzer_;
}
