#ifndef Parser_h
#include "Parser.h"
#endif

#ifndef File_h
#include "File.h"
#endif

#ifndef AstDebug_h
#include "AstDebug.h"
#endif

#include <stdio.h>

using namespace std;

/**
 * Yacc parser.
 */
extern int yyparse();

using namespace doctorj;

Parser::Parser() :
        currentFile_(NULL),
     lastWsPosition_(NULL),
     position_(NULL),
     tabWidth_(8),
     javaVersion_(1.3),
     hadError_(false)
{
}

Parser::~Parser()
{
    vector<File*>::iterator stop = files_.end();
    for (vector<File*>::iterator it = files_.begin(); it != stop; ++it) {
        File* f = *it;
        delete f;
    }
}

File* Parser::currentFile() const
{
    return currentFile_;
}

char* Parser::position() const
{
    return position_;
}

void Parser::setPosition(char* pos)
{
    position_ = pos;
}

void Parser::incrementPosition(int pos)
{
    position_ += pos;
}

void Parser::point(const string& tokenType) const
{
    // printf("%*s%s\n", 1 + position_, "^", tokenType.c_str());;
}

AstNoncode* Parser::makeNoncode() const
{
    DEBUG_NONCODE(cout << "Noncode("
                  << "from " << (unsigned int)(lastWsPosition_)
                  << " to " << (unsigned int)(position_)
                  << ". current_file " << currentFile_->name() << endl);
    if (lastWsPosition_ != position_) {
        AstNoncode* nc = new AstNoncode(lastWsPosition_, position_, currentFile_);
        return nc;
    }
    else {
        DEBUG_NONCODE(cout << "not creating empty noncode" << endl);
        return NULL;
    }
}

void Parser::display(const string& tokenType) const
{
    DEBUG_SCANNER({
        cout << currentFile()->fullLine(position_) << endl;
        point(tokenType);
    });
}

void Parser::reset(AstLeaf* const leaf, int len)
{
    AstNoncode* nc = makeNoncode();
    DEBUG_PARSER(cout << "setting leading noncode for <" << leaf->type() << "> to " << nc << endl);
    leaf->setLeadingNoncode(nc);
    lastWsPosition_ = position_ + len;

    // should we map from position_ to leaf? I guess not, for now.

    position_ += len;
}

void Parser::parse(AstCompilationUnit*& compUnit, const string& fileName) 
{
    File* file = new File(fileName);
    parse(compUnit, file);
}

void Parser::parse(AstCompilationUnit*& compUnit, File* const file) 
{
    DEBUG_PARSER(cout << "parsing '" << file->name() << "'" << endl);

    hadError_ = false;

    currentFile_ = file;
    position_ = currentFile_->head();
    lastWsPosition_ = currentFile_->head();

    yyparse();

    if (hadError_) {
        compUnit = NULL;
    }
    else if (compUnit) {
        AstNoncode* nc = makeNoncode();
        compUnit->setTrailingNoncode(nc);
    }
    else {
        DEBUG_PARSER(cout << "ERROR: " << file->name() << ": no compilation unit -- parsing failed." << endl);
    }
}

void Parser::report(const string& errType, const string& message)
{
    cerr << errType << ": " << currentFile()->name() 
         << "[" << currentFile()->lineOf(position_) << "]: " << endl;
    cerr << currentFile()->fullLine(position_) << endl;
    cerr << message << endl;
}

void Parser::reportError(const string& message)
{
    hadError_ = true;
    report("ERROR", message);
}

void Parser::reportWarning(const string& message)
{
    report("WARNING", message);
}

void Parser::setTabWidth(int tabWidth)
{
    tabWidth_ = tabWidth;
}

int Parser::tabWidth() const
{
    return tabWidth_;
}

void Parser::setJavaVersion(double version)
{
    javaVersion_ = version;
}

double Parser::javaVersion() const
{
    return javaVersion_;
}
