%{
/**
 * Grammar for Doctor J (Java code analyzer).
 *   author: Jeffrey E. Pace (jpace@erols.com)
 */
    
#ifndef AST_h
#include "AST.h"
#endif

#ifndef File_h
#include "File.h"
#endif

#ifndef Parser_h
#include "Parser.h"
#endif

#include <stdio.h>

#ifndef std_string
#define std_string
#include <string>
#endif

#ifndef std_iostream
#define std_iostream
#include <iostream>
#endif

#define PCF parser.currentFile()

using namespace doctorj;

Parser parser;

/*
 * Writes the error message.
 */
void yyerror(char* msg) 
{
    parser.reportError(msg);
}

extern int yylex();

doctorj::AstCompilationUnit* compUnit = NULL;

%}

%union {
    doctorj::AstAbstract* n_abstract;
    doctorj::AstAnd* n_and;
    doctorj::AstAndand* n_andand;
    doctorj::AstAndeq* n_andeq;
    doctorj::AstArrayAccess* t_array_access;
    doctorj::AstArrayInitializer* t_array_initializer;
    doctorj::AstAssert* n_assert;
    doctorj::AstAssignmentOperator* n_assignment_operator;
    doctorj::AstBlock* t_block;
    doctorj::AstBlockStatementList* t_block_statement_list;
    doctorj::AstBoolean* n_boolean;
    doctorj::AstBooleanLiteral* n_boolean_literal;
    doctorj::AstBreak* n_break;
    doctorj::AstByte* n_byte;
    doctorj::AstCase* n_case;
    doctorj::AstCatch* n_catch;
    doctorj::AstCatchClause* t_catch_clause;
    doctorj::AstCatchClauseList* t_catch_clause_list;
    doctorj::AstChar* n_char;
    doctorj::AstCharacterLiteral* n_character_literal;
    doctorj::AstClass* n_class;
    doctorj::AstClassBody* t_class_body;
    doctorj::AstClassBodyDeclarationList* t_class_body_declaration_list;
    doctorj::AstClassDeclaration* t_class_declaration;
    doctorj::AstClassInstanceCreationExpression* t_class_instance_creation_expression;
    doctorj::AstColon* n_colon;
    doctorj::AstComma* n_comma;
    doctorj::AstComp* n_comp;
    doctorj::AstCompilationUnit* t_compilation_unit;
    doctorj::AstConst* n_const;
    doctorj::AstConstructorBody* t_constructor_body;
    doctorj::AstConstructorDeclarator* t_constructor_declarator;
    doctorj::AstContinue* n_continue;
    doctorj::AstDefault* n_default;
    doctorj::AstDimension* t_dimension;
    doctorj::AstDimensionExpression* t_dimension_expression;
    doctorj::AstDimensionExpressionList* t_dimension_expression_list;
    doctorj::AstDimensionList* t_dimension_list;
    doctorj::AstDiv* n_div;
    doctorj::AstDiveq* n_diveq;
    doctorj::AstDo* n_do;
    doctorj::AstDot* n_dot;
    doctorj::AstDouble* n_double;
    doctorj::AstElse* n_else;
    doctorj::AstEq* n_eq;
    doctorj::AstEqeq* n_eqeq;
    doctorj::AstExplicitConstructorInvocation* t_explicit_constructor_invocation;
    doctorj::AstExpressionList* t_expression_list;
    doctorj::AstExtends* n_extends;
    doctorj::AstFieldAccess* t_field_access;
    doctorj::AstFieldDeclaration* t_field_declaration;
    doctorj::AstFinal* n_final;
    doctorj::AstFinally* n_finally;
    doctorj::AstFinallyBlock* t_finally_block;
    doctorj::AstFloat* n_float;
    doctorj::AstFloatingPointLiteral* n_floating_point_literal;
    doctorj::AstFor* n_for;
    doctorj::AstFormalParameter* t_formal_parameter;
    doctorj::AstFormalParameterList* t_formal_parameter_list;
    doctorj::AstGoto* n_goto;
    doctorj::AstGt* n_gt;
    doctorj::AstGteq* n_gteq;
    doctorj::AstIdentifier* n_identifier;
    doctorj::AstIf* n_if;
    doctorj::AstImplements* n_implements;
    doctorj::AstImplementsInterfaceList* t_implements_interface_list;
    doctorj::AstImport* n_import;
    doctorj::AstImportDeclaration* t_import_declaration;
    doctorj::AstImportDeclarationList* t_import_declaration_list;
    doctorj::AstInstanceof* n_instanceof;
    doctorj::AstInt* n_int;
    doctorj::AstIntegerLiteral* n_integer_literal;
    doctorj::AstInterface* n_interface;
    doctorj::AstInterfaceBody* t_interface_body;
    doctorj::AstInterfaceDeclaration* t_interface_declaration;
    doctorj::AstInterfaceMemberDeclarationList* t_interface_member_declaration_list;
    doctorj::AstItem* t_item;
    doctorj::AstLbrace* n_lbrace;
    doctorj::AstLbracket* n_lbracket;
    doctorj::AstLeaf* n_leaf;
    doctorj::AstLocalVariableDeclaration* t_local_variable_declaration;
    doctorj::AstLong* n_long;
    doctorj::AstLparen* n_lparen;
    doctorj::AstLshift* n_lshift;
    doctorj::AstLshifteq* n_lshifteq;
    doctorj::AstLt* n_lt;
    doctorj::AstLteq* n_lteq;
    doctorj::AstMethodDeclarator* t_method_declarator;
    doctorj::AstMethodHeader* t_method_header;
    doctorj::AstMethodInvocation* t_method_invocation;
    doctorj::AstMinus* n_minus;
    doctorj::AstMinuseq* n_minuseq;
    doctorj::AstMinusminus* n_minusminus;
    doctorj::AstMod* n_mod;
    doctorj::AstModeq* n_modeq;
    doctorj::AstModifierList* t_modifier_list;
    doctorj::AstMult* n_mult;
    doctorj::AstMulteq* n_multeq;
    doctorj::AstName* t_name;
    doctorj::AstNameList* t_name_list;
    doctorj::AstNative* n_native;
    doctorj::AstNew* n_new;
    doctorj::AstNode* t_node;
    doctorj::AstNot* n_not;
    doctorj::AstNoteq* n_noteq;
    doctorj::AstNullLiteral* n_null_literal;
    doctorj::AstOr* n_or;
    doctorj::AstOreq* n_oreq;
    doctorj::AstOror* n_oror;
    doctorj::AstPackage* n_package;
    doctorj::AstPackageDeclaration* t_package_declaration;
    doctorj::AstPlus* n_plus;
    doctorj::AstPluseq* n_pluseq;
    doctorj::AstPlusplus* n_plusplus;
    doctorj::AstPrimitiveType* n_primitive_type;
    doctorj::AstPrivate* n_private;
    doctorj::AstProtected* n_protected;
    doctorj::AstPublic* n_public;
    doctorj::AstQuestion* n_question;
    doctorj::AstRbrace* n_rbrace;
    doctorj::AstRbracket* n_rbracket;
    doctorj::AstReturn* n_return;
    doctorj::AstRparen* n_rparen;
    doctorj::AstRshift* n_rshift;
    doctorj::AstRshifteq* n_rshifteq;
    doctorj::AstSemicolon* n_semicolon;
    doctorj::AstShort* n_short;
    doctorj::AstStatementExpressionList* t_statement_expression_list;
    doctorj::AstStatic* n_static;
    doctorj::AstStrictfp* n_strictfp;
    doctorj::AstStringLiteral* n_string_literal;
    doctorj::AstSuper* n_super;
    doctorj::AstSwitch* n_switch;
    doctorj::AstSwitchBlock* t_switch_block;
    doctorj::AstSwitchBlockStatementGroup* t_switch_block_statement_group;
    doctorj::AstSwitchBlockStatementGroupList* t_switch_block_statement_group_list;
    doctorj::AstSwitchLabel* t_switch_label;
    doctorj::AstSwitchLabelList* t_switch_label_list;
    doctorj::AstSynchronized* n_synchronized;
    doctorj::AstThis* n_this;
    doctorj::AstThrow* n_throw;
    doctorj::AstThrows* n_throws;
    doctorj::AstThrowsNameList* t_throws_name_list;
    doctorj::AstTransient* n_transient;
    doctorj::AstTry* n_try;
    doctorj::AstTypeDeclarationList* t_type_declaration_list;
    doctorj::AstUrshift* n_urshift;
    doctorj::AstUrshifteq* n_urshifteq;
    doctorj::AstVariableDeclarator* t_variable_declarator;
    doctorj::AstVariableDeclaratorId* t_variable_declarator_id;
    doctorj::AstVariableDeclaratorList* t_variable_declarator_list;
    doctorj::AstVariableInitializerList* t_variable_initializer_list;
    doctorj::AstVoid* n_void;
    doctorj::AstVolatile* n_volatile;
    doctorj::AstWhile* n_while;
    doctorj::AstXor* n_xor;
    doctorj::AstXoreq* n_xoreq;
}


/* token types */
%token <n_abstract> ABSTRACT
%token <n_and> AND
%token <n_andand> ANDAND
%token <n_andeq> ANDEQ
%token <n_assert> ASSERT
%token <n_boolean> BOOLEAN
%token <n_boolean_literal> BOOLEAN_LITERAL
%token <n_break> BREAK
%token <n_byte> BYTE
%token <n_case> CASE
%token <n_catch> CATCH
%token <n_char> CHAR
%token <n_character_literal> CHARACTER_LITERAL
%token <n_class> CLASS
%token <n_colon> COLON
%token <n_comma> COMMA
%token <n_comp> COMP
%token <n_const> CONST
%token <n_continue> CONTINUE
%token <n_default> DEFAULT
%token <n_div> DIV
%token <n_diveq> DIVEQ
%token <n_do> DO
%token <n_dot> DOT
%token <n_double> DOUBLE
%token <n_else> ELSE
%token <n_eq> EQ
%token <n_eqeq> EQEQ
%token <n_extends> EXTENDS
%token <n_final> FINAL
%token <n_finally> FINALLY
%token <n_float> FLOAT
%token <n_floating_point_literal> FLOATING_POINT_LITERAL
%token <n_for> FOR
%token <n_goto> GOTO
%token <n_gt> GT
%token <n_gteq> GTEQ
%token <n_identifier> IDENTIFIER
%token <n_if> IF
%token <n_implements> IMPLEMENTS
%token <n_import> IMPORT
%token <n_instanceof> INSTANCEOF
%token <n_int> INT
%token <n_integer_literal> INTEGER_LITERAL
%token <n_interface> INTERFACE
%token <n_lbrace> LBRACE
%token <n_lbracket> LBRACKET
%token <n_long> LONG
%token <n_lparen> LPAREN
%token <n_lshift> LSHIFT
%token <n_lshifteq> LSHIFTEQ
%token <n_lt> LT
%token <n_lteq> LTEQ
%token <n_minus> MINUS
%token <n_minuseq> MINUSEQ
%token <n_minusminus> MINUSMINUS
%token <n_mod> MOD
%token <n_modeq> MODEQ
%token <n_mult> MULT
%token <n_multeq> MULTEQ
%token <n_native> NATIVE
%token <n_new> NEW
%token <n_not> NOT
%token <n_noteq> NOTEQ
%token <n_null_literal> NULL_LITERAL
%token <n_or> OR
%token <n_oreq> OREQ
%token <n_oror> OROR
%token <n_package> PACKAGE_
%token <n_plus> PLUS
%token <n_pluseq> PLUSEQ
%token <n_plusplus> PLUSPLUS
%token <n_private> PRIVATE
%token <n_protected> PROTECTED
%token <n_public> PUBLIC
%token <n_question> QUESTION
%token <n_rbrace> RBRACE
%token <n_rbracket> RBRACKET
%token <n_return> RETURN
%token <n_rparen> RPAREN
%token <n_rshift> RSHIFT
%token <n_rshifteq> RSHIFTEQ
%token <n_semicolon> SEMICOLON
%token <n_short> SHORT
%token <n_static> STATIC
%token <n_strictfp> STRICTFP
%token <n_string_literal> STRING_LITERAL
%token <n_super> SUPER
%token <n_switch> SWITCH
%token <n_synchronized> SYNCHRONIZED
%token <n_this> THIS
%token <n_throw> THROW
%token <n_throws> THROWS
%token <n_transient> TRANSIENT
%token <n_try> TRY
%token <n_urshift> URSHIFT
%token <n_urshifteq> URSHIFTEQ
%token <n_void> VOID
%token <n_volatile> VOLATILE
%token <n_while> WHILE
%token <n_xor> XOR
%token <n_xoreq> XOREQ
%type <n_assignment_operator> assignment_operator;
%type <n_leaf> modifier;
%type <n_primitive_type> primitive_type;
%type <t_array_access> array_access;
%type <t_array_initializer> array_initializer;
%type <t_block> block;
%type <t_block_statement_list> block_statement_list;
%type <t_catch_clause> catch_clause;
%type <t_catch_clause_list> catch_clause_list;
%type <t_class_body> class_body;
%type <t_class_body_declaration_list> class_body_declaration_list;
%type <t_class_declaration> class_declaration;
%type <t_class_instance_creation_expression> class_instance_creation_expression;
%type <t_compilation_unit> compilation_unit;
%type <t_constructor_body> constructor_body;
%type <t_constructor_declarator> constructor_declarator;
%type <t_dimension> dimension;
%type <t_dimension_expression> dimension_expression;
%type <t_dimension_expression_list> dimension_expression_list;
%type <t_dimension_list> dimension_list;
%type <t_explicit_constructor_invocation> explicit_constructor_invocation;
%type <t_expression_list> expression_list;
%type <t_field_access> field_access;
%type <t_field_declaration> field_declaration;
%type <t_finally_block> finally_block;
%type <t_formal_parameter> formal_parameter;
%type <t_formal_parameter_list> formal_parameter_list;
%type <t_implements_interface_list> implements_interface_list;
%type <t_import_declaration> import_declaration;
%type <t_import_declaration_list> import_declaration_list;
%type <t_interface_body> interface_body;
%type <t_interface_declaration> interface_declaration;
%type <t_interface_member_declaration_list> interface_member_declaration_list;
%type <t_item> additive_expression;
%type <t_item> and_expression;
%type <t_item> block_statement;
%type <t_item> conditional_and_expression;
%type <t_item> conditional_expression;
%type <t_item> conditional_or_expression;
%type <t_item> equality_expression;
%type <t_item> exclusive_or_expression;
%type <t_item> expression;
%type <t_item> inclusive_or_expression;
%type <t_item> multiplicative_expression;
%type <t_item> postfix_expression;
%type <t_item> primary;
%type <t_item> primary_no_new_array;
%type <t_item> reference_type;
%type <t_item> relational_expression;
%type <t_item> shift_expression;
%type <t_item> statement;
%type <t_item> statement_no_short_if;
%type <t_item> statement_without_trailing_substatement;
%type <t_item> type;
%type <t_item> type_declaration;
%type <t_item> unary_expression;
%type <t_item> unary_expression_not_plus_minus;
%type <t_item> variable_initializer;
%type <t_local_variable_declaration> local_variable_declaration;
%type <t_method_declarator> method_declarator;
%type <t_method_header> method_header;
%type <t_method_invocation> method_invocation;
%type <t_modifier_list> modifier_list;
%type <t_name> name;
%type <t_name_list> name_list;
%type <t_node> assignment;
%type <t_item> class_body_declaration;
%type <t_node> class_member_declaration;
%type <t_node> interface_member_declaration;
%type <t_node> postdecrement_expression;
%type <t_node> postincrement_expression;
%type <t_node> predecrement_expression;
%type <t_node> preincrement_expression;
%type <t_node> statement_expression;
%type <t_package_declaration> package_declaration;
%type <t_statement_expression_list> statement_expression_list;
%type <t_switch_block> switch_block;
%type <t_switch_block_statement_group> switch_block_statement_group;
%type <t_switch_block_statement_group_list> switch_block_statement_group_list;
%type <t_switch_label> switch_label;
%type <t_switch_label_list> switch_label_list;
%type <t_throws_name_list> throws_name_list;
%type <t_type_declaration_list> type_declaration_list;
%type <t_variable_declarator> variable_declarator;
%type <t_variable_declarator_id> variable_declarator_id;
%type <t_variable_declarator_list> variable_declarator_list;
%type <t_variable_initializer_list> variable_initializer_list;

%start compilation_unit


%%

additive_expression:
	  multiplicative_expression          { $$ = $1; }
	| additive_expression PLUS multiplicative_expression
		{ $$ = new AstAdditiveExpressionPlus($1, $2, $3, PCF); }
	| additive_expression MINUS multiplicative_expression
		{ $$ = new AstAdditiveExpressionMinus($1, $2, $3, PCF); }
	;

and_expression:
	  equality_expression                { $$ = $1; }
	| and_expression AND equality_expression
		{ $$ = new AstAndExpression($1, $2, $3, PCF); }
	;

array_access:
	  name dimension_expression          { $$ = new AstArrayAccessName($1, $2, PCF); }
	| primary_no_new_array dimension_expression
		{ $$ = new AstArrayAccessPrimary($1, $2, PCF); }
	;

array_initializer:
	  LBRACE variable_initializer_list COMMA RBRACE
		{ $$ = new AstArrayInitializerVarsComma($1, $2, $3, $4, PCF); }
	| LBRACE variable_initializer_list RBRACE
		{ $$ = new AstArrayInitializerVars($1, $2, $3, PCF); }
	| LBRACE COMMA RBRACE                { $$ = new AstArrayInitializerComma($1, $2, $3, PCF); }
	| LBRACE RBRACE                      { $$ = new AstArrayInitializerEmpty($1, $2, PCF); }
	;

assignment:
	  name assignment_operator expression
		{ $$ = new AstAssignmentName($1, $2, $3, PCF); }
	| field_access assignment_operator expression
		{ $$ = new AstAssignmentField($1, $2, $3, PCF); }
	| array_access assignment_operator expression
		{ $$ = new AstAssignmentArray($1, $2, $3, PCF); }
	;

assignment_operator:
	  ANDEQ                              { $$ = $1; }
	| DIVEQ                              { $$ = $1; }
	| EQ                                 { $$ = $1; }
	| LSHIFTEQ                           { $$ = $1; }
	| MINUSEQ                            { $$ = $1; }
	| MODEQ                              { $$ = $1; }
	| MULTEQ                             { $$ = $1; }
	| OREQ                               { $$ = $1; }
	| PLUSEQ                             { $$ = $1; }
	| RSHIFTEQ                           { $$ = $1; }
	| URSHIFTEQ                          { $$ = $1; }
	| XOREQ                              { $$ = $1; }
	;

block:
	  LBRACE block_statement_list RBRACE { $$ = new AstBlockWStatements($1, $2, $3, PCF); }
	| LBRACE RBRACE                      { $$ = new AstBlockNoStatements($1, $2, PCF); }
	;

block_statement:
	  class_declaration                    { $$ = $1; }
	| interface_declaration                { $$ = $1; }
	| local_variable_declaration SEMICOLON
		{ $$ = new AstLocalVariableDeclarationStatement($1, $2, PCF); }
	| statement                            { $$ = $1; }
	;

block_statement_list:
	  block_statement                      { $$ = new AstBlockStatementList($1, PCF); }
	| block_statement_list block_statement { $1->add($2); }
	;

catch_clause:
	  CATCH LPAREN formal_parameter RPAREN block
		{ $$ = new AstCatchClause($1, $2, $3, $4, $5, PCF); }
	;

catch_clause_list:
	  catch_clause                       { $$ = new AstCatchClauseList($1, PCF); }
	| catch_clause_list catch_clause     { $1->add($2); }
	;

class_body:
	  LBRACE class_body_declaration_list RBRACE { $$ = new AstClassBodyDecls($1, $2, $3, PCF); }
	| LBRACE RBRACE                      { $$ = new AstClassBodyEmpty($1, $2, PCF); }
	;

class_body_declaration:
	  SEMICOLON                          { $$ = $1; }
	| class_member_declaration           { $$ = $1; }
	| constructor_declarator /* no throws_name_list */ constructor_body
		{ $$ = new AstConstructorDeclarationNoModsNoThrows($1, $2, PCF); }
	| constructor_declarator throws_name_list constructor_body
		{ $$ = new AstConstructorDeclarationNoModsThrows($1, $2, $3, PCF); }
	| modifier_list constructor_declarator /* no throws_name_list */ constructor_body
		{ $$ = new AstConstructorDeclarationModsNoThrows($1, $2, $3, PCF); }
	| modifier_list constructor_declarator throws_name_list constructor_body
		{ $$ = new AstConstructorDeclarationModsThrows($1, $2, $3, $4, PCF); }
	| STATIC block                       { $$ = new AstStaticInitializer($1, $2, PCF); }
	| block                              { $$ = $1; }
	;

class_body_declaration_list:
	  class_body_declaration             { $$ = new AstClassBodyDeclarationList($1, PCF); }
	| class_body_declaration_list class_body_declaration
		{ $1->add($2); }
	;

class_declaration:
	  CLASS IDENTIFIER class_body
		{ $$ = new AstClassDeclarationNoModsBaseless($1, $2, $3, PCF); }
	| CLASS IDENTIFIER implements_interface_list class_body
		{ $$ = new AstClassDeclarationNoModsImplements($1, $2, $3, $4, PCF); }
	| CLASS IDENTIFIER EXTENDS name class_body
		{ $$ = new AstClassDeclarationNoModsExtends($1, $2, $3, $4, $5, PCF); }
	| CLASS IDENTIFIER EXTENDS name implements_interface_list class_body
		{ $$ = new AstClassDeclarationNoModsExtendsImplements($1, $2, $3, $4, $5, $6, PCF); }
	| modifier_list CLASS IDENTIFIER class_body
		{ $$ = new AstClassDeclarationModsBaseless($1, $2, $3, $4, PCF); }
	| modifier_list CLASS IDENTIFIER implements_interface_list class_body
		{ $$ = new AstClassDeclarationModsImplements($1, $2, $3, $4, $5, PCF); }
	| modifier_list CLASS IDENTIFIER EXTENDS name class_body
		{ $$ = new AstClassDeclarationModsExtends($1, $2, $3, $4, $5, $6, PCF); }
	| modifier_list CLASS IDENTIFIER EXTENDS name implements_interface_list class_body
		{ $$ = new AstClassDeclarationModsExtendsImplements($1, $2, $3, $4, $5, $6, $7, PCF); }
	;

class_instance_creation_expression:
	  NEW name LPAREN RPAREN
		{ $$ = new AstClassInstanceCreationExpressionNameNoExprNoBody($1, $2, $3, $4, PCF); }
	| NEW name LPAREN expression_list RPAREN
		{ $$ = new AstClassInstanceCreationExpressionNameExprNoBody($1, $2, $3, $4, $5, PCF); }
	| NEW name LPAREN RPAREN class_body
		{ $$ = new AstClassInstanceCreationExpressionNameNoExprBody($1, $2, $3, $4, $5, PCF); }
	| NEW name LPAREN expression_list RPAREN class_body
		{ $$ = new AstClassInstanceCreationExpressionNameExprBody($1, $2, $3, $4, $5, $6, PCF); }
	| primary DOT NEW IDENTIFIER LPAREN RPAREN
		{ $$ = new AstClassInstanceCreationExpressionPrimaryNoExprNoBody($1, $2, $3, $4, $5, $6, PCF); }
	| primary DOT NEW IDENTIFIER LPAREN expression_list RPAREN
		{ $$ = new AstClassInstanceCreationExpressionPrimaryExprNoBody($1, $2, $3, $4, $5, $6, $7, PCF); }
	| primary DOT NEW IDENTIFIER LPAREN RPAREN class_body
		{ $$ = new AstClassInstanceCreationExpressionPrimaryNoExprBody($1, $2, $3, $4, $5, $6, $7, PCF); }
	| primary DOT NEW IDENTIFIER LPAREN expression_list RPAREN class_body
		{ $$ = new AstClassInstanceCreationExpressionPrimaryExprBody($1, $2, $3, $4, $5, $6, $7, $8, PCF); }
	  /* handle x.new Inner(): */
	| name DOT NEW IDENTIFIER LPAREN RPAREN
		{ $$ = new AstClassInstanceCreationExpressionNameIdNoExprNoBody($1, $2, $3, $4, $5, $6, PCF); }
	| name DOT NEW IDENTIFIER LPAREN expression_list RPAREN
		{ $$ = new AstClassInstanceCreationExpressionNameIdExprNoBody($1, $2, $3, $4, $5, $6, $7, PCF); }
	| name DOT NEW IDENTIFIER LPAREN RPAREN class_body
		{ $$ = new AstClassInstanceCreationExpressionNameIdNoExprBody($1, $2, $3, $4, $5, $6, $7, PCF); }
	| name DOT NEW IDENTIFIER LPAREN expression_list RPAREN class_body
		{ $$ = new AstClassInstanceCreationExpressionNameIdExprBody($1, $2, $3, $4, $5, $6, $7, $8, PCF); }
	;

class_member_declaration:
	  field_declaration                  { $$ = $1; }
	| method_header block                { $$ = new AstMethodDeclarationBlock($1, $2, PCF); }
	| method_header SEMICOLON            { $$ = new AstMethodDeclarationSemicolon($1, $2, PCF); }
	/* repeat the prod for 'class_declaration' here: */
	| CLASS IDENTIFIER class_body
		{ $$ = new AstClassMemberDeclarationNoModsNoExtendsNoImplements($1, $2, $3, PCF); }
	| CLASS IDENTIFIER implements_interface_list class_body
		{ $$ = new AstClassMemberDeclarationNoModsNoExtendsImplements($1, $2, $3, $4, PCF); }
	| CLASS IDENTIFIER EXTENDS name class_body
		{ $$ = new AstClassMemberDeclarationNoModsExtendsNoImplements($1, $2, $3, $4, $5, PCF); }
	| CLASS IDENTIFIER EXTENDS name implements_interface_list class_body
		{ $$ = new AstClassMemberDeclarationNoModsExtendsImplements($1, $2, $3, $4, $5, $6, PCF); }
	| modifier_list CLASS IDENTIFIER class_body
		{ $$ = new AstClassMemberDeclarationModsNoExtendsNoImplements($1, $2, $3, $4, PCF); }
	| modifier_list CLASS IDENTIFIER implements_interface_list class_body
		{ $$ = new AstClassMemberDeclarationModsNoExtendsImplements($1, $2, $3, $4, $5, PCF); }
	| modifier_list CLASS IDENTIFIER EXTENDS name class_body
		{ $$ = new AstClassMemberDeclarationModsExtendsNoImplements($1, $2, $3, $4, $5, $6, PCF); }
	| modifier_list CLASS IDENTIFIER EXTENDS name implements_interface_list class_body
		{ $$ = new AstClassMemberDeclarationModsExtendsImplements($1, $2, $3, $4, $5, $6, $7, PCF); }
	| interface_declaration              { $$ = $1; }
	;

compilation_unit:
	  package_declaration import_declaration_list type_declaration_list
		{ compUnit = $$ = new AstCompilationUnitPkgImpTypes($1, $2, $3, PCF); }
	| package_declaration type_declaration_list
		{ compUnit = $$ = new AstCompilationUnitPkgTypes($1, $2, PCF); }
	| package_declaration import_declaration_list
		{ compUnit = $$ = new AstCompilationUnitPkgImp($1, $2, PCF); }
	|  import_declaration_list type_declaration_list
		{ compUnit = $$ = new AstCompilationUnitImpTypes($1, $2, PCF); }
	| import_declaration_list            { compUnit = $$ = new AstCompilationUnitImp($1, PCF); }
	| type_declaration_list              { compUnit = $$ = new AstCompilationUnitTypes($1, PCF); }
	|                                    { compUnit = $$ = new AstCompilationUnitEmpty(PCF); }
	| error                              { compUnit = NULL; }
	;

conditional_and_expression:
	  inclusive_or_expression            { $$ = $1; }
	| conditional_and_expression ANDAND inclusive_or_expression
		{ $$ = new AstConditionalAndExpression($1, $2, $3, PCF); }
	;

conditional_expression:
	  conditional_or_expression          { $$ = $1; }
	| conditional_or_expression QUESTION expression COLON conditional_expression
		{ $$ = new AstConditionalArithExpression($1, $2, $3, $4, $5, PCF); }
	;

conditional_or_expression:
	  conditional_and_expression         { $$ = $1; }
	| conditional_or_expression OROR conditional_and_expression
		{ $$ = new AstConditionalOrExpression($1, $2, $3, PCF); }
	;

constructor_body:
	  LBRACE explicit_constructor_invocation block_statement_list RBRACE
		{ $$ = new AstConstructorBodyExplicitBlock($1, $2, $3, $4, PCF); }
	| LBRACE explicit_constructor_invocation RBRACE
		{ $$ = new AstConstructorBodyExplicit($1, $2, $3, PCF); }
	| LBRACE block_statement_list RBRACE { $$ = new AstConstructorBodyBlock($1, $2, $3, PCF); }
	| LBRACE RBRACE                      { $$ = new AstConstructorBodyEmpty($1, $2, PCF); }
	;

constructor_declarator:
	  IDENTIFIER LPAREN RPAREN           { $$ = new AstConstructorDeclaratorEmpty($1, $2, $3, PCF); }
	| IDENTIFIER LPAREN formal_parameter_list RPAREN
		{ $$ = new AstConstructorDeclaratorParams($1, $2, $3, $4, PCF); }
	;

dimension:
	  LBRACKET RBRACKET                  { $$ = new AstDimension($1, $2, PCF); }
	;

dimension_expression:
	  LBRACKET expression RBRACKET       { $$ = new AstDimensionExpression($1, $2, $3, PCF); }
	;

dimension_expression_list:
	  dimension_expression               { $$ = new AstDimensionExpressionList($1, PCF); }
	| dimension_expression_list dimension_expression
		{ $1->add($2); }
	;

dimension_list:
	  dimension                          { $$ = new AstDimensionList($1, PCF); }
	| dimension_list dimension           { $1->add($2); }
	;

equality_expression:
	  relational_expression              { $$ = $1; }
	| equality_expression EQEQ relational_expression
		{ $$ = new AstEqualityExpressionEq($1, $2, $3, PCF); }
	| equality_expression NOTEQ relational_expression
		{ $$ = new AstEqualityExpressionNe($1, $2, $3, PCF); }
	;

exclusive_or_expression:
	  and_expression                     { $$ = $1; }
	| exclusive_or_expression XOR and_expression
		{ $$ = new AstXorExpression($1, $2, $3, PCF); }
	;

explicit_constructor_invocation:
	  THIS LPAREN /* no expression_list */ RPAREN SEMICOLON
		{ $$ = new AstExplicitConstructorInvocationThis($1, $2, $3, $4, PCF); }
	| THIS LPAREN expression_list RPAREN SEMICOLON
		{ $$ = new AstExplicitConstructorInvocationThisExpressions($1, $2, $3, $4, $5, PCF); }
	| SUPER LPAREN /* no expression_list */ RPAREN SEMICOLON
		{ $$ = new AstExplicitConstructorInvocationSuper($1, $2, $3, $4, PCF); }
	| SUPER LPAREN expression_list RPAREN SEMICOLON
		{ $$ = new AstExplicitConstructorInvocationSuperExpressions($1, $2, $3, $4, $5, PCF); }
	| primary DOT THIS LPAREN /* no expression_list */ RPAREN SEMICOLON
		{ $$ = new AstExplicitConstructorInvocationPrimaryThis($1, $2, $3, $4, $5, $6, PCF); }
	| primary DOT THIS LPAREN expression_list RPAREN SEMICOLON
		{ $$ = new AstExplicitConstructorInvocationPrimaryThisExpressions($1, $2, $3, $4, $5, $6, $7, PCF); }
	| primary DOT SUPER LPAREN /* no expression_list */ RPAREN SEMICOLON
		{ $$ = new AstExplicitConstructorInvocationPrimarySuper($1, $2, $3, $4, $5, $6, PCF); }
	| primary DOT SUPER LPAREN expression_list RPAREN SEMICOLON
		{ $$ = new AstExplicitConstructorInvocationPrimarySuperExpressions($1, $2, $3, $4, $5, $6, $7, PCF); }
	;

expression:
	  assignment                         { $$ = $1; }
	| conditional_expression             { $$ = $1; }
	;

/* this is always used as an argument list */
expression_list:
	  expression                         { $$ = new AstExpressionList($1, PCF); }
	| expression_list COMMA expression   { $1->add($2); $1->add($3); }
	;

field_access:
	  SUPER DOT IDENTIFIER               { $$ = new AstFieldAccessSuper($1, $2, $3, PCF); }
	| name DOT SUPER DOT IDENTIFIER      { $$ = new AstFieldAccessNameSuper($1, $2, $3, $4, $5, PCF); }
	| primary DOT IDENTIFIER             { $$ = new AstFieldAccessPrimary($1, $2, $3, PCF); }
	;

field_declaration:
	  modifier_list type variable_declarator_list SEMICOLON
		{ $$ = new AstFieldDeclarationMods($1, $2, $3, $4, PCF); }
	| type variable_declarator_list SEMICOLON
		{ $$ = new AstFieldDeclarationNoMods($1, $2, $3, PCF); }
	;

finally_block:
	  FINALLY block                      { $$ = new AstFinallyBlock($1, $2, PCF); }
	;

formal_parameter:
	  type variable_declarator_id        { $$ = new AstFormalParameterNonFinal($1, $2, PCF); }
	| FINAL type variable_declarator_id  { $$ = new AstFormalParameterFinal($1, $2, $3, PCF); }
	;

formal_parameter_list:
	  formal_parameter                   { $$ = new AstFormalParameterList($1, PCF); }
	| formal_parameter_list COMMA formal_parameter
		{ $1->add($2); $1->add($3); }
	;

implements_interface_list:
	  IMPLEMENTS name_list               { $$ = new AstImplementsInterfaceList($1, $2, PCF); }
	;

import_declaration:
	  /* single_type_import_declaration */
	  IMPORT name SEMICOLON              { $$ = new AstImportDeclarationSingle($1, $2, $3, PCF); }
	  /* type_import_on_demand_declaration */
	| IMPORT name DOT MULT SEMICOLON     { $$ = new AstImportDeclarationOnDemand($1, $2, $3, $4, $5, PCF); }
	;

import_declaration_list:
	  import_declaration                 { $$ = new AstImportDeclarationList($1, PCF); }
	| import_declaration_list import_declaration
		{ $1->add($2); }
	;

inclusive_or_expression:
	  exclusive_or_expression            { $$ = $1; }
	| inclusive_or_expression OR exclusive_or_expression
		{ $$ = new AstOrExpression($1, $2, $3, PCF); }
	;

interface_body:
	  LBRACE interface_member_declaration_list RBRACE
		{ $$ = new AstInterfaceBodyDecls($1, $2, $3, PCF); }
	| LBRACE RBRACE                      { $$ = new AstInterfaceBodyNoDecls($1, $2, PCF); }
	;

interface_declaration:
	  INTERFACE IDENTIFIER interface_body
		{ $$ = new AstInterfaceDeclarationNoModsNoExtends($1, $2, $3, PCF); }
	| INTERFACE IDENTIFIER EXTENDS name_list interface_body
		{
                  AstExtendsInterfaceList* eil = new AstExtendsInterfaceList($3, $4, PCF);
		  $$ = new AstInterfaceDeclarationNoModsExtends($1, $2, eil, $5, PCF);
                }
	| modifier_list INTERFACE IDENTIFIER interface_body
		{ $$ = new AstInterfaceDeclarationModsNoExtends($1, $2, $3, $4, PCF); }
	| modifier_list INTERFACE IDENTIFIER EXTENDS name_list interface_body
		{
                  AstExtendsInterfaceList* eil = new AstExtendsInterfaceList($4, $5, PCF);
                  $$ = new AstInterfaceDeclarationModsExtends($1, $2, $3, eil, $6, PCF);
                }
	;

interface_member_declaration:
	  method_header SEMICOLON            { $$ = new AstAbstractMethodDeclaration($1, $2, PCF); }
	| class_declaration                  { $$ = $1; }
	| field_declaration                  { $$ = $1; }
	| interface_declaration              { $$ = $1; } 
	;

interface_member_declaration_list:
	  interface_member_declaration       { $$ = new AstInterfaceMemberDeclarationList($1, PCF); }
	| interface_member_declaration_list interface_member_declaration
		{ $1->add($2); }
	;

local_variable_declaration:
	  type variable_declarator_list      { $$ = new AstLocalVariableDeclarationNonFinal($1, $2, PCF); }
	| FINAL type variable_declarator_list { $$ = new AstLocalVariableDeclarationFinal($1, $2, $3, PCF); }
	;

method_declarator:
	  IDENTIFIER LPAREN RPAREN
		{ $$ = new AstMethodDeclaratorNoArgs($1, $2, $3, PCF); }
	| IDENTIFIER LPAREN formal_parameter_list RPAREN
		{ $$ = new AstMethodDeclaratorWArgs($1, $2, $3, $4, PCF); }
	| method_declarator dimension // deprecated "void foo() []"
		{ $1->add($2); }
	;

method_header:
	  type method_declarator
		{ $$ = new AstMethodHeaderTypeNoModsNoThrows($1, $2, PCF); }
	| type method_declarator throws_name_list
		{ $$ = new AstMethodHeaderTypeNoModsThrows($1, $2, $3, PCF); }
	| VOID method_declarator
		{ $$ = new AstMethodHeaderVoidNoModsNoThrows($1, $2, PCF); }
	| VOID method_declarator throws_name_list
		{ $$ = new AstMethodHeaderVoidNoModsThrows($1, $2, $3, PCF); }
	| modifier_list type method_declarator
		{ $$ = new AstMethodHeaderTypeModsNoThrows($1, $2, $3, PCF); }
	| modifier_list type method_declarator throws_name_list
		{ $$ = new AstMethodHeaderTypeModsThrows($1, $2, $3, $4, PCF); }
	| modifier_list VOID method_declarator
		{ $$ = new AstMethodHeaderVoidModsNoThrows($1, $2, $3, PCF); }
	| modifier_list VOID method_declarator throws_name_list
		{ $$ = new AstMethodHeaderVoidModsThrows($1, $2, $3, $4, PCF); }
	;

method_invocation:
	  name LPAREN RPAREN
		{ $$ = new AstMethodInvocationNameNoArgs($1, $2, $3, PCF); }
	| name LPAREN expression_list RPAREN
		{ $$ = new AstMethodInvocationNameArgs($1, $2, $3, $4, PCF); }
	| primary DOT IDENTIFIER LPAREN RPAREN
		{ $$ = new AstMethodInvocationPrimaryNoArgs($1, $2, $3, $4, $5, PCF); }
	| primary DOT IDENTIFIER LPAREN expression_list RPAREN
		{ $$ = new AstMethodInvocationPrimaryArgs($1, $2, $3, $4, $5, $6, PCF); }
	| SUPER DOT IDENTIFIER LPAREN RPAREN
		{ $$ = new AstMethodInvocationSuperNoArgs($1, $2, $3, $4, $5, PCF); }
	| SUPER DOT IDENTIFIER LPAREN expression_list RPAREN
		{ $$ = new AstMethodInvocationSuperArgs($1, $2, $3, $4, $5, $6, PCF); }
	| name DOT SUPER DOT IDENTIFIER LPAREN RPAREN
		{ $$ = new AstMethodInvocationNameSuperNoArgs($1, $2, $3, $4, $5, $6, $7, PCF); }
	| name DOT SUPER DOT IDENTIFIER LPAREN expression_list RPAREN
		{ $$ = new AstMethodInvocationNameSuperArgs($1, $2, $3, $4, $5, $6, $7, $8, PCF); }
	;

modifier:
	  ABSTRACT                           { $$ = $1; }
	| FINAL                              { $$ = $1; }
	| NATIVE                             { $$ = $1; }
	| PRIVATE                            { $$ = $1; }
	| PROTECTED                          { $$ = $1; }
	| PUBLIC                             { $$ = $1; }
	| STATIC                             { $$ = $1; }
	| STRICTFP                           { $$ = $1; }
	| SYNCHRONIZED                       { $$ = $1; }
	| TRANSIENT                          { $$ = $1; }
	| VOLATILE                           { $$ = $1; }
	;

modifier_list:
	  modifier                           { $$ = new AstModifierList($1, PCF); }
	| modifier_list modifier             { $1->add($2); }
	;

multiplicative_expression:
	  unary_expression                   { $$ = $1; }
	| multiplicative_expression MULT unary_expression
		{ $$ = new AstMultiplicativeExpressionMult($1, $2, $3, PCF); }
	| multiplicative_expression DIV unary_expression
		{ $$ = new AstMultiplicativeExpressionDiv($1, $2, $3, PCF); }
	| multiplicative_expression MOD unary_expression
		{ $$ = new AstMultiplicativeExpressionMod($1, $2, $3, PCF); }
	;

name:
	  IDENTIFIER                         { $$ = new AstName($1, PCF); }
	| name DOT IDENTIFIER                { $1->append($2); $1->append($3); }
	;

name_list:
	  name                               { $$ = new AstNameList($1, PCF); }
	| name_list COMMA name               { $1->add($2); $1->add($3); }
	;

package_declaration:
	  PACKAGE_ name SEMICOLON            { $$ = new AstPackageDeclaration($1, $2, $3, PCF); }
	;

postfix_expression:
	  name                               { $$ = $1; }
	| postdecrement_expression           { $$ = $1; }
	| postincrement_expression           { $$ = $1; }
	| primary                            { $$ = $1; }
	;

postincrement_expression:
	  postfix_expression PLUSPLUS        { $$ = new AstPostincrementExpression($1, $2, PCF); }
	;

postdecrement_expression:
	  postfix_expression MINUSMINUS      { $$ = new AstPostdecrementExpression($1, $2, PCF); }
	;

preincrement_expression:
	  PLUSPLUS unary_expression          { $$ = new AstPreincrementExpression($1, $2, PCF); }
	;

predecrement_expression:
	  MINUSMINUS unary_expression        { $$ = new AstPredecrementExpression($1, $2, PCF); }
	;

primary:
	  NEW primitive_type dimension_expression_list
		{ $$ = new AstArrayCreationExpressionPrimitiveExprNoDims($1, $2, $3, PCF); }
	| NEW primitive_type dimension_expression_list dimension_list
		{ $$ = new AstArrayCreationExpressionPrimitiveExprDims($1, $2, $3, $4, PCF); }
	| NEW name dimension_expression_list
		{ $$ = new AstArrayCreationExpressionNameExprNoDims($1, $2, $3, PCF); }
	| NEW name dimension_expression_list dimension_list
		{ $$ = new AstArrayCreationExpressionNameExprDims($1, $2, $3, $4, PCF); }
	| NEW primitive_type dimension_list array_initializer
		{ $$ = new AstArrayCreationExpressionPrimitiveDimsInit($1, $2, $3, $4, PCF); }
	| NEW name dimension_list array_initializer
		{ $$ = new AstArrayCreationExpressionNameDimsInit($1, $2, $3, $4, PCF); }
	| primary_no_new_array
	;

primary_no_new_array:
	  BOOLEAN_LITERAL                    { $$ = $1; }
	| CHARACTER_LITERAL                  { $$ = $1; }
	| INTEGER_LITERAL                    { $$ = $1; }
	| NULL_LITERAL                       { $$ = $1; }
	| STRING_LITERAL                     { $$ = $1; }
	| FLOATING_POINT_LITERAL             { $$ = $1; }
	| THIS                               { $$ = $1; }
	| LPAREN expression RPAREN           { $$ = new AstPrimaryParenExpression($1, $2, $3, PCF); }
	| class_instance_creation_expression { $$ = $1; }
	| field_access                       { $$ = $1; }
	| method_invocation                  { $$ = $1; }
	| array_access                       { $$ = $1; }
	| primitive_type DOT CLASS           { $$ = new AstPrimitiveDotClass($1, $2, $3, PCF); }
	| VOID DOT CLASS                     { $$ = new AstVoidDotClass($1, $2, $3, PCF); }
	| primitive_type dimension_list DOT CLASS
		{ 
	          AstArrayPrimitiveType* apt = new AstArrayPrimitiveType($1, $2, PCF);
	          $$ = new AstArrayTypeDotClass(apt, $3, $4, PCF);
	        }
	| name dimension_list DOT CLASS
		{ 
	          AstArrayNameType* ant = new AstArrayNameType($1, $2, PCF);
	          $$ = new AstArrayTypeDotClass(ant, $3, $4, PCF);
	        }
	| name DOT CLASS                     { $$ = new AstNameDotClass($1, $2, $3, PCF); }
	| name DOT THIS                      { $$ = new AstNameDotThis($1, $2, $3, PCF); }
	;

primitive_type:
	  BOOLEAN                            { $$ = $1; }
	| BYTE                               { $$ = $1; }
	| CHAR                               { $$ = $1; }
	| DOUBLE                             { $$ = $1; }
	| FLOAT                              { $$ = $1; }
	| INT                                { $$ = $1; }
	| LONG                               { $$ = $1; }
	| SHORT                              { $$ = $1; }
	;

reference_type:
	  primitive_type dimension_list      { $$ = new AstArrayPrimitiveType($1, $2, PCF); }
	| name                               { $$ = $1; }
	| name dimension_list                { $$ = new AstArrayNameType($1, $2, PCF); }
	;

relational_expression:
	  shift_expression                   { $$ = $1; }
	| relational_expression LT shift_expression
		{ $$ = new AstRelationalExpressionLt($1, $2, $3, PCF); }
	| relational_expression GT shift_expression
		{ $$ = new AstRelationalExpressionGt($1, $2, $3, PCF); }
	| relational_expression LTEQ shift_expression
		{ $$ = new AstRelationalExpressionLteq($1, $2, $3, PCF); }
	| relational_expression GTEQ shift_expression
		{ $$ = new AstRelationalExpressionGteq($1, $2, $3, PCF); }
	| relational_expression INSTANCEOF reference_type
		{ $$ = new AstRelationalExpressionInstanceof($1, $2, $3, PCF); }
	;

shift_expression:
	  additive_expression                { $$ = $1; }
	| shift_expression LSHIFT additive_expression
		{ $$ = new AstShiftExpressionLshift($1, $2, $3, PCF); }
	| shift_expression RSHIFT additive_expression
		{ $$ = new AstShiftExpressionRshift($1, $2, $3, PCF); }
	| shift_expression URSHIFT additive_expression
		{ $$ = new AstShiftExpressionUrshift($1, $2, $3, PCF); }
	;

statement:
	  FOR LPAREN SEMICOLON SEMICOLON RPAREN statement
		{ $$ = new AstForStatementNoInitNoConditionNoUpdate($1, $2, $3, $4, $5, $6, PCF); }
	| FOR LPAREN statement_expression_list SEMICOLON SEMICOLON RPAREN statement
		{ $$ = new AstForStatementStmtsNoConditionNoUpdate($1, $2, $3, $4, $5, $6, $7, PCF); }
	| FOR LPAREN local_variable_declaration SEMICOLON SEMICOLON RPAREN statement
		{ $$ = new AstForStatementVarsNoConditionNoUpdate($1, $2, $3, $4, $5, $6, $7, PCF); }
	| FOR LPAREN SEMICOLON SEMICOLON statement_expression_list RPAREN statement
		{ $$ = new AstForStatementNoInitNoConditionUpdate($1, $2, $3, $4, $5, $6, $7, PCF); }
	| FOR LPAREN statement_expression_list SEMICOLON SEMICOLON statement_expression_list RPAREN statement
		{ $$ = new AstForStatementStmtsNoConditionUpdate($1, $2, $3, $4, $5, $6, $7, $8, PCF); }
	| FOR LPAREN local_variable_declaration SEMICOLON SEMICOLON statement_expression_list RPAREN statement
		{ $$ = new AstForStatementVarsNoConditionUpdate($1, $2, $3, $4, $5, $6, $7, $8, PCF); }
	| FOR LPAREN SEMICOLON expression SEMICOLON RPAREN statement
		{ $$ = new AstForStatementNoInitConditionNoUpdate($1, $2, $3, $4, $5, $6, $7, PCF); }
	| FOR LPAREN statement_expression_list SEMICOLON expression SEMICOLON RPAREN statement
		{ $$ = new AstForStatementStmtsConditionNoUpdate($1, $2, $3, $4, $5, $6, $7, $8, PCF); }
	| FOR LPAREN local_variable_declaration SEMICOLON expression SEMICOLON RPAREN statement
		{ $$ = new AstForStatementVarsConditionNoUpdate($1, $2, $3, $4, $5, $6, $7, $8, PCF); }
	| FOR LPAREN SEMICOLON expression SEMICOLON statement_expression_list RPAREN statement
		{ $$ = new AstForStatementNoInitConditionUpdate($1, $2, $3, $4, $5, $6, $7, $8, PCF); }
	| FOR LPAREN statement_expression_list SEMICOLON expression SEMICOLON statement_expression_list RPAREN statement
		{ $$ = new AstForStatementStmtsConditionUpdate($1, $2, $3, $4, $5, $6, $7, $8, $9, PCF); }
	| FOR LPAREN local_variable_declaration SEMICOLON expression SEMICOLON statement_expression_list RPAREN statement
		{ $$ = new AstForStatementVarsConditionUpdate($1, $2, $3, $4, $5, $6, $7, $8, $9, PCF); }
	| IF LPAREN expression RPAREN statement_no_short_if ELSE statement
		{ $$ = new AstIfThenElseStatement($1, $2, $3, $4, $5, $6, $7, PCF); }
	| IF LPAREN expression RPAREN statement
		{ $$ = new AstIfThenStatement($1, $2, $3, $4, $5, PCF); }
	| IDENTIFIER COLON statement         { $$ = new AstLabeledStatement($1, $2, $3, PCF); }
	| statement_without_trailing_substatement
	| WHILE LPAREN expression RPAREN statement
		{ $$ = new AstWhileStatement($1, $2, $3, $4, $5, PCF); }
	;

statement_expression:
	  assignment
	| class_instance_creation_expression      { $$ = $1; }
	| method_invocation                  { $$ = $1; }
	| postdecrement_expression
	| postincrement_expression
	| predecrement_expression
	| preincrement_expression
	;

statement_expression_list:
	  statement_expression               { $$ = new AstStatementExpressionList($1, PCF); }
	| statement_expression_list COMMA statement_expression
		{ $1->add($2); $1->add($3); }
	;

statement_no_short_if:
	  FOR LPAREN SEMICOLON SEMICOLON RPAREN statement_no_short_if
		{ $$ = new AstForStatementNoInitNoConditionNoUpdate($1, $2, $3, $4, $5, $6, PCF); }
	| FOR LPAREN statement_expression_list SEMICOLON SEMICOLON RPAREN statement_no_short_if
		{ $$ = new AstForStatementStmtsNoConditionNoUpdate($1, $2, $3, $4, $5, $6, $7, PCF); }
	| FOR LPAREN local_variable_declaration SEMICOLON SEMICOLON RPAREN statement_no_short_if
		{ $$ = new AstForStatementVarsNoConditionNoUpdate($1, $2, $3, $4, $5, $6, $7, PCF); }
	| FOR LPAREN SEMICOLON SEMICOLON statement_expression_list RPAREN statement_no_short_if
		{ $$ = new AstForStatementNoInitNoConditionUpdate($1, $2, $3, $4, $5, $6, $7, PCF); }
	| FOR LPAREN statement_expression_list SEMICOLON SEMICOLON statement_expression_list RPAREN statement_no_short_if
		{ $$ = new AstForStatementStmtsNoConditionUpdate($1, $2, $3, $4, $5, $6, $7, $8, PCF); }
	| FOR LPAREN local_variable_declaration SEMICOLON SEMICOLON statement_expression_list RPAREN statement_no_short_if
		{ $$ = new AstForStatementVarsNoConditionUpdate($1, $2, $3, $4, $5, $6, $7, $8, PCF); }
	| FOR LPAREN SEMICOLON expression SEMICOLON RPAREN statement_no_short_if
		{ $$ = new AstForStatementNoInitConditionNoUpdate($1, $2, $3, $4, $5, $6, $7, PCF); }
	| FOR LPAREN statement_expression_list SEMICOLON expression SEMICOLON RPAREN statement_no_short_if
		{ $$ = new AstForStatementStmtsConditionNoUpdate($1, $2, $3, $4, $5, $6, $7, $8, PCF); }
	| FOR LPAREN local_variable_declaration SEMICOLON expression SEMICOLON RPAREN statement_no_short_if
		{ $$ = new AstForStatementVarsConditionNoUpdate($1, $2, $3, $4, $5, $6, $7, $8, PCF); }
	| FOR LPAREN SEMICOLON expression SEMICOLON statement_expression_list RPAREN statement_no_short_if
		{ $$ = new AstForStatementNoInitConditionUpdate($1, $2, $3, $4, $5, $6, $7, $8, PCF); }
	| FOR LPAREN statement_expression_list SEMICOLON expression SEMICOLON statement_expression_list RPAREN statement_no_short_if
		{ $$ = new AstForStatementStmtsConditionUpdate($1, $2, $3, $4, $5, $6, $7, $8, $9, PCF); }
	| FOR LPAREN local_variable_declaration SEMICOLON expression SEMICOLON statement_expression_list RPAREN statement_no_short_if
		{ $$ = new AstForStatementVarsConditionUpdate($1, $2, $3, $4, $5, $6, $7, $8, $9, PCF); }
	| IF LPAREN expression RPAREN statement_no_short_if ELSE statement_no_short_if
		{ $$ = new AstIfThenElseStatement($1, $2, $3, $4, $5, $6, $7, PCF); }
	| IDENTIFIER COLON statement_no_short_if
		{ $$ = new AstLabeledStatement($1, $2, $3, PCF); }
	| statement_without_trailing_substatement
	| WHILE LPAREN expression RPAREN statement_no_short_if
		{ $$ = new AstWhileStatement($1, $2, $3, $4, $5, PCF); }
	;

statement_without_trailing_substatement:
	  ASSERT expression SEMICOLON        { $$ = new AstAssertStatementOneExpression($1, $2, $3, PCF); }
 	| ASSERT expression COLON expression SEMICOLON
		{ $$ = new AstAssertStatementTwoExpressions($1, $2, $3, $4, $5, PCF); }
	| block                              { $$ = $1; }
	| BREAK SEMICOLON                    { $$ = new AstBreakStatementNoId($1, $2, PCF); }
	| BREAK IDENTIFIER SEMICOLON         { $$ = new AstBreakStatementId($1, $2, $3, PCF); }
	| CONTINUE SEMICOLON                 { $$ = new AstContinueStatementNoId($1, $2, PCF); }
	| CONTINUE IDENTIFIER SEMICOLON      { $$ = new AstContinueStatementId($1, $2, $3, PCF); }
	| DO statement WHILE LPAREN expression RPAREN SEMICOLON
		{ $$ = new AstDoStatement($1, $2, $3, $4, $5, $6, $7, PCF); }
	| statement_expression SEMICOLON     { $$ = new AstExpressionStatement($1, $2, PCF); }
	| RETURN SEMICOLON                   { $$ = new AstReturnStatementEmpty($1, $2, PCF); }
	| RETURN expression SEMICOLON        { $$ = new AstReturnStatementExpr($1, $2, $3, PCF); }
	| SWITCH LPAREN expression RPAREN switch_block
		{ $$ = new AstSwitchStatement($1, $2, $3, $4, $5, PCF); }
	| SYNCHRONIZED LPAREN expression RPAREN block
		{ $$ = new AstSynchronizedStatement($1, $2, $3, $4, $5, PCF); }
	| THROW expression SEMICOLON         { $$ = new AstThrowStatement($1, $2, $3, PCF); }
	| TRY block catch_clause_list        { $$ = new AstTryStatementCatches($1, $2, $3, PCF); }
	| TRY block catch_clause_list finally_block
		{ $$ = new AstTryStatementCatchesFinally($1, $2, $3, $4, PCF); }
	| TRY block finally_block            { $$ = new AstTryStatementFinally($1, $2, $3, PCF); }
	| SEMICOLON                          { $$ = $1; }
	;

switch_block:
	  LBRACE switch_block_statement_group_list switch_label_list RBRACE
		{ $$ = new AstSwitchBlockGroupsLabels($1, $2, $3, $4, PCF); }
	| LBRACE switch_block_statement_group_list RBRACE
		{ $$ = new AstSwitchBlockGroups($1, $2, $3, PCF); }
	| LBRACE switch_label_list RBRACE
		{ $$ = new AstSwitchBlockLabels($1, $2, $3, PCF); }
	| LBRACE RBRACE
		{ $$ = new AstSwitchBlockEmpty($1, $2, PCF); }
	;

switch_block_statement_group:
	  switch_label_list block_statement_list
		{ $$ = new AstSwitchBlockStatementGroup($1, $2, PCF); }
	;

switch_block_statement_group_list:
	  switch_block_statement_group  { $$ = new AstSwitchBlockStatementGroupList($1, PCF); }
	| switch_block_statement_group_list switch_block_statement_group
		{ $1->add($2); }
	;

switch_label:
	  CASE expression COLON              { $$ = new AstSwitchLabelCase($1, $2, $3, PCF); }
	| DEFAULT COLON                      { $$ = new AstSwitchLabelDefault($1, $2, PCF); }
	;

switch_label_list:
	  switch_label                       { $$ = new AstSwitchLabelList($1, PCF); }
	| switch_label_list switch_label     { $1->add($2); }
	;

throws_name_list:
	  THROWS name_list                   { $$ = new AstThrowsNameList($1, $2, PCF); }
	;

type:
	  primitive_type                     { $$ = $1; }
	| reference_type 
	;

type_declaration:
	  class_declaration                  { $$ = $1; }
	| interface_declaration              { $$ = $1; }
	| SEMICOLON                          { $$ = $1; }
	;

type_declaration_list:
	  type_declaration                 { $$ = new AstTypeDeclarationList($1, PCF); }
	| type_declaration_list type_declaration
		{ $1->add($2); }
	;

unary_expression:
	  MINUS unary_expression             { $$ = new AstUnaryExpressionMinus($1, $2, PCF); }
	| PLUS unary_expression              { $$ = new AstUnaryExpressionPlus($1, $2, PCF); }
	| predecrement_expression            { $$ = $1; }
	| preincrement_expression            { $$ = $1; }
	| unary_expression_not_plus_minus    { $$ = $1; }
	;

unary_expression_not_plus_minus:
	  COMP unary_expression              { $$ = new AstUnaryExpressionComp($1, $2, PCF); }
	| NOT unary_expression               { $$ = new AstUnaryExpressionNot($1, $2, PCF); }
	| LPAREN primitive_type RPAREN unary_expression
		{ $$ = new AstCastExpressionPrimitiveNoDims($1, $2, $3, $4, PCF); }
	| LPAREN primitive_type dimension_list RPAREN unary_expression
		{ $$ = new AstCastExpressionPrimitiveDims($1, $2, $3, $4, $5, PCF); }
	| LPAREN expression RPAREN unary_expression_not_plus_minus
		{ $$ = new AstCastExpressionExpr($1, $2, $3, $4, PCF); }
	| LPAREN name dimension_list RPAREN unary_expression_not_plus_minus
		{ $$ = new AstCastExpressionNameDims($1, $2, $3, $4, $5, PCF); }
	| postfix_expression                 { $$ = $1; }
	;

variable_declarator:
	  variable_declarator_id             { $$ = new AstVariableDeclaratorNoAssign($1, PCF); }
	| variable_declarator_id EQ variable_initializer
		{ $$ = new AstVariableDeclaratorAssign($1, $2, $3, PCF); }
	;

variable_declarator_id:
	  IDENTIFIER                         { $$ = new AstVariableDeclaratorIdNoDimensions($1, PCF); }
	| IDENTIFIER dimension_list          { $$ = new AstVariableDeclaratorIdDimensions($1, $2, PCF); }
	;

variable_declarator_list:
	  variable_declarator                { $$ = new AstVariableDeclaratorList($1, PCF); }
	| variable_declarator_list COMMA variable_declarator
		{ $1->add($2); $1->add($3); }
	;

variable_initializer:
	  array_initializer                  { $$ = $1; }
	| expression                         { $$ = $1; }
	;

variable_initializer_list:
	  variable_initializer               { $$ = new AstVariableInitializerList($1, PCF); }
	| variable_initializer_list COMMA variable_initializer
		{ $1->add($2); $1->add($3); }
	;

%%
