#ifndef Test_h
#include "Test.h"
#endif

#ifndef Java_h
#include "Java.h"
#endif

#ifndef Log_h
#include "Log.h"
#endif

#ifndef std_fstream
#define std_fstream
#include <fstream>
#endif

using namespace doctorj;

static void spew(const string& name, AstItem* const item, int indent = 0)
{
    while (--indent >= 0) {
        cout << "    ";
    }
    cout << name << " = " << item->type() << " (" << item->text() << ")" << endl;
}

class JavaSymbolTest : public Test 
{
public:
    JavaSymbolTest();
    virtual void run();
    virtual void testLocalVariable();
    virtual void testCatchClause();
    virtual void testForStatement();
    virtual void testConstructorParameter();
    virtual void testMethodParameter();
    virtual void testFieldSameClass();
    virtual void testFieldSameInterface();
    virtual void testFieldSuperClass();
    virtual void testFieldSuperInterface();
    virtual void testFieldOtherClass();
    virtual void testFieldOtherInterface();
};

JavaSymbolTest::JavaSymbolTest() : Test(true)
{
    Log::setEnabled();
}

void JavaSymbolTest::testLocalVariable()
{
    // write out a file
    string cname("JSymTestLocalVariable");
    string fname("/tmp/" + cname + ".java");
    ofstream of(fname.c_str());
    of << "public class " << cname << " {" << endl;
    of << "    public " << cname << "(int a) {" << endl;
    of << "        int i = 4;" << endl;
    of << "        i += 3;" << endl;
    of << "    }" << endl;
    of << "}" << endl;
    of.close();

    vector<char*> unused;
    unused.push_back(const_cast<char*>(fname.c_str()));
    AstProject project(1.3, unused);

    AstCompilationUnit* cu = project.getCompilationUnit(0);
    AstTypeDeclarationList* types = cu->getTypeDeclarationList();

    AstItem* type = types->getTypeDeclaration(0);
    spew("type", type);

    AstClassDeclaration* cd = dynamic_cast<AstClassDeclaration*>(type);
    spew("class declaration", cd);

    AstClassBody* classbody = cd->getClassBody();
    spew("class body", classbody);

    AstClassBodyDeclarationList* contents = classbody->getClassBodyDeclarationList();
    spew("class contents", contents);

    AstItem* decl = contents->getClassBodyDeclaration(0);
    spew("decl", decl);

    AstConstructorDeclaration* ctor = dynamic_cast<AstConstructorDeclaration*>(decl);
    spew("ctor", ctor, 1);

    AstConstructorBody* ctorbody = ctor->getConstructorBody();
    spew("ctor body", ctorbody, 2);

    AstBlockStatementList* statements = ctorbody->getBlockStatementList();
    spew("ctor statements", statements, 3);

    AstItem* stitem = statements->getBlockStatement(0);

    AstLocalVariableDeclarationStatement* lvdstmt = dynamic_cast<AstLocalVariableDeclarationStatement*>(stitem);
    spew("local var decl stmt", lvdstmt, 4);

    AstLocalVariableDeclaration* lvardec = lvdstmt->getLocalVariableDeclaration();
    spew("local var decl", lvardec, 5);
    
    AstVariableDeclaratorList* varlist = lvardec->getVariableDeclaratorList();

    AstVariableDeclarator* vardec = varlist->getVariableDeclarator(0);

    AstExpressionStatement* exprst = dynamic_cast<AstExpressionStatement*>(statements->getBlockStatement(1));
    spew("expr st", exprst, 6);

    AstItem* expr = exprst->getExpression();
    spew("expr", expr, 7);
    
    AstAssignmentName* assname = dynamic_cast<AstAssignmentName*>(expr);
    spew("ass name", assname, 8);

    AstName* name = assname->getName();
    spew("name", name, 9);
    
    AstItem* declaration = JavaSymbol::findVariableDeclaration(name);

    spew("declaration", declaration, 1);

    TEST(declaration != NULL, "declaration is not NULL");
    TESTEQ(declaration, lvardec, "declaration is the variable declaration");
}

void JavaSymbolTest::testCatchClause()
{
    // write out a file
    string cname("JSymCatchClause");
    string fname("/tmp/" + cname + ".java");
    ofstream of(fname.c_str());
    of << "public class " << cname << " {" << endl;
    of << "    void f() {" << endl;
    of << "        try {" << endl;
    of << "            int i = 3;" << endl;
    of << "        }" << endl;
    of << "        catch (NullPointerException n) {" << endl;
    of << "            n = null;" << endl;
    of << "        }" << endl;
    of << "    }" << endl;
    of << "}" << endl;
    of.close();

    vector<char*> unused;
    unused.push_back(const_cast<char*>(fname.c_str()));
    AstProject project(1.3, unused);

    AstCompilationUnit* cu = project.getCompilationUnit(0);
    AstTypeDeclarationList* types = cu->getTypeDeclarationList();

    AstItem* type = types->getTypeDeclaration(0);
    spew("type", type);

    AstClassDeclaration* cd = dynamic_cast<AstClassDeclaration*>(type);
    spew("class declaration", cd);

    AstClassBody* classbody = cd->getClassBody();
    spew("class body", classbody);

    AstClassBodyDeclarationList* cbdl = classbody->getClassBodyDeclarationList();
    spew("class declaration list", cbdl);

    AstItem* fitem = cbdl->getClassBodyDeclaration(0);
    spew("fitem", fitem);

    AstMethodDeclarationBlock* method = dynamic_cast<AstMethodDeclarationBlock*>(fitem);
    spew("method", method);

    AstBlock* block = method->getBlock();
    spew("block", block);
    
    AstBlockStatementList* stmts = block->getBlockStatementList();
    spew("stmts", stmts);

    AstItem* stmt = stmts->getBlockStatement(0);
    spew("stmt", stmt);
    
    AstTryStatement* trystmt = dynamic_cast<AstTryStatement*>(stmt);
    spew("try stmt", trystmt);

    AstCatchClauseList* cclist = trystmt->getCatchClauseList();
    
    AstCatchClause* cc = cclist->getCatchClause(0);

    AstFormalParameter* ccparam = cc->getFormalParameter();
    
    AstItem* bitem = cc->getBlock()->getBlockStatementList()->getBlockStatement(0);
    spew("block item", bitem);

    AstExpressionStatement* exprst = dynamic_cast<AstExpressionStatement*>(bitem);
    spew("expr st", exprst, 1);

    AstItem* expr = exprst->getExpression();
    spew("expr", expr, 2);

    AstAssignmentName* assname = dynamic_cast<AstAssignmentName*>(expr);
    spew("ass name", assname, 3);

    AstName* name = assname->getName();
    spew("name", name, 4);

    AstItem* declaration = JavaSymbol::findVariableDeclaration(name);
    
    spew("declaration", declaration, 1);

    TEST(declaration != NULL, "declaration is not NULL");
    TESTEQ(declaration, ccparam, "declaration is the catch clause parameter");
}

void JavaSymbolTest::testForStatement()
{
    // write out a file
    string cname("JSymTestForStatement");
    string fname("/tmp/" + cname + ".java");
    ofstream of(fname.c_str());
    of << "public class " << cname << " {" << endl;
    of << "    void f() {" << endl;
    of << "        for (int i = 0; i < 34; ++i) {" << endl;
    of << "            i += 4;" << endl;
    of << "        }" << endl;
    of << "    }" << endl;
    of << "}" << endl;
    of.close();

    vector<char*> unused;
    unused.push_back(const_cast<char*>(fname.c_str()));
    AstProject project(1.3, unused);

    AstCompilationUnit* cu = project.getCompilationUnit(0);
    AstTypeDeclarationList* types = cu->getTypeDeclarationList();

    AstItem* type = types->getTypeDeclaration(0);
    spew("type", type);

    AstClassDeclaration* cd = dynamic_cast<AstClassDeclaration*>(type);
    spew("class declaration", cd);

    AstClassBody* classbody = cd->getClassBody();
    spew("class body", classbody);

    AstClassBodyDeclarationList* cbdl = classbody->getClassBodyDeclarationList();
    spew("class declaration list", cbdl);

    AstItem* fitem = cbdl->getClassBodyDeclaration(0);
    spew("fitem", fitem);

    AstMethodDeclarationBlock* method = dynamic_cast<AstMethodDeclarationBlock*>(fitem);
    spew("method", method);

    AstBlock* block = method->getBlock();
    spew("block", block);

    AstBlockStatementList* stmts = block->getBlockStatementList();
    spew("stmts", stmts);

    AstItem* stmt = stmts->getBlockStatement(0);
    spew("stmt", stmt);
    
    AstForStatement* forstmt = dynamic_cast<AstForStatement*>(stmt);
    spew("for stmt", forstmt);

    AstForStatementVarsConditionUpdate* fsvcu = dynamic_cast<AstForStatementVarsConditionUpdate*>(forstmt);
    spew("fsvcu", fsvcu);

    AstLocalVariableDeclaration* lvardec = fsvcu->getLocalVariableDeclaration();
    spew("lvardec", lvardec, 1);

    AstItem* vartype = lvardec->getType();
    spew("var type", vartype, 2);
    
    AstVariableDeclaratorList* vars = lvardec->getVariableDeclaratorList();
    spew("vars", vars, 3);

    AstVariableDeclarator* vardec = vars->getVariableDeclarator(0);
    spew("vardec", vardec, 4);

    AstVariableDeclaratorId* decid = vardec->getVariableDeclaratorId();
    spew("dec id", decid, 5);

    AstIdentifier* id = decid->getIdentifier();
    spew("id", id, 6);

    AstItem* fbitem = forstmt->getStatement();
    spew("fb item", fbitem);
    
    AstBlockWStatements* forblock = dynamic_cast<AstBlockWStatements*>(fbitem);
    spew("for block", forblock);

    AstBlockStatementList* fstmtlist = forblock->getBlockStatementList();
    spew("for statement list", fstmtlist, 1);

    AstItem* fstitem = fstmtlist->getBlockStatement(0);
    spew("statement item", fstitem, 2);

    AstItem* fstmt = fstmtlist->getBlockStatement(0);
    spew("statement", fstmt, 2);

    AstExpressionStatement* exprst = dynamic_cast<AstExpressionStatement*>(fstmt);
    spew("expr statement", exprst, 3);

    AstItem* exitem = exprst->getExpression();
    spew("expression item", exitem, 4);

    AstAssignmentName* assname = dynamic_cast<AstAssignmentName*>(exitem);
    spew("assignment name", assname, 5);

    AstName* name = assname->getName();
    spew("name", name, 6);

    AstItem* declaration = JavaSymbol::findVariableDeclaration(name);

    spew("declaration", declaration, 1);

    TEST(declaration != NULL, "declaration is not NULL");
    TESTEQ(declaration, lvardec, "declaration is the variable declaration");
}

void JavaSymbolTest::testConstructorParameter()
{
    // write out a file
    string cname("JSymTestConstructorParameter");
    string fname("/tmp/" + cname + ".java");
    ofstream of(fname.c_str());
    of << "public class " << cname << " {" << endl;
    of << "    " << cname << "(int i1, int i) {" << endl;
    of << "        i += 4;" << endl;
    of << "    }" << endl;
    of << "}" << endl;
    of.close();

    vector<char*> unused;
    unused.push_back(const_cast<char*>(fname.c_str()));
    AstProject project(1.3, unused);

    AstCompilationUnit* cu = project.getCompilationUnit(0);
    AstTypeDeclarationList* types = cu->getTypeDeclarationList();

    AstItem* type = types->getTypeDeclaration(0);
    spew("type", type);

    AstClassDeclaration* cd = dynamic_cast<AstClassDeclaration*>(type);
    spew("class declaration", cd);

    AstClassBody* classbody = cd->getClassBody();
    spew("class body", classbody);

    AstClassBodyDeclarationList* cbdl = classbody->getClassBodyDeclarationList();
    spew("class declaration list", cbdl);

    AstItem* ctoritem = cbdl->getClassBodyDeclaration(0);
    spew("ctor item", ctoritem);

    AstConstructorDeclaration* ctor = dynamic_cast<AstConstructorDeclaration*>(ctoritem);
    spew("ctor", ctor, 1);

    AstConstructorDeclarator* ctordtor = ctor->getConstructorDeclarator();
    spew("ctor decltor", ctordtor, 2);

    AstFormalParameterList* params = ctordtor->getFormalParameterList();
    spew("formal param list", params, 3);

    AstFormalParameter* param = params->getFormalParameter(1);
    spew("formal param", param);

    AstConstructorBody* ctorbody = ctor->getConstructorBody();
    spew("ctor body", ctorbody, 2);

    AstBlockStatementList* statements = ctorbody->getBlockStatementList();
    spew("ctor statements", statements, 3);

    AstItem* stitem = statements->getBlockStatement(0);
    spew("statement item #0", stitem, 4);

    AstExpressionStatement* exprst = dynamic_cast<AstExpressionStatement*>(stitem);
    spew("expression statement", exprst, 5);

    AstItem* expr = exprst->getExpression();
    spew("expr item", expr, 6);
    
    AstAssignmentName* assname = dynamic_cast<AstAssignmentName*>(expr);
    spew("ass name", assname, 7);
    
    AstName* name = assname->getName();
    spew("name", name, 8);
    
    AstItem* declaration = JavaSymbol::findVariableDeclaration(name);

    spew("declaration", declaration, 1);

    TEST(declaration != NULL, "declaration is not NULL");
    TESTEQ(declaration, param, "declaration is the ctor parameter");
}

void JavaSymbolTest::testMethodParameter()
{
    // write out a file
    string cname("JSymTestMethodParameter");
    string fname("/tmp/" + cname + ".java");
    ofstream of(fname.c_str());
    of << "public class " << cname << " {" << endl;
    of << "    void f(int i1, int i) {" << endl;
    of << "        i += 4;" << endl;
    of << "    }" << endl;
    of << "}" << endl;
    of.close();

    vector<char*> unused;
    unused.push_back(const_cast<char*>(fname.c_str()));
    AstProject project(1.3, unused);

    AstCompilationUnit* cu = project.getCompilationUnit(0);
    AstTypeDeclarationList* types = cu->getTypeDeclarationList();

    AstItem* type = types->getTypeDeclaration(0);
    spew("type", type);

    AstClassDeclaration* cd = dynamic_cast<AstClassDeclaration*>(type);
    spew("class declaration", cd);

    AstClassBody* classbody = cd->getClassBody();
    spew("class body", classbody);

    AstClassBodyDeclarationList* cbdl = classbody->getClassBodyDeclarationList();
    spew("class declaration list", cbdl);

    AstItem* mitem = cbdl->getClassBodyDeclaration(0);
    spew("method item", mitem);

    AstMethodDeclaration* method = dynamic_cast<AstMethodDeclaration*>(mitem);
    spew("method", method, 1);

    AstMethodHeader* header = method->getMethodHeader();
    spew("method header", header, 2);

    AstMethodDeclarator* methoddtor = header->getMethodDeclarator();
    spew("method decltor", methoddtor, 3);

    AstFormalParameterList* params = methoddtor->getFormalParameterList();
    spew("formal param list", params, 4);

    AstFormalParameter* param = params->getFormalParameter(1);
    spew("formal param", param);

    AstMethodDeclarationBlock* mblock = dynamic_cast<AstMethodDeclarationBlock*>(method);
    spew("method block", mblock);

    AstBlock* block = mblock->getBlock();
    spew("block", block);

    AstBlockStatementList* stmts = block->getBlockStatementList();
    spew("stmts", stmts, 1);

    AstItem* stmt = stmts->getBlockStatement(0);
    spew("stmt", stmt, 2);
    
    AstExpressionStatement* exprst = dynamic_cast<AstExpressionStatement*>(stmt);
    spew("expr statement", exprst, 3);

    AstItem* exitem = exprst->getExpression();
    spew("expression item", exitem, 4);

    AstAssignmentName* assname = dynamic_cast<AstAssignmentName*>(exitem);
    spew("assignment name", assname, 5);

    AstName* name = assname->getName();
    spew("name", name, 6);

    AstItem* declaration = JavaSymbol::findVariableDeclaration(name);

    spew("declaration", declaration, 1);

    TEST(declaration != NULL, "declaration is not NULL");
    TESTEQ(declaration, param, "declaration is a method parameter");
}

void JavaSymbolTest::testFieldSameClass()
{
    // write out a file
    string cname("JSymTestFieldSameClass");
    string fname("/tmp/" + cname + ".java");
    ofstream of(fname.c_str());
    of << "public class " << cname << " {" << endl;
    of << "    int i;" << endl;
    of << "    public " << cname << "() {" << endl;
    of << "        i = 3;" << endl;
    of << "    }" << endl;
    of << "}" << endl;
    of.close();

    vector<char*> unused;
    unused.push_back(const_cast<char*>(fname.c_str()));
    AstProject project(1.3, unused);

    AstCompilationUnit* cu = project.getCompilationUnit(0);
    AstTypeDeclarationList* types = cu->getTypeDeclarationList();

    AstItem* type = types->getTypeDeclaration(0);
    spew("type", type);

    AstClassDeclaration* cd = dynamic_cast<AstClassDeclaration*>(type);
    spew("class declaration", cd);

    AstClassBody* classbody = cd->getClassBody();
    spew("class body", classbody);

    AstClassBodyDeclarationList* cbdl = classbody->getClassBodyDeclarationList();
    spew("class declaration list", cbdl);

    AstItem* flditem = cbdl->getClassBodyDeclaration(0);
    spew("field item", flditem);

    AstFieldDeclaration* fielddec = dynamic_cast<AstFieldDeclaration*>(flditem);
    spew("field decl", fielddec, 1);

    AstVariableDeclaratorList* fldvars = fielddec->getVariableDeclaratorList();
    spew("field var list", fldvars, 2);

    AstVariableDeclarator* field = fldvars->getVariableDeclarator(0);
    spew("field", field, 3);

    AstItem* ctoritem = cbdl->getClassBodyDeclaration(1);
    spew("ctor item", ctoritem);

    AstConstructorDeclaration* ctor = dynamic_cast<AstConstructorDeclaration*>(ctoritem);
    spew("ctor", ctor, 1);

    AstConstructorBody* ctorbody = ctor->getConstructorBody();
    spew("ctor body", ctorbody, 2);

    AstBlockStatementList* statements = ctorbody->getBlockStatementList();
    spew("ctor statements", statements, 2);

    AstItem* stitem = statements->getBlockStatement(0);
    spew("statement item #0", stitem, 3);

    AstExpressionStatement* exprst = dynamic_cast<AstExpressionStatement*>(stitem);
    spew("expression statement", exprst, 4);

    AstItem* expr = exprst->getExpression();
    spew("expr item", expr, 5);
    
    AstAssignmentName* assname = dynamic_cast<AstAssignmentName*>(expr);
    spew("ass name", assname, 6);
    
    AstName* name = assname->getName();
    spew("name", name, 7);
    
    AstItem* declaration = JavaSymbol::findVariableDeclaration(name);

    spew("declaration", declaration, 1);    

    TEST(declaration != NULL, "declaration is not NULL");
    TESTEQ(declaration, fielddec, "declaration is the class field");
}

void JavaSymbolTest::testFieldSameInterface()
{
    // write out a file
    string cname("JSymTestFieldSameInterface");
    string fname("/tmp/" + cname + ".java");
    ofstream of(fname.c_str());
    of << "public interface " << cname << " {" << endl;
    of << "    static final int i = 4;" << endl;
    of << "    static final int j = i * i;" << endl;
    of << "}" << endl;
    of.close();

    vector<char*> unused;
    unused.push_back(const_cast<char*>(fname.c_str()));
    AstProject project(1.3, unused);

    AstCompilationUnit* cu = project.getCompilationUnit(0);
    AstTypeDeclarationList* types = cu->getTypeDeclarationList();

    AstItem* type = types->getTypeDeclaration(0);
    spew("type", type);

    AstInterfaceDeclaration* id = dynamic_cast<AstInterfaceDeclaration*>(type);
    spew("interface declaration", id);

    AstInterfaceBody* ibody = id->getInterfaceBody();
    spew("interface body", ibody);

    AstInterfaceMemberDeclarationList* imdl = ibody->getInterfaceMemberDeclarationList();
    spew("interface member declaration list", imdl);

    AstItem* iflditem = imdl->getInterfaceMemberDeclaration(0);
    spew("i field item", iflditem);
  
    AstFieldDeclaration* ifielddec = dynamic_cast<AstFieldDeclaration*>(iflditem);
    spew("i field decl", ifielddec, 1);

    AstVariableDeclaratorList* ifldvars = ifielddec->getVariableDeclaratorList();
    spew("i field var list", ifldvars, 2);

    AstVariableDeclarator* ifield = ifldvars->getVariableDeclarator(0);
    spew("i field", ifield, 3);

    AstVariableDeclaratorId* ivdid = ifield->getVariableDeclaratorId();
    spew("i var dec id", ivdid, 4);

    AstItem* jflditem = imdl->getInterfaceMemberDeclaration(1);
    spew("j field item", jflditem);
  
    AstFieldDeclaration* jfielddec = dynamic_cast<AstFieldDeclaration*>(jflditem);
    spew("j field decl", jfielddec, 1);

    AstVariableDeclaratorList* jfldvars = jfielddec->getVariableDeclaratorList();
    spew("j field var list", jfldvars, 2);

    AstVariableDeclarator* jfield = jfldvars->getVariableDeclarator(0);
    spew("j field", jfield, 3);

    AstItem* jvarinit = jfield->getVariableInitializer();
    spew("j var init", jvarinit, 4);

    AstMultiplicativeExpression* jmex = dynamic_cast<AstMultiplicativeExpression*>(jvarinit);
    spew("j mult expression", jmex, 5);

    AstItem* ivar = jmex->getLhsExpression();
    spew("ivar", ivar, 6);

    AstName* i = dynamic_cast<AstName*>(ivar);
    spew("i", i, 7);

    AstItem* declaration = JavaSymbol::findVariableDeclaration(i);

    spew("declaration", declaration, 1);
    
    TEST(declaration != NULL, "declaration is not NULL");
    TESTEQ(declaration, ifielddec, "declaration is the interface field declaration");
}

void JavaSymbolTest::testFieldSuperClass()
{
}

void JavaSymbolTest::testFieldSuperInterface()
{
}

void JavaSymbolTest::testFieldOtherClass()
{
}

void JavaSymbolTest::testFieldOtherInterface()
{
}

void JavaSymbolTest::run() 
{
    testLocalVariable();
    testCatchClause();
    testForStatement();
    testConstructorParameter();
    testMethodParameter();
    testFieldSameClass();
    testFieldSameInterface();
    testFieldSuperClass();
    testFieldSuperInterface();
    testFieldOtherClass();
    testFieldOtherInterface();
}

int main(int argc, char** argv) 
{
    JavaSymbolTest t;
    t.run();
    cout << "symbol lookup test results: " << t << endl;
    return t.nerrors();
}
