#ifndef StatisticsReporter_h
#include "StatisticsReporter.h"
#endif

#ifndef ProjectStatistics_h
#include "ProjectStatistics.h"
#endif

#ifndef CompilationUnitStatistics_h
#include "CompilationUnitStatistics.h"
#endif

#ifndef InterfaceStatistics_h
#include "InterfaceStatistics.h"
#endif

#ifndef ConcreteClassStatistics_h
#include "ConcreteClassStatistics.h"
#endif

#ifndef AbstractClassStatistics_h
#include "AbstractClassStatistics.h"
#endif

#ifndef std_vector
#define std_vector
#include <vector>
#endif

using namespace std;
using namespace doctorj;


StatisticsReporter::StatisticsReporter(ostream& os) : Writer(os)
{
}

StatisticsReporter::~StatisticsReporter()
{
}

void StatisticsReporter::count(const InnerTypeStatistics* const cts,
                               int* const nConcretes,
                               int* const nAbstracts,
                               int* const nInterfaces,
                               vector<int>* const typeLines,
                               int* const nStaticMethods,
                               int* const nInstanceMethods,
                               vector<int>* const methodLines,
                               int* const nStaticFields,
                               int* const nInstanceFields)
{
    if (dynamic_cast<const ConcreteClassStatistics*>(cts)) {
        ++(*nConcretes);
    }
    else if (dynamic_cast<const InterfaceStatistics*>(cts)) {
        ++(*nInterfaces);
    }
    else if (dynamic_cast<const AbstractClassStatistics*>(cts)) {
        ++(*nAbstracts);
    }
    else {
        cerr << "contained type not understood: " << cts->name() << endl;
    }

    vector<int> methods = cts->getMethods();
    methodLines->insert(methodLines->end(), methods.begin(), methods.end());

    *nStaticMethods   += cts->getStaticMethodsCount();
    *nInstanceMethods += cts->getInstanceMethodsCount();

    *nStaticFields   += cts->getStaticFieldsCount();
    *nInstanceFields += cts->getInstanceFieldsCount();

    typeLines->push_back(cts->getLineCount());

    const vector<InnerTypeStatistics*>&          ctss  = cts->innerTypes();
    vector<InnerTypeStatistics*>::const_iterator cit   = ctss.begin();
    vector<InnerTypeStatistics*>::const_iterator cstop = ctss.end();
    while (cit != cstop) {
        const InnerTypeStatistics* const c = *cit;

        count(c,
              nConcretes,
              nAbstracts,
              nInterfaces,
              typeLines,
              nStaticMethods,
              nInstanceMethods,
              methodLines,
              nStaticFields,
              nInstanceFields);

        ++cit;
    }    
}

void StatisticsReporter::count(const CompilationUnitStatistics* const cus,
                               int* const nConcretes,
                               int* const nAbstracts,
                               int* const nInterfaces,
                               vector<int>* const typeLines,
                               int* const nStaticMethods,
                               int* const nInstanceMethods,
                               vector<int>* const methodLines,
                               int* const nStaticFields,
                               int* const nInstanceFields)
{
    const vector<InnerTypeStatistics*>&          ctss  = cus->types();
    vector<InnerTypeStatistics*>::const_iterator cit   = ctss.begin();
    vector<InnerTypeStatistics*>::const_iterator cstop = ctss.end();
    while (cit != cstop) {
        const InnerTypeStatistics* const cts = *cit;

        count(cts,
              nConcretes,
              nAbstracts,
              nInterfaces,
              typeLines,
              nStaticMethods,
              nInstanceMethods,
              methodLines,
              nStaticFields,
              nInstanceFields);

        ++cit;
    }
}

void StatisticsReporter::count(const ProjectStatistics* const ps,
                               int* const nConcretes,
                               int* const nAbstracts,
                               int* const nInterfaces,
                               vector<int>* const typeLines,
                               int* const nStaticMethods,
                               int* const nInstanceMethods,
                               vector<int>* const methodLines,
                               int* const nStaticFields,
                               int* const nInstanceFields)
{
    const vector<CompilationUnitStatistics*>&          cus   = ps->getCompilationUnitsStatistics();
    vector<CompilationUnitStatistics*>::const_iterator cit   = cus.begin();
    vector<CompilationUnitStatistics*>::const_iterator cstop = cus.end();
    while (cit != cstop) {
        const CompilationUnitStatistics* const cus = *cit;

        count(cus,
              nConcretes,
              nAbstracts,
              nInterfaces,
              typeLines,
              nStaticMethods,
              nInstanceMethods,
              methodLines,
              nStaticFields,
              nInstanceFields);

        ++cit;
    }
}

void StatisticsReporter::writeSummary(ProjectStatistics* const ps)
{
    const vector<CompilationUnitStatistics*>& cus = ps->getCompilationUnitsStatistics();
    vector<CompilationUnitStatistics*>::const_iterator cit   = cus.begin();
    vector<CompilationUnitStatistics*>::const_iterator cstop = cus.end();
    int nLines = 0;
    while (cit != cstop) {
        CompilationUnitStatistics* c = *cit;
        nLines += c->getLineCount();
        ++cit;
    }

    int nConcretes = 0;
    int nAbstracts = 0;
    int nInterfaces = 0;
    vector<int> typeLines;

    int nStaticMethods = 0;
    int nInstanceMethods = 0;
    vector<int> methodLines;

    int nStaticFields = 0;
    int nInstanceFields = 0;

    count(ps, 
          &nConcretes,
          &nAbstracts,
          &nInterfaces,
          &typeLines,
          &nStaticMethods,
          &nInstanceMethods,
          &methodLines,
          &nStaticFields,
          &nInstanceFields);

    outputSummary(nLines, 
                  cus.size(),
                  nConcretes,
                  nAbstracts,
                  nInterfaces,
                  typeLines,
                  nStaticMethods,
                  nInstanceMethods,
                  methodLines,
                  nStaticFields,
                  nInstanceFields);
}

void StatisticsReporter::writeInnerType(InnerTypeStatistics* const cts)
{
    string type = dynamic_cast<InterfaceStatistics*>(cts) ? "interface" : "class";

    outputInnerType(type, 
                    cts->name(),
                    cts->innerTypes(), 
                    cts->getLineCount(), 
                    cts->getStaticMethodsCount(), 
                    cts->getInstanceMethodsCount(),
                    cts->getMethods(),
                    cts->getStaticFieldsCount(),
                    cts->getInstanceFieldsCount());
}

