#ifndef StatisticsReporterText_h
#include "StatisticsReporterText.h"
#endif

#ifndef ProjectStatistics_h
#include "ProjectStatistics.h"
#endif

#ifndef CompilationUnitStatistics_h
#include "CompilationUnitStatistics.h"
#endif

using namespace doctorj;


StatisticsReporterText::StatisticsReporterText(ostream& os) : StatisticsReporter(os)
{
}

StatisticsReporterText::~StatisticsReporterText()
{
}

void StatisticsReporterText::writeString(const string& label, const string& str)
{
    indent();
    int    i = INDENT_SPACES * level_ + label.length() + (str.length() > 0 ? 2 : 1);
    string dots;
    int nDots = PAGE_WIDTH - i - str.length();
    if (nDots > 0) {
        dots.append(nDots, PERIOD);
    }
    os_ << label << " " << dots << " " << str << endl;
}

void StatisticsReporterText::outputSummary(int nLines,
                                           int nTypes,
                                           int nConcretes,
                                           int nAbstracts,
                                           int nInterfaces,
                                           const vector<int>& typeLines,
                                           int nStaticMethods,
                                           int nInstanceMethods,
                                           const vector<int>& methodLines,
                                           int nStaticFields,
                                           int nInstanceFields)
{
    writeLeft("summary");

    incrementLevel();

    write("#files",      nTypes);
    write("total lines", nLines);
    blankLine();

    write("type count",       nConcretes + nAbstracts + nInterfaces);
    incrementLevel();

    write("concrete classes", nConcretes);
    write("abstract classes", nAbstracts);
    write("interfaces",       nInterfaces);

    writeStats(typeLines);

    decrementLevel();
    blankLine();

    write("methods",  nStaticMethods + nInstanceMethods);
    incrementLevel();
    write("static",   nStaticMethods);
    write("instance", nInstanceMethods);

    writeStats(methodLines);

    decrementLevel();
    blankLine();

    write("fields",   nStaticFields + nInstanceFields);
    incrementLevel();
    write("static",   nStaticFields);
    write("instance", nInstanceFields);
    decrementLevel();
    blankLine();

    decrementLevel();
}


void StatisticsReporterText::outputInnerType(const string& type, 
                                             const string& name,
                                             const vector<InnerTypeStatistics*>& innerTypes,
                                             int nLines,
                                             int nStaticMethods,
                                             int nInstanceMethods,
                                             const vector<int>& methods,
                                             int nStaticFields,
                                             int nInstanceFields)
{
    incrementLevel();

    writeLeft(type + " " + name);

    incrementLevel();

    write("#types", innerTypes.size());
    blankLine();

    vector<InnerTypeStatistics*>::const_iterator tsit   = innerTypes.begin();
    vector<InnerTypeStatistics*>::const_iterator tsstop = innerTypes.end();
    while (tsit != tsstop) {
        InnerTypeStatistics* type = *tsit;
        writeInnerType(type);
        ++tsit;
    }

    decrementLevel();
    
    write("total lines", nLines);
    blankLine();

    int nMethods = nStaticMethods + nInstanceMethods;

    write("methods", nMethods);
    if (nMethods > 0) {
        incrementLevel();
        write("static",    nStaticMethods);
        write("instance",  nInstanceMethods);
        writeStats(methods);
        decrementLevel();
        blankLine();
    }

    int nFields = nStaticFields + nInstanceFields;

    write("fields", nFields);
    if (nFields > 0) {
        incrementLevel();
        write("static",   nStaticFields);
        write("instance", nInstanceFields);
        decrementLevel();
        blankLine();
    }

    decrementLevel();
}

void StatisticsReporterText::writeFullReport(ProjectStatistics* const ps)
{
    writeLeft("project statistics");

    const vector<CompilationUnitStatistics*>& cus = ps->getCompilationUnitsStatistics();
    vector<CompilationUnitStatistics*>::const_iterator cit   = cus.begin();
    vector<CompilationUnitStatistics*>::const_iterator cstop = cus.end();
    incrementLevel();
    while (cit != cstop) {
        const CompilationUnitStatistics* cus = *cit;
        
        fill('.');
        writeLeft("file " + cus->name());
        write("#imports", cus->getImportCount());

        const vector<InnerTypeStatistics*>& types = cus->types();
        write("#types", types.size());
        blankLine();

        vector<InnerTypeStatistics*>::const_iterator tsit   = types.begin();
        vector<InnerTypeStatistics*>::const_iterator tsstop = types.end();
        while (tsit != tsstop) {
            InnerTypeStatistics* type = *tsit;
            writeInnerType(type);
            ++tsit;
        }
        
        blankLine();
        ++cit;
    }
    
    if (cus.size() > 1) {
        writeSummary(ps);
    }

    decrementLevel();
}
