#ifndef FileByteStream_h
#define FileByteStream_h

#ifndef Types_h
#include "Types.h"
#endif

#ifndef std_string
#define std_string
#include <string>
#endif

#include <stdio.h>

using namespace std;

namespace doctorj
{
    /**
     * Wraps a file in a byte stream.
     */
    class FileByteStream
    {
    public:
        FileByteStream(const string& name);

        FileByteStream(FILE* file);

        virtual ~FileByteStream();

        /**
         * Reads nBytes number of bytes into the buffer, at the given offset.
         * See <code>fseek</code> for information about the offset and whence
         * parameters.
         */
        bool read(int nBytes, int offset, int whence);

        /**
         * Reads nBytes number of bytes into the buffer, at the current file
         * position.
         */
        bool read(int nBytes);

        /**
         * Gets the next byte.
         */
        u1 getU1();
        
        /**
         * Gets the next word.
         */
        virtual u2 getU2() = 0;

        /**
         * Gets the next int.
         */
        virtual u4 getU4() = 0;

        /**
         * Gets n bytes and returns a pointer to the first. The caller must
         * delete the returned bytes, which should be done as "delete []".
         */
        char* getBytes(int nBytes);

        /**
         * Skips the next n bytes.
         */
        void skip(int n);

        /**
         * Returns the current position in the buffer.
         */
        char* position() const;

        /**
         * Returns the file stream.
         */
        FILE* file() const;

    private:
        FILE* file_;

        char* buffer_;
        
        char* bufptr_;

        char* bufend_;
        
    };


    /**
     * Wraps a file in a byte stream. Words and ints are retrieved as big
     * endian, that is, high bytes first.
     */
    class FileByteStreamBigEndian : public FileByteStream
    {
    public:
        FileByteStreamBigEndian(const string& name);

        FileByteStreamBigEndian(FILE* file);

        virtual ~FileByteStreamBigEndian();
        
        /**
         * Gets the next word.
         */
        u2 getU2();

        /**
         * Gets the next int.
         */
        u4 getU4();

    };


    /**
     * Wraps a file in a byte stream. Words and ints are retrieved as little
     * endian, that is, low bytes first.
     */
    class FileByteStreamLittleEndian : public FileByteStream
    {
    public:
        FileByteStreamLittleEndian(const string& name);

        FileByteStreamLittleEndian(FILE* file);

        virtual ~FileByteStreamLittleEndian();
        
        /**
         * Gets the next word.
         */
        u2 getU2();

        /**
         * Gets the next int.
         */
        u4 getU4();

    };

}

#endif //! FileByteStream_h
