#ifndef SpellChecker_h
#define SpellChecker_h

#ifndef std_map
#define std_map
#include <map>
#endif

#ifndef std_vector
#define std_vector
#include <vector>
#endif

#ifndef std_string
#define std_string
#include <string>
#endif

using namespace std;

namespace doctorj
{
    /**
     * Checks spelling against a list of words.
     */
    class SpellChecker
    {
    public:
        static const int DEFAULT_MAX_DISTANCE;

        SpellChecker();

        virtual ~SpellChecker();

        virtual bool isCorrect(const string& word, int maxEditDistance, multimap<int, string>* const nearMatches = NULL) const;

        /**
         * Returns whether the word is correct, using the default maximum distance.
         */
        virtual bool isCorrect(const string& word, multimap<int, string>* const nearMatches = NULL) const;

        /**
         * Reads the contents of the dictionary, which is an ASCII file of words
         * separated by whitespace. For example, /usr/dict/words under RedHat 6.2
         * Linux.
         */
        virtual void addDictionary(const string& dictName);

        /**
         * Adds the word as a valid spelling.
         */
        virtual void addWord(const string& word);

        /**
         * Adds the word as a valid spelling.
         */
        virtual void addWord(const char* const word);

        /**
         * Computes and returns the Levenstein edit distance between the two words.
         */
        virtual int editDistance(const char* const str1, size_t len1, const char* const str2, size_t len2, int maximum = 3) const;

        /**
         * Returns whether the exact word exists.
         */
        virtual bool hasWord(const string& word) const;

        /**
         * Returns whether the two words are a close match, i.e., within the default
         * maximum distance.
         */
        virtual bool nearMatch(const string& str1, const string& str2) const;

    protected:

        /**
         * Compares the two strings.
         */
        virtual int compare(const char* const str1, int len1, const char* const str2, int len2) const;

        // vector<string> words_;
        char** words_;

        int nWords_;

        int maxWords_;

    };


    /**
     * Checks spelling without regard to case.
     */
    class NoCaseSpellChecker : public SpellChecker
    {
    public:
        NoCaseSpellChecker();

        virtual ~NoCaseSpellChecker();

        /**
         * Adds the word as a valid spelling.
         */
        virtual void addWord(const string& word);
        
    protected:
        /**
         * Compares the two characters without regard to case.
         */
        virtual int compare(const char* const str1, int len1, const char* const str2, int len2) const;

        /**
         * Returns whether the given word exists.
         */
        virtual bool hasWord(const string& word) const;

    private:
        /**
         * Lowercase words.
         */
        vector<string> lcwords_;

    };

}

#endif //! SpellChecker_h
