#ifndef Yagol_h
#include "Yagol.h"
#endif

#ifndef RCFile_h
#include "RCFile.h"
#endif

#ifndef yagol_stdlib_h
#define yagol_stdlib_h
#include <stdlib.h>
#endif

#ifndef yagol_string_h
#define yagol_string_h
#include <string.h>
#endif

#ifndef std_iostream
#define std_iostream
#include <iostream>
#endif

#ifndef std_algorithm
#define std_algorithm
#include <algorithm>
#endif

#include <sys/stat.h>
#include <unistd.h>
#include <iostream>

using namespace std;
using namespace yagol;
using namespace doctorj;

namespace yagol
{
    class VersionDisplayer : public yagol::ArgCallback<bool> 
    {
    public:
        VersionDisplayer(const string& package, const string& version);

        virtual ~VersionDisplayer();

        void setPackage(const string& package);

        void setVersion(const string& version);

        virtual bool onProcessed(const bool& ver);

        // virtual void onComplete(const OptionList& opts);

    private:
        bool* display_;

        string package_;

        string version_;
    };
}


VersionDisplayer::VersionDisplayer(const string& package, const string& version)
        : package_(package), version_(version)
{
}

VersionDisplayer::~VersionDisplayer()
{
}

void VersionDisplayer::setVersion(const string& version)
{
    version_ = version;
}

void VersionDisplayer::setPackage(const string& package)
{
    package_ = package;
}

bool VersionDisplayer::onProcessed(const bool& ver) 
{
    cout << package_ << ", version " << version_ << endl;
    // cout << "Written by <author> (<email>)." << endl;
    // cout << "Released under <license>." << endl;
    // displayed_ = true;
    return true;
}


AppOptionSet::AppOptionSet(const string& progName,
                           const string& progDesc /* = string() */,
                           const string& usageSummary /* = string() */)
        : versionDisplayer_(NULL)
{
    init(progName, progDesc, true, usageSummary);
}

AppOptionSet::AppOptionSet(bool strictArgs,
                           bool helpOnError,
                           const string& progName,
                           const string& progDesc /* = string() */,
                           const string& usageSummary /* = string() */)
        : OptionList(strictArgs), versionDisplayer_(NULL)
{
    init(progName, progDesc, helpOnError, usageSummary);
}

AppOptionSet::AppOptionSet(int options,
                           const string& progName,
                           const string& progDesc /* = string() */,
                           const string& usageSummary /* = string() */)
        : OptionList((options & STRICT_ARGUMENTS) != 0), versionDisplayer_(NULL)
{
    init(progName, progDesc, (options & HELP_ON_ERROR) != 0, usageSummary);
}

AppOptionSet::~AppOptionSet()
{
    if (versionDisplayer_) {
        delete versionDisplayer_;
    }
}

void AppOptionSet::init(const string& progName,
                        const string& progDesc,
                        bool helpOnError, 
                        const string& usageSummary)
{
    progName_ = progName;
    progDesc_ = progDesc;
    hadError_ = false;
    usageSummary_ = usageSummary;
    displayVersion_ = false;
    helpOnError_ = helpOnError;
}

template <typename T>
struct indent : public unary_function<T, void>
{
    indent(ostream& out) : os(out) {}
    void operator()(T x) { os << "    " << x << endl; }
    ostream& os;
};

void AppOptionSet::setHelpOnError(bool helpOnError)
{
    helpOnError_ = helpOnError;
}

void AppOptionSet::writeUsage(ostream& os) const
{
    os << "Usage: " << programName() << " " << usageSummary() << endl;
    if (programDescription() != "") {
        os << "    " << programDescription() << endl;
    }
    os << endl;

    os << "Options:" << endl;
    OptionList::writeUsage(os);
    
    if (envVars_.size() > 0) {
        os << endl;
        os << "Environment Variables" << endl;
        for_each(envVars_.begin(), envVars_.end(), indent< string >(os));
    }

    if (configFiles_.size() > 0) {
        os << endl;
        os << "Configuration Files" << endl;
        for_each(configFiles_.begin(), configFiles_.end(), indent< string >(os));
    }
}
    
string AppOptionSet::programName() const
{
    return progName_;
}

string AppOptionSet::programDescription() const
{
    return progDesc_;
}

void AppOptionSet::process(int argc, char** argv) throw(yagol::Exception)
{
    processConfigFiles();
    processEnvironmentVariables();

    try {
        // and do the normal command-line processing
        OptionList::process(argc, argv);
    }
    catch (const yagol::Exception& e) {
        onError(e);
    }
}

string AppOptionSet::expandFileName(const string& fname) const
{
    string exp = fname;
    if (fname[0] == '~') {
        char* home = getenv("HOME");
        if (home) {
            exp = string(home) + fname.substr(1);
        }
        else {
            // maybe this is Windows
            char* homedrive = getenv("HOMEDRIVE");
            char* homepath  = getenv("HOMEPATH");
            if (homedrive && homepath) {
                // do I need to escape backslashes?
                exp = string(homedrive) + string(homepath) + fname.substr(1);
            }
        }
    }
    return exp;
}

void AppOptionSet::processConfigFiles()
{
    for (vector<string>::iterator it = configFiles_.begin(), stop = configFiles_.end();
         it != stop; ++it) {
        string file = *it;
        file = expandFileName(file);
        readConfigFile(file);
    }
}

void AppOptionSet::processEnvironmentVariables()
{
    for (vector<string>::iterator it = envVars_.begin(), stop = envVars_.end();
         it != stop; ++it) {
        string envVar = *it;
        readEnvironmentVariable(envVar);
    }
}

void AppOptionSet::onError(const yagol::Exception& e) throw(yagol::Exception)
{
    cerr << "ERROR: " << e << endl;
    hadError_ = true;
    if (helpOnError_) {
        writeUsage(cerr);
    }
    else {
        // propagate it
        throw;                  // not "throw e"
    }
}

bool AppOptionSet::hadError() const
{
    return hadError_;
}

void AppOptionSet::addConfigFile(const string& file)
{
    configFiles_.push_back(file);
}

void AppOptionSet::readConfigFile(const string& file)
{
    // Read configuration files and set the name/value pairs. Note that if the
    // file doesn't exist, no exception is thrown--we just won't get any data.
    RCFile rcfile(file);
    int ct = rcfile.count();
    for (int i = 0; i < ct; ++i) {
        pair<string, string> pr = rcfile.get(i);
        string label = pr.first;
        string value = pr.second;
        
        for (vector<Option*>::iterator optit = options().begin(), stop = options().end(); optit != stop; ++optit) {
            Option* opt = *optit;
            
            if (opt->labelMatches(label)) {
                opt->setValue(value);
                opt->onRead();
            }
        }
    }
}

void AppOptionSet::readEnvironmentVariable(const string& envVar)
{
    char* envValue = getenv(envVar.c_str());
    if (envValue) {
        string envValueStr(envValue);
        vector<string> words;
        doctorj::StringUtilities::getWords(envValueStr, &words);

        // we have to do some trickery to account that the argc/argv set
        // contains the app name, whereas obviously this array doesn't.
        int len = words.size();
        char** ary = new char*[len + 1];
        for (int i = 0; i < len; ++i) {
            char* dup = strdup(words[i].c_str());
            ary[i + 1] = dup;
        }
        
        OptionList::process(len + 1, ary);

        // old fashioned (C) delete
        for (int i = 1; i <= len; ++i) {
            free(ary[i]);
        }
        // and a C++ delete
        delete [] ary;
    }
    else {
        DEBUG_YAGOL(cout << "readEnvironmentVariable(" << envVar << "): not found" << endl);
    }
}

void AppOptionSet::addEnvironmentVariable(const string& envVar)
{
    envVars_.push_back(envVar);
}

string AppOptionSet::usageSummary() const
{
    return usageSummary_;
}

void AppOptionSet::setVersion(const string& package, const string& version)
{
    displayVersion_ = false;
    versionDisplayer_ = new VersionDisplayer(package, version);
    addOptionCb("version", "output version information", versionDisplayer_);
}

bool AppOptionSet::displayedVersion() const
{
    return displayedVersion_;
}


ostream& yagol::operator<<(ostream& os, const yagol::Exception& e)
{
    e.print(os);
    return os;
}

Exception::Exception() : message_("yagol Exception")
{
}

Exception::Exception(const string& msg) : message_(msg)
{
}

Exception::Exception(const yagol::Exception& rhs)
{
    *this = rhs;
}

Exception::~Exception()
{
}

Exception& Exception::operator=(const yagol::Exception& rhs)
{
    if (this != &rhs) {
        message_ = rhs.message_;
    }
    return *this;
}

void Exception::print(ostream& os) const
{
    os << message();
}

string Exception::message() const
{
    return message_;
}


FileNotReadableException::FileNotReadableException(const string& filename)
        : Exception(string("file not readable: ") + filename)
{
}

FileNotReadableException::FileNotReadableException(const FileNotReadableException& rhs)
{
    *this = rhs;
}

FileNotReadableException::~FileNotReadableException()
{
}

FileNotReadableException&
FileNotReadableException::operator=(const FileNotReadableException& rhs)
{
    Exception::operator=(rhs);
    return *this;
}


FileNotWritableException::FileNotWritableException(const string& filename)
        : Exception(string("file not writable: ") + filename)
{
}

FileNotWritableException::FileNotWritableException(const FileNotWritableException& rhs)
{
    *this = rhs;
}

FileNotWritableException::~FileNotWritableException()
{
}

FileNotWritableException&
FileNotWritableException::operator=(const FileNotWritableException& rhs)
{
    Exception::operator=(rhs);
    return *this;
}


NoSuchFileException::NoSuchFileException(const string& filename)
        : Exception(string("no such file: ") + filename)
{
}

NoSuchFileException::NoSuchFileException(const NoSuchFileException& rhs)
{
    *this = rhs;
}

NoSuchFileException::~NoSuchFileException()
{
}

NoSuchFileException&
NoSuchFileException::operator=(const NoSuchFileException& rhs)
{
    Exception::operator=(rhs);
    return *this;
}


NotAFileException::NotAFileException(const string& name)
        : yagol::Exception(string("not a file: ") + name)
{
}

NotAFileException::NotAFileException(const NotAFileException& rhs)
{
    *this = rhs;
}

NotAFileException::~NotAFileException()
{
}

NotAFileException& NotAFileException::operator=(const NotAFileException& rhs)
{
    Exception::operator=(rhs);
    return *this;
}


static bool tagMatch(const string& name, const string& tag)
{
    // also handle 'namespace1.namespace2.<tag>'
    int lastdot = name.rfind('.');
    if (lastdot == -1) {
        // just match the tag
        return tag == name || (name.substr(0, tag.length()) == tag);
    }
    else {
        // it has to be the entire name
        return tag == name || tag == string(name, lastdot + 1);
    }
}

const string Option::DELIMITER = "--";

ostream& yagol::operator<<(ostream& os, const Option& opt)
{
    return opt.print(os);
}

Option::Option(const string& name, 
               const string& description,
               bool strictArgs /* = false */) :
        name_(name),
     description_(description),
     strictArgs_(strictArgs),
     configOnly_(false),
     cmdLineOnly_(false)
{
}

Option::Option(const Option& rhs) : 
        name_(rhs.name_),
     description_(rhs.description_),
     configOnly_(rhs.configOnly_),
     cmdLineOnly_(rhs.cmdLineOnly_)
{
}

Option::~Option()
{
}

string Option::name() const
{
    return name_;
}

string Option::description() const
{
    return description_;
}

ostream& Option::print(ostream& os) const
{
    os << "--" << name() << "\t";
    printDescription(os);
    os << endl;
    return os;
}

ostream& Option::printDescription(ostream& os) const
{
    os << description();
    return os;
}

bool Option::matchesDelimiter(const string& arg)
{
    return arg.substr(0, DELIMITER.length()) == DELIMITER;
}

int Option::delimiterLength()
{
    return DELIMITER.length();
}

bool Option::strictArgs() const
{
    return strictArgs_;
}

void Option::checkArgument(const string& arg) throw(yagol::Exception)
{
    if (matchesDelimiter(arg) && strictArgs()) {
        string errmsg = "argument '" + arg + "' for option " + name() + " appears to be a tag";
        throw yagol::Exception(errmsg);
    }
}

bool Option::onProcessed()
{
    return true;
}

bool Option::onRead()
{
    return true;
}

void Option::onComplete(const OptionList& opts)
{
}

bool Option::tagMatches(const string& tag) const
{
    if (tagMatch(name(), tag)) {
        return true;
    }
    else {
        int naliases = aliases_.size();
        for (int i = 0; i < naliases; ++i) {
            if (tagMatch(aliases_[i], tag)) {
                return true;
            }
        }
        return false;
    }
}

bool Option::labelMatches(const string& label) const
{
    return label == name();
}

void Option::setValueFromConfigFile(const string& value)
{
    setValue(value);
}

void Option::addLinkedOption(Option* const opt)
{
    linkedOptions_.push_back(opt);
}

vector<Option*>& Option::getLinkedOptions()
{
    return linkedOptions_;
}

void Option::setConfigOnly(bool cfgOnly)
{
    configOnly_ = cfgOnly;
}

bool Option::isConfigOnly() const
{
    return configOnly_;
}

void Option::setCommandLineOnly(bool clOnly)
{
    cmdLineOnly_ = clOnly;
}

bool Option::isCommandLineOnly() const
{
    return cmdLineOnly_;
}

void Option::writeConfig(ostream& os) const
{
    os << "# " << description() << endl;;
    // should probably insert # after newlines in the description:
    os << name() << " = " << defaultValueConfigString() << endl;
}

void Option::addAlias(const string& alias)
{
    aliases_.push_back(alias);
}


OptionFile::OptionFile(const string& name, 
                       const string& description,
                       string* const filename,
                       bool strictArgs /* = false */) :
        OptionGeneric<string>(name, description, filename, strictArgs)
{
}

OptionFile::OptionFile(const OptionFile& rhs) :
        OptionGeneric<string>(rhs)
{
}

OptionFile::~OptionFile()
{
}

bool OptionFile::onProcessed()
{
    validate(value());
    return true;
}


OptionInputFile::OptionInputFile(const string& name, 
                                 const string& description,
                                 string* const filename,
                                 bool strictArgs /* = false */) :
        OptionFile(name, description, filename, strictArgs)
{
}

OptionInputFile::OptionInputFile(const OptionInputFile& rhs) :
        OptionFile(rhs)
{
}

OptionInputFile::~OptionInputFile()
{
}

void OptionInputFile::validate(const string& filename) const throw(yagol::Exception)
{
    const char* rawname = filename.c_str();

    bool exists = 0 == access(rawname, F_OK);
    if (exists) {
        struct stat status;
        stat(rawname, &status);
        bool isfile = S_ISREG(status.st_mode);
        if (isfile) {
            bool readable = 0 == access(rawname, R_OK);
            if (!readable) {
                // we can't read it
                throw FileNotReadableException(filename);
            }
        }
        else {
            // it isn't a file
            throw yagol::NotAFileException(filename);
        }
    }
    else {
        // couldn't find the file
        throw yagol::NoSuchFileException(filename);
    }
}


ostream& yagol::operator<<(ostream& os, const yagol::OptionList& opts) 
{
    return opts.print(os);
}

OptionList::OptionList(bool strictArgs /* = false */) : strictArgs_(strictArgs), showedHelp_(false)
{
}

OptionList::~OptionList()
{
    vector<Option*>::iterator it = options_.begin();
    vector<Option*>::iterator stop = options_.end();
    for (; it != stop; ++it) {
        Option* opt = *it;
        delete opt;
    }
}

void OptionList::process(int argc, char** argv) throw(yagol::Exception)
{
    DEBUG_YAGOL(cout << "OptionList::process(): processing arguments" << endl);

    // skip the first argument, since that's the name of the program:
    int i = 1;

    while (i < argc) {
        DEBUG_YAGOL(cout << "processing argv[" << i << "] '" << argv[i] << "'" << endl);
        if (Option::matchesDelimiter(argv[i])) {
            DEBUG_YAGOL(cout << "option #" << i << " '" << argv[i] << "' matches the delimiter" << endl);

            string arg = string(string(argv[i]), Option::delimiterLength());

            Option* opt = findExactMatch(arg);
            if (!opt) {
                vector<string> otherMatches;
                opt = findApproximateMatch(arg, &otherMatches);
                if (otherMatches.size() > 1) {
                    complainAboutMatches(otherMatches, argv[i]);
                }
            }

            if (opt) {
                i += opt->consume(arg, i, argc, argv);
                
                // not doing anything with the return value here.
                opt->onProcessed();
            }
            else {
                DEBUG_YAGOL(cout << "option #" << i << " '" << argv[i] << "' did not match anything" << endl);
                
                string argstr = string(argv[i]);
                // maybe it was a cry for help
                if (argstr == "--usage" || 
                    argstr == "--help"  || 
                    argstr == "--h"     || 
                    argstr == "-h"      || 
                    argstr == "--?"     || 
                    argstr == "-?"      || 
                    argstr == "?") {
                    writeUsage(cout);
                    return;
                }
                if (argstr == "--help-config") {
                    writeConfigUsage(cout);
                    return;
                }
                // the argument was not consumed -- did it look like an option?
                else if (strictArgs_) {
                    DEBUG_YAGOL(cout << "option #" << i << " '" << argv[i] << "' seemed to be an option" << endl);
                    string msg("no option '" + string(argv[i]) + string("'"));
                    throw yagol::Exception(msg);
                }
                else {
                    // argument not consumed
                }

                DEBUG_YAGOL(cout << "storing option #" << i << " '" << argv[i] << "' as unprocessed" << endl);
                unprocessedArgs_.push_back(argv[i]);
                i++;
            }
        }
        else {
            // arg was not consumed, and does not look like an option
            unprocessedArgs_.push_back(argv[i]);
            i++;
        }
    }

    // OK, now we've processed every one, so fire the onComplete methods. Note
    // that we're not distinguishing between those options that were updated,
    // and those that weren't.
    vector<Option*>::iterator oit = options_.begin();
    vector<Option*>::iterator ostop = options_.end();
    for (; oit != ostop; ++oit) {
        Option* opt = *oit;
        DEBUG_YAGOL(cout << "calliing opt(" << opt->name() << ")->onComplete()" << endl);
        opt->onComplete(*this);
    }
    
    vector<Processor*>::iterator pit = postProcessors_.begin();
    vector<Processor*>::iterator pstop = postProcessors_.end();
    for (; pit != pstop; ++pit) {
        Processor* p = *pit;
        p->onComplete(*this);
    }
}

void OptionList::writeUsage(ostream& os) const
{
    showedHelp_ = true;

    // length of the longest name, for alignment
    size_t maxlen = 0;

    vector<Option*>::const_iterator it = options_.begin();
    vector<Option*>::const_iterator stop = options_.end();

    bool hasConfigOnlyOptions = false;

    for (; it != stop; ++it) {
        Option* opt = *it;
        if (opt->isConfigOnly()) {
            hasConfigOnlyOptions = true;
        }
        else {
            maxlen = std::max(maxlen, opt->name().length());
        }
    }

    // give it some more space (literally)
    maxlen += 4;

    // reset to head.
    it = options_.begin();

    for (; it != stop; ++it) {
        Option* opt = *it;
        if (!opt->isConfigOnly()) {
            string nm = opt->name();
            os << "    ";
            os << Option::DELIMITER << nm;
            size_t len = nm.length();
            for (size_t i = len; i < maxlen; ++i) {
                os << " ";
            }
            opt->printDescription(os);
            os << endl;
        }
    }

    if (hasConfigOnlyOptions) {
        os << "For configuration variables, run --help-config" << endl;
    }
}

void OptionList::writeConfigUsage(ostream& os) const
{
    showedHelp_ = true;

    vector<Option*>::const_iterator it = options_.begin();
    vector<Option*>::const_iterator stop = options_.end();

    os << "# resource file" << endl;

    for (; it != stop; ++it) {
        Option* opt = *it;
        if (!opt->isCommandLineOnly()) {
            opt->writeConfig(os);
            os << endl;
        }
    }
}

Option* OptionList::addOption(Option* const opt)
{
    //cout << "OPLST  ::addOption(" << opt << ")" << endl;
    checkName(opt->name());
    options_.push_back(opt);
    cout << flush;              // yes, we have a memory leak, at least in GCC 3.2
    return opt;
}

void OptionList::addLink(const string& from, const string& to) 
{
    Option* fromopt = findFirstByName(from);
    Option* toopt   = findFirstByName(to);
    if (fromopt && toopt) {
        toopt->addLinkedOption(fromopt);
    }
    else {
        cerr << "ERROR: OptionList::addLink(" << from << ", " << to << "): "
             << "no such option(s)" << endl;
    }
}

vector<Option*>& OptionList::options()
{
    return options_;
}

vector<char*> OptionList::getUnprocessedArgs() const
{
    return unprocessedArgs_;
}

vector<Option*> OptionList::findByName(const string& name) const
{
    vector<Option*> matches;

    vector<Option*>::const_iterator it = options_.begin();
    vector<Option*>::const_iterator stop = options_.end();

    for (; it != stop; ++it) {
        Option* opt = *it;
        if (opt->name() == name) {
            matches.push_back(opt);
        }
    }

    return matches;
}

Option* OptionList::findFirstByName(const string& name) const
{
    return findExactMatch(name);
}

Option* OptionList::findExactMatch(const string& name) const
{
    vector<Option*>::const_iterator it = options_.begin();
    vector<Option*>::const_iterator stop = options_.end();

    for (; it != stop; ++it) {
        Option* opt = *it;
        if (opt->name() == name) {
            return opt;
        }
    }
    return NULL;
}

Option* OptionList::findApproximateMatch(const string& name, 
                                         vector<string>* const others) const
{
    vector<Option*>::const_iterator it = options_.begin();
    vector<Option*>::const_iterator stop = options_.end();

    for (; it != stop; ++it) {
        Option* opt = *it;
        if (opt->tagMatches(name)) { // an approximate match
            *others = findMatches(name);
            return opt;
        }
    }
    return NULL;
}

void OptionList::checkName(const string& name) const throw(yagol::Exception)
{
    // any other matches for this one?
    vector<Option*> matches = findByName(name);

    unsigned long nMatches = matches.size();
    if (nMatches > 0) {
        string errmsg = string("Error: ") + name + " already matches ";
        if (nMatches == 1) {
            errmsg += matches[0]->name();
        }
        else if (nMatches == 2) {
            errmsg += "both ";
            errmsg += matches[0]->name() + " and " + matches[1]->name();
        }
        else {
            for (unsigned long mi = 0; mi < nMatches - 2; ++mi) {
                errmsg += matches[mi]->name() + ", ";
            }
            errmsg += matches[nMatches - 2]->name() + " and " + matches[nMatches - 1]->name();
        }
        cerr << "error in option setup: " << errmsg << endl;
        throw yagol::Exception(errmsg);
    }
}

vector<string> OptionList::findMatches(const string& tag) const
{
    vector<string> matches;

    vector<Option*>::const_iterator it = options_.begin();
    vector<Option*>::const_iterator stop = options_.end();

    for (; it != stop; ++it) {
        Option* opt = *it;
        if (opt->tagMatches(tag)) {
            matches.push_back(opt->name());
        }
    }
    return matches;
}

void OptionList::complainAboutMatches(const vector<string>& matches, const string& tag) const
     throw(yagol::Exception)
{
    if (matches.size() > 1) {
        string errmsg = tag + " is ambiguous. ";
        errmsg += "It could match ";
        if (matches.size() == 2) {
            errmsg += "either " + matches[0] + " or " + matches[1];
        }
        else {
            unsigned int nMatches = matches.size();
            for (unsigned int mi = 0; mi < nMatches - 2; ++mi) {
                errmsg += matches[mi] + ", ";
            }
            errmsg += matches[nMatches - 2] + " or " + matches[nMatches - 1];
        }
        throw yagol::Exception(errmsg);
    }
}

void OptionList::addProcessor(Processor* const proc)
{
    postProcessors_.push_back(proc);
}

ostream& OptionList::print(ostream& os) const
{
    vector<Option*>::const_iterator it = options_.begin();
    vector<Option*>::const_iterator stop = options_.end();

    for (; it != stop; ++it) {
        Option* opt = *it;
        opt->debugPrint(os);
        os << endl;
    }
    return os;
}

bool OptionList::showedHelp() const
{
    return showedHelp_;
}

void OptionList::setStrictArguments(bool strictArgs)
{
    strictArgs_ = strictArgs;
}


OptionOutputFile::OptionOutputFile(const string& name, 
                                   const string& description,
                                   string* const filename,
                                   bool strictArgs /* = false */) :
        OptionFile(name, description, filename, strictArgs)
{
}

OptionOutputFile::OptionOutputFile(const OptionOutputFile& rhs) :
        OptionFile(rhs)
{
}

OptionOutputFile::~OptionOutputFile()
{
}

void OptionOutputFile::validate(const string& filename) const throw(yagol::Exception)
{
    const char* rawname = filename.c_str();

    bool exists = 0 == access(rawname, F_OK);
    if (exists) {
        struct stat status;
        stat(rawname, &status);
        bool isfile = S_ISREG(status.st_mode);
        if (isfile) {
            bool writable = 0 == access(rawname, W_OK);
            if (!writable) {
                // can't write to the file
                throw FileNotWritableException(filename);
            }
        }
        else {
            // it exists, but it isn't a file
            throw NotAFileException(filename);
        }
    }
}


Processor::Processor()
{
}

Processor::~Processor()
{
}

void Processor::onComplete(const OptionList& opts)
{
    // nothing.
}
