"This module provides plotting functionality (wrapper for Viper)."

# Copyright (C) 2008 Joachim B. Haga
#
# This file is part of DOLFIN.
#
# DOLFIN is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# DOLFIN is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with DOLFIN.  If not, see <http://www.gnu.org/licenses/>.
#
# Modified by Martin Sandve Alnaes, 2008.
# Modified by Anders Logg, 2008-2010.
#
# First added:  2008-03-05
# Last changed: 2010-12-08

import os
import dolfin.cpp as cpp
from donothing import DoNothing
from dolfin.functions.function import Function
import ufl

__all__ = ['Viper', 'plot', 'update', 'interactive', 'save_plot', 'figure']

def make_viper_object(object):
    for plottype,classes in viper_dolfin.plottable:
        if isinstance(object, classes):
            return object

    if isinstance(object, cpp.DirichletBC):
        bc = object
        V = bc.function_space()
        v = Function(V)
        bc.apply(v.vector())
        return v

    # Try projecting function
    from dolfin.fem.project import project
    try:
        u = project(object)
        cpp.info("Object cannot be plotted directly, projecting to piecewise linears.")
        return u
    except:
        raise RuntimeError, ("Don't know how to plot given object and projection failed: " + str(object))


# Intelligent plot command that handles projections and different objects
# (aliased as 'plot')
def dolfin_plot(object, *args, **kwargs):
    "Plot given object (Mesh, MeshFunction or Function)"

    # Plot element
    if isinstance(object, ufl.FiniteElementBase):
        import ffc
        return ffc.plot(object, *args, **kwargs)

    # Check expression
    if isinstance(object, cpp.Expression) and "mesh" not in kwargs:
        raise TypeError, "expected a mesh when plotting an expression."

    return viper_dolfin.plot(make_viper_object(object), *args, **kwargs)


# Check DOLFIN_NOPLOT
do_nothing = False
if os.environ.has_key('DOLFIN_NOPLOT'):
    cpp.info("DOLFIN_NOPLOT set, plotting disabled.")
    do_nothing = True
else:

    # Check for Viper
    try:
        from viper import viper_dolfin
        for x in __all__:
            exec ('from viper.viper_dolfin import %s' % x)
        plot = dolfin_plot
    except ImportError, details:
        cpp.warning(str(details))
        cpp.warning("Unable to import Viper, plotting disabled.")
        do_nothing= True

# Ignore all plot calls
if do_nothing:
    for x in __all__:
        exec('%s = DoNothing' % x)
