"""This module provides a simple way to compute various norms of
Functions, including the standard L2 norm and other norms"""

__author__ = "Anders Logg (logg@simula.no)"
__date__ = "2008-03-19 -- 2009-12-11"
__copyright__ = "Copyright (C) 2008 Anders Logg"
__license__  = "GNU LGPL Version 2.1"

__all__ = ['norm']

from ufl import inner, grad, div, curl, dx
from math import sqrt

from dolfin.cpp import GenericVector, GenericFunction, Function, Mesh, error
from dolfin.fem.assemble import assemble

def norm(v, norm_type="L2", mesh=None):
    """Return norm of given vector or function. Example usage includes:

        norm(x)
        norm(x, 'linf')
        norm(v)
        norm(u0 - u1)
        norm(v, 'Hdiv')

    If the norm type is not specified, the standard L^2 norm is
    computed. Possible norm types include:

    Vectors:

      l^2:     norm(x, 'l2')      (default)
      l^1:     norm(x, 'l1')
      l^oo:    norm(x, 'linf')

    Functions:

      L^2:     norm(v, 'L2')
      H^1:     norm(v, 'H1')      includes L^2 term
      H^1_0:   norm(v, 'H10')     does not include L^2 term
      H(div):  norm(v, 'Hdiv')    includes L^2 term
      H(div):  norm(v, 'Hdiv0')   does not include L^2 term
      H(curl): norm(v, 'Hcurl')   includes L^2 term
      H(curl): norm(v, 'Hcurl0')  does not include L^2 term
    """

    if not isinstance(v, (GenericVector, GenericFunction)):
        raise TypeError, "expected a GenericVector or GenericFunction"

    # Check arguments
    if not isinstance(norm_type, str):
        error("Norm type must be a string, not " + str(type(norm_type)) + ".")
    if mesh is not None and not isinstance(mesh, Mesh):
        error("Expecting a Mesh, not " + str(type(mesh)) + ".")

    # Use vector norm if we get a vector
    if isinstance(v, GenericVector):
        if norm_type.lower() == "l2":
            return v.norm("l2")
        return v.norm(norm_type.lower())

    # Choose functional
    if norm_type.lower() == "l2":
        M = inner(v, v)*dx
    elif norm_type.lower() == "h1":
        M = inner(v, v)*dx + inner(grad(v), grad(v))*dx
    elif norm_type.lower() == "h10":
        M = inner(grad(v), grad(v))*dx
    elif norm_type.lower() == "hdiv":
        M = inner(v, v)*dx + div(v)*div(v)*dx
    elif norm_type.lower() == "hdiv0":
        M = div(v)*div(v)*dx
    elif norm_type.lower() == "hcurl":
        M = inner(v, v)*dx + inner(curl(v), curl(v))*dx
    elif norm_type.lower() == "hcurl0":
        M = inner(curl(v), curl(v))*dx
    else:
        error("Unknown norm type: "\
              "%s\nSee pydoc dolfin.norm for valid types." % str(norm_type))

    # Get mesh
    if isinstance(v, Function) and mesh is None:
        mesh = v.function_space().mesh()

    # Assemble value
    r = assemble(M, mesh=mesh, form_compiler_parameters={"representation": "quadrature"})

    # Check value
    if r < 0.0:
        error("Unable to compute norm, square of norm is negative (round-off errors?).")
    elif r == 0.0:
        return 0.0
    else:
        return sqrt(r)
