/***************************************************************************
 *   Copyright (C) 2006 by Peter Penz                                      *
 *   peter.penz@gmx.at                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "viewproperties.h"
#include <qdatetime.h>
#include <qfile.h>
#include <qdir.h>
#include <kurl.h>
#include <assert.h>
#include <klocale.h>

#include "dolphinsettings.h"

#define FILE_NAME "/.dolphinview"

ViewProperties::ViewProperties(const KURL& url)
    : m_changedProps(false),
      m_autoSave(true),
      m_subDirValidityHidden(false),
      m_url(url)
{
    QString fileURL(m_url.path());
    if ((fileURL.length() < 1) || (fileURL.at(0) != QChar('/'))) {
        return;
    }

    fileURL.append(FILE_NAME);

    QDir dir(m_url.path());
    QFile file(dir.path() + FILE_NAME);
    PropertiesNode node(&file);

    const bool isValidForSubDirs = !node.isEmpty() && node.isValidForSubDirs();
    while (dir.cdUp()) {
        QFile file(dir.path() + FILE_NAME);
        PropertiesNode parentNode(&file);
        if (!parentNode.isEmpty()) {
            const bool inheritProps = parentNode.isValidForSubDirs() &&
                                      (parentNode.subDirProperties().timeStamp() >
                                       node.localProperties().timeStamp());

            if (inheritProps) {
                node.setLocalProperties(parentNode.subDirProperties());
            }
        }
    }

    m_node = node;

    if (isValidForSubDirs) {
        m_subDirValidityHidden = true;
    }
    m_node.setValidForSubDirs(false);
}

ViewProperties::~ViewProperties()
{
    if (m_changedProps && m_autoSave) {
        save();
    }
}

void ViewProperties::setViewMode(DolphinView::Mode mode)
{
    if (m_node.localProperties().viewMode() != mode) {
        m_node.setViewMode(mode);
        updateTimeStamp();
    }
}

DolphinView::Mode ViewProperties::viewMode() const
{
    return m_node.localProperties().viewMode();
}

void ViewProperties::setShowHiddenFilesEnabled(bool show)
{
    if (m_node.localProperties().isShowHiddenFilesEnabled() != show) {
        m_node.setShowHiddenFilesEnabled(show);
        updateTimeStamp();
    }
}

bool ViewProperties::isShowHiddenFilesEnabled() const
{
    return m_node.localProperties().isShowHiddenFilesEnabled();
}

void ViewProperties::setValidForSubDirs(bool valid)
{
    if (m_node.isValidForSubDirs() != valid) {
        m_node.setValidForSubDirs(valid);
        updateTimeStamp();
    }
}

bool ViewProperties::isValidForSubDirs() const
{
    return m_node.isValidForSubDirs();
}

void ViewProperties::setAutoSaveEnabled(bool autoSave)
{
    m_autoSave = autoSave;
}

bool ViewProperties::isAutoSaveEnabled() const
{
    return m_autoSave;
}

void ViewProperties::save()
{
    QFile file(m_url.path() + FILE_NAME);
    if (!file.open(IO_WriteOnly)) {
        return;
    }

    const Properties& props = m_node.localProperties();
    char viewMode = static_cast<char>('0') + static_cast<char>(props.viewMode());

    const bool isValidForSubDirs = m_node.isValidForSubDirs() || m_subDirValidityHidden;

    QTextStream stream(&file);
    stream << viewMode
           << (props.isShowHiddenFilesEnabled() ? '1' : '0')
           << props.timeStamp().toString("yyyyMMddhhmmss")
           << (isValidForSubDirs ? '1' : '0');

    if (m_node.isValidForSubDirs()) {
        m_node.setSubDirProperties(props);
    }

    if (isValidForSubDirs) {
        const Properties& subDirProps = m_node.subDirProperties();
        viewMode = static_cast<char>('0') + static_cast<char>(subDirProps.viewMode());
        stream << viewMode
               << (subDirProps.isShowHiddenFilesEnabled() ? '1' : '0')
               << subDirProps.timeStamp().toString("yyyyMMddhhmmss");
    }
    file.flush();
    file.close();

    m_changedProps = false;
}

void ViewProperties::updateTimeStamp()
{
    m_changedProps = true;
    m_node.setTimeStamp(QDateTime::currentDateTime());
}

ViewProperties::Properties::Properties()
    : m_showHiddenFiles(false),
      m_viewMode(DolphinView::IconsView)
{
    m_timeStamp.setDate(QDate(1999, 12, 31));
    m_timeStamp.setTime(QTime(23, 59, 59));

    m_viewMode = DolphinSettings::instance().defaultViewMode();
}

ViewProperties::Properties::~Properties()
{
}

ViewProperties::PropertiesNode::PropertiesNode(QFile* file)
    : m_empty(true)
{
    m_isValidForSubDirs = false;

    if ((file != 0) && file->open(IO_ReadOnly)) {
        m_empty = false;

        const int max_len = 34;
        static char buffer[max_len];

        // TODO: use memset
        for (int i = 0; i < max_len; ++i) {
            buffer[i] = 0;
        }

        int readBytes = file->readLine(buffer, max_len);
        readBytes = readProperties(m_props, buffer);

        // check whether sub directory properties are available
        m_isValidForSubDirs = (buffer[readBytes] != '0');
        if (m_isValidForSubDirs) {
            readBytes = readProperties(m_subDirProps, &buffer[readBytes + 1]);
        }
        file->close();

        m_empty = (readBytes <= 0);
    }
}

ViewProperties::PropertiesNode::~PropertiesNode()
{
}

ViewProperties::PropertiesNode&
    ViewProperties::PropertiesNode::operator = (const PropertiesNode& node)
{
    if (&node != this) {
        m_empty = node.m_empty;
        m_isValidForSubDirs = node.m_isValidForSubDirs;
        m_props = node.m_props;
        m_subDirProps = node.m_subDirProps;
    }

    return *this;
}

int ViewProperties::PropertiesNode::toInt(const char* buffer, int count) const
{
    assert(buffer != 0);

    int value = 0;
    for (int i = 0; i < count; ++i) {
        value = value * 10 + static_cast<int>(buffer[i] - '0');
    }

    return value;
}

int ViewProperties::PropertiesNode::readProperties(Properties& props,
                                                   const char* buffer)
{
    props.setViewMode(static_cast<DolphinView::Mode>(buffer[0] - '0'));
    props.setShowHiddenFilesEnabled((buffer[1] != '0'));

    // read date
    QDateTime timeStamp;
    const int year  = toInt(&(buffer[2]), 4);
    const int month = toInt(&(buffer[6]), 2);
    const int day   = toInt(&(buffer[8]), 2);
    QDate date(year, month, day);
    timeStamp.setDate(date);

    // read time
    const int hour   = toInt(&(buffer[10]), 2);
    const int minute = toInt(&(buffer[12]), 2);
    const int second = toInt(&(buffer[14]), 2);
    QTime time(hour, minute, second);
    timeStamp.setTime(time);

    props.setTimeStamp(timeStamp);

    return (date.isValid() && time.isValid()) ? 16 : -1;
}


