/***************************************************************************
 *   Copyright (C) 2006 by Peter Penz                                      *
 *   peter.penz@gmx.at                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef VIEWPROPERTIES_H
#define VIEWPROPERTIES_H

#include <dolphinview.h>
#include <kurl.h>
#include <qdatetime.h>
class QFile;

/**
 * @short Maintains the view properties like 'view mode' or 'show hidden files' for a directory.
 *
 * The view properties are automatically stored inside
 * the directory as hidden file called '.dolphinview'. To read out the view properties
 * just construct an instance by passing the URL of the directory:
 * \code
 * ViewProperties props(KURL("/home/peter/Documents"));
 * const DolphinView::Mode mode = props.viewMode();
 * const bool showHiddenFiles = props.isShowHiddenFilesEnabled();
 * \endcode
 * When modifying a view property, the '.dolphinview' file is automatically updated
 * inside the destructor.
 *
 * @author Peter Penz
 */
// TODO: provide detailled design description, as mapping the user model to
// the physical modal is not trivial.
class ViewProperties
{
public:
    ViewProperties(const KURL& url);
    virtual ~ViewProperties();

    void setViewMode(DolphinView::Mode mode);
    DolphinView::Mode viewMode() const;

    void setShowHiddenFilesEnabled(bool show);
    bool isShowHiddenFilesEnabled() const;

    void setValidForSubDirs(bool valid);
    bool isValidForSubDirs() const;

    void setAutoSaveEnabled(bool autoSave);
    bool isAutoSaveEnabled() const;

    void updateTimeStamp();
    void save();

private:
    class Properties
    {
    public:
        Properties();
        ~Properties();  // non virtual
        void setShowHiddenFilesEnabled(bool show) { m_showHiddenFiles = show; }
        bool isShowHiddenFilesEnabled() const { return m_showHiddenFiles; }
        void setViewMode(DolphinView::Mode mode) { m_viewMode = mode; }
        DolphinView::Mode viewMode() const { return m_viewMode; }
        void setTimeStamp(const QDateTime& timeStamp) { m_timeStamp = timeStamp; }
        const QDateTime& timeStamp() const { return m_timeStamp; }

    private:
        bool m_showHiddenFiles;
        DolphinView::Mode m_viewMode;
        QDateTime m_timeStamp;
    };

    class PropertiesNode
    {
    public:
        PropertiesNode(QFile* file = 0);
        ~PropertiesNode();
        PropertiesNode& operator = (const PropertiesNode& node);
        bool isEmpty() const { return m_empty; }
        void setValidForSubDirs(bool valid) { m_isValidForSubDirs = valid; }
        bool isValidForSubDirs() const { return m_isValidForSubDirs; }


        void setLocalProperties(const Properties& props) { m_props = props; }
        const Properties& localProperties() const { return m_props; }
        void setShowHiddenFilesEnabled(bool show) { m_props.setShowHiddenFilesEnabled(show); }
        void setViewMode(DolphinView::Mode mode) { m_props.setViewMode(mode); }
        void setTimeStamp(const QDateTime timeStamp) { m_props.setTimeStamp(timeStamp); }
        const QDateTime& timeStamp() const { return m_props.timeStamp(); }

        void setSubDirProperties(const Properties& props) { m_subDirProps = props; }
        const Properties& subDirProperties() const { return m_subDirProps; }

    private:
        int toInt(const char* buffer, int count) const;
        int readProperties(Properties& props, const char* buffer);

        bool m_empty;
        bool m_isValidForSubDirs;
        Properties m_props;
        Properties m_subDirProps;
    };

    bool m_changedProps;
    bool m_autoSave;
    bool m_subDirValidityHidden;
    KURL m_url;
    PropertiesNode m_node;
};

#endif
