
import java.util.*;


/**
 * The Settings class maintains type information about the variables in the
 * configuration file.
 *
 * To make settings files more readable by people of various languages, the
 * actual names of the settings have been localized.  The settings are still
 * stored using an internal representation for each variable.  This class
 * maintains a mapping between the localized name of the variable and its
 * internal representation.
 *
 * <p>Instructions for adding settings:</p>
 * <ol>
 *   <li>Increment the setting count (NUM_SETTINGS)</li>
 *   <li>Add the variable name and possible enumerations to the proper
 *     language files for Settings.properties</li>
 *   <li>Add a lookup of the variable name along with its type to the
 *     var2type hashmap</li>
 *   <li>Create a hash set of enumerations for the variable if it takes a
 *     limited set of values</li>
 *   <li>Add comments describing the setting and its expected values to the
 *     resource file Comments.properties</li>
 * </ol>
 *
 * @version $Id: Settings.java,v 1.22 2002/09/29 05:42:20 blsecres Exp $
 * @author Ben Secrest
 */
public class Settings {
    /** A hashtable containing all the valid variables and their types.  */
    private HashMap var2type;

    /**
     * A hashtable containing all the valid variables and their localized
     * names.
     */
    private HashMap name2var;

    /**
     * A hashtable containing all the valid place holders and their localized
     * names.
     */
    private HashMap placeholder2local;

    /**
     * A hashtable containing a vector of possible enumeration values
     * for enumerated variables.
     */
    private HashMap enumerations;

    /** The localized version of <tt>true</tt> */
    private String localTrue;

    /** The localized version of <tt>false</tt> */
    private String localFalse;

    /** An object for logging error and progress */
    private IGLog log;

    /** The total number of settings */
    private static final int NUM_SETTINGS = 32;

    /** The total number of placeholders */
    public static final int NUM_PLACEHOLDERS = 15;

    /** This value indicates that the setting does not exist */
    public static final byte DNE = 0;

    /** This value indicates that the setting is a Boolean */
    public static final byte BOOLEAN = 1;

    /** This value indicates that the setting is an Integer */
    public static final byte INTEGER = 2;

    /** This value indicates that the setting is a String */
    public static final byte STRING = 3;

    /** This value indicates that the setting is an ArrayList */
    public static final byte ARRAY = 4;

    /** This value indicates that the setting is an enumeration */
    public static final byte ENUM = 5;

    /** The default logging level for this module */
    private static final int LOGLEVEL = 9;


    /**
      * The constructor fills the hashtable with the names and types of each
      * variable.  It also provides enumeration values for enum variables
      */
    public Settings(IGLog logObj, Locale locale) {
	log = logObj;
	if (LOGLEVEL >= IGLog.PROCEDURE)
	    log.add(IGLog.PROCEDURE, "Settings.Settings(IGLog, Locale)");

	ResourceBundle variableName =
	    ResourceBundle.getBundle("Settings", locale);

	// create a mapping of internal representation of variables to types
	if (LOGLEVEL >= IGLog.PROGRESS)
	    log.addResource(IGLog.PROGRESS, "SETTINGS_BUILD_VARMAP",
			    new String[]{log.getString("SETTINGS_TYPE")});
	var2type = new HashMap(NUM_SETTINGS);
	var2type.put(new String("LOCALE"), new Byte(STRING));
	var2type.put(new String("SETTINGS_VERSION"), new Byte(STRING));
	var2type.put(new String("GET_METHOD"), new Byte(ENUM));
	var2type.put(new String("FILE_SYSTEM_SEARCH"), new Byte(ENUM));
	var2type.put(new String("SCAN_ROOT"), new Byte(ARRAY));
	var2type.put(new String("HTTP_HEADERS"), new Byte(ARRAY));
	var2type.put(new String("SCAN_INCLUDE_FILTERS"), new Byte(ARRAY));
	var2type.put(new String("SCAN_EXCLUDE_FILTERS"), new Byte(ARRAY));
	var2type.put(new String("INDEX_INCLUDE_FILTERS"), new Byte(ARRAY));
	var2type.put(new String("INDEX_EXCLUDE_FILTERS"), new Byte(ARRAY));
	var2type.put(new String("OUTPUT_FILENAME"), new Byte(STRING));
	var2type.put(new String("INDEX_TYPE"), new Byte(ENUM));
	var2type.put(new String("FILES_TO_COPY"), new Byte(ARRAY));
	var2type.put(new String("FILE_COPY_TARGET"), new Byte(STRING));
	var2type.put(new String("SM_START_TXT"), new Byte(STRING));
	var2type.put(new String("SM_ENTER_FOLDER"), new Byte(STRING));
	var2type.put(new String("SM_LEAVE_FOLDER"), new Byte(STRING));
	var2type.put(new String("SM_LINKS_TXT"), new Byte(STRING));
	var2type.put(new String("SM_END_TXT"), new Byte(STRING));
	var2type.put(new String("OL_SORT_KEY"), new Byte(ENUM));
	var2type.put(new String("OL_ARTICLES"), new Byte(ARRAY));
	var2type.put(new String("OL_START_TXT"), new Byte(STRING));
	var2type.put(new String("OL_NAV_START"), new Byte(STRING));
	var2type.put(new String("OL_NAV_LINK"), new Byte(STRING));
	var2type.put(new String("OL_NAV_END"), new Byte(STRING));
	var2type.put(new String("OL_SEC_START"), new Byte(STRING));
	var2type.put(new String("OL_SEC_END"), new Byte(STRING));
	var2type.put(new String("OL_LINK"), new Byte(STRING));
	var2type.put(new String("OL_END_TXT"), new Byte(STRING));
	var2type.put(new String("PROJECT_NAME"), new Byte(STRING));
	var2type.put(new String("CREATED"), new Byte(INTEGER));
	var2type.put(new String("LOAD_AS_THEME"), new Byte(BOOLEAN));

	// create a mapping of localized variable names to internal variable
	// representations
	if (LOGLEVEL >= IGLog.PROGRESS)
	    log.addResource(IGLog.PROGRESS, "SETTINGS_BUILD_VARMAP",
			    new String[]{log.getString("SETTINGS_NAME")});
	name2var = new HashMap(NUM_SETTINGS);
	name2var.put(new String("locale"),
		new String("LOCALE"));
	name2var.put(variableName.getString("SETTINGS_VERSION"),
		new String("SETTINGS_VERSION"));
	name2var.put(variableName.getString("GET_METHOD"),
		new String("GET_METHOD"));
	name2var.put(variableName.getString("FILE_SYSTEM_SEARCH"),
		new String("FILE_SYSTEM_SEARCH"));
	name2var.put(variableName.getString("SCAN_ROOT"),
		new String("SCAN_ROOT"));
	name2var.put(variableName.getString("HTTP_HEADERS"),
		new String("HTTP_HEADERS"));
	name2var.put(variableName.getString("SCAN_INCLUDE_FILTERS"),
		new String("SCAN_INCLUDE_FILTERS"));
	name2var.put(variableName.getString("SCAN_EXCLUDE_FILTERS"),
		new String("SCAN_EXCLUDE_FILTERS"));
	name2var.put(variableName.getString("INDEX_INCLUDE_FILTERS"),
		new String("INDEX_INCLUDE_FILTERS"));
	name2var.put(variableName.getString("INDEX_EXCLUDE_FILTERS"),
		new String("INDEX_EXCLUDE_FILTERS"));
	name2var.put(variableName.getString("OUTPUT_FILENAME"),
		new String("OUTPUT_FILENAME"));
	name2var.put(variableName.getString("INDEX_TYPE"),
		new String("INDEX_TYPE"));
	name2var.put(variableName.getString("FILES_TO_COPY"),
		new String("FILES_TO_COPY"));
	name2var.put(variableName.getString("FILE_COPY_TARGET"),
		new String("FILE_COPY_TARGET"));
	name2var.put(variableName.getString("SM_START_TXT"),
		new String("SM_START_TXT"));
	name2var.put(variableName.getString("SM_ENTER_FOLDER"),
		new String("SM_ENTER_FOLDER"));
	name2var.put(variableName.getString("SM_LEAVE_FOLDER"),
		new String("SM_LEAVE_FOLDER"));
	name2var.put(variableName.getString("SM_LINKS_TXT"),
		new String("SM_LINKS_TXT"));
	name2var.put(variableName.getString("SM_END_TXT"),
		new String("SM_END_TXT"));
	name2var.put(variableName.getString("OL_SORT_KEY"),
		new String("OL_SORT_KEY"));
	name2var.put(variableName.getString("OL_ARTICLES"),
		new String("OL_ARTICLES"));
	name2var.put(variableName.getString("OL_START_TXT"),
		new String("OL_START_TXT"));
	name2var.put(variableName.getString("OL_NAV_START"),
		new String("OL_NAV_START"));
	name2var.put(variableName.getString("OL_NAV_LINK"),
		new String("OL_NAV_LINK"));
	name2var.put(variableName.getString("OL_NAV_END"),
		new String("OL_NAV_END"));
	name2var.put(variableName.getString("OL_SEC_START"),
		new String("OL_SEC_START"));
	name2var.put(variableName.getString("OL_SEC_END"),
		new String("OL_SEC_END"));
	name2var.put(variableName.getString("OL_LINK"),
		new String("OL_LINK"));
	name2var.put(variableName.getString("OL_END_TXT"),
		new String("OL_END_TXT"));
	name2var.put(variableName.getString("PROJECT_NAME"),
		new String("PROJECT_NAME"));
	name2var.put(variableName.getString("CREATED"),
		new String("CREATED"));
	name2var.put(variableName.getString("LOAD_AS_THEME"),
		new String("LOAD_AS_THEME"));

	// load boolean variables
	localTrue = variableName.getString("BOOLEAN_TRUE");
	localFalse = variableName.getString("BOOLEAN_FALSE");

	// load placeholders
	placeholder2local = new HashMap(NUM_PLACEHOLDERS);
	placeholder2local.put(new String("PH_GENERATE_DATE"),
			      variableName.getString("PH_GENERATE_DATE"));
	placeholder2local.put(new String("PH_DATE_FORMAT"),
			      variableName.getString("PH_DATE_FORMAT"));
	placeholder2local.put(new String("PH_IG_URL"),
			      variableName.getString("PH_IG_URL"));
	placeholder2local.put(new String("PH_IG_VERSION"),
			      variableName.getString("PH_IG_VERSION"));
	placeholder2local.put(new String("PH_OUTPUT_FILENAME"),
			      variableName.getString("PH_OUTPUT_FILENAME"));
	placeholder2local.put(new String("PH_OUTPUT_PATH"),
			      variableName.getString("PH_OUTPUT_PATH"));
	placeholder2local.put(new String("PH_FULL_DIR"),
			      variableName.getString("PH_FULL_DIR"));
	placeholder2local.put(new String("PH_TOP_DIR"),
			      variableName.getString("PH_TOP_DIR"));
	
	placeholder2local.put(new String("PH_FILE_COUNT"),
			      variableName.getString("PH_FILE_COUNT"));
	placeholder2local.put(new String("PH_FILE_AUTHOR"),
			      variableName.getString("PH_FILE_AUTHOR"));
	placeholder2local.put(new String("PH_FILE_TITLE"),
			      variableName.getString("PH_FILE_TITLE"));
	placeholder2local.put(new String("PH_FILE_DESCRIPTION"),
			      variableName.getString("PH_FILE_DESCRIPTION"));
	placeholder2local.put(new String("PH_FILE_KEYWORDS"),
			      variableName.getString("PH_FILE_KEYWORDS"));
	placeholder2local.put(new String("PH_FILE_KEYWORDS_SEPARATOR"),
			      variableName.getString("PH_FILE_KEYWORDS_SEPARATOR"));
	placeholder2local.put(new String("PH_FILE_SIZE"),
			      variableName.getString("PH_FILE_SIZE"));
	placeholder2local.put(new String("PH_FILE_SIZE_HUMAN"),
			      variableName.getString("PH_FILE_SIZE_HUMAN"));
	placeholder2local.put(new String("PH_FILE_SIZE_BYTES"),
			      variableName.getString("PH_FILE_SIZE_BYTES"));
	placeholder2local.put(new String("PH_FILE_SIZE_KB"),
			      variableName.getString("PH_FILE_SIZE_KB"));
	placeholder2local.put(new String("PH_FILE_SIZE_MB"),
			      variableName.getString("PH_FILE_SIZE_MB"));
	placeholder2local.put(new String("PH_FILE_TYPE"),
			      variableName.getString("PH_FILE_TYPE"));
	placeholder2local.put(new String("PH_FILE_PARSER"),
			      variableName.getString("PH_FILE_PARSER"));
	placeholder2local.put(new String("PH_FILE_NAME"),
			      variableName.getString("PH_FILE_NAME"));
	placeholder2local.put(new String("PH_FILE_REAL_LOCATION"),
			      variableName.getString("PH_FILE_REAL_LOCATION"));
	placeholder2local.put(new String("PH_FILE_REL_LOCATION"),
			      variableName.getString("PH_FILE_REL_LOCATION"));
	placeholder2local.put(new String("PH_FORMAT"),
			      variableName.getString("PH_FORMAT"));
	
	placeholder2local.put(new String("PH_SECTION_LETTER"),
			      variableName.getString("PH_SECTION_LETTER"));

	if (LOGLEVEL >= IGLog.PROGRESS)
	    log.addResource(IGLog.PROGRESS, "SETTINGS_BUILD_VARMAP",
			    new String[]{log.getString("SETTINGS_ENUM")});
	// map localized enumerations to internal representation
	// provide enumerations for getMethod
	name2var.put(variableName.getString("GET_METHOD_SERVER"),
		     new String("GET_METHOD_SERVER"));
	name2var.put(variableName.getString("GET_METHOD_FILESYSTEM"),
		     new String("GET_METHOD_FILESYSTEM"));
	enumerations = new HashMap(3);
	HashSet hs = new HashSet(2);
	hs.add(new String("GET_METHOD_SERVER"));
	hs.add(new String("GET_METHOD_FILESYSTEM"));
	enumerations.put(new String("GET_METHOD"), hs);

	// provide enumerations for fileSystemSearch
	name2var.put(variableName.getString("FILE_SYSTEM_SEARCH_LINK_FOLLOW"),
		     new String("FILE_SYSTEM_SEARCH_LINK_FOLLOW"));
	name2var.put(variableName.getString("FILE_SYSTEM_SEARCH_DIR_SCAN"),
		     new String("FILE_SYSTEM_SEARCH_DIR_SCAN"));
	hs = new HashSet(2);
	hs.add(new String("FILE_SYSTEM_SEARCH_LINK_FOLLOW"));
	hs.add(new String("FILE_SYSTEM_SEARCH_DIR_SCAN"));
	enumerations.put(new String("FILE_SYSTEM_SEARCH"), hs);

	// provide enumerations for indexType
	name2var.put(variableName.getString("INDEX_TYPE_SITE_MAP"),
		     new String("INDEX_TYPE_SITE_MAP"));
	name2var.put(variableName.getString("INDEX_TYPE_ORDERED_LIST"),
		     new String("INDEX_TYPE_ORDERED_LIST"));
	hs = new HashSet(2);
	hs.add(new String("INDEX_TYPE_SITE_MAP"));
	hs.add(new String("INDEX_TYPE_ORDERED_LIST"));
	enumerations.put(new String("INDEX_TYPE"), hs);

	// provide enumerations for olSortKey
	// NOTE: reusing the placeholder values since they're the same
	name2var.put(variableName.getString("PH_FILE_AUTHOR"),
		new String("OL_SORT_KEY_AUTHOR"));
	name2var.put(variableName.getString("PH_FILE_TITLE"),
		new String("OL_SORT_KEY_TITLE"));
	name2var.put(variableName.getString("PH_FILE_DESCRIPTION"),
		new String("OL_SORT_KEY_DESCRIPTION"));
	name2var.put(variableName.getString("PH_FILE_KEYWORDS"),
		new String("OL_SORT_KEY_KEYWORDS"));
	name2var.put(variableName.getString("PH_FILE_SIZE"),
		new String("OL_SORT_KEY_FILE_SIZE"));
	name2var.put(variableName.getString("PH_FILE_TYPE"),
		new String("OL_SORT_KEY_FILE_TYPE"));
	name2var.put(variableName.getString("PH_FILE_NAME"),
		new String("OL_SORT_KEY_FILE_NAME"));
	hs = new HashSet(7);
	hs.add(new String("OL_SORT_KEY_TITLE"));
	hs.add(new String("OL_SORT_KEY_AUTHOR"));
	hs.add(new String("OL_SORT_KEY_DESCRIPTION"));
	hs.add(new String("OL_SORT_KEY_KEYWORDS"));
	hs.add(new String("OL_SORT_KEY_FILE_SIZE"));
	hs.add(new String("OL_SORT_KEY_FILE_TYPE"));
	hs.add(new String("OL_SORT_KEY_FILE_NAME"));
	enumerations.put(new String("OL_SORT_KEY"), hs);
    }


    /**
      * Look up the setting type in the hashtable
      * @param variable The internal representation of the variable
      * @return The setting type
      * @see #DNE
      * @see #BOOLEAN
      * @see #INTEGER
      * @see #STRING
      * @see #ENUM
      * @see #ARRAY
      */
    public byte getTypeByVar(String variable) {
	if (LOGLEVEL >= IGLog.PROCEDURE)
	    log.add(IGLog.PROCEDURE, "byte Settings.getTypeByVar(String[" +
		    variable + "])");

	if (var2type.containsKey(variable))
	    return(((Byte) var2type.get(variable)).byteValue());
	else
	    return DNE;
    };


    /**
      * Retrieve a variable's type based on its localized name
      * @param name The localized name of the variable
      * @return The setting type
      */
    public byte getTypeByName(String name) {
	if (LOGLEVEL >= IGLog.PROCEDURE)
	    log.add(IGLog.PROCEDURE, "byte Settings.getTypeByName(String[" +
		    name + "])");

	if (name2var.containsKey(name))
	    return getTypeByVar(name2var.get(name).toString());
	else
	    return DNE;
    };


    /**
      * Provide access to internal representation of variables
      * @param name The localized name of the variable
      * @return The internal representation of a variable
      */
    public String getVarByName(String name) {
	if (LOGLEVEL >= IGLog.PROCEDURE)
	    log.add(IGLog.PROCEDURE, "String Settings.getVarByName(String[" +
		    name + "])");

	if (name2var.containsKey(name))
	    return name2var.get(name).toString();
	else
	    return null;
    }


    /**
      * Access localized variable names by internal representation
      * @param var The internal representation of a variable
      * @return The localized name of the variable
      */
    public String getNameByVar(String var) {
	if (LOGLEVEL >= IGLog.PROCEDURE)
	    log.add(IGLog.PROCEDURE, "String Settings.getNameByVar(String[" +
		    var + "])");

	// there's no easy way to do a reverse mapping other than iterating
	// over the elements in the hashtable
	if (name2var.containsValue(var))
	    for (Iterator i = name2var.entrySet().iterator(); i.hasNext(); ) {
		Map.Entry e = (Map.Entry) i.next();

		if (e.getValue().toString().equals(var))
		    return e.getKey().toString();
	    }

	return null;
    }


    /**
      * Check enumerated values
      * @param name The name of the setting in question
      * @param value The value being assigned to the setting
      * @return true if the value is valid, false otherwise
      */
    public boolean validEnumeration(String name, String value) {
	if (LOGLEVEL >= IGLog.PROCEDURE)
	    log.add(IGLog.PROCEDURE,
		    "boolean Settings.validEnumeration(String[" + name +
		    "], String[" + value + "])");

	if (name == null || value == null)
	    return false;

	HashSet set = (HashSet) enumerations.get(name);
	if (set != null && set.contains(value))
	    return true;
	else
	    return false;
    }


    /**
     * Convert boolean values from localized string to java boolean string
     * @param value The string value of a boolean setting
     * @return A string representation that can be used to initialize a Boolean
     * 	object or <tt>null</tt> if the value isn't a valid boolean
     */
    public String getBoolean(String value) {
	if (LOGLEVEL >= IGLog.PROCEDURE)
	    log.add(IGLog.PROCEDURE,
		    "boolean Settings.getBoolean(String[" + value + "])");

	if (value == null)
	    return null;

	if (localTrue.equals(value))
	    return "true";
	else if (localFalse.equals(value))
	    return "false";
	else
	    return null;
    }


    /**
     * Access localized versions of boolean values
     * @param val <tt>true</tt> or <tt>false</tt>
     * @return A localized representation of truth
     */
    public String getLocalBoolean(boolean val) {
	if (val)
	    return localTrue;
	else
	    return localFalse;
    }


    /**
     * Retrieve the localized version of a placeholder
     * @param key The internal representation of the placeholder
     * @return The localized placeholder or null if not found
     */
    public String getPlaceHolder(String key) {
	if (LOGLEVEL >= IGLog.PROCEDURE)
	    log.add(IGLog.PROCEDURE,
		    "String Settings.getPlaceHolder(String[" + key + "])");

	if (placeholder2local.containsKey(key))
	    return placeholder2local.get(key).toString();
	else
	    return null;
    }
}
