try:
    set
except NameError:
    from sets import Set as set

from dosage.util import fetchUrl, fetchManyUrls
from dosage.comic import Comic

class BasicComicModule(object):
    '''Base class with functions for comic modules.

    @type latestUrl: C{string}
    @cvar latestUrl: The URL for the latest comic strip.
    @type imageUrl: C{string}
    @cvar imageUrl: A string that is interpolated with the strip index
        to yield the URL for a particular strip.
    @type imageSearch: C{regex}
    @cvar imageSearch: A compiled regex that will locate the strip image URL
        when applied to the strip page.
    @type prevSearch: C{regex}
    @cvar prevSearch: A compiled regex that will locate the URL for the
        previous strip when applied to a strip page.
    '''
    referrer = None
    help = 'Sorry, no help for this module yet.'

    def __init__(self):
        self.name = self.__class__.__name__.split('.')[-1]
        self.currentUrl = None
        self.urls = set()

    def getFilename(self, imageUrl, pageUrl):
        return None

    def getReferrer(self, imageUrl, pageUrl):
        return self.referrer or self.latestUrl

    def getComic(self, url, pageUrl):
        if not url:
            return None
        return Comic(self.name, url, filename=self.getFilename(url, pageUrl), referrer=self.getReferrer(url, pageUrl))

    def getCurrentComics(self):
        self.currentUrl = self.latestUrl
        return self.getNextComics()

    def getNextComics(self):
        comics = []
        while not comics and self.currentUrl and self.currentUrl not in self.urls:
            comicUrlGroups, prevUrl = fetchManyUrls(self.currentUrl, (self.imageSearch, self.prevSearch))
            prevUrl = prevUrl and prevUrl[0] or None
            for comicUrl in comicUrlGroups:
                comics.append(self.getComic(comicUrl, self.currentUrl))
            self.urls.update([self.currentUrl])
            self.currentUrl = (prevUrl, None)[prevUrl in self.urls]
        return comics

    def setStrip(self, index):
        self.currentUrl = self.imageUrl % index

    def getHelp(self):
        return self.help

    def __iter__(self):
        """Iterate through the strips, starting from the current one and going backward."""
        if not self.currentUrl:
            self.currentUrl = self.latestUrl

        comics = True
        while comics:
            comics = self.getNextComics()
            if comics:
                yield comics

class BounceMixin(object):
    '''
    Mixin for comics modules needing the "bounce" trick; basically simulating
    going backwards, and then forwards again to get the canonical page URL.

    @type baseUrl: C{string}
    @cvar baseUrl: the URL to apply prevSearch to.
    @type prevSearch: C{regex}
    @cvar prevSearch: a compiled regex for finding the 'previous' URL.
    @type nextSearch: C{regex}
    @cvar nextSearch: a compiled regex for finding the 'next' URL.
    '''

    __latestUrl = None

    def getLatestUrl(self):
        if not self.__latestUrl:
            self.__latestUrl = fetchUrl(self.baseUrl, self.prevSearch)
            self.__latestUrl = fetchUrl(self.__latestUrl, self.nextSearch)
        return self.__latestUrl

    latestUrl = property(getLatestUrl)

class IndirectLatestMixin(object):
    '''
    Mixin for comics modules that link to the latest comic from a base page of
    some kind. This also supports comics which don't link to the last comic
    from the base page, but the beginning of the latest chapter or similiar
    schemes. It simulates going forward until it can't find a 'next' link as
    specified by the 'nextSearch' regex.

    @type baseUrl: C{string}
    @cvar baseUrl: the URL where the link to the latest comic is found.
    @type latestSearch C{regex}
    @cvar latestSearch: a compiled regex for finding the 'latest' URL.
    @type nextSearch C{regex}
    @cvar nextSearch: a compiled regex for finding the 'next' URL.
    '''

    __latestUrl = None

    def getLatestUrl(self):
        if not self.__latestUrl:
            self.__latestUrl = fetchUrl(self.baseUrl, self.latestSearch)
            if hasattr(self, "nextSearch"):
                nextUrl = fetchUrl(self.__latestUrl, self.nextSearch)
                while nextUrl:
                    self.__latestUrl = nextUrl
                    nextUrl = fetchUrl(self.__latestUrl, self.nextSearch)
        return self.__latestUrl

    latestUrl = property(getLatestUrl)
