/* doscan - Denial Of Service Capable Auditing of Networks
 * Copyright (C) 2003 Florian Weimer
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"
#include "ipv4.h"
#include "opt.h"
#include "proto.h"
#include "results.h"
#include "scan.h"
#include "subnets.h"
#include "ticks.h"

#include "getopt.h"

#include <cstdlib>
#include <cstdio>
#include <cstring>

static void version (void);
static void help ();

int
main (int argc, char **argv)
{
  int c;
  subnets nets;
  bool needs_shuffle = false;

  opt_program = argv[0];

  ticks_init ();
  proto_register_all ();

  while (1) {
    static struct option long_options[] = {
      {"add-burst", required_argument, 0, 'A'},
      {"add-timeout", required_argument, 0, 'a'},
      {"banner", required_argument, 0, 'b'},
      {"connections", required_argument, 0, 'c'},
      {"help", no_argument, 0, 'h'},
      {"file", required_argument, 0, 'f'},
      {"indicator", no_argument, 0, 'i'},
      {"net-errors", no_argument, 0, 'n'},
      {"no-epoll", no_argument, 0, 'E'},
      {"output", required_argument, 0, 'o'},
      {"port", required_argument, 0, 'p'},
      {"protocol", required_argument, 0, 'P'},
      {"receive", required_argument, 0, 'r'},
      {"send", required_argument, 0, 's'},
      {"style", required_argument, 0, 'S'},
      {"timeout", required_argument, 0, 't'},
      {"verbose", no_argument, 0, 'v'},
      {"version", no_argument, 0, 'V'},
      {0, 0, 0, 0}
    };
    int option_index = 0;

    c = getopt_long (argc, argv, "a:A:b:c:Ehino:p:P:r:s:S:t:vV",
                     long_options, &option_index);

    if (c == -1)
      break;

    switch (c) {
    case 'a':
      if ((sscanf (optarg, "%u", &opt_add_timeout) != 1) || (opt_add_timeout == 0)) {
        fprintf (stderr, "%s: illegal add timeout '%s'\n",
                 opt_program, optarg);
        exit(EXIT_FAILURE);
      }
      break;

    case 'A':
      if ((sscanf (optarg, "%u", &opt_add_burst) != 1) || (opt_add_burst == 0)) {
        fprintf (stderr, "%s: illegal add burst '%s'\n",
                 opt_program, optarg);
        exit(EXIT_FAILURE);
      }
      break;

    case 'b':
      if (sscanf (optarg, "%u", &opt_banner_size) != 1) {
        fprintf (stderr, "%s: illegal banner size '%s'\n",
                 opt_program, optarg);
        exit(EXIT_FAILURE);
      }
      break;

    case 'c':
      if ((sscanf (optarg, "%u", &opt_fd_count) != 1) || (opt_fd_count == 0)) {
        fprintf (stderr, "%s: illegal file descriptor count '%s'\n",
                 opt_program, optarg);
        exit(EXIT_FAILURE);
      }
      break;

    case 'E':
      opt_no_epoll = 1;
      break;

    case 'h':
      help ();
      exit (EXIT_SUCCESS);
      break;

    case 'f':
      nets.add_file(optarg);
      // We might have read lots of small prefixes from the file, so
      // better randomize them.
      needs_shuffle = true;
      break;

    case 'i':
      opt_indicator = 1;
      break;

    case 'n':
      opt_net_errors = 1;
      break;

    case 'o':
      opt_output = optarg;
      break;

    case 'p':
      if ((sscanf (optarg, "%u", &opt_port) != 1) || (opt_port == 0) || (opt_port > 65535)) {
        fprintf (stderr, "%s: illegal port '%s'\n",
                 opt_program, optarg);
        exit(EXIT_FAILURE);
      }
      break;

    case 'P':
      opt_protocol = optarg;
      break;

    case 'r':
      opt_receive = optarg;
      break;

    case 's':
      opt_send = optarg;
      break;

    case 'S':
      opt_output_style = optarg;
      break;

    case 't':
      if ((sscanf (optarg, "%u", &opt_connect_timeout) != 1) || (opt_connect_timeout == 0)) {
        fprintf (stderr, "%s: illegal connect timeout '%s'\n",
                 opt_program, optarg);
        exit(EXIT_FAILURE);
      }
      /* Sets all timeouts. */
      opt_read_timeout = opt_connect_timeout;
      opt_write_timeout = opt_connect_timeout;
      break;

    case 'v':
      opt_verbose = 1;
      break;

    case 'V':
      version();
      exit (EXIT_SUCCESS);
      break;

    case '?':
      /* command line parsing error */
      fprintf (stderr, "%s: invoke with --help to see all available options\n",
               opt_program);
      exit (EXIT_FAILURE);

    default:
      abort ();
    }
  }

  proto_select (opt_protocol);
  results_style (opt_output_style);
  results_format (opt_output);

  if (opt_port == 0) {
    fprintf (stderr, "%s: mandatory --port argument is missing\n",
             opt_program);
    exit (EXIT_FAILURE);
  }

  nets.add(argv + optind, argc - optind);
  if (needs_shuffle) {
    nets.shuffle();
  }

  scan(nets);
  results_print();

  return EXIT_SUCCESS;
}

static void
version (void)
{
  puts (PACKAGE_NAME " " PACKAGE_VERSION);
  puts ("Copyright (C) 2003 Florian Weimer.\n");
  puts ("This is free software; see the source for copying conditions.  There is NO");
  puts ("warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.");
}

static void
help (void)
{
  printf ("Usage: %s [OPTION]... {--port | -p} PORT [PREFIX]...\n",
          opt_program);
  puts ("Scan PREFIX for TCP services on port PORT and record responses.\n");

  puts ("Mandatory arguments to long options are mandatory "
        "for short options too.");

  puts ("  -a, --add-timeout=TIMEOUT  add new connections every TIMEOUT milliseconds");
  puts ("  -A, --add-burst=COUNT      add COUNT connections at once");
  puts ("  -b, --banner=COUNT         read at most COUNT bytes from host");
  puts ("  -c, --connections=COUNT    open COUNT parallel connections");
  puts ("  -E, --no-epoll             do not use epoll even if available");
  puts ("  -f, --file FILE            read network prefixes from FILE");
  puts ("  -i, --indicator            show progress indicator");
  puts ("  -n, --net-errors           do not suppress network errors in report");
  puts ("  -o, --output=FORMAT        change output format (see manual page)");
  puts ("  -p, --port=PORT            connect to PORT on remote hosts");
  puts ("  -P, --protocol=PROTOCOL    select protocol module PROTOCOL");
  puts ("  -r, --receive=REGEXP       match received data using REGEXP");
  puts ("  -s, --send=STRING          send STRING to remote host");
  puts ("  -S, --style=STRING         set output option to STRING");
  puts ("  -t, --timeout=TIMEOUT      set read and connect timeout");
  puts ("  -v, --verbose              turn on verbose mode\n");

  puts ("  -h, --help                 display this help message and exit");
  puts ("  -V, --version              output version information and exit");

  puts ("\nAll options besides -p/--port are optional.");

  puts ("\nTIMEOUT is measured in milliseconds.  REGEXP is a Perl-compatible regular");
  puts ("expression.  STRING is a string in C syntax (with escapes for binary");
  puts ("characters");

  puts ("\nReport bugs to <doscan-bugs@lists.enyo.de>.");
}

/* arch-tag: 11aa635c-311b-4044-b391-6554b67a9e66
 */
