/* doscan - Denial Of Service Capable Auditing of Networks       -*- C++ -*-
 * Copyright (C) 2003 Florian Weimer
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * The detection code is based on Nessus plugin 11890:
 *
 *   # This script is (C) Tenable Network Security
 *   # 10/22/2003 updated by KK Liu 10/22/2003
 *   #  - check messenger service, if not on - exit
 *   #  - check Windows OS
 */

#include "config.h"
#include "scan_windows.h"

#include <string>

scan_windows_version::scan_windows_version(event_queue& q, ipv4_t host,
                                           unsigned short port)
  : tcp_half_duplex_handler(q, host, port)
{
}

bool
scan_windows_version::enough_dcom_data(int state, const std::string& data)
{
  if (data.size() < 9) {
    request_more_data(state);
    return false;
  }

  unsigned char length = data[8];
  if (length > data.size()) {
    request_more_data(state, length - data.size());
    return false;
  }

  // FIXME: Maybe we should discard additional data.
  return true;
}

void
scan_windows_version::ready(int state, int error, const std::string& data)
{
  // Different bind requests for determining the Windows version.
  static const char ch_bind_windows_me[] = {
    0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00,
    0x53, 0x53, 0x56, 0x41, 0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0xe6, 0x73, 0x0c, 0xe6,
    0xf9, 0x88, 0xcf, 0x11, 0x9a, 0xf1, 0x00, 0x20, 0xaf, 0x6e, 0x72, 0xf4,
    0x02, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
    0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
  };
  static const std::string bind_windows_me(ch_bind_windows_me,
                                           sizeof(ch_bind_windows_me));

  static const char ch_bind_windows_xp[] = {
    0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00, 0xcc, 0x00, 0x00, 0x00,
    0x84, 0x67, 0xbe, 0x18, 0x31, 0x14, 0x5c, 0x16, 0x00, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0xb8, 0x4a, 0x9f, 0x4d,
    0x1c, 0x7d, 0xcf, 0x11, 0x86, 0x1e, 0x00, 0x20, 0xaf, 0x6e, 0x7c, 0x57,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
    0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x01, 0x00, 0xa0, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 0x00, 0x00, 0x00, 0x00,
    0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
    0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x03, 0x00, 0x01, 0x00,
    0x0a, 0x42, 0x24, 0x0a, 0x00, 0x17, 0x21, 0x41, 0x2e, 0x48, 0x01, 0x1d,
    0x13, 0x0b, 0x04, 0x4d, 0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
    0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
    0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x01, 0x00, 0xb0, 0x01, 0x52, 0x97,
    0xca, 0x59, 0xcf, 0x11, 0xa8, 0xd5, 0x00, 0xa0, 0xc9, 0x0d, 0x80, 0x51,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
    0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
  };
  static const std::string bind_windows_xp(ch_bind_windows_xp,
                                           sizeof(ch_bind_windows_xp));

  static const char ch_request_windows_xp[] = {
    0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00, 0xaa, 0x00, 0x00, 0x00,
    0x41, 0x41, 0x41, 0x41, 0x80, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x05, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x28, 0x63, 0x29, 0x20, 0x75, 0x65, 0x72, 0x84, 0x20, 0x73, 0x73, 0x53,
    0x20, 0x82, 0x80, 0x67, 0x00, 0x00, 0x00, 0x00, 0x80, 0x1d, 0x94, 0x5e,
    0x96, 0xbf, 0xcd, 0x11, 0xb5, 0x79, 0x08, 0x00, 0x2b, 0x30, 0xbf, 0xeb,
    0x01, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x5c, 0x00, 0x5c, 0x00, 0x41, 0x00, 0x00, 0x00,
    0x41, 0x00, 0x41, 0x00, 0x5c, 0x00, 0x43, 0x00, 0x24, 0x00, 0x5c, 0x00,
    0x41, 0x00, 0x2e, 0x00, 0x74, 0x00, 0x78, 0x00, 0x74, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
    0x01, 0x00, 0x00, 0x00, 0x58, 0x73, 0x0b, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x31, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x46, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x07, 0x00,
  };
  static const std::string request_windows_xp(ch_request_windows_xp,
                                              sizeof(ch_request_windows_xp));

  static const char ch_bind_windows_nt[] = {
    0x05, 0x00, 0x0B, 0x03, 0x10, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00,
    0x7F, 0x00, 0x00, 0x00, 0xD0, 0x16, 0xD0, 0x16, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0xA0, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xC0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x5D, 0x88, 0x8a, 0xEB, 0x1C, 0xC9, 0x11,
    0x9F, 0xE8, 0x08, 0x00, 0x2B, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
  };
  static const std::string bind_windows_nt(ch_bind_windows_nt,
                                           sizeof(ch_bind_windows_nt));

  static const char ch_request_windows_nt[] = {
    0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00, 0xaa, 0x00, 0x00, 0x00,
    0x41, 0x41, 0x41, 0x41, 0x80, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x05, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x28, 0x63, 0x29, 0x20, 0x75, 0x65, 0x72, 0x84, 0x20, 0x73, 0x73, 0x53,
    0x20, 0x82, 0x80, 0x67, 0x00, 0x00, 0x00, 0x00, 0x80, 0x1d, 0x94, 0x5e,
    0x96, 0xbf, 0xcd, 0x11, 0xb5, 0x79, 0x08, 0x00, 0x2b, 0x30, 0xbf, 0xeb,
    0x01, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x5c, 0x00, 0x5c, 0x00, 0x41, 0x00, 0x00, 0x00,
    0x41, 0x00, 0x41, 0x00, 0x5c, 0x00, 0x43, 0x00, 0x24, 0x00, 0x5c, 0x00,
    0x41, 0x00, 0x2e, 0x00, 0x74, 0x00, 0x78, 0x00, 0x74, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
    0x01, 0x00, 0x00, 0x00, 0x58, 0x73, 0x0b, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x31, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x46, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x07, 0x00,
  };
  static const std::string request_windows_nt(ch_request_windows_nt,
                                              sizeof(ch_request_windows_nt));

  if (error != 0) {
    // We ignore connect errors.
    if (state >= 1) {
      result(unknown);
    }
    return;
  }

  switch (state) {
  case 0:
    // Check for pending connection error.  FIXME: The base class
    // should do this.
    if (get_error()) {
      return;
    }

    // Send Windows Me bind request.
    send(state + 1, bind_windows_me);
    break;

  case 1:
    // Receive anser to Windows Me bind request.
    if (enough_dcom_data(state, data)) {
      if (data.size() > 24) {
        static const char ref_ch[] = {0x02, 0x00, 0x01, 0x00};
        static const std::string ref(ref_ch, sizeof(ref_ch));

        if (data.substr(data.size() - 24, 4) == ref) {
          result(windows_me);
        } else {
          // Reconnect for Windows XP bind request.
          reconnect(state + 1, false);
        }
      } else {
        result(unknown);
      }
    };
    break;

  case 2:
    // Send Windows XP bind request.
    send(state + 1, bind_windows_xp);
    break;

  case 3:
    // Send real Windows XP request.
    if (enough_dcom_data(state, data)) {
      send(state + 1, request_windows_xp);
    }
    break;

  case 4:
    // Process Windows XP answer.
    if (enough_dcom_data(state, data)) {
      if (data.length() == 32) {
        // The length is not a strong indicator, we have to check for
        // the RPC error code, too.

        static const char xp_err_ch[] = {0x03, 0x00, 0x01, 0x1c};
        static const std::string xp_err(xp_err_ch, sizeof(xp_err_ch));

        static const char unix_err_ch[] = {0x1c, 0x00, 0x00, 0x1c};
        static const std::string unix_err(unix_err_ch, sizeof(unix_err_ch));

        const std::string err(data, data.size() - 8, 4);

        if (err == xp_err) {
          result(windows_xp);
        } else if (err == unix_err) {
          result(unix_dce);
        } else {
          result(other_dce);
        }

      } else {
        reconnect(state + 1, false);
      }
    }
    break;

  case 5:
    // Send Windows NT bind request.
    send(state + 1, bind_windows_nt);
    break;

  case 6:
    // Send real Windows NT request.
    if (enough_dcom_data(state, data)) {
      send(state + 1, request_windows_nt);
    }
    break;

  case 7:
    // Process answer to Windows NT request.
    if (enough_dcom_data(state, data)) {
      if (data.length() == 32) {
        result(windows_nt);
      } else if (data.length() == 36) {
        result(windows_2000);
      } else {
        result(unknown);
      }
    }
    break;

  default:
    abort();
  }
}

// arch-tag: 58e2633a-9bfe-49ff-a0a1-544b5e85bd85
