/* Copyright (C) 2008, 2009 by Intevation GmbH
 * Authors:
 * Bernhard Herzog <bh@intevation.de>
 *
 * This program is free software under the LGPL (>=v2.1)
 * Read the file COPYING coming with the software for details.
 */

#include "lib.h"
#include "dict.h"
#include "common.h"
#include "commands-util.h"
#include "module-context.h"
#include "mailbox-list-private.h"

#include "metadata-plugin.h"

#include <stdlib.h>


#define METADATA_LIST_CONTEXT(obj) \
	MODULE_CONTEXT((obj), metadata_mailbox_list_module)
static MODULE_CONTEXT_DEFINE_INIT(metadata_mailbox_list_module,
				  &mailbox_list_module_register);

struct metadata_mailbox_list {
	union mailbox_list_module_context module_ctx;
};

static void (*metadata_next_hook_mailbox_list_created)(struct mailbox_list *);


static struct dict *metadata_dict;
static bool metadata_allow_private = FALSE;
static bool metadata_debug = FALSE;


bool metadata_private_allowed(void) {
	return metadata_allow_private;
}


static const char *mailbox_key_path(struct mailbox_list *list,
				    const char *mailboxname)
{
	return mailbox_list_get_path(list, mailboxname,
				     MAILBOX_LIST_PATH_TYPE_MAILBOX);
}


static const char *create_dict_key(struct mail_storage *storage,
				   const char *mailboxname, const char *entry,
				   bool private)
{
	return t_strconcat(private ? DICT_PATH_PRIVATE : DICT_PATH_SHARED,
			   mailbox_key_path(mail_storage_get_list(storage),
					    mailboxname),
			   "/", entry, NULL);
}

static const char *renamed_key(const char *oldkey, const char *oldpath,
			       const char *newpath)
{
	const char *first_slash;
	size_t oldlen;

	first_slash = strchr(oldkey, '/');
	if (!first_slash)
		return NULL;

	oldlen = strlen(oldpath);
	if (strncmp(first_slash + 1, oldpath, oldlen) == 0) {
		return t_strconcat(t_strdup_until(oldkey, first_slash + 1),
				   newpath,
				   first_slash + 1 + oldlen,
				   NULL);
	}

	return NULL;
}

static void rename_metadata_of_mailbox(struct dict_transaction_context *dt,
				       struct mailbox_list *list,
				       const char *oldname,
				       const char *newname)
{
	struct dict_iterate_context *iter;
	const char *oldpath;
	const char *newpath;
	const char *key;
	const char *value;
	const char *newkey;

	oldpath = mailbox_key_path(list, oldname);
	newpath = mailbox_key_path(list, newname);

	iter = dict_iterate_init(metadata_dict,
				 t_strconcat(DICT_PATH_SHARED, oldpath, NULL),
				 DICT_ITERATE_FLAG_RECURSE
				 | DICT_ITERATE_FLAG_SORT_BY_KEY);
	while (dict_iterate(iter, &key, &value) > 0) {
		T_BEGIN {
			newkey = renamed_key(key, oldpath, newpath);
			dict_set(dt, newkey, value);
			dict_unset(dt, key);
		} T_END;
	}
	dict_iterate_deinit(&iter);
}

static int rename_metadata_of_children(struct dict_transaction_context *dt,
				       struct mailbox_list *list,
				       const char *oldname,
				       const char *newname)
{
	struct mailbox_list_iterate_context *iter;
	const struct mailbox_info *info;
	const char *pattern;
	size_t oldnamelen;
	int ret;

	ret = 0;

	oldnamelen = strlen(oldname);
	pattern = t_strdup_printf("%s%c*", oldname,
				  mailbox_list_get_hierarchy_sep(list));
	iter = mailbox_list_iter_init(list, pattern,
				      MAILBOX_LIST_ITER_RETURN_NO_FLAGS);
	while ((info = mailbox_list_iter_next(iter)) != NULL) {
		const char *renamed;

		/* verify that the prefix matches, otherwise we could have
		   problems with mailbox names containing '%' and '*' chars */
		if (strncmp(info->name, oldname, oldnamelen) == 0 &&
		    info->name[oldnamelen] ==
		    mailbox_list_get_hierarchy_sep(list)) {
			T_BEGIN {
				renamed = t_strconcat(newname,
						      info->name + oldnamelen,
						      NULL);
				rename_metadata_of_mailbox(dt, list,
							   info->name, renamed);
			} T_END;
		}
	}

	if (mailbox_list_iter_deinit(&iter) < 0) {
		ret = -1;
	}

	return ret;
}

static int metadata_mailbox_list_rename(struct mailbox_list *list,
					const char *oldname,
					const char *newname)
{
	struct metadata_mailbox_list *mlist = METADATA_LIST_CONTEXT(list);
	struct dict_transaction_context *dt;
	int success;

	dt = dict_transaction_begin(metadata_dict);

	T_BEGIN {
		rename_metadata_of_mailbox(dt, list, oldname, newname);
		rename_metadata_of_children(dt, list, oldname, newname);
	} T_END;


	success = mlist->module_ctx.super.rename_mailbox(list, oldname,
							 newname);
	if (success < 0) {
		dict_transaction_rollback(&dt);
	} else {
		if (dict_transaction_commit(&dt) < 0) {
			i_error("metadata_mailbox_list_rename:"
				" could not update metadata dict");
		}
	}

	return success;
}

static int metadata_mailbox_list_delete(struct mailbox_list *list,
					const char *name)
{
	struct metadata_mailbox_list *mlist = METADATA_LIST_CONTEXT(list);
	struct dict_transaction_context *dt;
	struct dict_iterate_context *iter;
	const char *keypath;
	const char *key;
	const char *value;
	int success;

	keypath = mailbox_key_path(list, name);

	dt = dict_transaction_begin(metadata_dict);
	iter = dict_iterate_init(metadata_dict, t_strconcat(DICT_PATH_SHARED,
							    keypath, NULL),
				 DICT_ITERATE_FLAG_RECURSE
				 | DICT_ITERATE_FLAG_SORT_BY_KEY);
	while (dict_iterate(iter, &key, &value) > 0) {
		dict_unset(dt, key);
	}
	dict_iterate_deinit(&iter);

	success = mlist->module_ctx.super.delete_mailbox(list, name);
	if (success < 0) {
		dict_transaction_rollback(&dt);
	} else {
		if (dict_transaction_commit(&dt) < 0) {
			i_error("metadata_mailbox_list_delete:"
				" could not update metadata dict");
		}
	}

	return success;
}


static void metadata_mailbox_list_created(struct mailbox_list *list)
{
	struct metadata_mailbox_list *mlist;

	mlist = p_new(list->pool, struct metadata_mailbox_list, 1);
	mlist->module_ctx.super = list->v;
	list->v.rename_mailbox = metadata_mailbox_list_rename;
	list->v.delete_mailbox = metadata_mailbox_list_delete;
	MODULE_CONTEXT_SET(list, metadata_mailbox_list_module, mlist);

	if (metadata_next_hook_mailbox_list_created != NULL)
		metadata_next_hook_mailbox_list_created(list);
}


bool metadata_get_metadata_entry(struct client_command_context *cmd,
				 const char *mailboxname, const char *entry,
				 const char **value_r, bool private)
{
	int success;
	struct mail_storage *storage;
	const char *key;

	if (metadata_debug)
		i_info("metadata_get_metadata_entry: mailboxname=%s, entry=%s,"
		       " private=%d", mailboxname, entry, private);

	if (!client_verify_mailbox_name(cmd, mailboxname,
					CLIENT_VERIFY_MAILBOX_SHOULD_EXIST))
		return FALSE;

	storage = client_find_storage(cmd, &mailboxname);
	if (!storage)
		return FALSE;

	key = create_dict_key(storage, mailboxname, entry, private);
	if (metadata_debug)
		i_info("metadata_get_metadata_entry: dict key=%s", key);

	success = dict_lookup(metadata_dict, cmd->pool, key, value_r);

	if (success == 0) {
		*value_r = NULL;
	} else if (success < 0) {
		client_send_tagline(cmd, "NO Lookup failed.");
	}

	return success >= 0;
}

bool metadata_set_metadata_entry(struct client_command_context *cmd,
				 const char *mailboxname, const char *entry,
				 const char *value, bool private)
{
	struct dict_transaction_context *dt;
	struct mail_storage *storage;
	const char *key;

	if (metadata_debug)
		i_info("metadata_set_metadata_entry: mailboxname=%s, entry=%s,"
		       " value=%s, private=%d", mailboxname, entry,
		       value != NULL ? value : "<NULL>", private);

	if (!client_verify_mailbox_name(cmd, mailboxname,
					CLIENT_VERIFY_MAILBOX_SHOULD_EXIST))
		return FALSE;

	storage = client_find_storage(cmd, &mailboxname);
	if (!storage)
		return FALSE;

	key = create_dict_key(storage, mailboxname, entry, private);
	if (metadata_debug)
		i_info("metadata_set_metadata_entry: dict key=%s", key);

	dt = dict_transaction_begin(metadata_dict);

	if (value != NULL) {
		dict_set(dt, key, value);
	} else {
		dict_unset(dt, key);
	}

	if (dict_transaction_commit(&dt) < 0)
	{
		client_send_tagline(cmd, "NO Setting meta-data failed.");
		return FALSE;
	}

	return TRUE;
}

void metadata_plugin_init(void)
{
	const char *username;
	const char *dict_uri;
	const char *base_dir;

	username = getenv("USER");
	if (username == NULL)
		i_fatal("metadata plugin: USER unset");

	dict_uri = getenv("METADATA_DICT");
	if (dict_uri == NULL)
		i_fatal("metadata plugin: METADATA_DICT unset");

	metadata_allow_private = getenv("METADATA_ALLOW_PRIVATE") != NULL;
	metadata_debug = getenv("METADATA_DEBUG") != NULL;

	base_dir = getenv("BASE_DIR");
	if (base_dir == NULL)
	    base_dir = PKG_RUNDIR;
	metadata_dict = dict_init(dict_uri, DICT_DATA_TYPE_STRING, username,
				  base_dir);

	if (metadata_dict == NULL)
		i_fatal("metadata plugin: dict_init() failed");

	metadata_next_hook_mailbox_list_created = hook_mailbox_list_created;
	hook_mailbox_list_created = metadata_mailbox_list_created;
}

void metadata_plugin_deinit(void)
{
	if (metadata_dict != NULL) {
		dict_deinit(&metadata_dict);
	}
	if (metadata_next_hook_mailbox_list_created) {
		hook_mailbox_list_created =
			metadata_next_hook_mailbox_list_created;
	}
}
