/* Drip - a transcoder for Unix
 * Copyright (C) 2001-2003 Jarl van Katwijk
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


/*
    Various bitmap colourspace conversion routines:
    - I420toRGB
    - RGBtoI420


*/


#include "../config.h"
#include "../src/drip.h"

#define SCALEBITS 8
#define ONE_HALF  (1 << (SCALEBITS - 1))
#define FIX(x)          ((int) ((x) * (1L<<SCALEBITS) + 0.5))

/* Lookup tables */
gdouble T1164[256];
gdouble T2018[256];
gdouble T0813[256];
gdouble T1596[256];
gdouble T0391[256];
gint T29900[256*4];
gint T58700[256*4];
gint T11400[256*4];
gint T16874[256*4];
gint T33126[256*4];
gint T50000[256*4];
gint T41869[256*4];
gint T08131[256*4];


/* Init done flag */
static gboolean init_done = FALSE;


/* Limit colour to valid range */
inline guint8 normalize_colour(gdouble colour) {
    static guint colour_norm;

    if (colour<0) {
        colour_norm = 0;
    } else if (colour>255) {
        colour_norm = 255;
    } else {
        colour_norm = (guint8)(colour);
    }
    return colour_norm;
}
       

/* TODO no chroma interpolating is done atm */
void I420toRGB24(guint8 *dst, guint8 *src, gint width, gint height) {
    register gint offset = 0;
    register gint offset4 = 0;
    register gint offset3 = 0;
    static gdouble offset4_f;
    register guint8 Y,CR,CB;
    register gdouble Bf,Gf,Rf;
    static guint8 *src0,*src1,*src2;
    register gdouble T1164_Y;
    gint WH = width*height;
    
    /* Init */
    src0 = src;
    src1 = (guint8*)((glong)src0+WH);
    src2 = (guint8*)((glong)src1+WH/4);
    offset = 0;
    offset3 = 0;

    /* Loop */
    for (register gint y=0;y<height;y++) {
        offset4_f = (y>>1) * (width>>1);
        for (register gint x=0;x<width;x++) {
             offset4 = (gint)offset4_f;

             Y  = src0[offset];
             CB = src1[offset4];
             CR = src2[offset4];
 
             T1164_Y = T1164[Y];
             Bf = T1164_Y + T2018[CB];
             Gf = T1164_Y - T0813[CR] - T0391[CB];
             Rf = T1164_Y + T1596[CR];
 
             dst[offset3++] = normalize_colour(Rf); //(B) + (G<<8) + (R<<16); // RGB 24bits
             dst[offset3++] = normalize_colour(Gf);
             dst[offset3++] = normalize_colour(Bf);
 
             offset++;
             offset4_f = offset4_f+0.5;
        }
    }

    /* Clean & Exit */
    return;
}   

void I420toRGB24_init(void) {
    /* Calculate lookup tables */
    for (gdouble i=-16;i<240;i++) {
        T1164[(gint)(i+16)] = i*1.164;
    }
    for (gdouble i=-128;i<128;i++) {
        T2018[(gint)(i+128)] = i*2.018;
        T0813[(gint)(i+128)] = i*0.813;
        T1596[(gint)(i+128)] = i*1.596;
        T0391[(gint)(i+128)] = i*0.391;
    }
    init_done = TRUE;
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"I420toRGB24 initialized");
    return;
}


/* TODO no chroma interpolating is done atm */
void RGB24toI420(guint8 *dst, guint8 *src, gint width, gint height) {
    register gint wrap, wrap3, x, y;
    register gint r, g, b, r1, g1, b1;
    register guint8 *cb, *cr;
    gint WH = width*height;

    cb = (guint8*)((glong)dst + WH);
    cr = (guint8*)((glong)cb + WH/4);
  
    wrap = width;
    wrap3 = width+width+width;
    for(y=0;y<height;y+=2) {
        for(x=0;x<width;x+=2) {

            r = src[0];
            g = src[1];
            b = src[2];
            r1 = r;
            g1 = g;
            b1 = b;
            dst[0] = (T29900[r] + T58700[g] + T11400[b] + ONE_HALF) >> SCALEBITS;
            r = src[3];
            g = src[4];
            b = src[5];
            r1 += r;
            g1 += g;
            b1 += b;
            dst[1] = (T29900[r] + T58700[g] + T11400[b] + ONE_HALF) >> SCALEBITS;

            src += wrap3;
            dst += wrap;
            r = src[0];
            g = src[1];
            b = src[2];
            r1 += r;
            g1 += g;
            b1 += b;
            dst[0] = (T29900[r] + T58700[g] + T11400[b] + ONE_HALF) >> SCALEBITS;
            r = src[3];
            g = src[4];
            b = src[5];
            r1 += r;
            g1 += g;
            b1 += b;
            dst[1] = (T29900[r] + T58700[g] + T11400[b] + ONE_HALF) >> SCALEBITS;

            cb[0] = (guint8)(((- T16874[r1] - T33126[g1] + T50000[b1] + 4*ONE_HALF - 1)>>(SCALEBITS+2)) + 128);
            cr[0] = (guint8)(((  T50000[r1] - T41869[g1] - T08131[b1] + 4*ONE_HALF - 1)>>(SCALEBITS+2)) + 128);

            cb++;
            cr++;
            src += -wrap3 + 6; //2 * 3;
            dst += -wrap + 2;
        }
        src += wrap3;
        dst += wrap;
    }
    return;
}  

void RGB24toI420_init(void) {
    /* Calculate lookup tables */
    for (gint i=0;i<256*4;i++) {
        T29900[i] = ((gint)(0.29900*256+0.5) * i);
        T58700[i] = ((gint)(0.58700*256+0.5) * i);
        T11400[i] = ((gint)(0.11400*256+0.5) * i);
        T16874[i] = ((gint)(0.16874*256+0.5) * i);
        T33126[i] = ((gint)(0.33126*256+0.5) * i);
        T50000[i] = ((gint)(0.50000*256+0.5) * i);
        T41869[i] = ((gint)(0.41869*256+0.5) * i);
        T08131[i] = ((gint)(0.08131*256+0.5) * i);
    }
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"RGB24toI420 initialized");
    return;
}

