/* Drip - a transcoder for Unix
 * Copyright (C) 2001-2003 Jarl van Katwijk
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

// ----------------------------------------------------------------------------------
//
// noisefilter, removes video noise for better compression quality.
//
// 26-10-2002 - Jarl van Katwijk
//
// ----------------------------------------------------------------------------------


#include "../config.h"
#include "../src/drip.h"
#include "../encoder/plugin-loader.hh"
#include <gmodule.h>

// Custom includes:
// none

#undef DEBUG


static gboolean init_done = FALSE;                 // Plugin initialised flag
static gint width,height;                          // Input Width\Height sizes
static gint xoff,yoff,xlen,ylen;                   // Offset and lenght values
static gdouble framerate;                           // Framerate (25.00, 29.97, etc)
static gdouble value1,value2,value3,value4,value5;  // Plugin defined values - these are loaded\saved by Drip
static guint8 *src_planes[3];                      // Input data planes (I420 colour space)
static guint8 *dst_planes[3];                      // Output data planes
// Custom vari's
#define MAX_SRATE 50000
#define MAX_CHANNELS 2
#define BYTES_PS 2
#define MAXPREVS 20
#define BANDS 8  /* The more bands, the better, up to a point.  
                    The quality of the filter algorithm imposes a limit */
gint oneside= 0;
gint32 threshold[BANDS];
gint filtcount[BANDS-1] = {10, 8, 6, 4, 3, 2, 1}; /* number of voxels back for filter */
static GtkObject  *filter_threshold_adj;
/* one per channel per band... */
typedef struct {
    #define SQUELCHMULT 11000
    gint32 prevs[MAXPREVS];
    /*   ...Previous voxels, averaged in to filter apart bands.
         One between each pair of adjacent bands */
    gulong squelchgain;
    /* ...*SQUELCHMULT allows less obstrusive squelch switching */
    guchar squelching;
} FilterABand;
/* one per channel... */
typedef struct {
    FilterABand B[BANDS];
} FilterAChannel;
FilterAChannel Channel[2];



extern "C" {


/* Return type (SPU,AUDIO,VIDEO) of filter */
G_MODULE_EXPORT module_type_e noisefilter_type(void) {
    return AUDIO;
}


/* Return phase (PRE,NORMAL,POST) of filter */
G_MODULE_EXPORT module_phase_e noisefilter_phase(void) {
    return POST;
}



/* here from gui.c */
void filter_set_threshold(gint athreshold) {
  gint band;

  /* take the exponent, more or less.. */
  gint32 ethreshold= 1;  ethreshold <<= (athreshold/2);
  if (athreshold& 1)  ethreshold= (ethreshold*3)/2;
  //printf("\nfilter_set_threshold(%i)\n", ethreshold);
  threshold[0]= 0;
  threshold[1]= ethreshold / 3;
  threshold[2]= ethreshold / 2;
  for (band= 3; band< BANDS; band++)
    threshold[band]= ethreshold;
}

static void noise_changed(GtkWidget *w, gpointer data) {
    value1 = GTK_ADJUSTMENT(filter_threshold_adj)->value; // bext_level
}


/* Function which implements a GtkPlug to let user configure this plugin */
G_MODULE_EXPORT GtkWidget* noisefilter_gui(void) {
    static GtkWidget *table = NULL;
    GtkWidget *button, *label, *hscale, *bbox;

    #ifdef DEBUG
    g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Plugin: noisefilter gui");
    #endif

    filter_threshold_adj = gtk_adjustment_new(value1, 0, 20 + 10, 2, 10, 10);

    table = gtk_table_new(2, 3, FALSE);
    gtk_table_set_col_spacings(GTK_TABLE(table), 5);
    gtk_container_set_border_width(GTK_CONTAINER(table), 5);
    gtk_widget_show(table);

    label = gtk_label_new("Threshold: ");
    gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);
    gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3, GTK_FILL, GTK_FILL, 0, 0);
    gtk_widget_show(label);


    hscale = gtk_hscale_new(GTK_ADJUSTMENT(filter_threshold_adj));
    gtk_widget_set_usize(hscale, 400, 35);
    gtk_scale_set_digits(GTK_SCALE(hscale), 0);
    gtk_table_attach_defaults(GTK_TABLE(table), hscale, 1, 2, 2, 3);
    gtk_signal_connect(GTK_OBJECT(hscale), "motion_notify_event", GTK_SIGNAL_FUNC(noise_changed), NULL);
    gtk_widget_show(hscale);

    /* Clean & Exit */
    return table;
}


/* This function is called for every (audio\video) frame.
   Here the actual filtering is done. */
G_MODULE_EXPORT guint8** noisefilter_apply(guint8** src, glong size, gulong SCR, gulong PTS) {
    #define PRECISION 256
    #define HEADROOM 128
    gint i;
    gint16 *data = (gint16*)src[0];
    static gint old_srate, old_nch;
    #ifdef DEBUG
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Plugin: noisefilter apply (PTS=%lu,SCR=%lu)",PTS,SCR);
    #endif

  for (i = 0; i < size/2; i+= 2)  {
    int channel, band;
    for (channel= 0; channel< 2; channel++) {
      gint32 in, data_, inf[BANDS];
      /* ...input voxels are 16 bits, but
         we'll use 32 to avoid rounding errors */
      in = data[i+channel]* PRECISION;   /* for better precision */

      data_= 0;
      if (channel && oneside) { /* do only one channel.  good for testing. */
        /* do nothing */
      }  else {
        register int j;

        /* filter each frequency band... */
        for (band=0; band< BANDS; band++) {
          gint32 out_; /* output voxel for this band */
          FilterABand* pB= &Channel[channel].B[band];
          register gint32 inf_;
          gint32 threshold_= threshold[band]* PRECISION;
          gint32* pprevs= pB->prevs;
          unsigned long squelchgain= pB->squelchgain;
          unsigned long safesquelch;
          if (band< (BANDS-1)) {
            register gint32 in_;
            register gint32 * pprev_;  int j0;
            inf_= inf[band];
            in_ = band? inf_ : in; /* previous filter's output */
            /* ...voxel input to the filter */
            inf_= in_; /* the lowpass part */
            j0= MAXPREVS-filtcount[band];  pprev_= &pprevs[j0];
            for (j= j0; j<MAXPREVS; j++)
              { inf_+= *pprev_;  pprev_++; }
            inf_ /= filtcount[band]+1;
            inf[band]= inf_;
            inf[band+1]= in_ - inf_;  /* the highpass part */
          }

          inf_ = inf[band];
          out_= inf_;
          if (!pB->squelching) {
            if (squelchgain< SQUELCHMULT)
              squelchgain+= SQUELCHMULT/256; /* come out of squelch quickly
                                               so we don't blunt attacks */
            if ((inf_< threshold_) && (inf_> -threshold_)) {
              /* squelchable voxel */
              pB->squelching= 1;
              if (squelchgain) squelchgain--;
            }
          }
          if (pB->squelching) {
            if (squelchgain) squelchgain--;
            if ((inf_> threshold_) || (inf_< -threshold_)) {
              pB->squelching= 0;  /* kick out of squelch mode */
            }
          }

          safesquelch= squelchgain*HEADROOM;  safesquelch/= SQUELCHMULT;
          out_ *= safesquelch;  out_/= HEADROOM;

          pB->squelchgain= squelchgain;
          data_ += out_;  /* reassemble the bands here, after filtering */

          if (band< (BANDS-1)) {
            /* shift the prevs for next time... */
            register gint32 * pprev_, * pprev_1;  int j0;
            j0= MAXPREVS-filtcount[band];
            pprev_= &pprevs[j0]; pprev_1= pprev_+1;
            for (j= j0; j< MAXPREVS-1; j++)
              { *pprev_= *pprev_1;  pprev_++; pprev_1++; }
            pprevs[MAXPREVS-1]= band? inf[band-1] : in;
          }
        }

        data[i+ channel] = data_ / PRECISION;  /* reverse *PRECISION above */

      }
    }
    }


    /* Clean & Exit */
    return src;             // returning input data, here dst=src, but dst!=src is OK too
}


G_MODULE_EXPORT void noisefilter_init(gint w, gint h, 
                                     gint xo, gint yo, gint xl, gint yl,
                                     gdouble fr) {
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Plugin: noisefilter init");
    /* Copy values */
    width = w; height = h;
    xoff = xo; yoff = yo; xlen = xl; ylen = yl;
    framerate = fr;
    /* Filter specific initialisation */
    //TODO

    /* Set init flag */
    init_done = TRUE;
    /* Clean & Exit */
    return;
}


/* Called when parameters need tobe reset to default, auto called when plugin used for 1st time */
G_MODULE_EXPORT void noisefilter_default(void) {
    g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Plugin: noisefilter default");
    /* Set value's to default */
    value1 = 9;
    value2 = 0;
    value3 = 0;
    value4 = 0;
    value5 = 0;
    /* Clean & Exit */
    return;
}


/* Called after encoding has ended. Variable can be reset etc. */
G_MODULE_EXPORT void noisefilter_destruct(void) {
    g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Plugin: noisefilter destruct");
    /* Clean & Exit */
    return;
}






/* ------------------------------------------------------------------------ */
/* FIXED FUNCTIONS AFTER THIS POINT, DONT EDIT, ONLY CHANGE FUNCTION NAMES! */
/* ------------------------------------------------------------------------ */


/* Return values of plugin (value1,value2,... value5) */
G_MODULE_EXPORT void noisefilter_values(gdouble* values) {
    values[0] = value1;
    values[1] = value2;
    values[2] = value3;
    values[3] = value4;
    values[4] = value5;
    // etc, there are VALUES values loaded\saved, see ./encoder/plugin-loader.hh
    return;
}

/* Set plugins values */
G_MODULE_EXPORT void noisefilter_values_function(gdouble *v) {
    value1 = v[0];
    value2 = v[1];
    value3 = v[2];
    value4 = v[3];
    value5 = v[4];
    return;
}

/* g_module_check_init is automatically executed upon loading */
G_MODULE_EXPORT const gchar* g_module_check_init(GModule *module) {
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Plugin: noisefilter selftest passed");
    return NULL;
}

} //extern "C"


