/* Drip - a transcoder for Unix
 * Copyright (C) 2001-2003 Jarl van Katwijk
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "../config.h"
#include <gnome.h>
#include "interface.h"
#include "support.h"
#include "drip.h"
#include <dirent.h>
#include <sys/stat.h>
#include <unistd.h>
#include <pthread.h>
#include <math.h>
#include <netdb.h>
#include <sys/types.h>
#include <ctype.h>
#include <fcntl.h>
#include <errno.h>
#include "callbacks.h"
#include "process.h"
#include <sys/time.h>
#include "../encoder/external.hh"
#include "../encoder/plugin-loader.hh"
#include "external.h"
// avifile version
#include <version.h>


// extra loggin
//#define DEBUG
#ifdef DEBUG
#warning         -------------------------------------
#warning
#warning
#warning
#warning
#warning
#warning         DEBUGGING WILL PRODUCE HUGE LOG FILES
#warning
#warning
#warning
#warning
#warning
#warning         -------------------------------------
#endif


/* local proto's */
void avifile_scan(void);

/* globals */
gboolean booted = FALSE;
gboolean logging_locked = FALSE;
GdkFont *logfont_debug,*logfont_warning,*logfont_error,*logfont_critical,*logfont_info,*logfont_message;
GdkColor col_debug,col_warning,col_error,col_critical,col_info,col_message;
struct timeval encoding_tv;
struct timezone encoding_tz={0,0};
gboolean encoding_time = FALSE;
gint log_file;
//drip_config Config;
GString *status_action;
guint status_perc;
gint linenr = 0;
guint log_handler_drip_id,log_handler_drip_cb_id;
pthread_t serverorbthread;
gint serverorbthread_id;
gboolean server_orb = FALSE;
GtkWidget *mainwindow;
GtkWidget *framewindow = NULL;
GtkWidget *druidtest = NULL;
// gpointer currentframebuffer;
gint xd = 0;
gboolean making_iso;
gboolean encoding;
gboolean encoding_paused;
gboolean stopped;
pthread_t encoderthread;
gint encoderthread_id;
GString *previous_location;
GMutex *LOCKED = NULL;
GCond *LOCKEDcondition = NULL;
gboolean display_unlock = FALSE;


/* GUI ENTRIES FOR ENCODER IMPLEMENTATIONS */

void gui_ready() {
    /* Done encoding, reset and notice user */
    GString *command = g_string_new("");

    /* Rename generated files */
    rename_divx_files();

    /* Notice user */
    g_log(DRIP_LD,G_LOG_LEVEL_MESSAGE,"-----   Encoding avi done   -----");

    /* Dealloc progress frame */
    drip_lock("GUI ready1");
    if (gpixmap!=NULL) {
        gtk_widget_destroy(gpixmap);
        gpixmap = NULL;
    }
    drip_unlock("GUI ready1");

    /* Hide progres frame buttons */
    drip_lock("GUI ready3");
    gtk_widget_hide(hbox35);
    drip_unlock("GUI ready3");

    /* Make ISO images out of generated DIVX's */
    if (Config.create_iso == TRUE) {
        pthread_t isothread;
        gint isothread_id;
        making_iso = TRUE;
        isothread_id = pthread_create(&isothread,NULL,make_iso_images,NULL);
        while (making_iso==TRUE) usleep(100000);
    }

    /* Set GUI statusbar */
    drip_lock("GUI ready2");
    gnome_appbar_set_status(GNOME_APPBAR(appbar1),"   Ready.");
    gnome_appbar_set_progress(GNOME_APPBAR(appbar1),0);
    drip_unlock("GUI ready2");

    /* Mark converting status */
    converting = FALSE;

    /* Clean & exit */
    g_string_free(command,TRUE);
    return;
}

void gui_progress(glong frame,glong total) {
    /* Display progress to user */
    static glong total_previous = 1000;
    GString *frames = g_string_new("");
    gdouble base = 1;//totalvobs;
    gdouble perc = 1;//totalvobs;
    gdouble part = 0;
    static gdouble integral = 1;
    GString *eta;
    struct timeval current_tv;
    struct timezone current_tz={0,0};
    gulong seconds_left = 59;
    gulong minutes_left = 59;
    gulong hours_left = 99;
    glong currentvob = 0;

    if (total_previous>total) {
        /* get start time */
        gettimeofday(&encoding_tv,&encoding_tz);
        encoding_time = TRUE;
    }
    total_previous = total;

    if (total>0) {
        /* Calc perc */
        if (perc>0) {
            perc = 1/perc*(currentvob+1);
            if (total>0) {
                part = perc*((gdouble)frame/(gdouble)total);
            }
        }
        if (base>0) {
            base = 1/base*currentvob;
        }

        /* Check if progress makes any sense, correct if not */
        if (perc>1)    { perc = 1;}
        if (part>perc) { part = perc;}
        if (part<base) { part = base;}
        if (part==0)   { part = 0.0000001;};

        /* Build ETA time string */
        if (encoding_time == FALSE) {
            /* get start time */
            gettimeofday(&encoding_tv,&encoding_tz);
            encoding_time = TRUE;
        }
        gettimeofday(&current_tv,&current_tz);
        seconds_left = (gulong)((current_tv.tv_sec - encoding_tv.tv_sec) * (1-part) / part);

        hours_left = (guint)(seconds_left/3600);
        seconds_left -= 3600*hours_left;

        minutes_left = (guint)(seconds_left/60);
        seconds_left -= 60*minutes_left;

        eta = g_string_new("");
        if (hours_left>0) {
            g_string_sprintf(eta,"%li hours %li minutes",hours_left,minutes_left);
        } else {
            g_string_sprintf(eta,"%li minutes %li seconds",minutes_left,seconds_left);
        }

        /* Show */
        drip_lock("GUI progress1");
        if (total==999999) {
            g_string_sprintf(frames,"   Frame %li (%.2ffps)      Time Remaining Unknown",frame,(gdouble)(frame/(gdouble)(current_tv.tv_sec - encoding_tv.tv_sec)));
            part = modf(part * (gdouble)1500,&integral);
            gnome_appbar_set_progress(GNOME_APPBAR(appbar1),part);
        } else {
            g_string_sprintf(frames,"   %.1f%%    Frame %li/%li (%.2ffps)    Remaining estimated %s",part*100,frame,total,(gdouble)(frame/(gdouble)(current_tv.tv_sec - encoding_tv.tv_sec)),eta->str);
            gnome_appbar_set_progress(GNOME_APPBAR(appbar1),part);
        }
        gnome_appbar_set_status(GNOME_APPBAR(appbar1),frames->str);
        drip_unlock("GUI progress1");
        g_string_free(eta,TRUE);
    } else {
        g_string_sprintf(frames,"Encoding frame %li",frame);
        drip_lock("GUI progress2");
        gnome_appbar_set_status(GNOME_APPBAR(appbar1),frames->str);
        gnome_appbar_set_progress(GNOME_APPBAR(appbar1),part);
        drip_unlock("GUI progress2");
    }

    /* Clean & exit */
    g_string_free(frames,TRUE);
    return;
}

/* Show progress frame that's been generate by the backend every N frames */
void gui_frame(gpointer buffer,glong frame,glong x,glong y) {
    gint yd;
    GString *frame_filename = g_string_new("");

    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"New progress frame %li is available (%li,%li)",frame,x,y);
    drip_lock("GUI frame1");

    /* Clean previous frame */
    if (gpixmap!=NULL) {
        gtk_widget_destroy(gpixmap);
    }

    if (Config.progres_size < (gdouble)0) {
        /* Dont draw progres frame, exit */
        gpixmap = NULL;
    } else {
        /* Build new pixmap widget */
        xd = hbox_main->allocation.width;                  // ADAPTIVE SIZE
        yd = (gint)(y*((gdouble)xd/x));
        gpixmap = (GtkWidget*)gnome_pixmap_new_from_rgb_d_at_size((guchar*)buffer,(guchar*)NULL,x,y,xd*Config.progres_size,yd*Config.progres_size); // TODO: ASCPECT
        gtk_box_pack_start (GTK_BOX (hbox_main), gpixmap, FALSE, FALSE, 0);
        gtk_widget_show(gpixmap);
    }
    /* Clean Exit */
    g_string_free(frame_filename,TRUE);
    drip_unlock("GUI frame1");
    return;
}

void gui_errorsignal(glong signal) {
    g_log(DRIP_LD,G_LOG_LEVEL_CRITICAL,"BACKEND: Signal %li has occured! Please send a detailed bug report to the Drip Project",signal);
    return;
}

/* Searches the /etc/fstab for either a cdrom or dvd entry 
 * If it finds one it returns the device for it. If not, 
 * default to /dev/cdrom ( a safe bet )
 */
gchar* detect_dvdrom() {
    GString * device;
    gint result;
    device = g_string_new("");
	
    /* Look for a device in mtab  */
    device = drip_system("grep dvd /etc/fstab | awk '{print $1}'",&result,FALSE,TRUE);
    if (device->len==0 || strstr(device->str,"{empty output}")!=NULL) {
        /* Couldn't find anything. Safe default */
        g_string_sprintf(device,"%s","/dev/cdrom");
    }
    return device->str;
}

void log_handler(const gchar *log_domain,GLogLevelFlags log_level,const gchar *message,gpointer user_data) {
    /* Handles logging. See glib logging system for more info */
    time_t Time = time(NULL);
    static GString *date = NULL;
    static GString *domain;
    static GString *level;
    static GString *logstring;
    static GString *displaystring;
    gboolean quit;
    gint bytes = 0;
    gchar *last_char;
    GdkColor *colour = NULL;
    GdkFont *logfont = NULL;

    if (date == NULL) {
        date = g_string_new("");
        domain = g_string_new("");
        level = g_string_new("");
        logstring = g_string_new("");
        displaystring = g_string_new("");
    }
    /* Check log domain */
    if (strcmp(log_domain,DRIP_LD)!=0 && strcmp(log_domain,DRIP_CB_LD)!=0) {
        g_error("Got a log for domain %s, which does not belong to the drip log domain.",log_domain);
    }

    /* Lock if not logging from a CallBack function */
    if (strcmp(log_domain,DRIP_LD)==0) {
        drip_lock("log handler1");
    }

    /* Format domain */
    domain = g_string_assign(domain,log_domain);
    while (domain->len < 8) {
        domain = g_string_append(domain," ");
    }

    /* Handle log_level, strip flags */
    log_level = (GLogLevelFlags)(log_level & 252);
    switch (log_level) {
        case G_LOG_LEVEL_ERROR:      g_string_sprintf(level,"Error");
                                     colour = &col_error;
                                     logfont = logfont_error;
                                     quit = TRUE;
                                     break;
        case G_LOG_LEVEL_CRITICAL:   g_string_sprintf(level,"Critical");
                                     colour = &col_critical;
                                     logfont = logfont_critical;
                                     quit = TRUE;
                                     break;
        case G_LOG_LEVEL_WARNING:    g_string_sprintf(level,"Warning");
                                     colour = &col_warning;
                                     logfont = logfont_warning;
                                     quit = FALSE;
                                     break;
        case G_LOG_LEVEL_MESSAGE:    g_string_sprintf(level,"Message");
                                     colour = &col_message;
                                     logfont = logfont_message;
                                     quit = FALSE;
                                     break;
        case G_LOG_LEVEL_INFO:       g_string_sprintf(level,"Info");
                                     colour = &col_info;
                                     logfont = logfont_info;
                                     quit = FALSE;
                                     break;
        case G_LOG_LEVEL_DEBUG:      g_string_sprintf(level,"Debug");
                                     colour = &col_debug;
                                     logfont = logfont_debug;
                                     quit = FALSE;
                                     break;
        default:                     g_string_sprintf(level,"Unknown");
                                     quit = FALSE;
                                     break;
    }

    /* Text formatting */
    while (level->len < 8) {
        level =  g_string_append(level," ");
    }

    /* Format date string */
    g_string_sprintf(date,ctime(&Time));
    date = g_string_truncate(date,(date->len)-1);

    /* Store log entry */
    g_string_sprintf(logstring,"%s | %i\t| %s | %s\n",date->str,linenr,level->str,message);
    g_string_sprintf(displaystring,"%s | %s\n",date->str,message);
    last_char = (gchar*)((glong)logstring->str + logstring->len - 2); /* we got 2x \n? */
    if (*last_char == 10) {
        logstring = g_string_truncate(logstring,logstring->len -1);
    }
    bytes = write(log_file,logstring->str,logstring->len); /* write log entry to log file */
    /* Write criticals to tty and new dialog window */
    #ifndef DEBUG
    if (log_level == G_LOG_LEVEL_CRITICAL) {
        drip_info(logstring->str);
    #endif
        g_print(logstring->str);
    #ifndef DEBUG
    }
    #endif

    /* Log to GUI */
    if (textscreen!=NULL) {
        if (log_level!=G_LOG_LEVEL_DEBUG || Config.debug_log==TRUE) {
            gtk_text_insert (GTK_TEXT(textscreen), logfont, colour, NULL, displaystring->str, -1);
        }
    }
    linenr++;

    /* Should we exit Drip? */
    if (quit) {
        /* Lock if not logging from a CallBack function */
        if (strcmp(log_domain,DRIP_LD)==0) {
            drip_unlock("log handler1");
        }
        drip_info(logstring->str);
        g_print("sleeping...");
        sleep(10);
        g_print(" done\n");
        terminate(TRUE,FALSE);
    }

    /* Free strings */
    logging_locked = FALSE;
    #ifdef DEBUG
    fflush(NULL);
    #endif
    /* Lock if not logging from a CallBack function */
    if (strcmp(log_domain,DRIP_LD)==0) {
        drip_unlock("log handler1");
    }

    return;
}


/* Traceback if Config.DVDdevice is a symbolic link */
gboolean DVDsymlink(void) {
    GString *command = g_string_new("");
    GString *GS_symlink = g_string_new("");
    guchar *symlink = NULL;
    guchar *toto = NULL;
    gint n;
    do {
        symlink = (guchar*)malloc(256);
        n = readlink(Config.DVDdevice->str,symlink,256);
        if (n!=-1 && n<255) { /* DVDdevice is a symlink, use real device instead */
            symlink[n] = 0; /* NULL terminate */
            if (strstr(symlink,"/dev/")==NULL) {
                if (strstr(symlink,"..") != NULL){
                    toto = symlink + 2;
                    g_string_sprintf(GS_symlink,"%s%s","/dev/",toto);
                } else {
                    g_string_sprintf(GS_symlink,"%s%s","/dev/",symlink);
                }
            } else {
                g_string_sprintf(GS_symlink,"%s",symlink);
            }
            g_string_sprintf(command,"Using DVD device %s because %s is a symlink",GS_symlink->str,Config.DVDdevice->str);
            g_log(DRIP_LD,G_LOG_LEVEL_MESSAGE,command->str);
            g_string_sprintf(Config.DVDdevice,"%s",GS_symlink->str);
            free(symlink);
        } else if (errno!=22) { /* Err=22 -> end looping, real device found */
            g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"Error while parsing DVD device |%s| : %s (%i)",Config.DVDdevice->str,strerror(errno),errno);
            g_string_sprintf(Config.DVDdevice,"/dev/cdrom");
        }
    } while (n>-1);

    g_string_free(command,TRUE);
    g_string_free(GS_symlink,TRUE);
    return TRUE;
}


/* Initialise Drip */
void drip_init() {
    GString *configfile;
    GString *HOME;
    GIOChannel *configfile_channel = NULL;
    glong configfile_size;
    guint bytesloaded;
    gint loaderror;
    gchar *buf;
    gint fd;
    gint i,n;
    glong bytes_to_read;
    struct stat file_stats;
    gboolean true_udf,dummy_bool,config_ok;
    GString *command = g_string_new("");
    GString *logfile = g_string_new("");
    GString *logfile_old = g_string_new("");
    GtkWidget *welcome;

    previous_location = g_string_new("");
    if (!g_thread_supported()) {
        terminate(TRUE,FALSE);
    }
    LOCKED = g_mutex_new();
    LOCKEDcondition = g_cond_new();
    HOME = g_string_new(getenv("HOME"));
    /* Initialize all drip defaults */
    g_string_sprintf(command,"%s/.drip",HOME->str);
    mkdir (command->str, S_IRUSR | S_IWUSR | S_IXUSR);
    g_string_sprintf(command,"%s/.drip/logs_old",HOME->str);
    mkdir (command->str, S_IRUSR | S_IWUSR | S_IXUSR);
    g_string_sprintf(command,"%s/.drip/plugins",HOME->str);
    mkdir (command->str, S_IRUSR | S_IWUSR | S_IXUSR);

    status_action = g_string_new("");
    configfile = g_string_new("");
    g_string_sprintf(configfile,"%s/.drip/drip.cfg",getenv("HOME"));
    for (n=0;n<256;n++) {
        for (i=0;i<8;i++) { Config.audio_channel_names[n][i] = g_string_new("");}
        for (i=0;i<32;i++) { Config.subpicture_names[n][i] = g_string_new("");}
    }

    /* Logging fonts and colours */
    logfont_info =    gdk_font_load("-B&H-LucidaTypewriter-Medium-R-Normal-Sans-12-120-75-75-M-70-ISO8859-1");
    logfont_message = gdk_font_load("-B&H-LucidaTypewriter-Bold-R-Normal-Sans-12-120-75-75-M-70-ISO8859-1");
    logfont_debug =   gdk_font_load("-Adobe-Courier-Medium-O-Normal--12-120-75-75-M-70-ISO8859-1");//-B&H-LucidaTypewriter-Medium-R-Normal-Sans-12-120-75-75-M-70-ISO8859-1");
    logfont_warning = gdk_font_load("-B&H-LucidaTypewriter-Bold-R-Normal-Sans-12-120-75-75-M-70-ISO8859-1");
    logfont_error =   gdk_font_load("-B&H-LucidaTypewriter-Medium-R-Normal-Sans-12-120-75-75-M-70-ISO8859-1");
    logfont_critical= gdk_font_load("-B&H-LucidaTypewriter-Medium-R-Normal-Sans-12-120-75-75-M-70-ISO8859-1");
    gdk_color_parse("DarkSlateGrey",&col_debug);
    gdk_color_parse("Red",&col_warning);
    gdk_color_parse("tomato3",&col_error);
    gdk_color_parse("darkRed",&col_critical);
    gdk_color_parse("black",&col_info);
    gdk_color_parse("blue",&col_message);

    /* Default dvd related settings */
    Config.DVDdevice = g_string_new("");
    Config.DVDsource = g_string_new("");
    Config.DIVXfileUser = g_string_new("");
    Config.DIVXfileFormatted = g_string_new("");
    Config.track_dir = g_string_new("");
    Config.audio_preferedA = g_string_new("");
    Config.audio_preferedB = g_string_new("");
    Config.spu_preferedA = g_string_new("");
    Config.spu_preferedB = g_string_new("");
    Config.CACHElocation = g_string_new("");
    for (n=0;n<256;n++) {
        Config.audio_channels_in_dvd[n] = 0;
        Config.subpictures_in_dvd[n] = -1;
    }
    Config.subpicture_stream = -1;
    Config.angle = 1;
    Config.clutfile = g_string_new("");
    Config.audio_2nd_enabled = FALSE;
    Config.two_audio_channels_support = FALSE;

    /* DVD volume settings */
    Config.DVDvolumeid = g_string_new("");
    Config.DVDpublisher = g_string_new("");
    Config.DVDpreparer = g_string_new("");
    Config.DVDcreationdate = g_string_new("");
    Config.DVDvolumesize_MB = 0;

    /* Drip logging domain */
    g_string_sprintf(logfile,"%s/.drip/drip.log",getenv("HOME"));
    if (stat(logfile->str,&file_stats)!=-1) {
        g_string_sprintf(logfile_old,"%s/.drip/logs_old/drip.%li.log",getenv("HOME"),file_stats.st_ctime);
        rename(logfile->str,logfile_old->str);
    } else {
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Could not stat log file (%s), removed it",logfile->str);
        unlink(logfile->str); /* delete old log file, TODO: log file rotation */
    }
    log_file = open(logfile->str,O_WRONLY|O_CREAT,S_IREAD|S_IWRITE); /* create log file */
    log_handler_drip_id = g_log_set_handler (DRIP_LD, (GLogLevelFlags)(G_LOG_LEVEL_WARNING |G_LOG_LEVEL_MESSAGE | G_LOG_LEVEL_INFO | G_LOG_LEVEL_DEBUG | G_LOG_LEVEL_CRITICAL | G_LOG_LEVEL_ERROR | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION),log_handler, NULL);
    log_handler_drip_cb_id = g_log_set_handler (DRIP_CB_LD, (GLogLevelFlags)(G_LOG_LEVEL_WARNING |G_LOG_LEVEL_MESSAGE | G_LOG_LEVEL_INFO | G_LOG_LEVEL_DEBUG | G_LOG_LEVEL_CRITICAL | G_LOG_LEVEL_ERROR | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION),log_handler, NULL);

    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Activated log handler and created log file %s",logfile->str);
    g_print("Activated log handler and created log file %s\n-------------------------------------------------------------------\n",logfile->str);

    /* Build main window */
    mainwindow = create_mainwindow();
    gtk_widget_show(mainwindow);

    /* Scan avifile for loaded codecs */
    avifile_scan();
    /* Any saved configuration? */
    config_ok = FALSE;
    fd = open(configfile->str,O_RDONLY);
    if (fd>-1) {
        configfile_channel = g_io_channel_unix_new(fd);
        stat(configfile->str, &file_stats);
        configfile_size = file_stats.st_size;
        if (configfile_size>0) {
            buf = (gchar*)malloc(configfile_size);
            memset(buf,0,configfile_size);
            /* Check config version */
            g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Checking config version");
            loaderror = g_io_channel_read(configfile_channel,(gchar*)&bytes_to_read,sizeof(glong),&bytesloaded);
            loaderror = g_io_channel_read(configfile_channel,buf,bytes_to_read,&bytesloaded);
            
            if (strstr(CONFIG_VERSION,buf)==NULL) {
                g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"Saved configuration (%s) is of incompatible version (%s)",buf,CONFIG_VERSION);
                free(buf);
                g_io_channel_close(configfile_channel);
                close(fd);
            } else {
                /* Read configuration from disk */
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&bytes_to_read,sizeof(glong),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,buf,bytes_to_read,&bytesloaded);
                    g_string_sprintf(Config.DVDdevice,"%s",(gchar*)buf);
                    memset(buf,0,configfile_size);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&bytes_to_read,sizeof(glong),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,buf,bytes_to_read,&bytesloaded);
                    g_string_sprintf(Config.DIVXfileUser,"%s",(gchar*)buf);
                    memset(buf,0,configfile_size);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.set_width,sizeof(glong),&bytesloaded);
                    memset(buf,0,configfile_size);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.set_height,sizeof(glong),&bytesloaded);
                    memset(buf,0,configfile_size);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.video_bitrate,sizeof(guint),&bytesloaded);
                    memset(buf,0,configfile_size);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.video_stream,sizeof(guint),&bytesloaded);
                    memset(buf,0,configfile_size);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.audio_bitrate,sizeof(gint),&bytesloaded);
                    memset(buf,0,configfile_size);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.audio_1st_stream,sizeof(guint),&bytesloaded);
                    memset(buf,0,configfile_size);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.encoding_video_codec,sizeof(gint),&bytesloaded);
                if (Config.codecs_amount<=Config.encoding_video_codec) {
                    Config.encoding_video_codec = 0;
                    Config.encoding_audio_codec = 0;
                    g_log(DRIP_LD,G_LOG_LEVEL_INFO,"The plugins of avifile have changed, resetted codecs");
                }
                    memset(buf,0,configfile_size);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.cliptop,sizeof(gint),&bytesloaded);
                    memset(buf,0,configfile_size);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.clipbottom,sizeof(gint),&bytesloaded);
                    memset(buf,0,configfile_size);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.blurradius,sizeof(gint),&bytesloaded);
                    memset(buf,0,configfile_size);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&dummy_bool,sizeof(gboolean),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.eject_dvdrom,sizeof(gboolean),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.use_dvddb,sizeof(gboolean),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.nice,sizeof(gint),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.conversion_mode,sizeof(cmode_t),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.maxsize,sizeof(glong),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&true_udf,sizeof(gboolean),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.encode,sizeof(gboolean),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&bytes_to_read,sizeof(glong),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,buf,bytes_to_read,&bytesloaded);
                    g_string_sprintf(Config.DVDsource,"%s",(gchar*)buf);
                    memset(buf,0,configfile_size);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.useDVDdevice,sizeof(gboolean),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.cpus,sizeof(gint),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.done_warning,sizeof(gboolean),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&bytes_to_read,sizeof(glong),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,buf,bytes_to_read,&bytesloaded);
                    g_string_sprintf(Config.CACHElocation,"%s",(gchar*)buf);
                    memset(buf,0,configfile_size);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.cache_delete_on_exit,sizeof(gboolean),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.cache_chunk,sizeof(gboolean),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.create_iso,sizeof(gboolean),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.hyper_zoom,sizeof(gboolean),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.autoclip,sizeof(gboolean),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.autoclip_frames,sizeof(gint),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.clipright,sizeof(gint),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.clipleft,sizeof(gint),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&dummy_bool,sizeof(gboolean),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.deinterlace,sizeof(gboolean),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.pulldown,sizeof(gboolean),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.encoding_audio_codec,sizeof(gint),&bytesloaded);
                if (Config.codecs_amount<=Config.encoding_audio_codec) {
                    Config.encoding_audio_codec = 0;
                    Config.encoding_video_codec = 0;
                    g_log(DRIP_LD,G_LOG_LEVEL_INFO,"The plugins of avifile have changed, resetted codecs");
                }

                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.mainwindow_X,sizeof(gint),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.mainwindow_Y,sizeof(gint),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.debug_log,sizeof(gboolean),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&Config.audio_norm,sizeof(gboolean),&bytesloaded);

                loaderror = g_io_channel_read(configfile_channel,(gchar*)&bytes_to_read,sizeof(glong),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,buf,bytes_to_read,&bytesloaded);
                    g_string_sprintf(Config.audio_preferedA,"%s",(gchar*)buf);
                    memset(buf,0,configfile_size);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&bytes_to_read,sizeof(glong),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,buf,bytes_to_read,&bytesloaded);
                    g_string_sprintf(Config.audio_preferedB,"%s",(gchar*)buf);
                    memset(buf,0,configfile_size);

                loaderror = g_io_channel_read(configfile_channel,(gchar*)&bytes_to_read,sizeof(glong),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,buf,bytes_to_read,&bytesloaded);
                    g_string_sprintf(Config.spu_preferedA,"%s",(gchar*)buf);
                    memset(buf,0,configfile_size);
                loaderror = g_io_channel_read(configfile_channel,(gchar*)&bytes_to_read,sizeof(glong),&bytesloaded);
                loaderror = g_io_channel_read(configfile_channel,buf,bytes_to_read,&bytesloaded);
                    g_string_sprintf(Config.spu_preferedB,"%s",(gchar*)buf);
                    memset(buf,0,configfile_size);

                free(buf);
                g_io_channel_close(configfile_channel);
                close(fd);
                config_ok = TRUE;
                g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Loaded saved configuration");
            }
        }
    } 

    if (config_ok == FALSE) {
        /* Default divx related settings */
        g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Setting up default configuration...");
        g_string_sprintf(Config.DVDdevice,"%s","/dev/cdrom");
        /* Traceback if Config.DVDdevice is a symbolic link */
        DVDsymlink();

        g_string_sprintf(Config.DVDsource,"./cache/");
        g_string_sprintf(Config.DIVXfileUser,"drip.avi");
        g_string_sprintf(Config.CACHElocation,"./cache/");
        Config.set_width = 512;
        Config.set_height = -1;
        Config.hyper_zoom = TRUE;
        Config.video_bitrate = 800;
        Config.video_stream = 0;
        Config.audio_bitrate = 128;
        Config.audio_1st_stream = 0;
        Config.audio_norm = TRUE;
        /* Video codec, scan for 1st valid codec */
        for (i=0;i<Config.codecs_amount;i++) {
            if (Config.codecs[i].cav == VIDEO_CODEC) {
                Config.encoding_video_codec = i;
                i = Config.codecs_amount;
            }
        } 
        /* Audio codec, scan for 1st valid codec */
        for (i=0;i<Config.codecs_amount;i++) {
            if (Config.codecs[i].cav == AUDIO_CODEC) {
                Config.encoding_audio_codec = i;
                i = Config.codecs_amount;
            }
        } 
        Config.cliptop = 0;
        Config.clipbottom = 0;
        Config.autoclip = TRUE;
        Config.autoclip_frames = 256;
        Config.deinterlace = TRUE;
        Config.pulldown = FALSE;
        Config.clipright = 0;
        Config.clipleft = 0;
        Config.blurradius = 0;
        /* Default various other settings */
        Config.cache_delete_on_exit = FALSE;
        Config.cache_chunk = TRUE;
        Config.eject_dvdrom = FALSE;
        Config.nice = 0;
        Config.conversion_mode = UNKNOWN;
        Config.use_dvddb = FALSE;
        Config.maxsize = 696;
        Config.encode = TRUE;
        Config.useDVDdevice = TRUE;
        Config.cpus = 1;
        Config.done_warning = TRUE;
        Config.create_iso = FALSE;
        Config.mainwindow_X = 500;
        Config.mainwindow_Y = 400;
        /* Language Preferences */
        g_string_sprintf(Config.audio_preferedA,"English");
        g_string_sprintf(Config.audio_preferedB,"*Mongolian");
        g_string_sprintf(Config.spu_preferedA,"Nederlands");
        g_string_sprintf(Config.spu_preferedB,"English");

#ifdef DEBUG
        Config.debug_log = TRUE;
#else
        Config.debug_log = FALSE;
#endif
        /* No save config around? -> Welcome user */
        welcome = create_messagebox_betanotice();
        gtk_widget_show(welcome);
    }

    /* Default progres size */
    Config.progres_size = (gdouble)7/(gdouble)10;

    /* Load plugins */
    if (init_plugins() == FALSE) {
        //g_log(DRIP_LD,G_LOG_LEVEL_ERROR,"Plugins not loaded");
    }

    /* Clean & Exit */
    if (Config.CACHElocation->len==0) {
        /* check if cache location is empty */
        g_string_sprintf(Config.CACHElocation,"./");
    }
    g_string_free(HOME,TRUE);
    g_string_free(logfile,TRUE);
    g_string_free(logfile_old,TRUE);
    g_string_free(configfile,TRUE);
    g_string_free(command,TRUE);
    return;
}

/* Check if host in on unwanted domain */
gboolean drip_domain(void) {
    struct hostent *host;
    gchar *name = (char*)malloc(1024);
    gchar *domain;
    gchar *extension;
    GtkWidget *dialog;
    size_t len = 1024;
 
    if (gethostname(name,len)==-1) {
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Error while getting hostname");
    } 
    host = gethostbyname(name);

    if (!(domain=strchr(host->h_name, '.'))) {
        return FALSE;
    }
    domain++;

    if (!(extension=strchr(domain, '.'))) {
        return FALSE;
    }
    extension++;

    if (strcmp(extension,"il")==0 || strcmp(extension,"mil")==0 || strcmp(extension,"gov")==0) {
        drip_lock("domain1");
        dialog = create_messagebox();
        gtk_widget_show(dialog);
        drip_unlock("domain1");
        sleep(5);
        drip_lock("domain2");
        terminate(TRUE,FALSE);
        drip_unlock("domain2");
    }

    return TRUE;
}

/* Lock */
void drip_lock(const gchar *location) {
    /* Mutex ready? */
    if (LOCKED == NULL) {
        #ifdef DEBUG
        printf("Mutex not yet initialized, skipping lock\n");
        #endif
        return;
    }
    /* Already locked? */
    if (g_mutex_trylock(LOCKED) == FALSE && previous_location->len>0) {
        display_unlock = TRUE;
        printf("Already locked from %s, waiting to become unlocked for %s\n",previous_location->str,location);
        while (g_mutex_trylock(LOCKED)==FALSE) {
            g_cond_wait(LOCKEDcondition,LOCKED);
        }
        #ifdef DEBUG
        printf("Engine got unlocked from %s and locked for %s\n",previous_location->str,location);
        #endif
    }
    /* Lock gdk */
    g_string_sprintf(previous_location,"%s",location); 
    gdk_threads_enter();
    #ifdef DEBUG
    printf("Locked, from %s\n",location);
    #endif
    return;
}

/* Unlock */
void drip_unlock(const gchar *location) {
    /* Mutex ready? */
    if (LOCKED == NULL) {
        #ifdef DEBUG
        printf("Mutex not yet initialized, skipping unlock\n");
        #endif
        return;
    }
    /* Unlocking from correct location? */
    if (strcmp(previous_location->str,location)!=0) {
        /* Display termination reason, DONT use g_log(DRIP_LD,..) for this! */
        printf("Unlocking from incorrect location %s, %s was exspected, this should not happened!\n",location,previous_location->str);
        terminate(TRUE,FALSE);
    }
    /* Unlock */
    g_mutex_unlock(LOCKED);
    g_cond_signal(LOCKEDcondition);
    gdk_threads_leave();
    if (display_unlock == TRUE) {
        display_unlock = FALSE;
        #ifdef DEBUG
        printf("Unlocked from %s\n",location);
        #endif
    }
    #ifdef DEBUG
    else {
        printf("Unlocked from %s\n",location);
    }
    #endif
    return;
}

/* Terminates drip, should do clean shutdown  */
void terminate(gboolean hard,gboolean lock) {
    GString *str = g_string_new("");
    GString *command = g_string_new("");

    if (converting && hard==FALSE) {
        GtkWidget* buzy;
        buzy = create_messagebox_buzy();
        gtk_widget_show(buzy);
    } else {
        //TODO Fix crash when this is enabled
        printf("Closing plugins..\n");fflush(NULL);
        close_plugins();

        printf("Saving config..\n");fflush(NULL);
        save_config(FALSE);
        printf("Removing cache..\n");fflush(NULL);
        remove_cache(FALSE);
        printf("Quitting encoder..\n");fflush(NULL);
        encoder_quit(FALSE);
        printf("Removing log handlers..\n");fflush(NULL);
        g_log_remove_handler(DRIP_CB_LD,log_handler_drip_cb_id);
        g_log_remove_handler(DRIP_LD,log_handler_drip_id);
        printf("Closing log file..\n");fflush(NULL);
        close(log_file);
        g_mutex_unlock(LOCKED); // Overrule locks
        gdk_threads_leave(); // Overrule locks
        printf("Exiting..\n");fflush(NULL);
        exit(0);
    }

    /* Clean & Exit */
    g_string_free(str,TRUE);
    g_string_free(command,TRUE);
    return;
}


/* Fetch available output codecs */
void avifile_scan() {
    codecS** codec_list;
    guint i;
    gint av_codecs = 0;
    gboolean mp3_codec_bundled = FALSE;
    gboolean mp3_codec_standalone = FALSE;

    Config.codecs = (codecS*)malloc(sizeof(codecS)*256);
    codec_list = encoder_codecs();
    Config.codecs_amount = 0;
    for (i=0;i<codec_list[0]->length;i++) {
        if (codec_list[i]->ctype->len>0 && isupper(codec_list[i]->ctype->str[0])) {
                /* Fill local config */
                Config.codecs[Config.codecs_amount].ctype = g_string_new(codec_list[i]->ctype->str);
                Config.codecs[Config.codecs_amount].ccodec = codec_list[i]->ccodec;
                Config.codecs[Config.codecs_amount].cmodule = g_string_new(codec_list[i]->cmodule->str);
                Config.codecs[Config.codecs_amount].cdescription = g_string_new(codec_list[i]->cdescription->str);
                if (strcmp(Config.codecs[Config.codecs_amount].cdescription->str,"Lame MPEG layer-3 encoder (runtime)")==0) {
                    Config.codecs[Config.codecs_amount].cdescription = g_string_append(Config.codecs[Config.codecs_amount].cdescription," (Dual audio channel supported)");
                    mp3_codec_standalone = TRUE;
                } else if (strncmp(Config.codecs[Config.codecs_amount].cdescription->str,"Lame ",5)==0) {
                    Config.codecs[Config.codecs_amount].cdescription = g_string_append(Config.codecs[Config.codecs_amount].cdescription," (Single audio channel supported)");
                    mp3_codec_bundled = TRUE;
                } 
                switch (codec_list[i]->cav) {
                    case 0: Config.codecs[Config.codecs_amount].cav = VIDEO_CODEC;
                            av_codecs++;
                            break;
                    case 1: Config.codecs[Config.codecs_amount].cav = AUDIO_CODEC;
                            av_codecs++;
                            break;
                    default:Config.codecs[Config.codecs_amount].cav = UNKNOWN_CODEC;
                            break;
                }
                g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Avifile: plugin %s loaded (%s), coded id %li",Config.codecs[Config.codecs_amount].cdescription->str,Config.codecs[Config.codecs_amount].cmodule->str,Config.codecs[Config.codecs_amount].ccodec);
                Config.codecs_amount++;
    //        }
        }
    }
    g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Loaded %i encoding codecs from Avifile %i.%i.%i",av_codecs,AVIFILE_MAJOR_VERSION,AVIFILE_MINOR_VERSION,AVIFILE_PATCHLEVEL);

    /* INFO to user about current capabilities */
    /* bundled = TRUE, standalone = FALSE */
    if ((mp3_codec_bundled == TRUE) && (mp3_codec_standalone == FALSE)) { 
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Audio codec supports a single audio channel at variable bitrate");
        Config.two_audio_channels_support = FALSE;
    }
    /* bundled = FALSE, standalone = TRUE */
    if ((mp3_codec_bundled == FALSE) && (mp3_codec_standalone == TRUE)) {
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Audio codec supports dual audio channel at fixed 128kb bitrate");
        Config.two_audio_channels_support = TRUE;
    }
    /* bundled = TRUE, standalone = TRUE */
    if ((mp3_codec_bundled == TRUE) && (mp3_codec_standalone == TRUE)) {
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Audio codec supports a single audio channel at variable bitrate");
        Config.two_audio_channels_support = FALSE;
    }
    /* bundled = FALSE, standalone = FALSE */
    if ((mp3_codec_bundled == FALSE) && (mp3_codec_standalone == FALSE)) {
        g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"Audio not supported, no audio codec available!");
        Config.two_audio_channels_support = FALSE;
    }




    /* Check Config.encoding_video_codec & Config.encoding_audio_codec are within range */
    if (Config.encoding_video_codec>=Config.codecs_amount) {
        Config.encoding_video_codec = 0;
    }
    if (Config.encoding_audio_codec>=Config.codecs_amount) {
        Config.encoding_audio_codec = 0;
    }

    return;
}


int main (int argc, char *argv[]) {
    GString *welcome,*line;
    gint i;
    /* Init's */
    g_thread_init(NULL);
    gtk_init(&argc,&argv);
    #ifdef ENABLE_NLS
    bindtextdomain (PACKAGE, PACKAGE_LOCALE_DIR);
    textdomain (PACKAGE);
    #endif
    gnome_init(PACKAGE, VERSION, argc, argv);
    drip_init();
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Drip %s, pid = %i",VERSION,getpid());

    /* We can log to the GUI now, show some info */
    welcome = g_string_new("");
    line    = g_string_new("");
    g_string_sprintf(welcome,"        Welcome to Drip %s        ",VERSION);
    for (i=0;i<welcome->len;i++) {
         g_string_sprintf(line,"-");
    }
    g_log(DRIP_LD,G_LOG_LEVEL_MESSAGE,"%s",line->str);
    g_log(DRIP_LD,G_LOG_LEVEL_MESSAGE,"%s",welcome->str);
    g_log(DRIP_LD,G_LOG_LEVEL_MESSAGE,"%s",line->str);
    g_string_free(welcome,TRUE);
    g_string_free(line,TRUE);

    /* Warn if libdvdcss isn't available */
    #ifdef DVDCSS_NOT_INSTALLED
    g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"libdvdcss is not installed, most DVD disks wont be readable!");
    g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"To get this working install libdvdcss, recompile libdvdread");
    g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"and recompile Drip. After that all DVD's should read just fine.");
    #endif

    /* Start glib */
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Frontend running inner loop");
    gdk_threads_enter();
    gtk_main();
    gdk_threads_leave();
    return 0;
}

