/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Copyright 2002-2004 Todd Kulesza, Davyd Madeley
 *
 * Authors:
 * 		Davyd Madeley <davyd@ucc.asn.au>
 */

#include <dlfcn.h>
#include <gnome.h>
#include <libbonobo.h>

#include <config.h>

#include "query_music_players.h"

#ifdef HAVE_RHYTHMBOX
#include "Rhythmbox.h"
#endif

/* ---- definitions ---- */
static void *xmms_dl = NULL; /* for dlopen() */

static void
add_music_entry (GtkListStore *store, gchar *entry, GdkPixbuf *pixbuf, gboolean *changed, gboolean prepend);

static gboolean
xmms_init (void);
static gchar
*xmms_query (void);

static gboolean
rhythmbox_init (gpointer **ptr);
static gchar
*rhythmbox_query (gpointer *ptr);
/* ---- end definitions ---- */

void
query_music_players (GtkListStore *playing)
{
	gboolean entries;
	GdkPixbuf *pixbuf;
	gpointer *ptr;
	
	gtk_list_store_clear (playing);
	entries = FALSE;

	if (xmms_init ())
	{
		pixbuf = gdk_pixbuf_new_from_file (DATADIR G_DIR_SEPARATOR_S "pixmaps"
				G_DIR_SEPARATOR_S "drivel" G_DIR_SEPARATOR_S "xmms.xpm", NULL);
		add_music_entry (playing, xmms_query (), pixbuf, &entries, FALSE);
	}
#ifdef HAVE_RHYTHMBOX
	if (rhythmbox_init (&ptr))
	{
		pixbuf = NULL;
		add_music_entry (playing, rhythmbox_query (ptr), pixbuf, &entries, FALSE);
	}
#endif
	if (entries)
		add_music_entry (playing, g_strdup (""), NULL, NULL, TRUE);
	else
		/* FIXME:
		 * I would like to make this menu item insensitive, however due to
		 * an API issue in GTK, this will not be possible yet.
		 * See GNOME bug #135875
		 */
		add_music_entry (playing, g_strdup (_("No song playing")), NULL, NULL, FALSE);
}

static void
add_music_entry (GtkListStore *store, gchar *entry, GdkPixbuf *pixbuf, gboolean *changed, gboolean prepend)
{
	GtkTreeIter iter;

	if (entry)
	{
		if (prepend)
			gtk_list_store_prepend (store, &iter);
		else
			gtk_list_store_append (store, &iter);
		gtk_list_store_set (store, &iter,
				0, entry,
				1, pixbuf,
				-1);
		if (changed)
			*changed = TRUE;
	}
}

/*
 * xmms_init() and xmms_query() are based off
 * music_xmms_grab() and xmms_load_ok() from logjam.
 *
 * logjam - a GTK client for LiveJournal.
 * Copyright (C) 2000-2003 Evan Martin <evan@livejournal.com>
 * 
 * Imported May 3, 2003 by Todd Kulesza <todd@dropline.net>
 * Modified May 5, 2004 by Davyd Madeley <davyd@ucc.asn.au>
 */
static gboolean
xmms_init (void)
{
	if (xmms_dl == NULL)
	{
		xmms_dl = dlopen ("libxmms.so", RTLD_LAZY);

		if (!xmms_dl)
			xmms_dl = dlopen ("libxmms.so.1", RTLD_LAZY);
		if (!xmms_dl)
			return FALSE; /* no library; don't use xmms. */
	}
	return TRUE;
}

static gchar
*xmms_query (void)
{
	gchar *raw_title, *normalized_title=NULL;
	gint pos, bytes_read, bytes_written;

	static gint (*xrgpp)(gint) = NULL;
	static gchar* (*xrgpt)(gint, gint) = NULL;

	if (xrgpp == NULL || xrgpt == NULL) {
		xrgpp = dlsym (xmms_dl, "xmms_remote_get_playlist_pos");
		if (dlerror ())
			return NULL;
		xrgpt = dlsym (xmms_dl, "xmms_remote_get_playlist_title");
		if (dlerror ())
			return NULL;
	}

	pos = xrgpp (0);
	raw_title = xrgpt (0, pos);
	/* the string from XMMS is in the locale encoding. We need to 
	 * convert this to UTF8
	 */
	if (raw_title) {
		normalized_title = g_locale_to_utf8(raw_title, strlen(raw_title), &bytes_read, &bytes_written, NULL);
		free(raw_title);
	}
	return normalized_title;
}

/*
 * routines for talking to Rhythmbox
 */
#ifdef HAVE_RHYTHMBOX
static gboolean
rhythmbox_init (gpointer **ptr)
{
	GNOME_Rhythmbox rb;
	CORBA_Environment ev;
	Bonobo_PropertyBag pb;
	CORBA_any *any;
		
	if (!bonobo_is_initialized ())
	{
		g_warning ("Bonobo has not been initialized!");
		return FALSE;
	}
	
	CORBA_exception_init (&ev);

	/* get the Rhythmbox component only if it is active */
	rb = bonobo_activation_activate ("((repo_ids.has('IDL:GNOME/Rhythmbox:1.0')) AND (_active == TRUE))",
			NULL, 0, NULL, &ev);

	if (rb == CORBA_OBJECT_NIL)
	{
		g_warning ("Could not create an instance of Rhythmbox");
		return FALSE;
	}
	
	CORBA_exception_init (&ev);

	pb = GNOME_Rhythmbox_getPlayerProperties (rb, &ev);
	if (BONOBO_EX (&ev))
	{
		char *err = bonobo_exception_get_text (&ev);
		g_warning ("An exception occurred '%s'", err);
		g_free (err);
		return FALSE;
	}

	any = bonobo_pbclient_get_value (pb, "song", NULL, NULL);

	if (!CORBA_TypeCode_equivalent (any->_type,
				TC_GNOME_Rhythmbox_SongInfo,
				NULL))
	{
		g_warning ("Unexpected type from Bonobo");
		return FALSE;
	}

	*ptr = any->_value;

	CORBA_exception_free (&ev);
	
	return TRUE;
}

static gchar
*rhythmbox_query (gpointer *ptr)
{
	GNOME_Rhythmbox_SongInfo *song_info;

	song_info = (GNOME_Rhythmbox_SongInfo *) ptr;

	if (song_info != NULL)
		return g_strdup_printf ("%s - %s", song_info->artist, song_info->title);
	else
		return NULL;
}
#endif
