/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Copyright 2002-2004 Todd Kulesza
 *
 * Authors:
 * 		Todd Kulesza <todd@dropline.net>
 */

#include <config.h>

#include <stdarg.h> /* variable-length functions */
#include <string.h>
#include <time.h>
#include <libxml/parser.h>
#include <glib.h>

#include "xmlrpc.h"

/* Parse 'args' into an XML tree of appropriate values */

static void
parse_parameter_types (va_list *args, xmlNodePtr node)
{
	xmlNodePtr xml_param, xml_value;
	gchar *value;
	gint type, value_int;
	gdouble value_double;
	GTimeVal value_time;
	
	g_return_if_fail (node);
	g_return_if_fail (*args);
	
	xml_param = NULL;
	xml_value = NULL;
	
	/* parse the parameters */
	while ((type = va_arg (*args, gint)) > -1)
	{
		xml_param = xmlNewChild (node, NULL, "param", NULL);
		if (type != XMLRPC_TYPE_NODE)
			xml_value = xmlNewChild (xml_param, NULL, "value", NULL);
		
		switch (type)
		{
			case XMLRPC_TYPE_INT:
			{
				value_int = va_arg (*args, gint);
				value = g_strdup_printf ("%d", value_int);
				xmlNewTextChild (xml_value, NULL, "i4", value);
				g_free (value);
				break;
			}
			case XMLRPC_TYPE_BOOL:
			{
				value_int = (gint)va_arg (*args, gboolean);
				if (value_int)
					value = g_strdup ("1");
				else
					value = g_strdup ("0");
				xmlNewTextChild (xml_value, NULL, "boolean", value);
				g_free (value);
				break;
			}
			case XMLRPC_TYPE_STRING:
			{
				value = va_arg (*args, gchar*);
				xmlNewTextChild (xml_value, NULL, "string", value);
				break;
			}
			case XMLRPC_TYPE_DOUBLE:
			{
				value_double = va_arg (*args, gdouble);
				value = g_strdup_printf ("%f", value_double);
				xmlNewTextChild (xml_value, NULL, "double", value);
				g_free (value);
				break;
			}
			case XMLRPC_TYPE_DATE:
			{
				value = g_new0 (gchar, 32);
				value_time = va_arg (*args, GTimeVal);
				strftime (value, 32, "%Y%m%dT%H:%M:%S", localtime (&value_time.tv_sec));
				xmlNewTextChild (xml_value, NULL, "dateTime.iso8601", value);
				g_free (value);
				break;
			}
			case XMLRPC_TYPE_BASE64:
			{
				value = va_arg (*args, gchar*);
				xmlNewTextChild (xml_value, NULL, "base64", value);
				break;
			}
			case XMLRPC_TYPE_NODE:
			{
				xml_value = va_arg (*args, xmlNodePtr);
				xmlAddChild (xml_param, xml_value);
				break;
			}
			default:
			{
				g_warning ("parse_parameter_types: Unknown XML-RPC type");
				break;
			}
		}
	}
	
	va_end (*args);
	
	return;
}

/* Parse an XML-RPC 'type' node into a list of names and values in a table */

static void
parse_xmlrpc_type_to_hashtable (xmlNodePtr node, GHashTable *table)
{
	xmlNodePtr child;
	xmlChar *key;
	gchar *name, *value;

	g_return_if_fail (node);
	g_return_if_fail (table);
	
	name = value = NULL;
	child = node->xmlChildrenNode;
	while (child)
	{
		/* descend into 'struct', 'member', 'array', and 'data' trees... */
		if (!xmlStrcmp (child->name, (const xmlChar *) "struct") ||
			!xmlStrcmp (child->name, (const xmlChar *) "member") ||
			!xmlStrcmp (child->name, (const xmlChar *) "array") ||
			!xmlStrcmp (child->name, (const xmlChar *) "data"))
		{
			parse_xmlrpc_type_to_hashtable (child, table);
		}
		/* extract the 'name' tag... */
		else if (!xmlStrcmp (child->name, (const xmlChar *) "name"))
		{
			key = xmlNodeGetContent (child);
			if (key)
			{
				gchar *lower;
				gint count = 0;
				
				lower = g_ascii_strdown (key, -1);
				name = g_strdup_printf ("%s%d", lower, count);
				while (g_hash_table_lookup (table, name))
				{
					g_free (name);
					name = g_strdup_printf ("%s%d", lower, ++count);
				}
				
				xmlFree (key);
				g_free (lower);
			}
		}
		/* if the value is a struct or array, descend; otherwise, extract it */
		else
		{
			xmlNodePtr grandchild;
			
			grandchild = child->xmlChildrenNode;
			while (grandchild)
			{
				if (!xmlStrcmp (grandchild->name, (const xmlChar *) "struct") ||
					!xmlStrcmp (grandchild->name, (const xmlChar *) "member") ||
					!xmlStrcmp (grandchild->name, (const xmlChar *) "array") ||
					!xmlStrcmp (grandchild->name, (const xmlChar *) "data"))
				{
					parse_xmlrpc_type_to_hashtable (child, table);
				}
				else 
				{
					key = xmlNodeGetContent (grandchild);
					if (key)
					{
						value = g_strdup (key);
						xmlFree (key);
					}
				}
				
				grandchild = grandchild->next;
			}
		}
		
		child = child->next;
	}
	
	if (name && value)
		g_hash_table_insert (table, name, value);
	else if (value)
		g_hash_table_insert (table, g_strdup ("response"), value);
	else
		g_free (name);
	
	return;
}

/* Build an XML-RPC packet from a -1-terminated list of types and values */

gchar*
xmlrpc_build_packet (const gchar *name, ...)
{
	va_list *args;
	xmlDocPtr doc;
	xmlNodePtr call, params;
	xmlChar *xml;
	gint bufsize;
	
	/* build the methodCall and methodName nodes */
	doc = xmlNewDoc ("1.0");
	call = xmlNewNode (NULL, "methodCall");
	xmlDocSetRootElement (doc, call);
	xmlNewTextChild (call, NULL, "methodName", name);
	
	/* build the params node */
	args = g_new0 (va_list, 1);
	va_start (*args, name);
	params = xmlNewChild (call, NULL, "params", NULL);
	parse_parameter_types (args, params);
	
	/* output the XML */
	xmlDocDumpFormatMemoryEnc (doc, &xml, &bufsize, "UTF-8", 0);
	
	xmlFreeDoc (doc);
	
	return xml;
}

/* Parse an XML-RPC packet into a hash table of name-value pairs */

GHashTable*
xmlrpc_parse_packet (const gchar *xml, gint length)
{
	GHashTable *table;
	xmlDocPtr doc;
	xmlNodePtr node;
	gboolean finished = FALSE;
	
	table = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free);
	
	doc = xmlReadMemory (xml, length, NULL, NULL, XML_PARSE_NOBLANKS);
	if (!doc)
	{
		g_warning ("xmlrpc_parse_packet: Packet is not XML");
		return NULL;
	}
	
	node = xmlDocGetRootElement (doc);
	if (!node || !node->name || 
			strncmp ("methodResponse", node->name, strlen (node->name)))
	{
		g_warning ("xmlrpc_parse_packet: XML is not valid XML-RPC");
		xmlFreeDoc (doc);
		return NULL;
	}
	
	/* descend to "methodReponse's" child */
	node = node->xmlChildrenNode;
	while (node && !finished)
	{
		/* we have a 'params' response */
		if (!strncmp ("params", node->name, strlen (node->name)))
		{
			g_hash_table_insert (table, g_strdup ("success"), g_strdup ("params"));
			/* descend from params to param */
			node = node->xmlChildrenNode;
			while (node && !finished)
			{
				if (!strncmp ("param", node->name, strlen (node->name)))
					finished = TRUE;
				else
					node = node->next;
			}
		}
		/* we have a 'fault' response */
		else if (!strncmp ("fault", node->name, strlen (node->name)))
		{
			g_hash_table_insert (table, g_strdup ("success"), g_strdup ("fault"));
			finished = TRUE;
		}
		/* otherwise keep parsing */
		else
			node = node->next;
	}
	
	/* descend from param or fault to value */
	node = node->xmlChildrenNode;
	finished = FALSE;
	while (node && !finished)
	{
		if (!strncmp ("value", node->name, strlen (node->name)))
			finished = TRUE;
		else
			node = node->next;
	}
	
	parse_xmlrpc_type_to_hashtable (node, table);
	
	xmlFreeDoc (doc);
	
	return table;
}

/* Build an XML node for an integer value */
xmlNodePtr
xmlrpc_build_value_int (gint value)
{
	xmlNodePtr parent;
	xmlChar *content;
	
	content = g_strdup_printf ("%d", value);
	parent = xmlNewNode (NULL, "value");
	xmlNewTextChild (parent, NULL, "i4", content);
	xmlFree (content);
	
	return parent;
}

/* Build an XML node for a boolean value */
xmlNodePtr
xmlrpc_build_value_bool (gboolean value)
{
	xmlNodePtr parent;
	xmlChar *content;
	
	if (value)
		content = g_strdup ("1");
	else
		content = g_strdup ("0");
	parent = xmlNewNode (NULL, "value");
	xmlNewTextChild (parent, NULL, "boolean", content);
	xmlFree (content);
	
	return parent;
}

/* Build an XML node for a string value */
xmlNodePtr
xmlrpc_build_value_string (const gchar *value)
{
	xmlNodePtr parent;
	xmlChar *content;
	
	content = g_strdup (value);
	parent = xmlNewNode (NULL, "value");
	xmlNewTextChild (parent, NULL, "string", content);
	xmlFree (content);
	
	return parent;
}

/* Build an XML node for a double value */
xmlNodePtr
xmlrpc_build_value_double (gdouble value)
{
	xmlNodePtr parent;
	xmlChar *content;
	
	content = g_strdup_printf ("%f", value);
	parent = xmlNewNode (NULL, "value");
	xmlNewTextChild (parent, NULL, "double", content);
	xmlFree (content);
	
	return parent;
}

/* Build an XML node for a date/time value */
xmlNodePtr
xmlrpc_build_value_date (GTimeVal value)
{
	xmlNodePtr parent;
	xmlChar content[32];
	
	strftime (content, 32, "%Y%m%dT%H:%M:%S", localtime (&value.tv_sec));
	parent = xmlNewNode (NULL, "value");
	xmlNewTextChild (parent, NULL, "dateTime.iso8601", content);
	
	return parent;
}

/* Build an XML node for a base64 value */
xmlNodePtr
xmlrpc_build_value_base64 (const gchar *value)
{
	xmlNodePtr parent;
	xmlChar *content;
	
	content = g_strdup (value);
	parent = xmlNewNode (NULL, "value");
	xmlNewTextChild (parent, NULL, "base64", content);
	xmlFree (content);
	
	return parent;
}

/* Build an XML node for a struct value-pair */
xmlNodePtr
xmlrpc_build_value_struct (void)
{
	xmlNodePtr parent, child;
	
	parent = xmlNewNode (NULL, "value");
	child = xmlNewChild (parent, NULL, "struct", NULL);
	
	return parent;
}

/* Add a membernode to a struct */
void
xmlrpc_struct_add (xmlNodePtr parent, const gchar *name, xmlNodePtr value)
{
	xmlNodePtr structure, child;
	
	structure = parent->xmlChildrenNode;
	child = xmlNewChild (structure, NULL, "member", NULL);
	xmlNewChild (child, NULL, "name", name);
	xmlAddChild (child, value);
	
	return;
}

/* Build an XML node for an array of values */
xmlNodePtr
xmlrpc_build_value_array (void)
{
	xmlNodePtr array, data;
	
	array = xmlNewNode (NULL, "value");
	data = xmlNewChild (array, NULL, "array", NULL);
	data = xmlNewChild (data, NULL, "data", NULL);
	
	return array;
}

/* Add a value to an XML array */
void
xmlrpc_array_add (xmlNodePtr array, xmlNodePtr value)
{
	xmlNodePtr data;
	
	data = array->xmlChildrenNode;
	data = data->xmlChildrenNode;
	xmlAddChild (data, value);
	
	return;
}

void
print_xml (xmlNodePtr node)
{
	xmlDocPtr doc;
	xmlNodePtr root;
	xmlChar *buffer;
	gint size;
	
	g_return_if_fail (node);
	
	doc = xmlNewDoc ("1.0");
	root = xmlNewNode (NULL, "testing");
	xmlDocSetRootElement (doc, root);
	g_print ("add child\n");
	xmlAddChild (root, node);
	g_print ("dump memory\n");
	xmlDocDumpFormatMemoryEnc (doc, &buffer, &size, "UTF-8", 1);
	
	g_print ("----- print_xml() -------\n%s\n\n", buffer);
	
	return;
}
