/*********************************************************************************/
/* $Id: AdiUsbAdslDriver.c,v 1.19 2003/06/18 21:58:41 sleeper Exp $              */
/*-------------------------------------------------------------------------------*/
/*										 */
/* Copyright (c) 2002, Analog Devices Inc., All Rights Reserved			 */
/* User space interface rewritten by C.Casteyde (casteyde.christian@free.fr)	 */
/* Multi-modem support added by Renaud Guerin (rguerin@freebox.fr)		 */
/* Memory leaks/boot desynchro by Fred Ros (sl33p3r@free.fr)			 */
/*										 */
/* AdiUsbAdslDriver.c								 */
/*										 */
/* Main code for Linux kernel driver.					         */
/*										 */
/*-------------------------------------------------------------------------------*/
/* This file is part of the "ADI USB ADSL Driver for Linux".   			 */
/*   										 */
/* "ADI USB ADSL Driver for Linux" is free software; you can redistribute it     */
/* and/or modify it under the terms of the GNU General Public License as         */
/* published by the Free Software Foundation; either version 2 of the License,   */
/* or (at your option) any later version.   				         */
/*									         */
/* "ADI USB ADSL Driver for Linux" is distributed in the hope that it will be    */
/* useful, but WITHOUT ANY WARRANTY; without even the implied warranty of        */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the   		 */
/* GNU General Public License for more details.   				 */
/*   										 */
/* You should have received a copy of the GNU General Public License   		 */
/* along with "ADI USB ADSL Driver for Linux"; if not, write to the Free Software*/
/* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA     */
/*********************************************************************************/

#include "Adiutil.h"
#include <linux/module.h>
#include <linux/init.h>
#include <linux/proc_fs.h>
#include <linux/list.h>
#include <asm/uaccess.h>
#include <linux/if_arp.h>
#include "AdiUsbAdslDriver.h"
#include "Util.h"
#include "Dsp.h"
#include "Msg.h"
#include "Boot.h"
#include "Pipes.h"
#include "Crc.h"
#include "Mpoa.h"
#include "Me.h"
#include "Uni.h"
#include "debug.h"


/*********************************************************************************/
/* Local prototypes								 */
/*********************************************************************************/

/* USB driver */
static void *adi_probe(struct usb_device *usb, unsigned int ifnum, const struct usb_device_id *id);
static void adi_disconnect(struct usb_device *usb, void *ptr);
static void adi_irq(struct urb *urb);
static int adi_user(struct usb_device *dev, unsigned int code, void *buf);
static int adi_read_proc(char *page, char **start, off_t off, int count, int *eof, void *data);

/* Ethernet device */
static int create_etherdev(Hardware *pHw);
static int adi_open(struct net_device *ether);
static int adi_close(struct net_device *ether);
static int adi_ioctl(struct net_device *ether, struct ifreq *rq, int cmd);
static struct net_device_stats *adi_stats(struct net_device *ether);
static int adi_start_xmit(struct sk_buff *skb, struct net_device *ether);
static void adi_set_multicast(struct net_device *ether);
static void adi_tx_timeout(struct net_device *ether);


/*********************************************************************************/
/* Globals and local utilities                                                   */
/*********************************************************************************/

/*Tell usb what VID/PID combinations we support*/
static const struct usb_device_id adi_ids[] =
{
    {  USB_DEVICE (kEAGLE_VID, kEAGLEI_PID_PREFIRMWARE) }, /* before USB firmware upload */
    {  USB_DEVICE (kEAGLE_VID, kEAGLEI_PID_PSTFIRMWARE) }, /* after  USB firmware upload */
    {  USB_DEVICE (kEAGLE_VID, kEAGLEII_PID_PREFIRMWARE) }, /* before USB firmware upload */
    {  USB_DEVICE (kEAGLE_VID, kEAGLEII_PID_PSTFIRMWARE) }, /* after  USB firmware upload */
    {  USB_DEVICE (kEAGLE_VID, kEAGLEIIC_PID_PREFIRMWARE) }, /* before USB firmware upload */
    {  USB_DEVICE (kEAGLE_VID, kEAGLEIIC_PID_PSTFIRMWARE) }, /* after  USB firmware upload */
    {  USB_DEVICE (kUSR_VID, kMILLER_A_PID_PREFIRMWARE) },
    {  USB_DEVICE (kUSR_VID, kMILLER_A_PID_PSTFIRMWARE) },
    {  USB_DEVICE (kUSR_VID, kMILLER_B_PID_PREFIRMWARE) },
    {  USB_DEVICE (kUSR_VID, kMILLER_B_PID_PSTFIRMWARE) },
    {  USB_DEVICE (kUSR_VID, kHEINEKEN_A_PID_PREFIRMWARE) },
    {  USB_DEVICE (kUSR_VID, kHEINEKEN_A_PID_PSTFIRMWARE) },
    {  USB_DEVICE (kUSR_VID, kHEINEKEN_B_PID_PREFIRMWARE) },
    {  USB_DEVICE (kUSR_VID, kHEINEKEN_B_PID_PSTFIRMWARE) },
    { }
};

/*send this to USB to register the driver*/
static struct usb_driver adi_driver =
{
    .name       = "adiusbadsl",
    .id_table   = adi_ids,
    .probe      = adi_probe,
    .disconnect = adi_disconnect,
    .ioctl      = adi_user
};

/* Linked list of modem Hardware structs */
LIST_HEAD(modem_list);

/* Our /proc dir entry */
struct proc_dir_entry* procdir;

/*Default driver options*/
static const DriverOptions DefaultOptions =
{
    { "OPTN0", DEFAULT_OPTN0 },
    { "OPTN2", DEFAULT_OPTN2 },
    { "OPTN3", DEFAULT_OPTN3 },
    { "OPTN4", DEFAULT_OPTN4 },
    { "OPTN5", DEFAULT_OPTN5 },
    { "OPTN6", DEFAULT_OPTN6 },
    { "OPTN7", DEFAULT_OPTN7 },
    { "OPTN15", DEFAULT_OPTN15 },
    { "VPI", DEFAULT_VPI },
    { "VCI", DEFAULT_VCI },
    { "Encapsulation", DEFAULT_ENCAPS },
    { "Linetype", DEFAULT_LINETYPE },
    { "RatePollFreq", DEFAULT_POLLFREQ },
};


static int CheckOptions(const DriverOptions aOptions)
{
    int result = 1;
    if (aOptions[ConfigEncapsulation].Value != MPOA_MODE_BRIDGED_ETH_LLC &&
	aOptions[ConfigEncapsulation].Value != MPOA_MODE_BRIDGED_ETH_VC &&
	aOptions[ConfigEncapsulation].Value != MPOA_MODE_ROUTED_IP_LLC &&
	aOptions[ConfigEncapsulation].Value != MPOA_MODE_ROUTED_IP_VC &&
	aOptions[ConfigEncapsulation].Value != MPOA_MODE_PPPOA_LLC &&
	aOptions[ConfigEncapsulation].Value != MPOA_MODE_PPPOA_VC)
	result = 0;
    return result;
}




struct EthArpPkt                /* copied from if_arp.h */
{
    unsigned char  h_dest[ETH_ALEN];	
    unsigned char  h_source[ETH_ALEN];	
    unsigned short h_proto;
    unsigned short ar_hrd;	/* format of hardware address	*/
    unsigned short ar_pro;	/* format of protocol address	*/
    unsigned char  ar_hln;	/* length of hardware address	*/
    unsigned char  ar_pln;	/* length of protocol address	*/
    unsigned short ar_op;	/* ARP opcode (command)		*/
    
    /*
     *	 Ethernet looks like this : This bit is variable sized however...
     */
    unsigned char ar_sha[ETH_ALEN]; /* sender hardware address	*/
    unsigned char ar_sip[4];	/* sender IP address		*/
    unsigned char ar_tha[ETH_ALEN]; /* target hardware address	*/
    unsigned char ar_tip[4];	/* target IP address		*/
};

static char *if_name = NULL;

unsigned int module_dbg_mask = 0x00;


/*********************************************************************************/
/* MODULE stuff									 */
/*********************************************************************************/
MODULE_AUTHOR ("Anoosh Naderi <anoosh.naderi@analog.com>");
MODULE_DESCRIPTION ("Analog Devices Inc. USB ADSL Modem driver");
MODULE_DEVICE_TABLE (usb, adi_ids);
MODULE_LICENSE("GPL");
MODULE_PARM (if_name,"s");
MODULE_PARM_DESC (if_name,"Exported ethernet interface name");
MODULE_PARM (module_dbg_mask,"i");
MODULE_PARM_DESC (module_dbg_mask,"Module Debug mask");

EXPORT_NO_SYMBOLS;



/*********************************************************************************/
/* Code that's part of the USB driver interface					 */
/*********************************************************************************/


/*********************************************************************************/
/* adi_init									 */
/*********************************************************************************/
static int __init adi_init (void)
{
    int result = 0;
    
    adi_enters (DBG_INIT);
    
    
    adi_report ("driver V"ADIDRIVERVERSION" loaded\n");

    CrcGenerateTable();
    procdir = proc_mkdir("driver/adimodem",NULL);
    
    if ( procdir )
    {
       procdir->owner = THIS_MODULE;
    }
    else
    {
        adi_report ("could not create /proc/driver/adimodem/\n");
        result = -ENOMEM;
    }


    usb_register(&adi_driver);
    
    
    adi_leaves (DBG_INIT);
    
    return 0;
}

module_init (adi_init);


/*********************************************************************************/
/* adi_exit									 */
/*********************************************************************************/
static void __exit adi_exit (void)
{
    adi_enters (DBG_INIT);
    
    /* This calls automatically the adi_disconnect method if necessary: */
    usb_deregister (&adi_driver);
    
    adi_report ("driver unloaded\n");
    remove_proc_entry("driver/adimodem",NULL);
    
    adi_leaves (DBG_INIT);
    
}

module_exit (adi_exit);


/*********************************************************************************/
/* adi_probe									 */
/*********************************************************************************/
static void *adi_probe(struct usb_device *usb, unsigned int ifnum, const struct usb_device_id *id)
{
    /* Allow multi-device by using an abstract pointer to hardware: */
    UInt32                  pid  = usb->descriptor.idProduct;
    void                   *pRet = NULL;
    char                    path[32];
    struct proc_dir_entry*  procfile;
    Hardware               *pHw  = NULL;
    int                     ret  = 0;
    
    adi_enters (DBG_INIT);
    
    adi_dbg (DBG_INIT,"vid (%#X) pid (%#X), ifnum (%d)\n",
         usb->descriptor.idVendor, usb->descriptor.idProduct, ifnum);

    /* This driver knows only pre and postfirmware devices. */
    if (!ISPREFIRMWARE(pid) && !ISPOSTFIRMWARE(pid))
    {
        adi_dbg (DBG_INIT," Not a supported modem\n");
        goto probe_exit;
    }

    /* Try to select the first USB configuration: */
    if (usb_set_configuration(usb, usb->config[0].bConfigurationValue) < 0)
    {
        adi_err ("adi_probe : set_configuration failed.\n");
        
        goto probe_exit;
    }
    

    switch (pid)
    {
        case kCASE_PREFIRMWARE:
	    /* Create a new pre-firmware device: */
            pHw = GET_KBUFFER (sizeof(Hardware));
            if ( !pHw )
            {
                adi_err ("Not enough memory to get new driver structure..\n");
                goto probe_exit;
            }

            adi_report ("New pre-firmware modem detected\n");

	    /* Reinitialize the modem structure: */
	    memset(pHw, 0, sizeof(Hardware));
	    pHw->usbdev = usb;
            pHw->MsgInitialized = FALSE;
            pHw->UnplugFlag = FALSE; // 04152003
            
            
           adi_report ("Uploading firmware..\n");
           
           ret = loadUsbFirmware ( pHw, pid );
           if ( ret < 0 ) 
           {
               adi_err ("Can't upload firmware to modem...\n");
               goto probe_exit;
           }
           
           adi_report ("Binding Hardware to USB %03d/%03d\n",
                  usb->bus->busnum, usb->devnum);
           
           list_add(&pHw->list,&modem_list);
           
           pRet = pHw;
           
           break;
            
#if 0        
    case kCASE_PREFIRMWARE:
	{
	    /* Create a new pre-firmware device: */
            pHw = GET_KBUFFER (sizeof(Hardware));
           if ( !pHw )
            {
                break;
            }

	    adi_report ("New modem detected, waiting for firmware...\n");
            
	    /* Reinitialize the modem structure: */
	    memset(pHw, 0, sizeof(Hardware));
	    pHw->usbdev = usb;
            pHw->MsgInitialized = FALSE;
            
           adi_report ("Binding Hardware to USB %03d/%03d\n",
                  usb->bus->busnum, usb->devnum);
            
           
           list_add(&pHw->list,&modem_list);
           
           pRet = pHw;
	}    
        break;
#endif /* 0 */        
    case kCASE_POSTFIRMWARE:
	{
	    int i;
	    int ret;
            struct usb_endpoint_descriptor *epindata;
            struct usb_endpoint_descriptor *epoutdata;
            struct usb_endpoint_descriptor *epoutidma;
	    /* Create a new post-firmware device: */
            pHw = GET_KBUFFER (sizeof (Hardware));
            
            if ( !pHw )
            {
                break;
            }
            
	    /* Reinitialize the modem structure (and the state machine): */
	    memset(pHw, 0, sizeof(Hardware));
	    pHw->usbdev = usb;
            pHw->MsgInitialized = FALSE;
            pHw->UnplugFlag = FALSE; // 04152003

	    init_waitqueue_head(&pHw->OperationalQueue);
	    pHw->DSPLock = RW_LOCK_UNLOCKED;
	    pHw->NetLock = SPIN_LOCK_UNLOCKED;
	    sema_init(&pHw->NetSem, 1);

            /* Add our new device's list hook to the modem_list */
            list_add(&pHw->list,&modem_list);
           
            adi_report ("New USB ADSL device detected, "
                   "waiting for DSP code...\n");
	    
            /**********************************************************************/
            /* Some one-time init stuff						  */
            /* For now we'll just choose the biggest pipe. This will insure       */
            /* that we'll be ok no matter what the train rate is. Unfortunately,  */
            /* the driver model doesn't fit well with delaying read pipe size     */
            /* figuring, etc., until after we're trained.			  */
            /**********************************************************************/

            /* FIXME : for several modems */
            pHw->IsoPipeSize     = kFASTEST_ISO_RATE;
            pHw->IsoFramesPerUrb = kFRAMES_PER_ISO_URB;

#ifdef USEBULK
		  pHw->LowRateFlag = FALSE; //Low bit Rate: 04042003
#else
		  pHw->LowRateFlag = TRUE; //Low bit Rate: 04042003
#endif

            /**********************************************************************/
            /*We use constants for the endpoint addresses - but this is the       */
            /*easiest way for now - and most of the other device-specific         */
            /*usb drivers do the same. Eliminates us having to parse through      */
            /*the configuration descriptor, etc., to get these numbers            */
            /**********************************************************************/
            pHw->pipeBulkIdmaOut = usb_sndbulkpipe(usb, kEP_BULK_IDMA_OUT);
            pHw->pipeBulkDataOut = usb_sndbulkpipe(usb, kEP_BULK_DATA_OUT);
            pHw->pipeBulkDataIn  = usb_rcvbulkpipe(usb, kEP_BULK_DATA_IN);
            pHw->pipeIsoDataIn   = usb_rcvisocpipe(usb, kEP_ISOC_DATA_IN);
            pHw->pipeIntIn       = usb_rcvintpipe(usb, kEP_INT_IN);

            /*Make sure we can get all the memory we need*/
            pHw->pInterruptData = GET_KBUFFER(sizeof(CDC_NOTIFY));
            pHw->pReadyData     = GET_KBUFFER(ETHERNET_PACKET_DATA_SIZE);
            pHw->pOutgoingData  = GET_KBUFFER(OUTGOING_DATA_SIZE);
            for (i=0; i<INCOMING_Q_SIZE; i++)
            {
#ifdef USEBULK
                pHw->pIncomingData[i] = GET_KBUFFER(INCOMING_DATA_SIZE);
#else
                pHw->pUrbReadIso[i]   = usb_alloc_urb(pHw->IsoFramesPerUrb);
                pHw->pIncomingData[i] = GET_KBUFFER((pHw->IsoPipeSize * pHw->IsoFramesPerUrb));
#endif
                if (!pHw->pIncomingData[i])
                {
                    adi_err ("adi_probe : Can't allocate memory\n");
                    goto probe_exit;
                }
            }

            /*If any of those buffers were not succesfully allocated, we*/
            /*cannot proceed succesfully                                */
            if ( (!pHw->pInterruptData) ||
                 (!pHw->pReadyData)     ||
                 (!pHw->pOutgoingData) )
            {
                adi_err ("adi_probe : Can't allocate memory\n");
                goto probe_exit;
            }

            pHw->pOAMCell = GET_KBUFFER(128);
            if (pHw->pOAMCell == 0 )
            {
                adi_err ("adi_probe : Can't allocate memory\n");
                goto probe_exit;
            }       

            /* Routed Mode */
            pHw->pPacket = GET_KBUFFER(ETHERNET_MAX_PACKET_SIZE);
            if (pHw->pPacket == 0 )
            {
                adi_err ("adi_probe : Can't allocate memory\n");
                goto probe_exit;
            }	

            
            /*************************************************************************/
            /* These buffers are used to make the Uni code happy. It was written     */
            /* for MacOS9 - where we had to keep an internal queue of incoming       */
            /* and outgoing packets. Since we don't have to do that here on Linux,   */
            /* we essentially create 1-deep queues and slightly modified the         */
            /* Uni code so it just uses these buffers instead of pulling from queues.*/
            /*************************************************************************/
            pHw->ReassemblyBuffer.GB.AllocedSize   = ETHERNET_PACKET_DATA_SIZE;
            pHw->ReassemblyBuffer.GB.pData         = pHw->pReadyData;
            pHw->SegmentationBuffer.GB.AllocedSize = OUTGOING_DATA_SIZE;
            pHw->SegmentationBuffer.GB.pData       = pHw->pOutgoingData;
            pHw->mru = 0;
            
            /*Initialize the CTRL URB queue*/
            ret = alloc_queued_urb_ctrl ( pHw, CTRL_URB_Q_SIZE );
            
            if (ret != 0)
            {
                adi_err ("adi_probe : alloc_queued_urb_ctrl out of memory\n");
                goto probe_exit;
            }

            INIT_LIST_HEAD(&pHw->ctrl_urb_ipg_q);
            pHw->ctrl_q_lock = SPIN_LOCK_UNLOCKED;
            pHw->ctrl_urb_failed= FALSE;
            
            /*************************************************************************/
            /* Initialize our kernel timers					     */
            /* WatchdogTimer is the "management entity", per the ADI documentation.  */
            /* It is responsible for initiating a regular status check of the modem, */
            /* and it is the main "driver" for the state machine.		     */
            /*************************************************************************/
            init_timer(&pHw->AdiModemSm.timerSM);
            pHw->AdiModemSm.timerSM.function = WatchdogTimer;
            pHw->AdiModemSm.timerSM.data     = (unsigned long) pHw;

            /*************************************************************************/
            /* The UHCI usb driver does not allow us to queue control urbs.          */
            /* Therefore, to make my own asynchronous control urb queue, I           */
            /* have a timer that goes off after 1 second after I sent a control      */
            /* urb to check to see if another control urb needs submitted. 	     */
            /*************************************************************************/
            init_timer(&pHw->CtrlUrbQueueTimer);
            pHw->CtrlUrbQueueTimer.function  = ctrl_urb_q_watcher;
            pHw->CtrlUrbQueueTimer.data      = (unsigned long) pHw; 

            init_timer (&pHw->ctrl_urb_retry);
            pHw->ctrl_urb_retry.function = ctrl_urb_retry_send;
            pHw->ctrl_urb_retry.data = (unsigned long) pHw;
            pHw->ctrl_urb_failed = NULL;
            
            
            /*Other init stuff*/
            pHw->AdiModemSm.HeartbeatCounter = 0; 
            pHw->AdiModemSm.CurrentAdiState  = STATE_JUST_PLUGGED_IN;

            /*************************************************************************
             * FFD102902: OAM Timer Init. 
             *************************************************************************/
            init_timer(&pHw->OAMTimer);
            pHw->OAMTimer.function  = OAMTimerFunction;
            pHw->OAMTimer.data      = (unsigned long)pHw; 
            /*************************************************************************/


            
#ifdef USEBULK
            /*****************************************************************************/
            /* Other drivers seem to be calling usb_set_interface, but we're not using an*/
            /* alternate interface on any of the interfaces, so we don't need to as far  */
            /* as I can understand. All we really need to do from a USB stack perspective*/
            /* is tell usb that we will handle all the interfaces for this device, so he */
            /* doesn't enumerate through all drivers for each interface.		 */
            /*****************************************************************************/
#endif
            usb_driver_claim_interface(&adi_driver, &(usb->actconfig->interface[0]), pHw);
            usb_driver_claim_interface(&adi_driver, &(usb->actconfig->interface[1]), pHw);
            usb_driver_claim_interface(&adi_driver, &(usb->actconfig->interface[2]), pHw);
            /* Get information about our endpoints*/
            epoutidma = usb->actconfig->interface[1].altsetting[0].endpoint + 0;
            epoutdata = usb->actconfig->interface[1].altsetting[0].endpoint + 1; 
#ifdef USEBULK
            epindata  = usb->actconfig->interface[2].altsetting[0].endpoint + 0;
#else /*ifdef USEBULK*/
            /*******************************************************************************/
            /* Need to eventually choose the alternate interface based on the train ds rate*/
            /* But, like I said above, the driver model doesn't fit well into the	   */
            /* concept of delaying the ISO pipe startup					   */
            /*******************************************************************************/

            /*FIXME : for several modems*/
            if (usb_set_interface(usb, kUSB_INTF_IN, kFASTEST_ISO_INTF) < 0)
            {                
                adi_err ("usb_set_interface failed on iso alt 7\n");
            }
            else
            {
                epindata=usb->actconfig->interface[kUSB_INTF_IN].altsetting[kFASTEST_ISO_INTF].endpoint+0; 
            }
#endif /*ifdef USEBULK else*/

            /* Setup our /proc interface */
            snprintf(path,32,"%03d-%03d",pHw->usbdev->bus->busnum, pHw->usbdev->devnum);            
            procfile=create_proc_read_entry(path,0, procdir, adi_read_proc, pHw);
            
            if (procfile)
            {
                procfile->owner=THIS_MODULE;
                adi_report ("created proc entry at : /proc/driver/adimodem/%s\n",
                            path);
            }
            else
            {
                adi_err ("failed to create proc entry at : /proc/driver/%s !\n",
                         path);
            }
	    
	    /*******************************************************************************/
	    /* Even though the rest of the drivers that claim multiple interfaces	   */
	    /* just go ahead and return NULL from the initial probe - that causes	   */
	    /* the usb stack to think the device isn't claimed, so it goes and tries       */
	    /* to load another driver. In our case, since the device advertises 	   */
	    /* itself as an ethernet control model comm class device, the acm and	   */
	    /* cdcether drivers get loaded. So, we'll return our private pointer           */
	    /* here too (in addition to the usb_driver_claim_interface above), to          */
	    /* keep usb from trying to load another driver			           */
	    /*******************************************************************************/
	    pRet = pHw;
        }
        break;
    }

probe_exit:
    if ( ( pRet == NULL ) &&
         ( pHw != NULL) )
    {
        int i;
        
        /*
          pRet NULL means that something wrong occurred.
          Thus we have to do some house-cleaning :(
        */
        
        if ( pHw->pInterruptData ) {
            FREE_KBUFFER (pHw->pInterruptData);
        }
        
        if ( pHw->pReadyData ) {
            FREE_KBUFFER (pHw->pReadyData);
        }
        
        if ( pHw->pOutgoingData ) {
            FREE_KBUFFER (pHw->pOutgoingData);
        }
        
        for (i=0; i<INCOMING_Q_SIZE; i++)
        {
            if ( pHw->pIncomingData[i] ) 
            {
#ifdef USEBULK
                FREE_KBUFFER (pHw->pIncomingData[i]);
#else
                usb_free_urb ( pHw->pUrbReadIso[i] );                
                FREE_KBUFFER (pHw->pIncomingData[i]);
#endif
            }
            else
            {
                /*
                  The allocation step stops as soon as
                  one alloc failed. So, past this point, no
                  pIncomingData has been allocated
                */
                break;
            }
        }
        
        FREE_KBUFFER (pHw);
    }        

    
    adi_leaves (DBG_INIT);
    return pRet;
}

/*******************************************************************************/
/* adi_disconnect							       */
/*******************************************************************************/
static void adi_disconnect(struct usb_device *usb, void *ptr)
{
    Hardware *pHw = (Hardware *)ptr;
    UInt32 pid = usb->descriptor.idProduct;
    char path[32];

    adi_enters (DBG_INIT);
    
    /*We can't do anything if we don't have a valid Hardware pointer*/
    if (pHw == NULL)
    {
        adi_err ("adi_disconnect : No device.\n");
        goto dis_done;
    }

    pHw->UnplugFlag = TRUE; 

    
    switch (pid)
    {
    case kCASE_PREFIRMWARE:
        list_del(&pHw->list);
        FREE_KBUFFER(pHw);
        
        adi_report ("Modem removed\n");
        
	break;
    case kCASE_POSTFIRMWARE:
	{
	    int i;
	    /* If timers are currently running, delete them: */
	    if (timer_pending(&pHw->AdiModemSm.timerSM))
            {
		del_timer(&pHw->AdiModemSm.timerSM);
            }
            
	    if (timer_pending(&pHw->CtrlUrbQueueTimer))
            {
		del_timer(&pHw->CtrlUrbQueueTimer);
            }
            
            if (timer_pending (&pHw->ctrl_urb_retry)) 
            {
		del_timer(&pHw->ctrl_urb_retry);
            }
            
            
	    read_lock(&pHw->DSPLock);
	    /* Unlink pending interrupt URBs: */
	    if (pHw->HasIntURB)
	    {
                pHw->urbInt.transfer_flags &= ~USB_ASYNC_UNLINK;
                usb_unlink_urb(&pHw->urbInt);
		pHw->HasIntURB = FALSE;
	    }
           /* Free the DSP code: */ /* FIXME : don't duplicate DSP code for
                                     * every modem ? */
	    FreeDspData(&pHw->MainPage, &pHw->pSwapPages, &pHw->SwapPageCount);
	    read_unlock(&pHw->DSPLock);
	    
            if (timer_pending(&pHw->OAMTimer))
                del_timer(&pHw->OAMTimer);
            
            /* Remove our network interface from the kernel: */
	    spin_lock(&pHw->NetLock);
	    if (pHw->pLinuxNet)
	    {
		/* This will call the adi_close method: */
		unregister_netdev(pHw->pLinuxNet);
		kfree(pHw->pLinuxNet);
		pHw->pLinuxNet = NULL;
	    }
	    spin_unlock(&pHw->NetLock);
    
	    /* Free memory we alloced: */
	    FREE_KBUFFER(pHw->pInterruptData);
	    FREE_KBUFFER(pHw->pReadyData);
	    for (i=0; i<INCOMING_Q_SIZE; i++)
	    {
		FREE_KBUFFER(pHw->pIncomingData[i]);
#ifndef USEBULK
		usb_free_urb(pHw->pUrbReadIso[i]);
#endif
	    }
            
            FREE_KBUFFER(pHw->pOAMCell); 
            FREE_KBUFFER(pHw->pPacket);

            unlink_ipg_ctrl_urb ( pHw );
            free_queued_urb_ctrl (&pHw->ctrl_urb_free_q);
            free_queued_urb_ctrl (&pHw->ctrl_urb_ipg_q);
	    
            /* Tell usb that we no longer claim these interfaces as our property: */
            usb_driver_release_interface(&adi_driver, &(usb->actconfig->interface[0]));
            usb_driver_release_interface(&adi_driver, &(usb->actconfig->interface[1]));
            usb_driver_release_interface(&adi_driver, &(usb->actconfig->interface[2]));

#if 0            
            if (pHw->pDriverCMVs) {
                FREE_VBUFFER (pHw->pDriverCMVs);
            }
#endif
            
            list_del(&pHw->list);
            
            snprintf(path,32,"%03d-%03d",pHw->usbdev->bus->busnum, pHw->usbdev->devnum);
            remove_proc_entry(path, procdir);
            
            FREE_KBUFFER(pHw);
            
            adi_report ("ADSL device removed\n");
	}
	break;
    }

dis_done:
    adi_leaves (DBG_INIT);
}

/*******************************************************************************/
/* adi_irq								       */
/*******************************************************************************/
static void adi_irq(struct urb *urb)
{
    Hardware *pHw;

    adi_enters (DBG_INTS);
    
    /*We put the modem pointer as the context*/
    pHw = (Hardware *)urb->context;
    if (NULL == pHw)
    {
        adi_err ("adi_irq : No device.\n");
	goto irq_done;
    }
    if (!pHw->HasIntURB)
    {
        adi_err ("adi_irq : Callback on unregistered interrupt handler!\n");
	goto irq_done;
    }

    if (urb->status ==  -ENOENT) 
    {
        /*
          User cancelled this URB : do nothing, and don't reset status to 0,
          or the URB will be re-used
        */
        adi_report ("adi_irq : URB canceled by user.\n");
        goto irq_done;
    }
        
    /*Lets check the status first, to make sure this was succesfull*/
    if ( urb->status < 0 )
    {
        adi_err ("adi_irq : URB status indicates error (%d)\n",
                 urb->status);
        /* These errors seem to be what we get when the device has been unplugged. */
	/* adi_disconnect will soon be called, meanwhile we must skip this interrupt: */
        urb->status = 0;
        goto irq_done;
    }

    /*All is well, we can process the data!*/
    if (pHw->pInterruptData->bmRequestType == 0x08) /* device-to-host interrupt */
    {
        DeviceInt *pInt;

        pInt = (DeviceInt *)(&(pHw->pInterruptData->Data[0])); 
        switch (cpu_to_le16(pInt->Interrupt))
        {
        case kADI_INT_LOADSWAPPAGE:
	    {
                UInt16 SwapData = cpu_to_le16(pInt->IntInfo.SwapIntInfo.SwapData);
                adi_dbg (DBG_INTS,"interrupt = kADI_INT_LOADSWAPPAGE\n");
                if (pHw->AdiModemSm.CurrentAdiState != STATE_HARD_RESET_INITIATE)
		{
		    read_lock(&pHw->DSPLock);
                    IdmaUploadSwapPage(pHw, SwapData);
		    read_unlock(&pHw->DSPLock);
		}
            }
            break;
        case kADI_INT_INCOMINGCMV:
            {
                adi_dbg (DBG_INTS,"interrupt = kADI_INT_INCOMINGCMV\n");
                
                ProcessIncomingCmv(pHw, pInt->IntInfo.CmvIntInfo.CmvData);
            }
        break;
        default:
            adi_dbg (DBG_INTS,"interrupt = unsupported(%d)\n", pInt->Interrupt);
            
            break;
        }
    }

irq_done:
    adi_leaves (DBG_INTS);
    
}

/* Walk the Hardware list to find the one related to the current usb_device */
static Hardware *find_hardware( struct usb_device *dev )
{
    struct list_head *ptr;
    Hardware         *pHw = NULL;
    
    for (ptr = modem_list.next; ptr != &modem_list; ptr = ptr->next)
    {
        Hardware *entry;
        entry=list_entry(ptr, Hardware, list);
        if (entry->usbdev == dev)
        {
            pHw=entry;
            break;
        }
    }
    return pHw;
}


/*******************************************************************************/
/* adi_user								       */
/*									       */
/* This is actually the ioctl handler for ioctls coming from the usermode      */
/* helper application. The other ioctl routine, adi_ioctl, will receive        */
/* ioctls coming from network socket applications.			       */
/*******************************************************************************/
static int adi_user(struct usb_device *dev, unsigned int code, void *buf)
{
    Hardware *pHw;
    int retval = -ENOTTY;
    UInt32 pid = dev->descriptor.idProduct;

    /*
      USB automatically transfers user ioctl structure in kernel
      address space (the size is already encoded in the ioctl code):
    */
    struct adi_ioctl_info *pIOCTLinfo = (struct adi_ioctl_info *) buf;
    
    MOD_INC_USE_COUNT;
    
    if (!(pHw = find_hardware(dev)))
    {
        adi_report ("Could not find Hardware for USB device %03d/%03d\n",
                    dev->bus->busnum, dev->devnum);
        MOD_DEC_USE_COUNT;
        return -ENOTTY;
    }

    /* Check this ioctl if for one of our devices: */
    switch (pid)
    {
        case kCASE_PREFIRMWARE:
            switch (code)
            {
                case ADIUSBADSLSETDBG:
                    if (NULL == pIOCTLinfo) 
                    {
                        adi_err ("ADIUSBADSLSETDBG : No data\n");
                        break;
                    }
                
                    module_dbg_mask = pIOCTLinfo->IDMAStart;
                    adi_report ("Debug mask set to 0x%x\n",module_dbg_mask);
                    retval = 0;                        
                    break;
                
                case ADIUSBADSLGETDBG:
                    if (NULL == pIOCTLinfo) 
                    {
                        adi_err ("ADIUSBADSLGETDBG : No data\n");
                        break;
                    }
                    pIOCTLinfo->IDMAStart = module_dbg_mask;
                    retval = 0;                
                    break;
                
/*                 case ADIUSBADSLFIRMWARE: */
/*                 { */
/*                     FIRMWARE_RECORD *pFirmware = NULL; */
                
/*                     retval = -ERESTARTSYS; */
                
                
/*                     retval = -EFAULT; */
                
/*                     if (NULL == pIOCTLinfo)  */
/*                     { */
/*                         adi_err ("ADIUSBADSLFIRMWARE : No data\n"); */
/*                         goto end_load_firmware; */
/*                     } */
                
                
/*                     pFirmware = (FIRMWARE_RECORD *) GET_VBUFFER(pIOCTLinfo->BufferSize); */
/*                     if (NULL == pFirmware) */
/*                     { */
/*                         retval = -ENOMEM; */
/*                         adi_err ("ADIUSBADSLFIRMWARE : No more memory\n"); */
                    
/*                         goto end_load_firmware; */
/*                     } */
                
/*                     if (copy_from_user(pFirmware, pIOCTLinfo->pBuffer, pIOCTLinfo->BufferSize) != 0) */
/*                     { */
/*                         adi_err ("ADIUSBADSLFIRMWARE : copy from user failed\n"); */
                    
/*                         goto end_free_firmware; */
/*                     } */
                
/*                     adi_report ("ioctl ADIUSBADSLFIRMWARE received\n"); */
                
/*                     /\* Load the firmware: *\/ */
/*                     adi_report ("Loading firmware to modem...\n"); */
                
/*                     if (loadUsbFirmware(pHw, pFirmware, pIOCTLinfo->BufferSize) < 0) */
/*                     { */
/*                         adi_err ("Unable to load firmware to modem!\n"); */
/*                         retval = -EIO; */
/*                     } */
/*                     else */
/*                     { */
/*                         adi_report ("Firmware successfully loaded to modem\n"); */
                    
/*                         retval = 0; */
/*                     } */
/*                   end_free_firmware: */
/*                     FREE_VBUFFER(pFirmware); */
/*                   end_load_firmware: */
/*                 } */
            }
            break;
        case kCASE_POSTFIRMWARE:
            switch (code)
            {
                    
                case ADIUSBADSLSETDBG:
                    if (NULL == pIOCTLinfo) 
                    {
                        adi_err ("ADIUSBADSLSETDBG : No data\n");
                        break;
                    }
                
                    module_dbg_mask = pIOCTLinfo->IDMAStart;
                    adi_report ("Debug mask set to 0x%x\n",module_dbg_mask);
                    retval = 0;                
                    break;
                
                case ADIUSBADSLGETDBG:
                    if (NULL == pIOCTLinfo) 
                    {
                        adi_err ("ADIUSBADSLGETDBG : No data\n");
                        break;
                    }
                    pIOCTLinfo->IDMAStart = module_dbg_mask;
                    retval = 0;                
                    break;
                    
#if 0            
                case ADIUSBADSLCMVS:
                    retval = -ERESTARTSYS;
                    if (down_interruptible(&pHw->NetSem)) 
                    {
                        adi_err ("ADIUSBADSLCMVS : Can't lock\n");
                        break;
                    }

                    if ( pHw->IsOpen ) 
                    {
                        retval = -EBUSY;
                        adi_err ("ADIUSBADSLCMVS : Eth device already open.\n");                        
                        goto end_set_cmvs;
                    }

                    
                    if (NULL == pIOCTLinfo) 
                    {
                        adi_err ("ADIUSBADSLCMVS : No data.\n");
                        retval = -EFAULT;
                        goto end_set_cmvs;
                    }

                    if (pIOCTLinfo->pBuffer == NULL ) 
                    {
                        adi_err ("ADIUSBADSLCMVS: Null input buffer\n");
                        retval = -EINVAL;
                        goto end_set_cmvs;
                    }

                    if ( pIOCTLinfo->BufferSize == 0 ) 
                    {
                        adi_err ("ADIUSBADSLCMVS: Invalid Buffersize\n");
                        retval = -EINVAL;
                        goto end_set_cmvs;    
                    }
                    
                    /* Get a buffer for CMVs */
                    adi_dbg (DBG_INIT,"Allocating %d bytes for CMVs\n",pIOCTLinfo->BufferSize);
                    
                    pHw->pDriverCMVs = GET_VBUFFER (pIOCTLinfo->BufferSize);
                    if ( pHw->pDriverCMVs == NULL )
                    {
                        adi_err ("ADIUSBADSLCMVS: Not enough memory to get %d bytes\n",
                                 pIOCTLinfo->BufferSize);
                        retval = -ENOMEM;
                        goto end_set_cmvs;
                    }                    
                    
                
                    if (copy_from_user(pHw->pDriverCMVs, pIOCTLinfo->pBuffer, pIOCTLinfo->BufferSize))
                    {
                        retval = -EFAULT;
                        FREE_VBUFFER (pHw->pDriverCMVs);
                        adi_err ("ADIUSBADSLCMVS : copy from user failed.\n");
                        goto end_set_cmvs;
                    }


                    adi_report ("ioctl ADIUSBADSLCMVS received and treated.\n");
                    retval = 0;
                    
              end_set_cmvs:
                    up(&pHw->NetSem);
                    break;
#endif
                    
                case ADIUSBADSLOPTIONS:		/* Set driver options */
                    /* Option cannot be set while network interface is up */
            
                    retval = -ERESTARTSYS;
                    if (down_interruptible(&pHw->NetSem)) 
                    {
                        adi_err ("ADIUSBADSLOPTIONS : Can't lock\n");
                        break;
                    }

#if 0
                    if ( pHw->pDriverCMVs == NULL ) 
                    {
                        /* User as not yet sent ADIUSBADSLCMVS */
                        adi_err ("ADIUSBADSLOPTIONS: CMVs not yet sent.\n");
                        goto end_set_options;
                    }
#endif                    
                    retval = -EBUSY;
            
                    if (!pHw->IsOpen)
                    {
                        DriverOptions aOptions;
                
                        memcpy(&aOptions, &DefaultOptions, sizeof(DriverOptions));
                
                        retval = -EFAULT;
                        if (NULL == pIOCTLinfo) 
                        {
                            adi_err ("ADIUSBADSLOPTIONS : No data.\n");
                    
                            goto end_set_options;
                        }
                
                        /* Check the option array size: */
                        retval = -EINVAL;
                        if (sizeof(aOptions) != pIOCTLinfo->BufferSize) 
                        {
                            adi_err ("ADIUSBADSLOPTIONS : Invalid Data size\n");
                    
                            goto end_set_options;
                        }
                
                        if (copy_from_user(&aOptions, pIOCTLinfo->pBuffer, sizeof(aOptions)))
                        {
                            retval = -EFAULT;
                            adi_err ("ADIUSBADSLOPTIONS : copy from user failed.\n");
                    
                            goto end_set_options;
                        }
                        /* Options retrieved, check them: */
                        if (CheckOptions(aOptions))
                        {
                            /* OK, initialize Mpoa and Msg: */
                            adi_report ("ioctl ADIUSBADSLOPTIONS received\n");
                    
                            /* Initialize the message handling */
                            MsgInitialize(pHw, aOptions);
                            MpoaInitialize(pHw, aOptions);
                    
                            pHw->MsgInitialized = TRUE;

                            retval = 0;
                        }
                    }
                    else
                    {
                        retval = -EBUSY;
                        adi_err ("ADIUSBADSLOPTIONS : Eth device already open.\n");
                
                    }
            
              end_set_options:
                    up(&pHw->NetSem);
                    break;
            
                case ADIUSBADSLDSP:		/* Load DSP code */
                    /* DSP cannot be loaded while network interface is up */
                    retval = -ERESTARTSYS;
                    if (down_interruptible(&pHw->NetSem)) 
                    {
                        adi_err ("ADIUSBADSLDSP : Can't lock\n");
                        break;
                    }

                    if ( !pHw->MsgInitialized ) 
                    {
                        adi_err ("Message handling not initialized."
                                 " Please send options.\n");
                        retval = -EINVAL;
                        goto end_load_adsl_dsp;
                    }
            
                    if (!pHw->IsOpen)
                    {
                        UInt8 *pBuf = NULL;
                        IDMAPage MainPage;
                        IDMAPage *pSwapPages;
                        UInt32   SwapPageCount;
                        long flags;
                
                        retval = -EFAULT;
                        if (NULL == pIOCTLinfo) 
                        {
                            adi_err ("ADIUSBADSLDSP : No data.\n");
                            goto end_load_adsl_dsp;
                        }
                
                        if (pIOCTLinfo->IDMAStart >= pIOCTLinfo->BufferSize) 
                        {
                            adi_err ("ADIUSBADSLDSP : Invalid Data Size\n");
                            goto end_load_adsl_dsp;
                        }
                
                        retval = -ENOMEM;
                        pBuf = GET_VBUFFER(pIOCTLinfo->BufferSize);
                        if (NULL == pBuf) 
                        {
                            adi_err ("ADIUSBADSLDSP : No memory.\n");
                    
                            goto end_load_adsl_dsp;
                        }
                
                        retval = -EFAULT;
                        if (copy_from_user(pBuf, pIOCTLinfo->pBuffer, pIOCTLinfo->BufferSize))
                        {
                            adi_err ("ADIUSBADSLDSP : copy from user failed.\n");
                    
                            goto free_adsl_dsp;
                        }
                
                        /* Get our DSP code ready for IDMA booting */
                        adi_report ("ioctl ADIUSBADSLDSP received\n");
                
                        retval = ProcessDSPCode(pHw, pBuf, pIOCTLinfo->IDMAStart, pIOCTLinfo->BufferSize,
                                                &MainPage, &pSwapPages, &SwapPageCount);
                        if (retval != 0)
                        {
                            adi_err ("ADIUSBADSLDSP : ProcessDSPCode failed (%d)\n",
                                     retval);
                            FreeDspData(&MainPage, &pSwapPages, &SwapPageCount);
                            goto free_adsl_dsp;
                        }
                
                        /* Ok, we got our DSP code, replace the old DSP code by the new one: */
                        write_lock_irqsave(&pHw->DSPLock, flags);
                        FreeDspData(&pHw->MainPage, &pHw->pSwapPages, &pHw->SwapPageCount);
                        pHw->MainPage.BlockCount = MainPage.BlockCount;
                        pHw->MainPage.Blocks = MainPage.Blocks;
                        pHw->pSwapPages = pSwapPages;
                        pHw->SwapPageCount = SwapPageCount;
                        write_unlock_irqrestore(&pHw->DSPLock, flags);
                
                        /* Load the DSP code if necessary: */
                        adi_report ("Loading DSP code to device...\n");
                
                        read_lock(&pHw->DSPLock);
                        retval = -EIO;
                        if (!pHw->HasIntURB)
                        {
                            /* Get the USB endpoint for IDMA uploading: */
                            struct usb_endpoint_descriptor *epint =
                                dev->actconfig->interface[0].altsetting[0].endpoint + 0;
                            /* Install the interrupt handler to send IDMA pages to the modem
                               and handle further incoming data: */
                            usb_fill_int_urb(&pHw->urbInt, dev, pHw->pipeIntIn, pHw->pInterruptData,
                                             sizeof(CDC_NOTIFY), adi_irq, pHw, epint->bInterval);
                            pHw->HasIntURB = TRUE;
                            if (usb_submit_urb(&pHw->urbInt) < 0)
                            {
                                adi_err ("Unable to submit interrupt URB!\n");
                        
                                pHw->HasIntURB = FALSE;
                                goto unlock_adsl_dsp;
                            }

                        }
                
                        /* Send the DSP code to the modem: */
                        if (!pHw->HasIntURB || BootTheModem(pHw, FALSE) < 0)
                        {
                            adi_err ("Unable to load DSP code to device!\n");
                    
                            goto unlock_adsl_dsp;
                        }
                        else
                        {
                            adi_report ("DSP code successfully loaded to device\n");
                        }
                
                        retval = 0;
                
                      unlock_adsl_dsp:
                        read_unlock(&pHw->DSPLock);
                      free_adsl_dsp:
                        FREE_VBUFFER(pBuf);
                    }
                    else
                    {
                        retval = -EBUSY;
                        adi_err ("ADIUSBADSLDSP : Eth device already open\n");
                    }
            
              end_load_adsl_dsp:
                    up(&pHw->NetSem);
                    break;
            
                case ADIUSBADSLGETITF:		/* Get the network interface name */
                {
                    uint32_t dwNameLen;
                    unsigned long flags;
                
                    retval = -EFAULT;
                    if (NULL == pIOCTLinfo) 
                    {
                        adi_err ("ADIUSBADSLGETITF : No data.\n");
                    
                        break;
                    }
                
                    /* Create the Ethernet interface if necessary: */
                    retval = -ERESTARTSYS;
                    if (down_interruptible(&pHw->NetSem)) 
                    {
                        adi_err ("ADIUSBADSLDSP : Can't lock\n");
                    
                        break;
                    }
                
                    spin_lock_irqsave(&pHw->NetLock, flags);
                    if (!pHw->pLinuxNet)
                    {
                        retval = create_etherdev(pHw);
                        if (retval < 0)
                        {
                            adi_err ("ADIUSBADSLDSP : eth dev creation failed (%d)\n",
                                     retval);
                            goto end_get_itf;
                        }                    
                    }
                
                    dwNameLen = strlen(pHw->pLinuxNet->name) + sizeof(char);
                    if (NULL == pIOCTLinfo->pBuffer)
                    {
                        /* Get the requested size */
                        pIOCTLinfo->BufferSize = dwNameLen;
                        retval = 0;
                    }
                    else
                    {
                        /* Get the interface name */
                        if (pIOCTLinfo->BufferSize < dwNameLen)
                        {
                            adi_err ("ADIUSBADSLGETITF : Invalid Buffer size\n");
                        
                            retval = -EINVAL;
                            goto end_get_itf;
                        }
                        if (copy_to_user(pIOCTLinfo->pBuffer, pHw->pLinuxNet->name, dwNameLen) != 0)
                        {
                            adi_err ("ADIUSBADSLGETITF : copy to user failed\n");
                        
                            retval = -EFAULT;
                            goto end_get_itf;
                        }
                        retval = 0;
                    }
                  end_get_itf:
                    spin_unlock_irqrestore(&pHw->NetLock, flags);
                    up(&pHw->NetSem);
                    break;
                }
                break;
            
                case ADIUSBADSLSYNC:		/* Wait for modem "operational" state */
                    if (wait_event_interruptible(pHw->OperationalQueue,
                                                 ((pHw->AdiModemSm.CurrentAdiState & STATE_OPERATIONAL) == STATE_OPERATIONAL)))
                        retval = -EINTR;
                    else
                        retval = 0;
                    break;
            
                default:
                    /* Bad ioctl */
                    break;
            }
            break;
        default:
            /* Bad device */
            break;
    }
    MOD_DEC_USE_COUNT;
    return retval;
}

/*******************************************************************************/
/* adi_read_proc							       */
/*******************************************************************************/
static int adi_read_proc(char *page, char **start, off_t off, int count, int *eof, void *data)
{
    Hardware *pHw = (Hardware *)data;
    int size;
    char *p = page;
    /*If this output needs to be larger than 4K (PAGE_SIZE), we need to do this differently*/
    p += sprintf(p, "\nAnalog Devices USB ADSL Modem Status Display\n");
    p += sprintf(p, "-------------------------------------------------------------\n");
    p += sprintf(p, "USB Bus : %03d\t USB Device : %03d Dbg mask: 0x%x\n",
                 pHw->usbdev->bus->busnum, pHw->usbdev->devnum, module_dbg_mask);
    if (pHw->pLinuxNet->name)
    {
        p += sprintf(p, "Ethernet Interface : %s\n",pHw->pLinuxNet->name);
    }
    else
    {
        p += sprintf(p, "Ethernet Interface : none\n");
    }
    p += sprintf(p, "Tx Rate  %10.10d  Rx Rate  %10.10d  Crc      %10.10d\n",
                 pHw->AdiModemSm.UpRate / 1024, 
                 pHw->AdiModemSm.DownRate / 1024, 
                 pHw->Statistics[STAT_CELLS_LOST_CRC]);
    p += sprintf(p, "FEC      %10.10d  Margin   %10.10d  Atten    %10.10d dB\n",
                 pHw->AdiModemSm.stats_Uncorr_Blks,
                 pHw->AdiModemSm.stats_Cur_SNR & 0xFF,
                 (pHw->AdiModemSm.stats_Cur_Atten & 0xFF)/2);
    p += sprintf(p, "VID-CPE  %10.10d  VID-CO   %10.10d  HEC      %10.10d\n",
                 pHw->AdiModemSm.INFO14,
                 pHw->AdiModemSm.INFO08,
                 pHw->AdiModemSm.DIAG03);

    p += sprintf(p, "VPI      %10.10d  VCI      %10.10d  Delin         ", 
                 pHw->Vc.Vpi, pHw->Vc.Vci);

    /*Delineation is the only one where we print a string instead of a number*/
    if (pHw->AdiModemSm.flags & 0x0C00)
	p += sprintf(p, "ERROR\n");
    else
	if (pHw->AdiModemSm.flags & 0x0030) 
	    p += sprintf(p, " LOSS\n");
	else
	    p += sprintf(p, " GOOD\n");
    p += sprintf(p, "Cells Rx %10.10d  Cells Tx %10.10d\n",
                 pHw->Statistics[STAT_CELLS_RX],
                 pHw->Statistics[STAT_CELLS_TX]);
    p += sprintf(p, "Pkts Rx  %10.10d  Pkts Tx  %10.10d\n",
                 pHw->Statistics[STAT_PAKTS_RX],
                 pHw->Statistics[STAT_PAKTS_TX]);
    p += sprintf(p, "OAM      %10.10d  Bad VPI  %10.10d  Bad CRC  %10.10d\n",
                  pHw->Statistics[STAT_CELLS_OAM_RCVD],
                  pHw->Statistics[STAT_CELLS_LOST_VPIVCI],
                  pHw->Statistics[STAT_CELLS_LOST_CRC]
                  );
    p += sprintf(p, "Oversiz. %10.10d\n\n",
                pHw->Statistics[STAT_CELLS_LOST_OTHER] );
    
    switch (pHw->AdiModemSm.CurrentAdiState)
    {
        case STATE_UNDEFINED:
            p += sprintf(p, "Modem is unplugged from USB.\n");
            break;
        case STATE_JUST_PLUGGED_IN:
            p += sprintf(p, "Modem waiting for driver response.\n");
            p += sprintf (p,"Please send DSP (adictrl -d)\n");
            break;
        case STATE_UNTRAIN:
        case STATE_UNTRAIN_TX:
        case STATE_UNTRAIN_RX:
            p += sprintf(p, "Modem is initializing(UNTRAIN)\n");
            break;
        case STATE_INITIALIZING:
        case STATE_INITIALIZING_TX:
        case STATE_INITIALIZING_RX:
            p += sprintf(p, "Modem is initializing(INITIALIZING)\n");
            break;
        case STATE_HARD_RESET_INITIATE:
        case STATE_HARD_RESET_END:
        case STATE_BOOT_WAIT:
        case STATE_BOOT_STAGE_1:
        case STATE_BOOT_STAGE_2:
        case STATE_BOOT_STAGE_3:
            p += sprintf(p, "Modem is booting\n");
            break;
        case STATE_OPERATIONAL:
        case STATE_OPERATIONAL_TX:
        case STATE_OPERATIONAL_RX:
            p += sprintf(p, "Modem is operational\n");
            break;
        case STATE_STALLED_FOREVER:
            p += sprintf(p, "Modem cannot boot. Unplug other devices and retry.\n");
            break;
        default:
            p += sprintf(p, "Unhandled state\n");
            adi_report ("Timer: unhandled state 0x%X\n", pHw->AdiModemSm.CurrentAdiState);            
            break;
    }
    p += sprintf(p, "\n");
    /*Figure out how much of the page we used*/
    size  = p - page;
    size -= off;
    
    if (size < count)
    {
	*eof = 1; 
	if (size <= 0)
	    return 0;
    }
    else
	size = count;
    /*Fool caller into thinking we started where he told us to in the page*/
    *start = page + off;
    return size;
}


/*******************************************************************************/
/* Code that's part of the ethernet driver interface			       */
/*******************************************************************************/

/*******************************************************************************/
/* create_etherdev							       */
/*******************************************************************************/
static int create_etherdev(Hardware *pHw)
{
    struct net_device *ether = init_etherdev(NULL, 0);
    
    if (!ether)
    {
	return -ENOMEM;
    }
    
    /* Initialize the Ethernet device: */
    ether->priv               = pHw;
    if ( if_name != NULL ) 
    {
        /* User supplied a name */
        strncpy (ether->name,if_name,IFNAMSIZ);
    }
    
    ether->open               = adi_open;
    ether->stop               = adi_close;
    ether->do_ioctl           = adi_ioctl;
    ether->get_stats          = adi_stats;
    ether->hard_start_xmit    = adi_start_xmit;
    ether->set_multicast_list = adi_set_multicast;
    ether->tx_timeout         = adi_tx_timeout;
    ether->watchdog_timeo     = kTRANSMIT_TIMEOUT;
    GetMacAddress(pHw);
    memcpy(ether->dev_addr, pHw->MAC, ETH_LENGTH_OF_ADDRESS);
    SET_MODULE_OWNER(ether);
    /*Link to be able to get back to the net_device from the Hardware*/
    pHw->pLinuxNet = ether;
    
    adi_report ("Modem ethernet interface is '%s' (mtu %d)\n",
	ether->name, ether->mtu);
    
    return 0;
}


/*******************************************************************************/
/* adi_open								       */
/*******************************************************************************/
static int adi_open(struct net_device *ether)
{
    Hardware *pHw = (Hardware *)ether->priv;

    MOD_INC_USE_COUNT;
    adi_enters (DBG_ENET);
    

    /*Tell ourselves that we're open for business!*/
    if (down_interruptible(&pHw->NetSem))
	return -ERESTARTSYS;
    pHw->IsOpen = TRUE;
    up(&pHw->NetSem);
    
    /* Create incoming data URBs: */
    StartReadPipe(pHw);

    /*Tell the kernel we're ready for network traffic flow*/
    pHw->IsWriting = FALSE;
    netif_start_queue(ether);

    adi_leaves (DBG_ENET);
    
    return 0;
}

/*******************************************************************************/
/* adi_close								       */
/*******************************************************************************/
static int adi_close(struct net_device *ether)
{
    Hardware *pHw = (Hardware *)ether->priv;

    adi_enters (DBG_ENET);
    
    if (down_interruptible(&pHw->NetSem))
	return -ERESTARTSYS;
    /*Tell the kernel that we do not want any more network traffic*/
    netif_stop_queue(ether);

    /* Stop our outgoing data urbs */
    if (pHw->IsWriting)
    {
	pHw->urbWrite.transfer_flags &= ~USB_ASYNC_UNLINK;
	usb_unlink_urb(&pHw->urbWrite);
    }
    
    /*Stop our incoming data urbs*/
    StopReadPipe(pHw);

    /*Tell ourselves that we're closed for business!*/
    pHw->IsOpen = FALSE;
    up(&pHw->NetSem);

    adi_leaves (DBG_ENET);
    
    MOD_DEC_USE_COUNT;
    return 0;
}

/*******************************************************************************/
/* adi_ioctl 								       */
/*******************************************************************************/
static int adi_ioctl(struct net_device *ether, struct ifreq *rq, int cmd)
{
    adi_enters (DBG_ENET);
    

    adi_leaves (DBG_ENET);
    
    /*This particular ioctl is accessed through the socket API*/
    /*For now, I don't know of any ioctls that we need to support*/
    return -ENOTTY;
}

/*******************************************************************************/
/* adi_stats 								       */
/*******************************************************************************/
static struct net_device_stats *adi_stats(struct net_device *ether)
{
    Hardware *pHw = (Hardware *)ether->priv;

    adi_enters (DBG_ENET);
    

    adi_leaves (DBG_ENET);
    
    return &pHw->LinuxStats;
}

/*******************************************************************************/
/* adi_start_xmit 							       */
/*******************************************************************************/
static int adi_start_xmit(struct sk_buff *skb, struct net_device *ether)
{
    Hardware *pHw = (Hardware *)ether->priv;
    ETHERNET_MAC_HEADER *pHdr = (ETHERNET_MAC_HEADER *)skb->data;
    int result=0, packet_sent=0;
    
    adi_enters (DBG_ENET);
    

    /*Tell the kernel to hold off until we're done sending this packet*/
    netif_stop_queue(ether);

    /*Send the packet through the SAR to get ATM cells*/
    switch (pHw->MpoaMode)
    {
    case MPOA_MODE_BRIDGED_ETH_LLC :
    case MPOA_MODE_BRIDGED_ETH_VC  :
        result = UniProcessOutboundPdu(pHw, skb->data, skb->len);
        break;
    case MPOA_MODE_ROUTED_IP_LLC   :
    case MPOA_MODE_ROUTED_IP_VC    :
        
        if (pHdr->EtherType != ETHERTYPE_IP)
        {
            if (pHdr->EtherType == ETHERTYPE_ARP)
            {
                EthernetArpReply(pHw, skb->data, skb->len);
                //goto xmit_exit;

                /***********************************************************
		  Note that in the MAC Code we go to xmit_exit but here it
		  seems that we have to somehow return the send function
		  successfully; for now we countiue the send function after 
		  sending back the ARP Request.This might be a potential BUG
                ************************************************************/
                result = UniProcessOutboundPdu(pHw, skb->data, skb->len);
            }
            else
            {
                // If the packet is not an ARP or IP, then its invalid, but
                // tell Linux that all is well
                adi_err ("Invalid ethernet in packet.\n");
                goto xmit_exit;
            } 
        }
        else
        {
            // Adjust the packet pointer and size so we
            // skip the ethernet header
            skb->data   += sizeof(ETHERNET_MAC_HEADER);
            skb->len -= sizeof(ETHERNET_MAC_HEADER);
            result = UniProcessOutboundPdu(pHw, skb->data, skb->len);
        }
        
        break;

        case MPOA_MODE_PPPOA_LLC       :
        case MPOA_MODE_PPPOA_VC        :
            result = UniProcessOutboundPdu(pHw, skb->data+16, skb->len-16);
            break;
    }
    /* If all is OK, send the data: */
    if (!result && !packet_sent)
    {
	/* Fix the timestamp of transaction for timeout detection: */
	ether->trans_start = jiffies;
	/* Store packet information for transmit statistics: */
	pHw->OutgoingPacketSize = skb->len;
	/* Uni says all is ok to proceed with sending the data: */
	result = WriteAtmData(pHw);
    }
    /* We've copied the data into our own internal buffers, so we can */ 
    /* go ahead and free the kernel's buffer for this packet. */
    
  xmit_exit:
    dev_kfree_skb(skb);
    if (0 != result)
    {
	/*******************************************************************************/
	/* hard_start_xmit must never return 1 in normal condition. Returning 1 means  */
	/* there is an unrecoverable hardware error and that the adapter cannot be used*/
	/* anymore. In this case, we must not destroy the socket buffer neither resume */
	/* packet transmission. I really don't know who will wake up the softnet layer */
	/* after, and indeed it seems the network interface is definitively dead. As we*/
	/* do not and cannot handle hardware errors at this level, all we can do is to */
	/* consider the socket buffer has been lost on the wire and resume	       */
	/* transmissions.							       */
	/*******************************************************************************/
	
	/*
          Update statistics: count only real errors (not "noy sent" that occured
          because the modem is (re)booting
        */
        if ( result != 1 ) 
        {
            adi_warn ("Packet transmission error: result=%d\n", result);
            
            ++pHw->LinuxStats.tx_errors;
            ++pHw->LinuxStats.tx_carrier_errors;
	}
        
	/* There will be no URB completion, so we must wake up the network layer ourself: */
	pHw->IsWriting = FALSE;
	netif_wake_queue(ether);
    }
    else
    {
	/*******************************************************************************/
	/* You'll notice that, in this case, we do not wake up the netif queue	       */
	/* .... that's because we're only using one write URB, so we have	       */
	/* to wait until the completion handler for the write is called		       */
	/* or a transmit timeout occur before we can tell kernel that we can handle    */
	/* more network data.							       */
	/* A potential performance improvement would be to have an internal	       */
	/* URB queue for outgoing data, so we could (theoretically) start	       */
	/* the software-processing of a packet while another one is being sent.	       */
	/*******************************************************************************/
    }
    adi_leaves (DBG_ENET);
    
    return 0;
}

/*******************************************************************************/
/* adi_set_multicast							       */
/*******************************************************************************/
static void adi_set_multicast(struct net_device *ether)
{
    adi_enters (DBG_ENET);
    

    /*******************************************************************************/
    /* Our ethernet side is really all software, so anything that		   */
    /* a normal ethernet card would do in hardware, we do here in the driver.	   */
    /* On a normal ethernet card, you would upload the filter address list	   */
    /* to the card - but we just loop through a fast software compare. 		   */
    /* Since we will just always check whats in the net_device *, we don't 	   */
    /* have to do anything here!						   */
    /*******************************************************************************/

    adi_leaves (DBG_ENET);
    
}


/*******************************************************************************/
/* adi_tx_timeout							       */
/*******************************************************************************/
static void adi_tx_timeout(struct net_device *ether)
{
    Hardware *pHw = (Hardware *)ether->priv;
    adi_warn ("Transmit timed out!\n");
    
    /*
    Other linux drivers don't seem to recover and resend
    the socket buffer Linux gives them. However, O'Reilly's
    Linux Device Driver says we must not lose the buffer.
    But, as packets may anyway be lost by media in general,
    it's quite certain the protocol will resend it sooner or
    later, or consider packet lost as normal. Therefore, this
    driver do not attempt anything more than removing the broken
    URB. This will call WriteCompletion, which will call
    netif_wakequeue:
    */
    pHw->urbWrite.transfer_flags |= USB_ASYNC_UNLINK;
    usb_unlink_urb(&pHw->urbWrite);

    pHw->urbOAMWrite.transfer_flags |= USB_ASYNC_UNLINK;
    usb_unlink_urb(&pHw->urbOAMWrite);
    
    /* We must reset the transaction time to keep the watchdog quiet: */
    ether->trans_start = jiffies;

    adi_leaves (DBG_ENET);
    
}


/**************************************************************************************
$Log: AdiUsbAdslDriver.c,v $
Revision 1.19  2003/06/18 21:58:41  sleeper
VERSION

Revision 1.18  2003/06/03 22:56:35  sleeper
Added debugging flags management + removing of firmware ioctl

Revision 1.17  2003/04/08 23:10:37  sleeper
Add module load variable if_name

Revision 1.16  2003/04/24 00:06:28  sleeper
fix a display bug

Revision 1.15  2003/04/23 22:23:03  sleeper
Change urb control submission + stats

Revision 1.14  2003/03/31 23:36:42  sleeper
Fix merge of routed ethernet modifs from 2.0.1

Revision 1.13  2003/03/31 22:01:35  sleeper
Change some /proc info

Revision 1.12  2003/03/21 00:22:41  sleeper
Msg Initialization modifs from 2.0.1

Revision 1.11  2003/03/20 22:07:55  sleeper
Merge routed mode modfis + info statements added

Revision 1.10  2003/03/17 00:23:45  sleeper
Add an msg in proc read handler

Revision 1.9  2003/03/11 00:38:03  sleeper
Add OAM modifs as present in Sagemn 2.0.1 driver

Revision 1.8  2003/03/10 23:19:39  sleeper
Change Control URB submission

Revision 1.7  2003/03/02 20:48:40  sleeper
Change attenuation display line to dB value.

Revision 1.6  2003/02/24 23:54:26  sleeper
Remove nasty mdelays

Revision 1.5  2003/02/21 21:52:14  sleeper
Add Renaud Guerin(rguerin@freebox.fr) support for multi-modem and fixed a bug
introduced by CC5, that caused a Oops when doing rmmod

Revision 1.4  2003/02/20 22:18:37  sleeper
Fix a bug that occured when options were not loader and DSP was sent to driver

Revision 1.3  2003/02/14 23:30:56  sleeper
Merge CC5 diffs

Revision 1.2  2003/02/10 23:40:08  sleeper
Fix memory leaks, error strings

Revision 1.30  2002/05/24 17:56:44  Anoosh Naderi
Clean up the code.
Added support for PPPoA (LLC & VC).

Added ISO support for incoming data.

Revision 1.27  2002/02/12 21:50:08  steve.urquhart
Added statement to /proc output for modem status.

Revision 1.26  2002/02/11 21:12:35  chris.edgington
Added support for possible multiple read urbs.

Revision 1.25  2002/01/23 17:23:02  chris.edgington
Moved code that claims usb interfaces to be after kernel memory
allocations, so we don't claim the interfaces until we're sure we
will use them.
Added verification of pointer validity for all memory allocs.
Added comments to adi_user.

Revision 1.24  2002/01/18 18:00:09  chris.edgington
Added include of Options.h.
Removed temporary MODULE_PARMs section, since we're using .conf file now.
Moved adi_user to bottom of file, cleaned it up some.
Write ioctl handler for options data transfer.
Moved calls to MsgInitialize and MpoaInitialize to options ioctl.

Revision 1.23  2002/01/18 00:01:20  steve.urquhart
Upgraded adi_user to handle 2 ioctls

Revision 1.22  2002/01/17 17:49:18  steve.urquhart
Fixed GPL Header, keyword subsititution

Revision 1.21  2002/01/16 22:59:33  steve.urquhart
Moved portion of DspFoodProcessor and invocation of BootTheModem to
adi_user ioctl handler.  Removed LoadAndPrepareDsp.  Cleaned up the ioctl
interface.  Decreased initial DSP vmalloc to less than .5MB.

Revision 1.20  2002/01/15 22:15:59  steve.urquhart
Second revision of user mode code, includes example of means necessary to
ioctl the driver.

Revision 1.19  2002/01/14 21:59:30  chris.edgington
Added GPL header.

Revision 1.18  2002/01/14 17:38:41  chris.edgington
Changed include order.

Revision 1.17  2002/01/10 17:14:15  chris.edgington
Added /proc/adimodem interface.

Revision 1.16  2002/01/09 22:15:56  chris.edgington
Added support for proper kernel symbol name importing through MODVERSIONS.

Revision 1.15  2002/01/09 20:16:57  chris.edgington
Added transmit timeout support via adi_tx_timeout.

Revision 1.14  2002/01/08 20:03:37  chris.edgington
Removed a few unused local vars.
Removed hardcoded MAC address setting code.

Revision 1.13  2002/01/08 16:22:25  chris.edgington
Fixed parameters to memcpy of MAC address in adi_probe.

Revision 1.12  2002/01/08 16:06:28  chris.edgington
Added some necessary comments.
Removed invalid DeAllocQueuedBuffers calls in adi_disconnect.
Finished details of adi_set_multicast.

Revision 1.11  2002/01/04 20:47:47  chris.edgington
Added initialization of data send/receive buffers in Hardware.
Removed initialization of send/receive queues - not needed for Linux.
Moved cancellation of timers in adi_disconnect to be right at the beginning.
Added calls to StartReadPipe and StopReadPipe in adi_open and adi_close.
Added transmit logic to adi_start_xmit (call WriteAtmData).

Revision 1.10  2002/01/02 21:59:00  chris.edgington
Added initialization of IdleOutQ, LiveOutQ, IdleWaitQ, LiveWaitQ.
Added calls to CrcGenerateTable and MpoaInitialize.

Revision 1.9  2001/12/28 22:08:26  chris.edgington
Added creation and initialization of net_device interface.
Added skeletons for adi_open, adi_close, adi_ioctl, adi_start_xmit,
      adi_stats, and adi_set_multicast.

Revision 1.8  2001/12/28 19:28:03  chris.edgington
Added initialization of CtrlUrbFreeQ and CtrlUrbPendQ.
Added initialization of CtrlUrbQueueTimer.
Make adi_disconnect only attemp free when pid = post firmware pid.
Only upload swap page when state != HARD_RESET_INITIATE.

Revision 1.7  2001/12/26 22:43:33  chris.edgington
Added initialization code for kernel timer.
Changed parameters to BootTheModem.

Revision 1.6  2001/12/22 21:43:23  chris.edgington
Added call to MsgInitialize.
Added call to ProcessIncomingCmv on CMV interrupt.

Revision 1.5  2001/12/22 19:53:00  chris.edgington
Moved utility routines to Util.c - loadUsbFirmware, EZUSBWrite.
Added loading of swap pages in response to swap page request interrupt.

Revision 1.4  2001/12/21 23:56:51  chris.edgington
Added interrupt handler and startup of interrupt pipe.
Added call to BootTheModem.

Revision 1.3  2001/12/20 22:39:25  chris.edgington
Removed call to usb_set_interface - I don't think we need it.
Added calls to usb_driver_claim_interface, to keep usb from calling adi_probe
for each interface exposed by the device ('cause we use them all and we don't
want another driver trying to claim an interface.)
Added call to LoadAndPrepareDspCode
Added calls to usb_driver_release_interface in adi_disconnect.

Revision 1.2  2001/12/17 22:21:37  chris.edgington
Wrote core driver init code, including USB firmware upload.
Use "insmod -f adiusbadsl.o" to get this version to load.

Revision 1.1.1.1  2001/12/14 21:27:07  chris.edgington
Initial import
**************************************************************************************/
