/*********************************************************************************/
/* $Id: Boot.c,v 1.4 2003/06/03 22:51:34 sleeper Exp $                           */
/*										 */
/* Copyright (c) 2001, Analog Devices Inc., All Rights Reserved			 */
/*										 */
/* Boot.c									 */
/*										 */
/* Code responsible for booting the modem					 */
/*										 */
/* This file is part of the "ADI USB ADSL Driver for Linux".			 */
/*										 */
/* "ADI USB ADSL Driver for Linux" is free software; you can redistribute it     */
/* and/or modify it under the terms of the GNU General Public License as         */
/* published by the Free Software Foundation; either version 2 of the License,   */
/* or (at your option) any later version.					 */
/*										 */
/* "ADI USB ADSL Driver for Linux" is distributed in the hope that it will be    */
/* useful, but WITHOUT ANY WARRANTY; without even the implied warranty of	 */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the		 */
/* GNU General Public License for more details.					 */
/*										 */
/* You should have received a copy of the GNU General Public License		 */
/* along with "ADI USB ADSL Driver for Linux"; if not, write to the Free Software*/
/* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA     */
/*********************************************************************************/

#include "Adiutil.h"
#include "Boot.h"
#include "Util.h"
#include "Me.h"
#include "debug.h"

/*********************************************************************************/
/* Local only prototypes							 */
/*********************************************************************************/
static void IdmaUploadMainPage(Hardware *pHw);
static int IDMAWrite(Hardware *pHw, void *pData, UInt32 Size);
static void IDMACompletion(struct urb *urb);

/*********************************************************************************/
/* BootModem									 */
/* Kinda like doing a ctrl-alt-delete on your PC. Reuploads (or does first	 */
/* upload) of the DSP code, and initializes the host's state machine. 		 */
/*********************************************************************************/
int BootTheModem (Hardware *pHw, Boolean Rebooting ) 
{
    int stat = USB_ST_NOERROR;

    adi_enters (DBG_BOOT);

    /*********************************************************************************/
    /* The steps in the boot process are not documented, they were "gleaned"	     */
    /* from the NDIS code. A different, simpler process might be possible,	     */
    /* but we have no way to prove it at this point.				     */
    /*********************************************************************************/

    if (!Rebooting)
    {
	/*To kickoff the boot process, do a custom command, SET_MODE to BOOTMODE_IDMA*/
	stat = AdiCustomWrite(pHw, kADI_CMD_SET_MODE, kMODE_BOOTMODE_IDMA, 0, NULL);
	if (stat != USB_ST_NOERROR)
	{
	    adi_err( "AdiCustomWrite error %x on SET_MODE BOOTMODE_IDMA\n", stat);
	    goto end_boot;
	}
	
	/*LOOPBACK_ON correlates to ENUM_FPGALBK in the NDIS code*/
	stat = AdiCustomWrite(pHw, kADI_CMD_SET_MODE, kMODE_LOOPBACK_ON, 0, NULL);
	if (stat != USB_ST_NOERROR)
	{
	    adi_err("AdiCustomWrite error %x on SET_MODE LOOPBACK_ON\n", stat);
	    goto end_boot;
	}
    }

    /*I know we've already done this - but this is what the NDIS code does!*/
    stat = AdiCustomWrite(pHw, kADI_CMD_SET_MODE, kMODE_BOOTMODE_IDMA, 0, NULL);
    if (stat != USB_ST_NOERROR)
    {
	adi_err("AdiCustomWrite error %x on SET_MODE BOOTMODE_IDMA\n", stat);
	goto end_boot;
    }

    /*
      At this point, we must wait for 200 ms. Original code was doing a
      non-interruptible wait !!! This leads to freeze and possibly odd
      behaviours.
      The new way of doing it is adding steps in the state machine, and schedule
      the WatchDog to beep in 200 ms
    */
    
    pHw->AdiModemSm.CurrentAdiState = STATE_BOOT_STAGE_1;
    pHw->AdiModemSm.ModemReplyExpected = EVENT_TIMER_TICK;
    SetTimerInterval(pHw, TRANS_TIME_2_BOOT_STAGE_1);
    
end_boot:
    adi_leaves (DBG_BOOT);
    return stat;
       
    
}

int BootTheModem_Stage_1 (Hardware *pHw) 
{
    int stat = USB_ST_NOERROR;

    adi_enters (DBG_BOOT);
    
    stat = AdiCustomWrite(pHw, kADI_CMD_SET_MODE, kMODE_START_RESET, 0, NULL);
    if (stat != USB_ST_NOERROR)
    {
	adi_err ("AdiCustomWrite error %x on SET_MODE START_RESET\n", stat);
	goto end_boot;
    }

    /* Advance to stage 2 and wait for 200 ms */
    pHw->AdiModemSm.CurrentAdiState = STATE_BOOT_STAGE_2;
    SetTimerInterval(pHw, TRANS_TIME_2_BOOT_STAGE_2);

end_boot:
    adi_leaves (DBG_BOOT);
    return stat;
}

int BootTheModem_Stage_2 (Hardware *pHw) 
{
    int stat = USB_ST_NOERROR;
    UInt16 myWord;

    adi_enters (DBG_BOOT);

    stat = AdiCustomWrite(pHw, kADI_CMD_SET_MODE, kMODE_END_RESET, 0, NULL);
    if (stat != USB_ST_NOERROR)
    {
	adi_err ("AdiCustomWrite error %x on SET_MODE END_RESET\n", stat);
	goto end_boot;
    }

    /*Clear all of the CMV mailboxes - this is a MUST!!!  [IIDos-Oct.18,01]*/
    myWord = cpu_to_le16(kMAILBOX_EMPTY);
    stat  = AdiCustomWrite(pHw, kADI_CMD_SET_2183_DATA, (kDSP_MP_TX_MAILBOX | 0x4000),
                           2, (UInt8 *)&myWord);
    stat  = AdiCustomWrite(pHw, kADI_CMD_SET_2183_DATA, (kDSP_MP_RX_MAILBOX | 0x4000),
                           2, (UInt8 *)&myWord);
    stat  = AdiCustomWrite(pHw, kADI_CMD_SET_2183_DATA, (kDSP_SWAP_MAILBOX  | 0x4000),
                           2, (UInt8 *)&myWord);

    pHw->AdiModemSm.CurrentAdiState = STATE_BOOT_STAGE_3;
    SetTimerInterval(pHw, TRANS_TIME_2_BOOT_STAGE_3);

end_boot:
    adi_leaves (DBG_BOOT);
    return stat;
}

int BootTheModem_Stage_3 (Hardware *pHw) 
{
    int stat = USB_ST_NOERROR;

    adi_enters (DBG_BOOT);
    
    /*Finally, we get to do something that seems substantial!*/
    IdmaUploadMainPage(pHw);
    /*********************************************************************************/
    /*This is the last operation in our sequence of tasks for device		     */
    /*initialization. Everything else will be driven by the hardware.		     */
    /*Once the main page is loaded, the hardware will start requesting		     */
    /*swap pages.								     */
    /*Put our modem in the BOOT_WAIT state and start our watchdog!		     */
    /*********************************************************************************/
    pHw->AdiModemSm.CurrentAdiState = STATE_BOOT_WAIT;
    SetTimerInterval(pHw, TRANS_TIME_4_BOOTWAIT);

    adi_leaves (DBG_BOOT);
    return stat;
}
    

#if 0
/*********************************************************************************/
/* BootModem									 */
/* Kinda like doing a ctrl-alt-delete on your PC. Reuploads (or does first	 */
/* upload) of the DSP code, and initializes the host's state machine. 		 */
/*********************************************************************************/
int BootTheModem(Hardware *pHw, Boolean Rebooting)
{
    int stat = USB_ST_NOERROR;
    UInt16 myWord;

    adi_enters (DBG_BOOT);

    /*********************************************************************************/
    /* The steps in the boot process are not documented, they were "gleaned"	     */
    /* from the NDIS code. A different, simpler process might be possible,	     */
    /* but we have no way to prove it at this point.				     */
    /*********************************************************************************/

    if (!Rebooting)
    {
	/*To kickoff the boot process, do a custom command, SET_MODE to BOOTMODE_IDMA*/
	stat = AdiCustomWrite(pHw, kADI_CMD_SET_MODE, kMODE_BOOTMODE_IDMA, 0, NULL);
	if (stat != USB_ST_NOERROR)
	{
	    adi_err ("AdiCustomWrite error %x on SET_MODE BOOTMODE_IDMA\n", stat);
	    goto end_boot;
	}
	
	/*LOOPBACK_ON correlates to ENUM_FPGALBK in the NDIS code*/
	stat = AdiCustomWrite(pHw, kADI_CMD_SET_MODE, kMODE_LOOPBACK_ON, 0, NULL);
	if (stat != USB_ST_NOERROR)
	{
	    adi_err ("AdiCustomWrite error %x on SET_MODE LOOPBACK_ON\n", stat);
	    goto end_boot;
	}
    }

    /*I know we've already done this - but this is what the NDIS code does!*/
    stat = AdiCustomWrite(pHw, kADI_CMD_SET_MODE, kMODE_BOOTMODE_IDMA, 0, NULL);
    if (stat != USB_ST_NOERROR)
    {
	adi_err ("AdiCustomWrite error %x on SET_MODE BOOTMODE_IDMA\n", stat);
	goto end_boot;
    }

    /*Analog folks say this delay is needed - but they couldn't tell us why!*/
    wait_ms(200);

    stat = AdiCustomWrite(pHw, kADI_CMD_SET_MODE, kMODE_START_RESET, 0, NULL);
    if (stat != USB_ST_NOERROR)
    {
	adi_err ("AdiCustomWrite error %x on SET_MODE START_RESET\n", stat);
	goto end_boot;
    }

    /*Analog folks say this delay is needed - but they couldn't tell us why!*/
    wait_ms(200);

    stat = AdiCustomWrite(pHw, kADI_CMD_SET_MODE, kMODE_END_RESET, 0, NULL);
    if (stat != USB_ST_NOERROR)
    {
	adi_err ("AdiCustomWrite error %x on SET_MODE END_RESET\n", stat);
	goto end_boot;
    }

    /*Clear all of the CMV mailboxes - this is a MUST!!!  [IIDos-Oct.18,01]*/
    myWord = cpu_to_le16(kMAILBOX_EMPTY);
    stat  = AdiCustomWrite(pHw, kADI_CMD_SET_2183_DATA, (kDSP_MP_TX_MAILBOX | 0x4000),
                           2, (UInt8 *)&myWord);
    stat  = AdiCustomWrite(pHw, kADI_CMD_SET_2183_DATA, (kDSP_MP_RX_MAILBOX | 0x4000),
                           2, (UInt8 *)&myWord);
    stat  = AdiCustomWrite(pHw, kADI_CMD_SET_2183_DATA, (kDSP_SWAP_MAILBOX  | 0x4000),
                           2, (UInt8 *)&myWord);

    /*Analog folks say this delay is needed - but they couldn't tell us why!*/
    wait_ms(1000);

    /*Finally, we get to do something that seems substantial!*/
    IdmaUploadMainPage(pHw);
    /*********************************************************************************/
    /*This is the last operation in our sequence of tasks for device		     */
    /*initialization. Everything else will be driven by the hardware.		     */
    /*Once the main page is loaded, the hardware will start requesting		     */
    /*swap pages.								     */
    /*Put our modem in the BOOT_WAIT state and start our watchdog!		     */
    /*********************************************************************************/
    pHw->AdiModemSm.CurrentAdiState = STATE_BOOT_WAIT;
    SetTimerInterval(pHw, TRANS_TIME_4_BOOTWAIT);

end_boot:
    adi_leaves (DBG_BOOT);
    return stat;
}
#endif /* 0 */
 
/*********************************************************************************/
/* IdmaUploadMainPage								 */
/*										 */
/* Sends the Main page blocks of the DSP code through the IDMA interface.        */
/* Note that the bulk of this logic is straight from Buffer.c in the NDIS code,  */
/* there was virtually no documentation on how this is supposed to work.	 */
/*********************************************************************************/
static void IdmaUploadMainPage(Hardware *pHw)
{
    int stat;
    UInt32 i, myBlockSize;
    UInt8 *pLastBits = NULL;
    UInt8 *pBlockData;
    IDMALoadBlockInfo myBLI;

    adi_enters (DBG_BOOT);

    adi_dbg (DBG_BOOT,"Loading 0x%#x blocks from main page.\n",
             pHw->MainPage.BlockCount);
    

    for (i=0; i<pHw->MainPage.BlockCount; i++)
    {
        /*Init data that is not block-location-specific*/
        myBLI.ADIHeader = cpu_to_le16(0xABCD);
        myBLI.Ovl       = 0;
        myBLI.OvlOffset = cpu_to_le16(0x8000);
        myBLI.LastBlock = 0;

        /*Is this NOT the first block?*/
        if (pHw->MainPage.Blocks[i].DSPAddr != 0)
        {
           myBLI.Size  = cpu_to_le16((UInt16)(pHw->MainPage.Blocks[i].DSPSize));
           myBLI.Addr  = cpu_to_le16(pHw->MainPage.Blocks[i].DSPAddr);
           myBlockSize = (UInt16)(pHw->MainPage.Blocks[i].DSPSize);
           pBlockData  = pHw->MainPage.Blocks[i].MemOffset;
        }
        /*This is block 0 - we want to send all but offset 0 - so we start at offset 1*/
        /*We should only get here once!*/
        else
        {
           myBLI.Size  = cpu_to_le16((UInt16)(pHw->MainPage.Blocks[i].DSPSize - 3));
           myBLI.Addr  = cpu_to_le16(1);
           myBlockSize = (UInt16)(pHw->MainPage.Blocks[i].DSPSize - 3);
           pBlockData  = pHw->MainPage.Blocks[i].MemOffset+3;
           pLastBits   = pHw->MainPage.Blocks[i].MemOffset;
        }

        /*Write the structure telling the DSP what to expect in the next block*/
        stat = IDMAWrite(pHw, (void *)&myBLI, sizeof(IDMALoadBlockInfo));
        if (stat != USB_ST_NOERROR)
	    adi_err ("Error %#X on IDMA write of BlockLoadInfo\n", stat);

        /*********************************************************************************/
        /* Occassionally the modem will freak out during idma boot and get in 		 */
        /* a weird mode where it requests wrong swap pages, etc.			 */
        /* Comparing the usb traces of linux and Windows reveals that linux		 */
        /* does things MUCH faster, so my theory is that we're overwhelming		 */
        /* the hardware. So, I'll inject a tiny delay here to keep things		 */
        /* under control								 */
        /*********************************************************************************/
        wait_ms(1); 

        /*Now write the actual block data*/
        stat = IDMAWrite(pHw, (void *)pBlockData, myBlockSize);
        if (stat != USB_ST_NOERROR)
	    adi_err ("Error %#X on IDMA write of block data\n", stat);

        wait_ms(1);
    }

    /*All blocks are loaded, now we can write out offset 0 of the first block,    */
    /*which will tell the hardware to start - resulting in requests for swap pages*/
    myBLI.ADIHeader = cpu_to_le16(0xABCD);
    myBLI.Ovl       = 0;
    myBLI.OvlOffset = cpu_to_le16(0x8000);
    myBLI.LastBlock = cpu_to_le16(1);
    myBLI.Size      = cpu_to_le16(3);
    myBLI.Addr      = 0;
    myBlockSize     = 3;
    pBlockData      = pLastBits;
    adi_dbg (DBG_BOOT,"sending last entry from main page, DSP will start soon!.\n");
    

    /*Write the structure telling the DSP what to expect in the next block*/
    stat = IDMAWrite(pHw, (void *)&myBLI, sizeof(IDMALoadBlockInfo));
    if (stat != USB_ST_NOERROR)
	adi_err ("Error %#X on IDMA write of BlockLoadInfo\n", stat);
    /*Now write the actual block data*/
    stat = IDMAWrite(pHw, (void *)pBlockData, myBlockSize);
    if (stat != USB_ST_NOERROR)
	adi_err ("Error %#X on IDMA write of block data\n", stat);

    adi_leaves (DBG_BOOT);
}

/*********************************************************************************/
/* IdmaUploadSwapPage								 */
/*										 */
/* Sends the blocks of the requested swap page through the IDMA interface.	 */
/* Note that the bulk of this logic is straight from Buffer.c in the NDIS code,  */
/* there was virtually no documentation on how this is supposed to work.	 */
/*********************************************************************************/
void IdmaUploadSwapPage(Hardware *pHw, UInt16 SwapInfo)
{
    int stat;
    UInt32 i, myBlockSize;
    UInt16 PageIndex, Ovl;
    IDMALoadBlockInfo myBLI;

    adi_enters (DBG_BOOT);

    PageIndex = (SwapInfo & 0x00FF) - 1;
    Ovl       = ((SwapInfo & 0xF000) >> 12) | ((SwapInfo & 0x0F00) >> 4);

    adi_dbg (DBG_BOOT,"Loading 0x%x blocks from swap page 0x%x, ovl=0x%x.\n", 
             pHw->pSwapPages[PageIndex].BlockCount, PageIndex, Ovl);
    
    if (PageIndex > pHw->SwapPageCount)
    {
       adi_err ("Request for swap page 0x%x when only 0x%x pages exist!\n",
               PageIndex, pHw->SwapPageCount);
       
       return;
    }
    
    /*For swap pages, we actually know we won't have more than 2 blocks*/
    for (i=0; i<pHw->pSwapPages[PageIndex].BlockCount; i++)
    {
        /*Init data that is not block-location-specific*/
        myBLI.ADIHeader = cpu_to_le16(0xABCD);
        myBLI.Ovl       = cpu_to_le16(Ovl);
        myBLI.OvlOffset = cpu_to_le16(0x8000 | Ovl);
        myBlockSize     = pHw->pSwapPages[PageIndex].Blocks[i].DSPExtendedSize;
        myBLI.Size      = cpu_to_le16((UInt16)myBlockSize);
        myBLI.Addr      = cpu_to_le16(pHw->pSwapPages[PageIndex].Blocks[i].DSPAddr);

        if ((pHw->pSwapPages[PageIndex].BlockCount-1) == i)
           myBLI.LastBlock = cpu_to_le16(1);
        else
           myBLI.LastBlock = cpu_to_le16(0);

        /*Write the structure telling the DSP what to expect in the next block*/
        stat = IDMAWrite(pHw, (void *)&myBLI, sizeof(IDMALoadBlockInfo));
        if (stat != USB_ST_NOERROR)
	    adi_err ("Error %#X on IDMA write of BlockLoadInfo\n", stat);

        /*********************************************************************************/
        /* Occassionally the modem will freak out during idma boot and get in 		 */
        /* a weird mode where it requests wrong swap pages, etc.			 */
        /* Comparing the usb traces of linux and Windows reveals that linux		 */
        /* does things MUCH faster, so my theory is that we're overwhelming		 */
        /* the hardware. So, I'll inject a tiny delay here to keep things		 */
        /* under control								 */
        /*********************************************************************************/
        wait_ms(1); 

        /*Now write the actual block data*/
        stat = IDMAWrite(pHw, (void *)(pHw->pSwapPages[PageIndex].Blocks[i].MemOffset), myBlockSize);
        if (stat != USB_ST_NOERROR)
	    adi_err ("Error %#X on IDMA write of block data\n", stat);

        wait_ms(1);
    }

    adi_leaves (DBG_BOOT);
}


/*********************************************************************************/
/* IDMAWrite									 */
/*										 */
/* Responsible for sending data to the IDMA interface via the BULK IDMA OUT pipe.*/
/*********************************************************************************/
static int IDMAWrite(Hardware *pHw, void *pData, UInt32 Size)
{
    int ret = USB_ST_NOERROR;
    struct urb *urb;
    UInt8 *xfer_buf = GET_KBUFFER(Size);

    adi_enters (DBG_BOOT);

    if (!xfer_buf)
    {
       adi_err ("kmalloc in IDMAWrite for %#X bytes failed\n", Size);
       
       ret = -ENOMEM;
       goto IDMA_exit;
    }

    memcpy(xfer_buf, pData, Size);

    /* Get an URB and prepare it for submission*/
    urb = usb_alloc_urb(0);

    usb_fill_bulk_urb(urb, pHw->usbdev, pHw->pipeBulkIdmaOut, xfer_buf, Size, IDMACompletion, pHw);
    urb->transfer_flags |= USB_QUEUE_BULK;

    adi_dbg (DBG_BOOT,"Submitting BULK URB %p\n", urb);
    
    ret = usb_submit_urb(urb);

IDMA_exit:
    adi_leaves (DBG_BOOT);
    return ret;
}


/*********************************************************************************/
/* IDMACompletion								 */
/*										 */
/* Called when an IDMAWrite urb has completed. All we have to do is free	 */
/* the memory allocated in IDMAWrite.						 */
/*********************************************************************************/
static void IDMACompletion(struct urb *urb)
{
    adi_enters (DBG_BOOT);

    /* All we really need to do is free the memory we alloced and free the urb*/
    FREE_KBUFFER(urb->transfer_buffer);
    usb_free_urb(urb);

    adi_dbg (DBG_BOOT,"Freed BULK URB %p\n", urb);
    

    adi_leaves (DBG_BOOT);
}

/*****************************************************************
$Log: Boot.c,v $
Revision 1.4  2003/06/03 22:51:34  sleeper
Changed ZAPS to adi_dbg/err macros

Revision 1.3  2003/04/23 22:23:52  sleeper
use of cpu_to_XX

Revision 1.2  2003/02/24 23:41:33  sleeper
Remove nasty mdelays

Revision 1.1.1.1  2003/02/10 23:29:49  sleeper
Imported sources

Revision 1.20  2002/05/24 21:59:30  Anoosh Naderi
Clean up the code

Revision 1.11  2002/01/14 21:59:30  chris.edgington
Added GPL header.

Revision 1.10  2002/01/08 16:27:47  chris.edgington
Fixed simple compile error.

Revision 1.9  2002/01/08 16:07:37  chris.edgington
Changed BootTheModem to be a void function.
Added some necessary comments.

Revision 1.8  2002/01/03 17:17:21  chris.edgington
Changed line-ending format to UNIX style.

Revision 1.7  2002/01/02 22:00:47  chris.edgington
Changed instances of STOLE macro to HSTOLE (Host Short To Little Endian).

Revision 1.6  2001/12/28 19:29:09  chris.edgington
Changed some debug messages from DEBUG_ATTN to DEBUG_INFO level.

Revision 1.5  2001/12/26 22:45:03  chris.edgington
Added "Rebooting" parm to BootTheModem.
Moved AdiCustomWrite to Utils.c.

Revision 1.4  2001/12/22 21:42:22  chris.edgington
Removed ifdef around setting of modem state after main page upload.

Revision 1.3  2001/12/22 19:51:34  chris.edgington
Added 1ms delay between blockinfo and blockdata to stabilize boot - linux just
does usb stuff too fast (relative to Windows).
Finished linux version of IdmaUploadSwapPage.
Rewrote IDMAWrite to use asynchronous completion of urbs.

Revision 1.2  2001/12/21 23:52:33  chris.edgington
Rewrote BootTheModem for Linux.
Rewrote IdmaUploadMainPage for Linux.
Added AdiCustomWrite.
Wrote IDMAWrite (needs more work - needs to pull urbs from a list.)
Added dump_urb.

Revision 1.1  2001/12/17 22:19:52  chris.edgington
Initial version - not part of build yet.

Revision 1.1  2001/12/10 03:27:03  chris.edgington
First version - ported from Boot.c in MACOS9 code.
********************************************************************************/
