/*********************************************************************************/
/* $Id: Buffers.h,v 1.2 2003/03/10 23:20:12 sleeper Exp $                 */
/*										 */
/* Copyright (c) 2001, Analog Devices Inc., All Rights Reserved			 */
/*										 */
/* Buffers.h									 */
/*										 */
/* Various types of "buffers" that will be used throughout the driver		 */
/*										 */
/* This file is part of the "ADI USB ADSL Driver for Linux".			 */
/* 										 */
/* "ADI USB ADSL Driver for Linux" is free software; you can redistribute it     */
/* and/or modify it under the terms of the GNU General Public License as         */
/* published by the Free Software Foundation; either version 2 of the License,   */
/* or (at your option) any later version.					 */
/*										 */
/* "ADI USB ADSL Driver for Linux" is distributed in the hope that it will be    */
/* useful, but WITHOUT ANY WARRANTY; without even the implied warranty of	 */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the		 */
/* GNU General Public License for more details.					 */
/*										 */
/* You should have received a copy of the GNU General Public License		 */
/* along with "ADI USB ADSL Driver for Linux"; if not, write to the Free Software*/
/* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA     */
/*********************************************************************************/

#ifndef BUFFERS_H
#define BUFFERS_H

/*********************************************************************************/
/* Structs that will be held in queues (manipulated with Enqueue and Dequeue)    */
/* In order to simplify and buffer management, each type of buffer will begin    */
/* with a generic buffer and follow with specific stuff. This will allow generic */
/* buffer manipulation routines to operate on any buffer. Note that if the       */
/* buffer needs associated with a Pb stored in the buffer (so you can get back   */
/* to the buffer pointer in a USB callback), the AdiUSBPB structure that the     */
/* pQueueEntry member will reside in needs to immediately follow the GB member.  */
/*********************************************************************************/

typedef struct
{
    QHdr         List;
    UInt8        *pData;
    UInt32       AllocedSize;
} GENERIC_BUFFER;

#define CTRL_URB_Q_SIZE 16

typedef struct
{
    GENERIC_BUFFER GB;
    void        *pHw;
    struct urb  *urb;
} QUEUED_URB_BUFFER;

typedef struct 
{
    struct list_head list;
    struct urb *urb;
    void *dev;
} queued_urb_t;


/*********************************************************************************/
/* This will hold network packet data that is waiting to be sent back to         */
/* the higher-level stack. Two queues of these will be maintained - a free       */
/* queue and a waiting queue							 */
/*********************************************************************************/
#define ETHERNET_PACKET_Q_SIZE    64
#define ETHERNET_PACKET_DATA_SIZE 1536
typedef struct
{
    GENERIC_BUFFER GB;
    UInt32         EncapSkipSize;
    UInt32         PacketSize;
    Boolean        DropThisPacket;
} ETHERNET_PACKET_BUFFER;

/************************************************************************************/
/* This will be the receiver buffer for incoming network data. We will queue        */
/* up a bunch of reads with USB with these buffers, then process them into          */
/* real ethernet packets as they come in. This enables us to support situations     */
/* where the entire ethernet packet does not come in in one USBBulkRead. We	    */
/* do require that the data received into these buffers is an exact number          */
/* of ATM cells									    */
/* LINUX NOTE :									    */
/* In kernel 2.4.7-10 (the stock kernel with RedHat 7.2), the usb-uhci module	    */
/* has a problem queuing multiple read bulk urbs). Therefore, we set this to one so */
/* we'll work with uhci or ohci. However, higher performance can be seen on ohci    */
/* if you allow multiple read urbs. So, if you're on an ohci-based machine,	    */
/* set INCOMING_Q_SIZE to > 1 to achieve greater performance.			    */
/************************************************************************************/
#ifdef USBULK
#define INCOMING_Q_SIZE 1
#else
/*We'll set it to 6 when we're dealing with ISOCHRONOUS data.*/
#define INCOMING_Q_SIZE 6
#endif

/************************************************************************************/
/* Here's how I got this - Max ethernet   = 1500 bytes + 			    */
/*          possible Ethernet FCS         = 4 bytes +				    */
/*          possible MPOA encapsulation   = 10 bytes +				    */
/*          CPCS PDU trailer              = 8 bytes +				    */
/*          at most 2 ATM payloads of pad = 96 bytes				    */
/*                                       ----------				    */
/*                                         1618 bytes				    */
/*          Divided among ATM cells = 34 ATM cells * 53 bytes			    */
/*                      Grand total = 1802					    */
/* NOTE NOTE NOTE NOTE								    */
/* This logic all makes sense IF the device received data in ATM cell sized chunks, */
/* however, it doesn't. The USB pipes receive 64bytes at a time, so we have         */
/* to have space for a size that is the LCM of 64 and 53, which is 64*53            */
/************************************************************************************/
#define INCOMING_DATA_SIZE (64 * 53)
typedef struct
{
    GENERIC_BUFFER GB;
    void          *pHw;
    struct urb    Urb;
} INCOMING_BUFFER;

/************************************************************************************/
/* This will be where we build our segmentation array of ATM cells to be sent	    */
/* out over the ADSL link.  OUTGOING_Q_SIZE used to be set at 16 - but when	    */
/* using WhatRoute to do large pings - we'd get too many ethernet packets too	    */
/* fast and we'd run out of outgoing buffers. So, I've upped it to 64 - we	    */
/* should never run out now.							    */
/************************************************************************************/
#define OUTGOING_Q_SIZE    64
#define OUTGOING_DATA_SIZE 1802 /* same logic as for INCOMING_DATA_SIZE */
typedef struct
{
    GENERIC_BUFFER GB;
    struct urb     Urb;
    UInt32         TotalBytes;
} OUTGOING_BUFFER;

#endif /* BUFFERS_H */

/********************************************************************************
$Log: Buffers.h,v $
Revision 1.2  2003/03/10 23:20:12  sleeper
Change Control URB submission

Revision 1.1.1.1  2003/02/10 23:29:49  sleeper
Imported sources

Revision 1.20  2002/05/24 17:56:44  Anoosh Naderi
Clean up the code.

Revision 1.11  2002/03/12 17:56:44  chris.edgington
Added ISO support for incoming data.

Revision 1.10  2002/02/11 21:14:51  chris.edgington
Added comment about linux kernel issues related to uhci, ohci, and performance.

Revision 1.9  2002/01/23 17:24:48  chris.edgington
Added code to set pointer to NULL in buffer free macros.

Revision 1.8  2002/01/14 21:59:30  chris.edgington
Added GPL header.

Revision 1.7  2002/01/04 20:51:38  chris.edgington
Added checking for NULL ptr before freeing in FREE_ macros.
Added pHw member to INCOMING_BUFFER.

Revision 1.6  2002/01/02 22:05:22  chris.edgington
Removed DSPLOAD_BUFFER and UTIL_BUFFER - they're not used in Linux and they
referred to structs that were not defined yet in the include order we needed.

Revision 1.5  2001/12/28 19:33:15  chris.edgington
Added GENERIC_BUFFER as header for QUEUED_URB_BUFFER.
Removed GENERIC_BUFFER_WITH_URB.

Revision 1.4  2001/12/26 22:49:05  chris.edgington
Modified GET_KBUFFER to be interrupt safe.

Revision 1.3  2001/12/22 19:47:28  chris.edgington
Added macros for getting/freeing kernel memory and virtual memory.
Fleshed out the details of the various queued buffer types.

Revision 1.2  2001/12/20 22:36:34  chris.edgington
Changed kmalloc and kfree to vmalloc and vfree. kmalloc cannot handle requests
larger than 128K - our first alloc is 1MB.

Revision 1.1.1.1  2001/12/14 21:27:09  chris.edgington
Initial import

---------------------------------------------------------------------------
 Log entries from MacOS9 project
---------------------------------------------------------------------------

Revision 1.6  2001/12/05 22:09:55  chris.edgington
Added Analog Devices copyright notice.

Revision 1.5  2001/11/21 01:49:07  chris.edgington
Increased number of incoming, outgoing, and ethernet packet buffers from 16 to 64. This fixed the bug with PING payload sizes > 20000 and increases our performance a little on FTP download.

Revision 1.4  2001/11/12 14:23:24  chris.edgington
Added some comments and did some simple reformatting in preparation for code review.

Revision 1.3  2001/10/30 02:58:35  ilko.dossev
Addeg GET_MEMORY/FREE_MEMORY and GET_BUFFER macros.

Revision 1.2  2001/10/10 20:34:20  chris.edgington
Changed incoming buffer size to be proper multiple of bulk pipe max size and ATM cell size.
Changed outgoing buffer size to hold one more ATM cell (could be two cells of padding, if last cell contains enough data to not hold the CPCS trailer.)

Revision 1.1  2001/10/08 21:25:47  chris.edgington
First version.
*************************************************************************************************/
