/**********************************************************************************/
/* $Id: Mpoa.c,v 1.7 2003/06/03 22:42:29 sleeper Exp $ 			  */
/*										  */
/* Copyright (c) 2001, Analog Devices Inc., All Rights Reserved			  */
/*										  */
/* Mpoa.c									  */
/*										  */
/* RFC2684(1483) "Multi-Protocol Encapsulation Over ATM AAL5" support module      */
/*										  */
/* This file is part of the "ADI USB ADSL Driver for Linux".			  */
/*										  */
/* "ADI USB ADSL Driver for Linux" is free software; you can redistribute it      */
/* and/or modify it under the terms of the GNU General Public License as 	  */
/* published by the Free Software Foundation; either version 2 of the License,    */
/* or (at your option) any later version.					  */
/*										  */
/* "ADI USB ADSL Driver for Linux" is distributed in the hope that it will be     */
/* useful, but WITHOUT ANY WARRANTY; without even the implied warranty of	  */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the		  */
/* GNU General Public License for more details.					  */
/*										  */
/* You should have received a copy of the GNU General Public License		  */
/* along with "ADI USB ADSL Driver for Linux"; if not, write to the Free Software */
/* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA      */
/**********************************************************************************/

#include "Adiutil.h"
#include "Mpoa.h"
#include "Uni.h"
#include "debug.h"

/**********************************************************************************/
/* This module provides support similar to Rfc1483.c from the NDIS code. 	  */
/* Therefore, the following text from Rfc1483.c is provided, modified slightly,   */
/* as an explanation for how the various encapsulation types are handled. It      */
/* should be noted that some of the resonsibility for encapsulation is shared     */
/* by the more generic UniSendPdu routine. It checks to see if an encapsulation   */
/* header has been specified and includes it went sending the Pdu. This eliminates*/
/* an additional call for the most common operations.				  */
/*										  */
/* The MPOA module supports four PDU options:					  */
/*      1. LLC Encapsulated Ethernet						  */
/*      2. VC Encapsulated Ethernet (without FCS)				  */
/*      3. LLC Encapsulated IP 							  */
/*      4. VC Encapsulated IP							  */
/*      									  */
/*      The type of encapsulation is set during initialization, based on the	  */
/*      MpoaMode configuration option. The default mode is LLC Encapsulated 	  */
/*      Ethernet. Each of the PDU options is described briefly below.  		  */
/*      See http://www.ietf.org/rfc/rfc2684.txt?number=2684 for details.	  */
/*      (Note that OT = Open Transport.)					  */
/*										  */
/*      1. LLC Encapsulated Ethernet						  */
/*										  */
/*      Ethernet frames are accepted from OT and encapsulated as shown below. 	  */
/*      Note that the PID value of 0x00-07 is used, indicating that the LAN	  */
/*      FCS (Frame Check Sequence) is not present.				  */
/*										  */
/*               +-------------------------------+				  */
/*               |       LLC 0xAA-AA-03          |				  */
/*               +-------------------------------+				  */
/*               |       OUI 0x00-80-C2          |				  */
/*               +-------------------------------+				  */
/*               |       PID 0x00-07             |				  */
/*               +-------------------------------+				  */
/*               |       PAD 0x00-00             |				  */
/*               +-------------------------------+				  */
/*               |    MAC destination address    |				  */
/*               +-------------------------------+				  */
/*               |                               |				  */
/*               |   (remainder of MAC frame)    |				  */
/*               |                               |				  */
/*               +-------------------------------+				  */
/*										  */
/*      Once encapsulated, the RFC2684 PDUs are sent using AAL5 on the PVC 	  */
/*      provisioned during initialization.					  */
/*										  */
/*      Inbound LLC encapsulated 802.3 frames are accepted from the ATM UNI	  */
/*      module.  Note that in the inbound direction, either PID value is 	  */
/*      allowed.  If the PID value is 0x00-01 (indicating LAN FCS is present),	  */
/*      the LAN FCS is simply discarded.  The LLC encapsulation is removed and	  */
/*      the Ethernet frame is indicated to OT.					  */
/*										  */
/*      2. VC Encapsulated Ethernet						  */
/*									 	  */
/*      Same as (1) above with the exception that the LLC header is ommitted	  */
/*      (except for the PAD field).  Thus the PDUs transmitted are as shown	  */
/*      below.									  */
/*										  */
/*               +-------------------------------+				  */
/*               |       PAD 0x00-00             |				  */
/*               +-------------------------------+				  */
/*               |    MAC destination address    |				  */
/*               +-------------------------------+				  */
/*               |                               |				  */
/*               |   (remainder of MAC frame)    |				  */
/*               |                               |				  */
/*               +-------------------------------+				  */
/*										  */
/*      Note that the inclusion of the LAN FCS is not supported with VC		  */
/*      Encapsulated Ethernet.							  */
/*										  */
/*      3. LLC Encapsulated IP						  	  */
/*      									  */
/*      Ethernet frames are accepted from OT and the Ethernet header is		  */
/*      discarded up to the EtherType.  If the EtherType is not IP, the	          */
/*      frame is discarded.  Otherwise, the frame is encapsulated and sent	  */
/*      as shown below.								  */
/*										  */
/*               +-------------------------------+				  */
/*               |       LLC 0xAA-AA-03          |				  */
/*               +-------------------------------+				  */
/*               |       OUI 0x00-00-00          |				  */
/*               +-------------------------------+				  */
/*               | EtherType 0x08-00 (IP)        |				  */
/*               +-------------------------------+				  */
/*               |                               |				  */
/*               |           IP PDU              |				  */
/*               |                               |				  */
/*               +-------------------------------+				  */
/*      									  */
/*      In addition, the MPOA module will generate responses to outbound	  */
/*      Ethernet ARP requests in order to satisfy the IP stack.			  */
/*      Note:								          */
/*      I do not know if the above statement will be true on the Mac platform,    */
/*      real debugging will determine whether or not we need to do this.	  */
/*									          */
/*      4. VC Encapsulated IP							  */
/*										  */
/*      Same as (3) above except that the LLC header and Ethertype are ommitted.  */
/*      Thus the PDUs transmitted are as shown below.				  */
/*										  */
/*               +-------------------------------+				  */
/*               |                               |				  */
/*               |           IP PDU              |				  */
/*               |                               |				  */
/*               +-------------------------------+				  */
/*										  */
/**********************************************************************************/

/**********************************************************************************/
/* Local prototypes								  */
/**********************************************************************************/
static Boolean MpoaFilterReceivedPacket(Hardware *pHw, ETHERNET_MAC_HEADER *pMac);

/**********************************************************************************/
/* These static data structures are the encapsulation headers we need for Mpoa    */
/**********************************************************************************/

#define IS_LLCBRIDGEDETHNOFCS(p) (\
    (cpu_to_le32(*((UInt32 *)p))     == 0x0003AAAA) && \
    (cpu_to_le32(*((UInt32 *)(p+4))) == 0x0700C280) && \
    (cpu_to_le32(*((UInt32 *)(p+6))) == 0x00000700))

static const LLC_HEADER MpoaHdr_LlcEncapBridgedEthNoFcs =
    { {0xAA,0xAA,0x03},
      {0x00,0x80,0xC2},
      {0x00,0x07},
      {0x00,0x00}
    };

#define IS_LLCBRIDGEDETHFCS(p) (\
    (cpu_to_le32(*((UInt32 *)p))     == 0x0003AAAA) && \
    (cpu_to_le32(*((UInt32 *)(p+4))) == 0x0100C280) && \
    (cpu_to_le32(*((UInt32 *)(p+6))) == 0x00000100))

#ifdef LATER_LATER_LATER /* this isn't used right now */
static const LLC_HEADER MpoaHdr_LlcEncapBridgedEthFcs =
    { {0xAA,0xAA,0x03},
      {0x00,0x80,0xC2},
      {0x00,0x01},
      {0x00,0x00}
    };
#endif

#define IS_LLCROUTEDIP(p) (\
    (cpu_to_le32(*((UInt32 *)p))     == 0x0003AAAA) && \
    (cpu_to_le32(*((UInt32 *)(p+4))) == 0x00080000))

static const LLC_HEADER MpoaHdr_LlcEncapRoutedIp =
    { {0xAA,0xAA,0x03},
      {0x00,0x00,0x00},
      {0x08,0x00}
    };

#define IS_VCBRIDGEDETH(p) (\
    (*((UInt16 *)p)     == 0x0000))

static const UInt8 MpoaHdr_VcEncapBridgedEth[ MPOA_VC_ENCAP_BRIDGED_ETH_SIZE] =
    { 0x00,0x00 };

/**********************************************************************************/
/* Added on April 30,2002 by Anoosh to support PPPoA                                        */
static const UInt8 MpoaHdr_LlcEncapPPPoA[ MPOA_LLC_ENCAP_PPPOA_SIZE ] = { 0xFE, 0xFE, 0x03, 0xCF };

#define IS_LLCPPPOA(p) (\
    (cpu_to_le32(*((UInt32 *)p)) == 0xCF03FEFE))
/**********************************************************************************/

static const UInt8 MacBroadcast[ ETH_LENGTH_OF_ADDRESS ] =
    { 0xFF,0xFF,0xFF,0xFF,0xFF,0xFF };

#define EQUAL_MACHDR(a1,a2) (\
    (*((UInt32 *)a1) == *((UInt32 *)a2)) && \
    (*((UInt32 *)(a1+2)) == *((UInt32 *)(a2+2))))


/**********************************************************************************/
/* MpoaInitialize								  */
/* Prepares for MPOA processing by setting up the proper encapsulation		  */
/**********************************************************************************/
void MpoaInitialize( Hardware *pHw, const DriverOptions aOptions)
{
    UInt16 HdrSize=0;
    UInt8 *pHdr=NULL;
    UInt32 MpoaMode = aOptions[ConfigEncapsulation].Value;

    adi_enters (DBG_MPOA);
    

    /*Setup the encapsulation header and function pointer based on the mode specified*/
    switch (MpoaMode)
    {
       case MPOA_MODE_BRIDGED_ETH_LLC:
          adi_dbg(DBG_MPOA,"MpoaInitialize - MPOA mode set at: MPOA_MODE_BRIDGED_ETH_LLC\n");
    
          pHdr    = (UInt8 *)&MpoaHdr_LlcEncapBridgedEthNoFcs;
          HdrSize = MPOA_LLC_ENCAP_BRIDGED_ETH_SIZE;
          break;
       case MPOA_MODE_BRIDGED_ETH_VC:
          adi_dbg (DBG_MPOA,"MpoaInitialize - MPOA mode set at: MPOA_MODE_BRIDGED_ETH_VC\n");
          
          pHdr    = (UInt8 *)&MpoaHdr_VcEncapBridgedEth;
          HdrSize = MPOA_VC_ENCAP_BRIDGED_ETH_SIZE;
          break;
       case MPOA_MODE_ROUTED_IP_LLC:
          adi_dbg (DBG_MPOA,"MpoaInitialize - MPOA mode set at: MPOA_MODE_ROUTED_IP_LLC\n");
          
          pHdr    = (UInt8 *)&MpoaHdr_LlcEncapRoutedIp;
          HdrSize = MPOA_LLC_ENCAP_ROUTED_IP_SIZE;
          break;
       case MPOA_MODE_ROUTED_IP_VC:
          adi_dbg (DBG_MPOA,"MpoaInitialize - MPOA mode set at: MPOA_MODE_ROUTED_IP_VC\n");
          
          /* Since there is no ethernet header and the encapsulation is already*/
          /* a part of the VC by definition, no additional header info needs   */
          /* stuffed into packets when in this mode			       */
          pHdr    = NULL;
          HdrSize = 0;
          break;
       /**************************************************************************/
       /*The following two cases added on April30,2002 by Anoosh to support PPPoA*/
       case MPOA_MODE_PPPOA_LLC:
          adi_dbg (DBG_MPOA,"MpoaInitialize - MPOA mode set at: MPOA_MODE_PPPOA_LLC\n");
          
          pHdr    = (UInt8 *)&MpoaHdr_LlcEncapPPPoA;
          HdrSize = MPOA_LLC_ENCAP_PPPOA_SIZE;
          break;
       case MPOA_MODE_PPPOA_VC:
          adi_dbg (DBG_MPOA,"MpoaInitialize - MPOA mode set at: MPOA_MODE_PPPOA_VC\n");
          
          pHdr    = NULL;
          HdrSize = 0;
          break;
       /**************************************************************************/
    }

    /**********************************************************************************/
    /* The NDIS code has a distinction between "creating" and "activating" the        */
    /* virtual channel. That code is only necessary when dealing with higher-level    */
    /* software that understands and manages virtual channels (like NDIS 5). 	      */
    /* Since this is the Mac ethernet driver, we know that nobody above us is going   */
    /* to talk ATM, so we combined both functions into one, UniEstablishVc.           */
    /**********************************************************************************/
    UniEstablishVc(pHw, pHdr, HdrSize, aOptions[ConfigVPI].Value, aOptions[ConfigVCI].Value);

    pHw->MpoaMode = MpoaMode;

    /*
     * We also need to set the reassembly siwe that will be used. By default it is
     * set to the size of the pReassemblyBuffer, which is 1536, but as this size is
     * used to control that the received PDU is not to big, we should set it up
     * accordingly to the encapsulation type
     */
    switch (MpoaMode)
    {
        case MPOA_MODE_BRIDGED_ETH_LLC:
        case MPOA_MODE_BRIDGED_ETH_VC:
            /* PDU is a full ethernet frame ..
             */
            pHw->mru = 1514;
            break;
            
        case MPOA_MODE_ROUTED_IP_LLC:
        case MPOA_MODE_ROUTED_IP_VC:
            /* Do not change it ... set to 1536 */
            pHw->mru = 1536;
            break;
            
        case MPOA_MODE_PPPOA_LLC:
        case MPOA_MODE_PPPOA_VC:
            /*
             * PPPoA packet should hold in an thernet frame. Thus it is limited
             * to 1500 bytes. As we'll add 2 bytes for packet siwe following
             * ethernet header, the payload size is 1498.
             */
            pHw->mru = 1498;
            break;
            
        default:
            adi_warn( "Unknown encapsulation type 0x%x\n",MpoaMode);
            break;
    }
    
    adi_report ("Reassembly buffer size set to %u\n",
                pHw->ReassemblyBuffer.GB.AllocedSize);
            
    
    adi_leaves (DBG_MPOA);
    
}

/**********************************************************************************/
/* MpoaProcessReceivedPdu							  */
/*										  */
/* This routine will be called by the receive data routine by the "user"	  */
/* of this driver. I thought it was best to delay as much processing so		  */
/* we could exit the data receiver handler as soon as possible			  */
/**********************************************************************************/
void MpoaProcessReceivedPdu(Hardware *pHw, ETHERNET_PACKET_BUFFER *pBuf)
{
    ETHERNET_MAC_HEADER *pMac;
    
    adi_dbg (DBG_MPOA,"MpoaProcessReceivedPdu\n");
    
    
    switch (pHw->MpoaMode)
    {
    case MPOA_MODE_BRIDGED_ETH_LLC:
	if (IS_LLCBRIDGEDETHNOFCS(pBuf->GB.pData))
	{
	    /* We've got an LLC encapsulated ethernet packet without FCS,*/
	    /* since our "user" is ethernet, skip over the LLC header    */
	    pBuf->EncapSkipSize = MPOA_LLC_ENCAP_BRIDGED_ETH_SIZE;
	}
	else
	{
	    if (IS_LLCBRIDGEDETHFCS(pBuf->GB.pData))
	    {
		/* We've got an LLC encapsulated ethernet packet with FCS,*/
		/* since our "user" is ethernet, skip over the LLC header */
		pBuf->EncapSkipSize = MPOA_LLC_ENCAP_BRIDGED_ETH_SIZE;
	    }
	    else
	    {
                adi_dbg (DBG_MPOA,"MpoaProcessReceivedPdu: invalid LLC header\n");                
		pBuf->DropThisPacket = TRUE;
	    }
	} 
	break;
    case MPOA_MODE_BRIDGED_ETH_VC:
	if (IS_VCBRIDGEDETH(pBuf->GB.pData))
	{
	    /* We've got a VC encapsulated ethernet packet,	     */	
	    /* since our "user" is ethernet, skip over the VC header*/
	    pBuf->EncapSkipSize = MPOA_VC_ENCAP_BRIDGED_ETH_SIZE;
	}
	else
	{
	    /* The header for this Pdu did not match our stored VC header,*/
	    /* so we cannot with good faith accept this Pdu		   */
            adi_dbg (DBG_MPOA,"MpoaProcessReceivedPdu: invalid VC header\n");
            
	    pBuf->DropThisPacket = TRUE;
	}
	break;
    case MPOA_MODE_ROUTED_IP_LLC:
	if(IS_LLCROUTEDIP(pBuf->GB.pData))
	{ 
	    pBuf->EncapSkipSize =  MPOA_LLC_ENCAP_ROUTED_IP_SIZE ;
	}
	else
	{
            adi_dbg (DBG_MPOA,"MpoaProcessReceivedPdu: invalid LLC header\n");
            
	    pBuf->DropThisPacket = TRUE;
	}
	break;
    case MPOA_MODE_ROUTED_IP_VC:
	pBuf->EncapSkipSize =  0;
	break;
    /****************************************************************************/
    /* The following two cases added on April30,2002 by Anoosh to support PPPoA */
    case MPOA_MODE_PPPOA_LLC:
	if (IS_LLCPPPOA(pBuf->GB.pData))
	{
	    /* We've got an LLC encapsulated PPPoA packet,      */
	    /* since our "user" is PPP, skip over the LLC header*/
	    pBuf->EncapSkipSize = MPOA_LLC_ENCAP_PPPOA_SIZE;
	}
	else
	{
	    /* We're in LLC PPPOA mode, but this packet does not have an LLC*/
	    /* header, so we cannot with good faith accept this Pdu         */
            adi_dbg (DBG_MPOA,"MpoaProcessReceivedPdu: invalid LLC header\n");
            
	    pBuf->DropThisPacket = TRUE;
	}
	break;
    case MPOA_MODE_PPPOA_VC:
	/* In PPPOA_VC mode, there is no MPOA header, so we don't do skip anything*/
	pBuf->EncapSkipSize = 0;
	break;
    /****************************************************************************/
    default:
        adi_err ("MpoaProcessReceivedPdu: unsupported MpoaMode\n");        
	pBuf->DropThisPacket = TRUE;
	break;
    }
    
    /**************************************************************************/
    /* This comment has been added on April 30,2002 by Anoosh to support PPPoA*/
    /* We do not do packet filtering in any of the PPP modes, therefore I need*/
    /* to add a flag to see if this isPPP then do not do packet filtering like*/
    /* MAC driver in Mpoa.c line 364                                          */
    /**************************************************************************/
    
    /****************************************************************************/
    /* We've determined any header size that we need to skip and verified that  */
    /* we have a valid header. If we determined that this Pdu should be dropped,*/
    /* we shouldn't do any more processing					*/
    /****************************************************************************/
    if (!pBuf->DropThisPacket)
    {
	pMac = (ETHERNET_MAC_HEADER *)(pBuf->GB.pData + pBuf->EncapSkipSize);
	
	/**************************************************************/
	/* Comment and added by Anoosh on April30,2002 to support     */
	/* PPPoA we do not do packet filtering in any of the PPP modes*/
	switch (pHw->MpoaMode)
	{
	case MPOA_MODE_ROUTED_IP_LLC: 
	case MPOA_MODE_ROUTED_IP_VC:
	case MPOA_MODE_PPPOA_LLC:
	case MPOA_MODE_PPPOA_VC:
	    /*****************************************************************************/
	    /*Comment by Anoosh on April30,2002 to support PPPoA                         */
	    /*This is when we are in PPPoA mode and we do not have any packet filtering  */
	    /*We do not do anything here. MAC driver do some stuff for PPPoE to make     */
	    /*sure about the type of ethernet packet and session ID and if that is       */
	    /*correct then it take out the extra header for PPPoE and passes to PPP stack*/
	    /*****************************************************************************/
	    break;
	default:
	    /************************************************************************/
	    /*Comment by Anoosh on April30,2002 to support PPPoA                    */
	    /*This is the normal situation for LAN driver that we filter that packet*/
	    /************************************************************************/
	    pBuf->DropThisPacket = MpoaFilterReceivedPacket(pHw, pMac);
	    if (pBuf->DropThisPacket)
		pHw->Statistics[STAT_PAKTS_FILTERED]++;
	    break;
	}
	/**************************************************************/
    }
    
    adi_leaves (DBG_MPOA);
    
}

/**********************************************************************************/
/* MpoaFilterReceivedPacket							  */
/*										  */
/* Checks the MAC header relative to our current filter mode. Returns		  */
/* TRUE if this packet should be filtered (not processed).			  */
/**********************************************************************************/
static Boolean MpoaFilterReceivedPacket(Hardware *pHw, ETHERNET_MAC_HEADER *pMac)
{
    struct net_device *ether = pHw->pLinuxNet;
    Boolean FilterIt = FALSE;
    
    adi_enters (DBG_MPOA);
    
    

    /* If we're in promiscuous mode - then all packets go through!*/
    if (ether->flags & IFF_PROMISC)
    {
        adi_dbg (DBG_MPOA,"IFF_PROMISC set.\n");
        
	FilterIt = FALSE;
    }
    else
    {
	/* Lets determine if this is a Directed, Broadcast, or Multicast packet*/
	if (pMac->DstAddress[0] & 0x01)
	{
	    /* This is either a Broadcast or Multicast packet*/
	    if (EQUAL_MACHDR(pMac->DstAddress, MacBroadcast))
	    {
		/* This is a broadcast packet*/
                adi_dbg (DBG_MPOA,"Broadcast packet.\n");                
		FilterIt = FALSE;
	    }
	    else
	    {
		/* This is a multicast packet*/
		if (ether->flags & IFF_ALLMULTI)
		{
                    adi_dbg (DBG_MPOA,"IFF_ALLMULTI set.\n");
                    
		    FilterIt = FALSE;
		}
		else
		{
                    adi_dbg (DBG_MPOA,"Multicast packet.\n");
                    
		    /* Now we have to compare the DstAddress with our list*/
		    /* of acceptable multicast addresses                  */
		    if (ether->mc_count > 0)
		    {
			int i;
			struct dev_mc_list *mc;
			
			/* Start out by assuming that we need to filter this*/
			/* packet, since we will encounter a match only when*/
			/* this is a valid packet to go through (not filter)*/
			FilterIt = TRUE;
			
			/* This loop works because z will evaluate to 0 or NULL*/
			/* when z->next is NULL, which will stop the loop      */
			for (i = 0, mc = ether->mc_list;
			    mc && i < ether->mc_count;
			    i++, mc = mc->next) 
			{
			    if (EQUAL_MACHDR(pMac->DstAddress, mc->dmi_addr))
			    {
				FilterIt = FALSE;
				break; /* We found it - get out of the loop*/
			    }
			}
		    }
		    else
		    {
			FilterIt = TRUE;
		    }
		}
	    }
	}
	else
	{
	    /* This is a directed packet*/
	    if (EQUAL_MACHDR(pMac->DstAddress, pHw->MAC))
	    {
                adi_dbg (DBG_MPOA,"Directed packet match.\n");
                
		/* This packet is directed to us*/
		FilterIt = FALSE;
	    }
	    else
	    {
                adi_dbg (DBG_MPOA,"Directed packet not a match.\n");
                
		/* This packet is not directed to us */
		FilterIt = TRUE;
	    }
	}
    }

    adi_leaves (DBG_MPOA);
    
    return FilterIt;
}

// FFD120602 : (BUG?) Added for routed mode. It seems that there is a problem with 
// indicating the linux system that the send is complete. I had to do some calls after
// ARP call to make it working for now. Should be inverstigated later.
//---------------------------------------------------------------------------
// EthernetArpReply
//
// Simulates an ARP reply, used when in routed IP mode. Note, the logic
// for this code was taken from the Windows NDIS code.
//---------------------------------------------------------------------------

void EthernetArpReply(Hardware *pHw, UInt8 *pBuf, UInt32 PacketSize)
{
    ETHERNET_MAC_HEADER *pRequestHdr;
    ETHERNET_MAC_HEADER *pReplyHdr;
    ARP_PACKET *pArp;
    UInt32 SrcIP, DstIP;

    adi_enters (DBG_ROUTEIP);
    

    pRequestHdr = (ETHERNET_MAC_HEADER *)pBuf;
    pReplyHdr   = (ETHERNET_MAC_HEADER *)pHw->pPacket;

    // First, copy the original ARP request into our reply packet
    memcpy(pHw->pPacket, pBuf, PacketSize);

    pArp  = (ARP_PACKET *)(pHw->pPacket + sizeof(ETHERNET_MAC_HEADER));
    SrcIP = *(UInt32 *)(pArp->Spa);
    DstIP = *(UInt32 *)(pArp->Tpa);

    adi_dbg (DBG_ROUTEIP,"ARP Src IP: ");
    adi_dbg (DBG_ROUTEIP,"%d.%d.%d.%d", pArp->Spa[0], pArp->Spa[1],  pArp->Spa[2],  pArp->Spa[3]);
    adi_dbg (DBG_ROUTEIP," Dst IP: ");
    adi_dbg (DBG_ROUTEIP,"%d.%d.%d.%d", pArp->Tpa[0], pArp->Tpa[1],  pArp->Tpa[2],  pArp->Tpa[3]);
    adi_dbg (DBG_ROUTEIP,"\nARP IPs, UInt32 versions - Src: %x, Dst: %x\n", SrcIP, DstIP);
    
        // If this is a "gratuitous ARP request", don't reply
        if ((SrcIP == DstIP) || (SrcIP == 0x0000))
        {
            adi_dbg (DBG_ROUTEIP,"Gratuitous ARP, no reply generated.\n");
        
            goto EthArpReplyExit;
        }

    // Build our ARP reply
    pArp->Op = ARP_OP_REPLY;
    // Swap source and dest IP addresses
    *(UInt32 *)pArp->Spa = DstIP;
    *(UInt32 *)pArp->Tpa = SrcIP;
    // Set the dest hardware address to the original source address
    memcpy(pArp->Tha, pArp->Sha, MAC_SIZE);
    // Set source hardware address to the original dest IP address
    *(UInt32 *)(pArp->Sha+2) = DstIP;
    // Do the same in the ethernet header
    memcpy(pReplyHdr->DstAddress, pReplyHdr->SrcAddress, MAC_SIZE);
    *(UInt32 *)(pReplyHdr->SrcAddress+2) = DstIP;
    
    // The packet is ready ... set the flag and tell OT 
    pHw->FakePacketSize  = PacketSize;
    pHw->FakePacketReady = TRUE;


    {	
        struct sk_buff *skb;
        if (!(skb = dev_alloc_skb(ETH_DATA_LEN+14+2)))
        {
            adi_dbg (DBG_ROUTEIP,"Error allocating skb!\n");
            
            goto EthArpReplyExit;
        }
        skb->dev = pHw->pLinuxNet;
        adi_dbg (DBG_ROUTEIP,"EthernetArpReply: ROUTED : CALLING eth_copy_and_sum\n");
                

        eth_copy_and_sum(skb, pHw->pPacket, PacketSize + 2 , 0);

        skb_put(skb, PacketSize + 2);
        skb->protocol = eth_type_trans(skb, pHw->pLinuxNet);
        netif_rx(skb);

    }  

  EthArpReplyExit:
    adi_leaves (DBG_ROUTEIP);
    
}


/*********************************************************************************
$Log: Mpoa.c,v $
Revision 1.7  2003/06/03 22:42:29  sleeper
Changed ZAPS to adi_dbg/err macros

Revision 1.6  2003/04/09 19:22:27  sleeper
Fix a bug in the size of mru for routed ip

Revision 1.5  2003/04/08 23:10:53  sleeper
Add mru

Revision 1.4  2003/04/05 20:01:56  sleeper
Fix a bug in the reassembly size for Routed IP encapsulation

Revision 1.3  2003/04/24 00:06:10  sleeper
Fix reassembly buffer siwe based on encapsulation

Revision 1.2  2003/03/31 23:37:21  sleeper
Add EthernetArpReply function

Revision 1.1.1.1  2003/02/10 23:29:49  sleeper
Imported sources

Revision 1.7  2002/01/14 21:58:21  chris.edgington
Initialize local vars to get rid of compiler warning.
Added GPL header.

Revision 1.6  2002/01/08 20:21:22  chris.edgington
Added more extensive debug output to filtering code.

Revision 1.5  2002/01/08 20:09:42  chris.edgington
Removed an unused local var.

Revision 1.4  2002/01/08 16:25:42  chris.edgington
Fixed simple compile error.

Revision 1.3  2002/01/08 15:54:21  chris.edgington
Removed non-Linux specific versions of header compare macros.
Fleshed out details of packet filtering code for Linux.
Removed MpoaSetPacketFilter and MpoaSetMulticastFilters - not needed in Linux.

Revision 1.2  2002/01/04 20:56:19  chris.edgington
Created little-endian versions of IS_* quick-compare macros for encapsulation.

Revision 1.1  2002/01/02 21:56:31  chris.edgington
First version - from MacOS9 project (with Linux mods to get to compile).


---------------------------------------------------------------------------
 Log from MacOS9 project
---------------------------------------------------------------------------
Revision 2.00  2002/05/27 22:09:55  Anoosh Naderi
Added support for PPPoA
Clean up the code

Revision 1.18  2001/12/05 22:09:55  chris.edgington
Added Analog Devices copyright notice.

Revision 1.17  2001/11/12 14:23:24  chris.edgington
Added some comments and did some simple reformatting in preparation for code review.

Revision 1.16  2001/10/30 19:50:19  chris.edgington
Removed DANGERs that were no longer necessary.

Revision 1.15  2001/10/30 02:54:39  ilko.dossev
Pool memory fxns used for memory manipulation.
There is no way to tell if they are invoked on USB callback.

Revision 1.14  2001/10/26 19:52:16  ilko.dossev
Removed a redundant <PoolDeallocate> call.

Revision 1.13  2001/10/25 19:05:47  chris.edgington
Added counter for packets filtered by Ethernet filter.

Revision 1.12  2001/10/19 20:24:58  chris.edgington
Added macros for fast compare of LLC headers and MAC addresses.
Replaced calls to CompareMemory to use fast compare macros.

Revision 1.11  2001/10/17 22:25:13  chris.edgington
Added note about an optimization opportunity.

Revision 1.10  2001/10/11 19:45:24  chris.edgington
Wrote MpoaFilterReceivedPacket, MpoaSetPacketFilter, and MpoaSetMulticastFilters.
Fixed memory compares to check for TRUE instead of 0!

Revision 1.9  2001/10/09 15:08:50  chris.edgington
Updated to reflect new buffer field names.
Modified function paramters to reflect removal of Uni.
Moved header stuff into .h files.

Revision 1.8  2001/10/05 16:04:41  chris.edgington
Added setting of Uni.MpoaMode.
Added more debug output.

Revision 1.7  2001/10/01 18:43:25  chris.edgington
Pass address of ethernet header structs to COMP_MEM, not the structures themselves.

Revision 1.6  2001/09/28 15:19:52  chris.edgington
Moved encapsulation mode defines to Atm.h.

Revision 1.5  2001/09/24 18:05:38  chris.edgington
Removed include of multiple files, added single include of Adiutil.h.

Revision 1.4  2001/09/10 18:18:07  chris.edgington
Fixed compile errors.
Wrote MpoaProcessReceivedPdu.

Revision 1.3  2001/09/05 22:25:48  chris.edgington
Work in progress ...

Revision 1.2  2001/09/04 22:29:58  chris.edgington
Wrote MpoaInitialize and MpoaSend.

Revision 1.1  2001/09/04 21:44:20  chris.edgington
Initial version, no real functioning code, just some snippets and some documentation.
***************************************************************************************/
